;; Copyright (c) Cognitect, Inc.
;; All rights reserved.

(ns cognitect.aws.detective.specs
  (:require [clojure.spec.alpha :as s] [clojure.spec.gen.alpha :as gen]))

(s/def :cognitect.aws/client map?)

(s/def :core.async/channel any?)

(s/def
  :cognitect.aws.detective/Severity
  (s/spec string? :gen #(s/gen #{"INFORMATIONAL" "CRITICAL" "LOW" "HIGH" "MEDIUM"})))

(s/def
  :cognitect.aws.detective/TagResourceRequest
  (s/keys
    :req-un
    [:cognitect.aws.detective.TagResourceRequest/ResourceArn
     :cognitect.aws.detective.TagResourceRequest/Tags]))

(s/def
  :cognitect.aws.detective/ErrorCode
  (s/spec string? :gen #(s/gen #{"INVALID_GRAPH_ARN" "INVALID_REQUEST_BODY" "INTERNAL_ERROR"})))

(s/def
  :cognitect.aws.detective/UpdateDatasourcePackagesRequest
  (s/keys
    :req-un
    [:cognitect.aws.detective.UpdateDatasourcePackagesRequest/GraphArn
     :cognitect.aws.detective.UpdateDatasourcePackagesRequest/DatasourcePackages]))

(s/def
  :cognitect.aws.detective/ServiceQuotaExceededException
  (s/keys
    :opt-un
    [:cognitect.aws.detective.ServiceQuotaExceededException/Message
     :cognitect.aws.detective.ServiceQuotaExceededException/Resources]))

(s/def
  :cognitect.aws.detective/InternalServerException
  (s/keys :opt-un [:cognitect.aws.detective.InternalServerException/Message]))

(s/def :cognitect.aws.detective/Aso string?)

(s/def
  :cognitect.aws.detective/FilterCriteria
  (s/keys
    :opt-un
    [:cognitect.aws.detective.FilterCriteria/Severity
     :cognitect.aws.detective.FilterCriteria/Status
     :cognitect.aws.detective.FilterCriteria/CreatedTime
     :cognitect.aws.detective.FilterCriteria/EntityArn
     :cognitect.aws.detective.FilterCriteria/State]))

(s/def
  :cognitect.aws.detective/TimestampForCollection
  (s/keys :opt-un [:cognitect.aws.detective.TimestampForCollection/Timestamp]))

(s/def :cognitect.aws.detective/MemberDetailList (s/coll-of :cognitect.aws.detective/MemberDetail))

(s/def
  :cognitect.aws.detective/TagMap
  (s/map-of
    :cognitect.aws.detective/TagKey
    :cognitect.aws.detective/TagValue
    :min-count
    1
    :max-count
    50))

(s/def
  :cognitect.aws.detective/BatchGetGraphMemberDatasourcesResponse
  (s/keys
    :opt-un
    [:cognitect.aws.detective.BatchGetGraphMemberDatasourcesResponse/MemberDatasources
     :cognitect.aws.detective.BatchGetGraphMemberDatasourcesResponse/UnprocessedAccounts]))

(s/def
  :cognitect.aws.detective/ListGraphsResponse
  (s/keys
    :opt-un
    [:cognitect.aws.detective.ListGraphsResponse/NextToken
     :cognitect.aws.detective.ListGraphsResponse/GraphList]))

(s/def
  :cognitect.aws.detective/ListMembersResponse
  (s/keys
    :opt-un
    [:cognitect.aws.detective.ListMembersResponse/NextToken
     :cognitect.aws.detective.ListMembersResponse/MemberDetails]))

(s/def :cognitect.aws.detective/TagResourceResponse (s/keys))

(s/def
  :cognitect.aws.detective/ImpossibleTravelDetail
  (s/keys
    :opt-un
    [:cognitect.aws.detective.ImpossibleTravelDetail/EndingLocation
     :cognitect.aws.detective.ImpossibleTravelDetail/EndingIpAddress
     :cognitect.aws.detective.ImpossibleTravelDetail/StartingLocation
     :cognitect.aws.detective.ImpossibleTravelDetail/HourlyTimeDelta
     :cognitect.aws.detective.ImpossibleTravelDetail/StartingIpAddress]))

(s/def
  :cognitect.aws.detective/LastIngestStateChangeDates
  (s/map-of
    :cognitect.aws.detective/DatasourcePackageIngestState
    :cognitect.aws.detective/TimestampForCollection))

(s/def :cognitect.aws.detective/IpAddress string?)

(s/def
  :cognitect.aws.detective/EnableOrganizationAdminAccountRequest
  (s/keys :req-un [:cognitect.aws.detective.EnableOrganizationAdminAccountRequest/AccountId]))

(s/def
  :cognitect.aws.detective/ListGraphsRequest
  (s/keys
    :opt-un
    [:cognitect.aws.detective.ListGraphsRequest/NextToken
     :cognitect.aws.detective.ListGraphsRequest/MaxResults]))

(s/def
  :cognitect.aws.detective/StartInvestigationRequest
  (s/keys
    :req-un
    [:cognitect.aws.detective.StartInvestigationRequest/GraphArn
     :cognitect.aws.detective.StartInvestigationRequest/EntityArn
     :cognitect.aws.detective.StartInvestigationRequest/ScopeStartTime
     :cognitect.aws.detective.StartInvestigationRequest/ScopeEndTime]))

(s/def
  :cognitect.aws.detective/ValidationException
  (s/keys
    :opt-un
    [:cognitect.aws.detective.ValidationException/ErrorCodeReason
     :cognitect.aws.detective.ValidationException/ErrorCode
     :cognitect.aws.detective.ValidationException/Message]))

(s/def
  :cognitect.aws.detective/AccessDeniedException
  (s/keys
    :opt-un
    [:cognitect.aws.detective.AccessDeniedException/SubErrorCodeReason
     :cognitect.aws.detective.AccessDeniedException/ErrorCodeReason
     :cognitect.aws.detective.AccessDeniedException/SubErrorCode
     :cognitect.aws.detective.AccessDeniedException/ErrorCode
     :cognitect.aws.detective.AccessDeniedException/Message]))

(s/def
  :cognitect.aws.detective/APISuccessCount
  (s/spec int? :gen #(gen/choose Long/MIN_VALUE Long/MAX_VALUE)))

(s/def
  :cognitect.aws.detective/MembershipDatasourcesList
  (s/coll-of :cognitect.aws.detective/MembershipDatasources))

(s/def
  :cognitect.aws.detective/UnprocessedAccount
  (s/keys
    :opt-un
    [:cognitect.aws.detective.UnprocessedAccount/AccountId
     :cognitect.aws.detective.UnprocessedAccount/Reason]))

(s/def :cognitect.aws.detective/ErrorMessage string?)

(s/def
  :cognitect.aws.detective/AccountId
  (s/spec #(re-matches (re-pattern "^[0-9]+$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.detective/ListMembersRequest
  (s/keys
    :req-un
    [:cognitect.aws.detective.ListMembersRequest/GraphArn]
    :opt-un
    [:cognitect.aws.detective.ListMembersRequest/NextToken
     :cognitect.aws.detective.ListMembersRequest/MaxResults]))

(s/def
  :cognitect.aws.detective/MemberDisabledReason
  (s/spec string? :gen #(s/gen #{"VOLUME_UNKNOWN" "VOLUME_TOO_HIGH"})))

(s/def
  :cognitect.aws.detective/UnprocessedGraph
  (s/keys
    :opt-un
    [:cognitect.aws.detective.UnprocessedGraph/GraphArn
     :cognitect.aws.detective.UnprocessedGraph/Reason]))

(s/def :cognitect.aws.detective/Id string?)

(s/def
  :cognitect.aws.detective/VolumeUsageByDatasourcePackage
  (s/map-of
    :cognitect.aws.detective/DatasourcePackage
    :cognitect.aws.detective/DatasourcePackageUsageInfo))

(s/def
  :cognitect.aws.detective/DescribeOrganizationConfigurationResponse
  (s/keys :opt-un [:cognitect.aws.detective.DescribeOrganizationConfigurationResponse/AutoEnable]))

(s/def
  :cognitect.aws.detective/DescribeOrganizationConfigurationRequest
  (s/keys :req-un [:cognitect.aws.detective.DescribeOrganizationConfigurationRequest/GraphArn]))

(s/def
  :cognitect.aws.detective/UpdateOrganizationConfigurationRequest
  (s/keys
    :req-un
    [:cognitect.aws.detective.UpdateOrganizationConfigurationRequest/GraphArn]
    :opt-un
    [:cognitect.aws.detective.UpdateOrganizationConfigurationRequest/AutoEnable]))

(s/def
  :cognitect.aws.detective/Field
  (s/spec string? :gen #(s/gen #{"CREATED_TIME" "STATUS" "SEVERITY"})))

(s/def
  :cognitect.aws.detective/EntityArn
  (s/spec #(re-matches (re-pattern "^arn:.*") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.detective/ListDatasourcePackagesRequest
  (s/keys
    :req-un
    [:cognitect.aws.detective.ListDatasourcePackagesRequest/GraphArn]
    :opt-un
    [:cognitect.aws.detective.ListDatasourcePackagesRequest/NextToken
     :cognitect.aws.detective.ListDatasourcePackagesRequest/MaxResults]))

(s/def
  :cognitect.aws.detective/AiPaginationToken
  (s/spec
    (s/and string? #(<= 1 (count %) 2048))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 1 2048) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.detective/EmailAddress
  (s/spec
    #(re-matches
      (re-pattern "^.+@(?:(?:(?!-)[A-Za-z0-9-]{1,62})?[A-Za-z0-9]{1}\\.)+[A-Za-z]{2,63}$")
      %)
    :gen
    #(gen/string)))

(s/def
  :cognitect.aws.detective/FlaggedIpAddressDetail
  (s/keys
    :opt-un
    [:cognitect.aws.detective.FlaggedIpAddressDetail/IpAddress
     :cognitect.aws.detective.FlaggedIpAddressDetail/Reason]))

(s/def
  :cognitect.aws.detective/GetMembersRequest
  (s/keys
    :req-un
    [:cognitect.aws.detective.GetMembersRequest/GraphArn
     :cognitect.aws.detective.GetMembersRequest/AccountIds]))

(s/def
  :cognitect.aws.detective/CreateGraphResponse
  (s/keys :opt-un [:cognitect.aws.detective.CreateGraphResponse/GraphArn]))

(s/def :cognitect.aws.detective/UnprocessedReason string?)

(s/def :cognitect.aws.detective/EntityType (s/spec string? :gen #(s/gen #{"IAM_USER" "IAM_ROLE"})))

(s/def
  :cognitect.aws.detective/DatasourcePackageIngestDetail
  (s/keys
    :opt-un
    [:cognitect.aws.detective.DatasourcePackageIngestDetail/DatasourcePackageIngestState
     :cognitect.aws.detective.DatasourcePackageIngestDetail/LastIngestStateChange]))

(s/def
  :cognitect.aws.detective/CreateMembersRequest
  (s/keys
    :req-un
    [:cognitect.aws.detective.CreateMembersRequest/GraphArn
     :cognitect.aws.detective.CreateMembersRequest/Accounts]
    :opt-un
    [:cognitect.aws.detective.CreateMembersRequest/DisableEmailNotification
     :cognitect.aws.detective.CreateMembersRequest/Message]))

(s/def :cognitect.aws.detective/Reason (s/spec string? :gen #(s/gen #{"AWS_THREAT_INTELLIGENCE"})))

(s/def
  :cognitect.aws.detective/MemberResultsLimit
  (s/spec (s/and int? #(<= 1 % 200)) :gen #(gen/choose 1 200)))

(s/def :cognitect.aws.detective/Boolean boolean?)

(s/def
  :cognitect.aws.detective/CreateGraphRequest
  (s/keys :opt-un [:cognitect.aws.detective.CreateGraphRequest/Tags]))

(s/def :cognitect.aws.detective/Tactic string?)

(s/def
  :cognitect.aws.detective/ListInvitationsResponse
  (s/keys
    :opt-un
    [:cognitect.aws.detective.ListInvitationsResponse/NextToken
     :cognitect.aws.detective.ListInvitationsResponse/Invitations]))

(s/def
  :cognitect.aws.detective/ListOrganizationAdminAccountsResponse
  (s/keys
    :opt-un
    [:cognitect.aws.detective.ListOrganizationAdminAccountsResponse/Administrators
     :cognitect.aws.detective.ListOrganizationAdminAccountsResponse/NextToken]))

(s/def
  :cognitect.aws.detective/GetInvestigationResponse
  (s/keys
    :opt-un
    [:cognitect.aws.detective.GetInvestigationResponse/Severity
     :cognitect.aws.detective.GetInvestigationResponse/InvestigationId
     :cognitect.aws.detective.GetInvestigationResponse/GraphArn
     :cognitect.aws.detective.GetInvestigationResponse/Status
     :cognitect.aws.detective.GetInvestigationResponse/CreatedTime
     :cognitect.aws.detective.GetInvestigationResponse/EntityArn
     :cognitect.aws.detective.GetInvestigationResponse/State
     :cognitect.aws.detective.GetInvestigationResponse/ScopeStartTime
     :cognitect.aws.detective.GetInvestigationResponse/ScopeEndTime
     :cognitect.aws.detective.GetInvestigationResponse/EntityType]))

(s/def
  :cognitect.aws.detective/InvestigationDetails
  (s/coll-of :cognitect.aws.detective/InvestigationDetail))

(s/def
  :cognitect.aws.detective/ListIndicatorsRequest
  (s/keys
    :req-un
    [:cognitect.aws.detective.ListIndicatorsRequest/GraphArn
     :cognitect.aws.detective.ListIndicatorsRequest/InvestigationId]
    :opt-un
    [:cognitect.aws.detective.ListIndicatorsRequest/NextToken
     :cognitect.aws.detective.ListIndicatorsRequest/IndicatorType
     :cognitect.aws.detective.ListIndicatorsRequest/MaxResults]))

(s/def
  :cognitect.aws.detective/DatasourcePackageIngestDetails
  (s/map-of
    :cognitect.aws.detective/DatasourcePackage
    :cognitect.aws.detective/DatasourcePackageIngestDetail))

(s/def
  :cognitect.aws.detective/GraphArn
  (s/spec
    #(re-matches
      (re-pattern "^arn:aws[-\\w]{0,10}?:detective:[-\\w]{2,20}?:\\d{12}?:graph:[abcdef\\d]{32}?$")
      %)
    :gen
    #(gen/string)))

(s/def
  :cognitect.aws.detective/AccountList
  (s/coll-of :cognitect.aws.detective/Account :min-count 1 :max-count 50))

(s/def
  :cognitect.aws.detective/StartMonitoringMemberRequest
  (s/keys
    :req-un
    [:cognitect.aws.detective.StartMonitoringMemberRequest/GraphArn
     :cognitect.aws.detective.StartMonitoringMemberRequest/AccountId]))

(s/def
  :cognitect.aws.detective/UpdateInvestigationStateRequest
  (s/keys
    :req-un
    [:cognitect.aws.detective.UpdateInvestigationStateRequest/GraphArn
     :cognitect.aws.detective.UpdateInvestigationStateRequest/InvestigationId
     :cognitect.aws.detective.UpdateInvestigationStateRequest/State]))

(s/def
  :cognitect.aws.detective/RelatedFindingGroupDetail
  (s/keys :opt-un [:cognitect.aws.detective.RelatedFindingGroupDetail/Id]))

(s/def
  :cognitect.aws.detective/DateFilter
  (s/keys
    :req-un
    [:cognitect.aws.detective.DateFilter/StartInclusive
     :cognitect.aws.detective.DateFilter/EndInclusive]))

(s/def
  :cognitect.aws.detective/Administrator
  (s/keys
    :opt-un
    [:cognitect.aws.detective.Administrator/GraphArn
     :cognitect.aws.detective.Administrator/AccountId
     :cognitect.aws.detective.Administrator/DelegationTime]))

(s/def
  :cognitect.aws.detective/InvitationType
  (s/spec string? :gen #(s/gen #{"ORGANIZATION" "INVITATION"})))

(s/def
  :cognitect.aws.detective/DeleteMembersRequest
  (s/keys
    :req-un
    [:cognitect.aws.detective.DeleteMembersRequest/GraphArn
     :cognitect.aws.detective.DeleteMembersRequest/AccountIds]))

(s/def
  :cognitect.aws.detective/TagValue
  (s/spec
    (s/and string? #(>= 256 (count %)))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 0 256) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.detective/CreateMembersResponse
  (s/keys
    :opt-un
    [:cognitect.aws.detective.CreateMembersResponse/Members
     :cognitect.aws.detective.CreateMembersResponse/UnprocessedAccounts]))

(s/def
  :cognitect.aws.detective/DisassociateMembershipRequest
  (s/keys :req-un [:cognitect.aws.detective.DisassociateMembershipRequest/GraphArn]))

(s/def
  :cognitect.aws.detective/AccountIdExtendedList
  (s/coll-of :cognitect.aws.detective/AccountId :min-count 1 :max-count 200))

(s/def
  :cognitect.aws.detective/IndicatorDetail
  (s/keys
    :opt-un
    [:cognitect.aws.detective.IndicatorDetail/RelatedFindingDetail
     :cognitect.aws.detective.IndicatorDetail/NewGeolocationDetail
     :cognitect.aws.detective.IndicatorDetail/TTPsObservedDetail
     :cognitect.aws.detective.IndicatorDetail/RelatedFindingGroupDetail
     :cognitect.aws.detective.IndicatorDetail/NewAsoDetail
     :cognitect.aws.detective.IndicatorDetail/ImpossibleTravelDetail
     :cognitect.aws.detective.IndicatorDetail/FlaggedIpAddressDetail
     :cognitect.aws.detective.IndicatorDetail/NewUserAgentDetail]))

(s/def
  :cognitect.aws.detective/ListTagsForResourceRequest
  (s/keys :req-un [:cognitect.aws.detective.ListTagsForResourceRequest/ResourceArn]))

(s/def
  :cognitect.aws.detective/GetMembersResponse
  (s/keys
    :opt-un
    [:cognitect.aws.detective.GetMembersResponse/UnprocessedAccounts
     :cognitect.aws.detective.GetMembersResponse/MemberDetails]))

(s/def :cognitect.aws.detective/UntagResourceResponse (s/keys))

(s/def
  :cognitect.aws.detective/UnprocessedGraphList
  (s/coll-of :cognitect.aws.detective/UnprocessedGraph))

(s/def
  :cognitect.aws.detective/BatchGetMembershipDatasourcesResponse
  (s/keys
    :opt-un
    [:cognitect.aws.detective.BatchGetMembershipDatasourcesResponse/MembershipDatasources
     :cognitect.aws.detective.BatchGetMembershipDatasourcesResponse/UnprocessedGraphs]))

(s/def
  :cognitect.aws.detective/ListInvitationsRequest
  (s/keys
    :opt-un
    [:cognitect.aws.detective.ListInvitationsRequest/NextToken
     :cognitect.aws.detective.ListInvitationsRequest/MaxResults]))

(s/def
  :cognitect.aws.detective/IndicatorType
  (s/spec
    string?
    :gen
    #(s/gen
      #{"RELATED_FINDING"
        "TTP_OBSERVED"
        "RELATED_FINDING_GROUP"
        "NEW_GEOLOCATION"
        "NEW_USER_AGENT"
        "NEW_ASO"
        "IMPOSSIBLE_TRAVEL"
        "FLAGGED_IP_ADDRESS"})))

(s/def
  :cognitect.aws.detective/Percentage
  (s/spec double? :gen #(gen/double* {:infinite? false, :NaN? false})))

(s/def
  :cognitect.aws.detective/TTPsObservedDetail
  (s/keys
    :opt-un
    [:cognitect.aws.detective.TTPsObservedDetail/Procedure
     :cognitect.aws.detective.TTPsObservedDetail/APISuccessCount
     :cognitect.aws.detective.TTPsObservedDetail/APIFailureCount
     :cognitect.aws.detective.TTPsObservedDetail/APIName
     :cognitect.aws.detective.TTPsObservedDetail/Technique
     :cognitect.aws.detective.TTPsObservedDetail/IpAddress
     :cognitect.aws.detective.TTPsObservedDetail/Tactic]))

(s/def
  :cognitect.aws.detective/NewGeolocationDetail
  (s/keys
    :opt-un
    [:cognitect.aws.detective.NewGeolocationDetail/IsNewForEntireAccount
     :cognitect.aws.detective.NewGeolocationDetail/Location
     :cognitect.aws.detective.NewGeolocationDetail/IpAddress]))

(s/def
  :cognitect.aws.detective/NewUserAgentDetail
  (s/keys
    :opt-un
    [:cognitect.aws.detective.NewUserAgentDetail/IsNewForEntireAccount
     :cognitect.aws.detective.NewUserAgentDetail/UserAgent]))

(s/def
  :cognitect.aws.detective/ListInvestigationsRequest
  (s/keys
    :req-un
    [:cognitect.aws.detective.ListInvestigationsRequest/GraphArn]
    :opt-un
    [:cognitect.aws.detective.ListInvestigationsRequest/NextToken
     :cognitect.aws.detective.ListInvestigationsRequest/FilterCriteria
     :cognitect.aws.detective.ListInvestigationsRequest/MaxResults
     :cognitect.aws.detective.ListInvestigationsRequest/SortCriteria]))

(s/def
  :cognitect.aws.detective/DatasourcePackageIngestState
  (s/spec string? :gen #(s/gen #{"DISABLED" "STOPPED" "STARTED"})))

(s/def
  :cognitect.aws.detective/HourlyTimeDelta
  (s/spec int? :gen #(gen/choose Long/MIN_VALUE Long/MAX_VALUE)))

(s/def
  :cognitect.aws.detective/Account
  (s/keys
    :req-un
    [:cognitect.aws.detective.Account/AccountId :cognitect.aws.detective.Account/EmailAddress]))

(s/def
  :cognitect.aws.detective/ByteValue
  (s/spec int? :gen #(gen/choose Long/MIN_VALUE Long/MAX_VALUE)))

(s/def :cognitect.aws.detective/IsNewForEntireAccount boolean?)

(s/def
  :cognitect.aws.detective/DatasourcePackage
  (s/spec string? :gen #(s/gen #{"ASFF_SECURITYHUB_FINDING" "EKS_AUDIT" "DETECTIVE_CORE"})))

(s/def
  :cognitect.aws.detective/DeleteGraphRequest
  (s/keys :req-un [:cognitect.aws.detective.DeleteGraphRequest/GraphArn]))

(s/def
  :cognitect.aws.detective/NewAsoDetail
  (s/keys
    :opt-un
    [:cognitect.aws.detective.NewAsoDetail/IsNewForEntireAccount
     :cognitect.aws.detective.NewAsoDetail/Aso]))

(s/def :cognitect.aws.detective/SortOrder (s/spec string? :gen #(s/gen #{"DESC" "ASC"})))

(s/def
  :cognitect.aws.detective/InvestigationId
  (s/spec #(re-matches (re-pattern "^[0-9]+$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.detective/AccountIdList
  (s/coll-of :cognitect.aws.detective/AccountId :min-count 1 :max-count 50))

(s/def
  :cognitect.aws.detective/EmailMessage
  (s/spec
    (s/and string? #(<= 1 (count %) 1000))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 1 1000) #(gen/vector (gen/char-alpha) %))))))

(s/def :cognitect.aws.detective/APIName string?)

(s/def
  :cognitect.aws.detective/ListDatasourcePackagesResponse
  (s/keys
    :opt-un
    [:cognitect.aws.detective.ListDatasourcePackagesResponse/NextToken
     :cognitect.aws.detective.ListDatasourcePackagesResponse/DatasourcePackages]))

(s/def
  :cognitect.aws.detective/GetInvestigationRequest
  (s/keys
    :req-un
    [:cognitect.aws.detective.GetInvestigationRequest/GraphArn
     :cognitect.aws.detective.GetInvestigationRequest/InvestigationId]))

(s/def :cognitect.aws.detective/Indicators (s/coll-of :cognitect.aws.detective/Indicator))

(s/def
  :cognitect.aws.detective/StartInvestigationResponse
  (s/keys :opt-un [:cognitect.aws.detective.StartInvestigationResponse/InvestigationId]))

(s/def
  :cognitect.aws.detective/Value
  (s/spec
    (s/and string? #(<= 1 (count %) 500))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 1 500) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.detective/MembershipDatasources
  (s/keys
    :opt-un
    [:cognitect.aws.detective.MembershipDatasources/GraphArn
     :cognitect.aws.detective.MembershipDatasources/DatasourcePackageIngestHistory
     :cognitect.aws.detective.MembershipDatasources/AccountId]))

(s/def
  :cognitect.aws.detective/RejectInvitationRequest
  (s/keys :req-un [:cognitect.aws.detective.RejectInvitationRequest/GraphArn]))

(s/def
  :cognitect.aws.detective/ListOrganizationAdminAccountsRequest
  (s/keys
    :opt-un
    [:cognitect.aws.detective.ListOrganizationAdminAccountsRequest/NextToken
     :cognitect.aws.detective.ListOrganizationAdminAccountsRequest/MaxResults]))

(s/def
  :cognitect.aws.detective/MemberStatus
  (s/spec
    string?
    :gen
    #(s/gen
      #{"INVITED"
        "VERIFICATION_FAILED"
        "VERIFICATION_IN_PROGRESS"
        "ACCEPTED_BUT_DISABLED"
        "ENABLED"})))

(s/def
  :cognitect.aws.detective/MemberDetail
  (s/keys
    :opt-un
    [:cognitect.aws.detective.MemberDetail/InvitationType
     :cognitect.aws.detective.MemberDetail/DisabledReason
     :cognitect.aws.detective.MemberDetail/GraphArn
     :cognitect.aws.detective.MemberDetail/VolumeUsageInBytes
     :cognitect.aws.detective.MemberDetail/Status
     :cognitect.aws.detective.MemberDetail/PercentOfGraphUtilization
     :cognitect.aws.detective.MemberDetail/VolumeUsageUpdatedTime
     :cognitect.aws.detective.MemberDetail/DatasourcePackageIngestStates
     :cognitect.aws.detective.MemberDetail/PercentOfGraphUtilizationUpdatedTime
     :cognitect.aws.detective.MemberDetail/AccountId
     :cognitect.aws.detective.MemberDetail/MasterId
     :cognitect.aws.detective.MemberDetail/UpdatedTime
     :cognitect.aws.detective.MemberDetail/AdministratorId
     :cognitect.aws.detective.MemberDetail/EmailAddress
     :cognitect.aws.detective.MemberDetail/InvitedTime
     :cognitect.aws.detective.MemberDetail/VolumeUsageByDatasourcePackage]))

(s/def
  :cognitect.aws.detective/GraphArnList
  (s/coll-of :cognitect.aws.detective/GraphArn :min-count 1 :max-count 50))

(s/def
  :cognitect.aws.detective/ListIndicatorsResponse
  (s/keys
    :opt-un
    [:cognitect.aws.detective.ListIndicatorsResponse/InvestigationId
     :cognitect.aws.detective.ListIndicatorsResponse/GraphArn
     :cognitect.aws.detective.ListIndicatorsResponse/NextToken
     :cognitect.aws.detective.ListIndicatorsResponse/Indicators]))

(s/def
  :cognitect.aws.detective/MaxResults
  (s/spec (s/and int? #(<= 1 % 100)) :gen #(gen/choose 1 100)))

(s/def
  :cognitect.aws.detective/UntagResourceRequest
  (s/keys
    :req-un
    [:cognitect.aws.detective.UntagResourceRequest/ResourceArn
     :cognitect.aws.detective.UntagResourceRequest/TagKeys]))

(s/def
  :cognitect.aws.detective/TagKey
  (s/spec #(re-matches (re-pattern "^(?!aws:)[a-zA-Z+-=._:/]+$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.detective/DatasourcePackageIngestHistory
  (s/map-of
    :cognitect.aws.detective/DatasourcePackage
    :cognitect.aws.detective/LastIngestStateChangeDates))

(s/def
  :cognitect.aws.detective/SortCriteria
  (s/keys
    :opt-un
    [:cognitect.aws.detective.SortCriteria/SortOrder :cognitect.aws.detective.SortCriteria/Field]))

(s/def
  :cognitect.aws.detective/DatasourcePackageList
  (s/coll-of :cognitect.aws.detective/DatasourcePackage :min-count 1 :max-count 25))

(s/def
  :cognitect.aws.detective/Resource
  (s/spec
    (s/and string? #(<= 1 (count %) 64))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 1 64) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.detective/Graph
  (s/keys :opt-un [:cognitect.aws.detective.Graph/CreatedTime :cognitect.aws.detective.Graph/Arn]))

(s/def :cognitect.aws.detective/Type string?)

(s/def
  :cognitect.aws.detective/ResourceList
  (s/coll-of :cognitect.aws.detective/Resource :min-count 1 :max-count 50))

(s/def :cognitect.aws.detective/Timestamp inst?)

(s/def :cognitect.aws.detective/State (s/spec string? :gen #(s/gen #{"ARCHIVED" "ACTIVE"})))

(s/def
  :cognitect.aws.detective/InvestigationDetail
  (s/keys
    :opt-un
    [:cognitect.aws.detective.InvestigationDetail/Severity
     :cognitect.aws.detective.InvestigationDetail/InvestigationId
     :cognitect.aws.detective.InvestigationDetail/Status
     :cognitect.aws.detective.InvestigationDetail/CreatedTime
     :cognitect.aws.detective.InvestigationDetail/EntityArn
     :cognitect.aws.detective.InvestigationDetail/State
     :cognitect.aws.detective.InvestigationDetail/EntityType]))

(s/def :cognitect.aws.detective/GraphList (s/coll-of :cognitect.aws.detective/Graph))

(s/def
  :cognitect.aws.detective/StringFilter
  (s/keys :req-un [:cognitect.aws.detective.StringFilter/Value]))

(s/def :cognitect.aws.detective/Procedure string?)

(s/def
  :cognitect.aws.detective/Indicator
  (s/keys
    :opt-un
    [:cognitect.aws.detective.Indicator/IndicatorDetail
     :cognitect.aws.detective.Indicator/IndicatorType]))

(s/def
  :cognitect.aws.detective/DeleteMembersResponse
  (s/keys
    :opt-un
    [:cognitect.aws.detective.DeleteMembersResponse/UnprocessedAccounts
     :cognitect.aws.detective.DeleteMembersResponse/AccountIds]))

(s/def :cognitect.aws.detective/Location string?)

(s/def
  :cognitect.aws.detective/APIFailureCount
  (s/spec int? :gen #(gen/choose Long/MIN_VALUE Long/MAX_VALUE)))

(s/def
  :cognitect.aws.detective/DatasourcePackageIngestStates
  (s/map-of
    :cognitect.aws.detective/DatasourcePackage
    :cognitect.aws.detective/DatasourcePackageIngestState))

(s/def
  :cognitect.aws.detective/RelatedFindingDetail
  (s/keys
    :opt-un
    [:cognitect.aws.detective.RelatedFindingDetail/Type
     :cognitect.aws.detective.RelatedFindingDetail/Arn
     :cognitect.aws.detective.RelatedFindingDetail/IpAddress]))

(s/def
  :cognitect.aws.detective/ListInvestigationsResponse
  (s/keys
    :opt-un
    [:cognitect.aws.detective.ListInvestigationsResponse/InvestigationDetails
     :cognitect.aws.detective.ListInvestigationsResponse/NextToken]))

(s/def
  :cognitect.aws.detective/ListTagsForResourceResponse
  (s/keys :opt-un [:cognitect.aws.detective.ListTagsForResourceResponse/Tags]))

(s/def
  :cognitect.aws.detective/PaginationToken
  (s/spec
    (s/and string? #(<= 1 (count %) 1024))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 1 1024) #(gen/vector (gen/char-alpha) %))))))

(s/def :cognitect.aws.detective/ErrorCodeReason string?)

(s/def
  :cognitect.aws.detective/DatasourcePackageUsageInfo
  (s/keys
    :opt-un
    [:cognitect.aws.detective.DatasourcePackageUsageInfo/VolumeUsageInBytes
     :cognitect.aws.detective.DatasourcePackageUsageInfo/VolumeUsageUpdateTime]))

(s/def
  :cognitect.aws.detective/BatchGetGraphMemberDatasourcesRequest
  (s/keys
    :req-un
    [:cognitect.aws.detective.BatchGetGraphMemberDatasourcesRequest/GraphArn
     :cognitect.aws.detective.BatchGetGraphMemberDatasourcesRequest/AccountIds]))

(s/def
  :cognitect.aws.detective/UnprocessedAccountList
  (s/coll-of :cognitect.aws.detective/UnprocessedAccount))

(s/def :cognitect.aws.detective/UserAgent string?)

(s/def
  :cognitect.aws.detective/AcceptInvitationRequest
  (s/keys :req-un [:cognitect.aws.detective.AcceptInvitationRequest/GraphArn]))

(s/def :cognitect.aws.detective/Technique string?)

(s/def
  :cognitect.aws.detective/TooManyRequestsException
  (s/keys :opt-un [:cognitect.aws.detective.TooManyRequestsException/Message]))

(s/def
  :cognitect.aws.detective/AdministratorList
  (s/coll-of :cognitect.aws.detective/Administrator))

(s/def
  :cognitect.aws.detective/BatchGetMembershipDatasourcesRequest
  (s/keys :req-un [:cognitect.aws.detective.BatchGetMembershipDatasourcesRequest/GraphArns]))

(s/def
  :cognitect.aws.detective/ResourceNotFoundException
  (s/keys :opt-un [:cognitect.aws.detective.ResourceNotFoundException/Message]))

(s/def
  :cognitect.aws.detective/ConflictException
  (s/keys :opt-un [:cognitect.aws.detective.ConflictException/Message]))

(s/def
  :cognitect.aws.detective/TagKeyList
  (s/coll-of :cognitect.aws.detective/TagKey :min-count 1 :max-count 50))

(s/def
  :cognitect.aws.detective/Status
  (s/spec string? :gen #(s/gen #{"FAILED" "RUNNING" "SUCCESSFUL"})))

(s/def :cognitect.aws.detective.TagResourceRequest/ResourceArn :cognitect.aws.detective/GraphArn)

(s/def :cognitect.aws.detective.TagResourceRequest/Tags :cognitect.aws.detective/TagMap)

(s/def
  :cognitect.aws.detective.UpdateDatasourcePackagesRequest/GraphArn
  :cognitect.aws.detective/GraphArn)

(s/def
  :cognitect.aws.detective.UpdateDatasourcePackagesRequest/DatasourcePackages
  :cognitect.aws.detective/DatasourcePackageList)

(s/def
  :cognitect.aws.detective.ServiceQuotaExceededException/Message
  :cognitect.aws.detective/ErrorMessage)

(s/def
  :cognitect.aws.detective.ServiceQuotaExceededException/Resources
  :cognitect.aws.detective/ResourceList)

(s/def
  :cognitect.aws.detective.InternalServerException/Message
  :cognitect.aws.detective/ErrorMessage)

(s/def :cognitect.aws.detective.FilterCriteria/Severity :cognitect.aws.detective/StringFilter)

(s/def :cognitect.aws.detective.FilterCriteria/Status :cognitect.aws.detective/StringFilter)

(s/def :cognitect.aws.detective.FilterCriteria/State :cognitect.aws.detective/StringFilter)

(s/def :cognitect.aws.detective.FilterCriteria/EntityArn :cognitect.aws.detective/StringFilter)

(s/def :cognitect.aws.detective.FilterCriteria/CreatedTime :cognitect.aws.detective/DateFilter)

(s/def :cognitect.aws.detective.TimestampForCollection/Timestamp :cognitect.aws.detective/Timestamp)

(s/def
  :cognitect.aws.detective.BatchGetGraphMemberDatasourcesResponse/MemberDatasources
  :cognitect.aws.detective/MembershipDatasourcesList)

(s/def
  :cognitect.aws.detective.BatchGetGraphMemberDatasourcesResponse/UnprocessedAccounts
  :cognitect.aws.detective/UnprocessedAccountList)

(s/def :cognitect.aws.detective.ListGraphsResponse/GraphList :cognitect.aws.detective/GraphList)

(s/def
  :cognitect.aws.detective.ListGraphsResponse/NextToken
  :cognitect.aws.detective/PaginationToken)

(s/def
  :cognitect.aws.detective.ListMembersResponse/MemberDetails
  :cognitect.aws.detective/MemberDetailList)

(s/def
  :cognitect.aws.detective.ListMembersResponse/NextToken
  :cognitect.aws.detective/PaginationToken)

(s/def
  :cognitect.aws.detective.ImpossibleTravelDetail/StartingIpAddress
  :cognitect.aws.detective/IpAddress)

(s/def
  :cognitect.aws.detective.ImpossibleTravelDetail/EndingIpAddress
  :cognitect.aws.detective/IpAddress)

(s/def
  :cognitect.aws.detective.ImpossibleTravelDetail/StartingLocation
  :cognitect.aws.detective/Location)

(s/def
  :cognitect.aws.detective.ImpossibleTravelDetail/EndingLocation
  :cognitect.aws.detective/Location)

(s/def
  :cognitect.aws.detective.ImpossibleTravelDetail/HourlyTimeDelta
  :cognitect.aws.detective/HourlyTimeDelta)

(s/def
  :cognitect.aws.detective.EnableOrganizationAdminAccountRequest/AccountId
  :cognitect.aws.detective/AccountId)

(s/def
  :cognitect.aws.detective.ListGraphsRequest/NextToken
  :cognitect.aws.detective/PaginationToken)

(s/def
  :cognitect.aws.detective.ListGraphsRequest/MaxResults
  :cognitect.aws.detective/MemberResultsLimit)

(s/def
  :cognitect.aws.detective.StartInvestigationRequest/GraphArn
  :cognitect.aws.detective/GraphArn)

(s/def
  :cognitect.aws.detective.StartInvestigationRequest/EntityArn
  :cognitect.aws.detective/EntityArn)

(s/def
  :cognitect.aws.detective.StartInvestigationRequest/ScopeStartTime
  :cognitect.aws.detective/Timestamp)

(s/def
  :cognitect.aws.detective.StartInvestigationRequest/ScopeEndTime
  :cognitect.aws.detective/Timestamp)

(s/def :cognitect.aws.detective.ValidationException/Message :cognitect.aws.detective/ErrorMessage)

(s/def :cognitect.aws.detective.ValidationException/ErrorCode :cognitect.aws.detective/ErrorCode)

(s/def
  :cognitect.aws.detective.ValidationException/ErrorCodeReason
  :cognitect.aws.detective/ErrorCodeReason)

(s/def :cognitect.aws.detective.AccessDeniedException/Message :cognitect.aws.detective/ErrorMessage)

(s/def :cognitect.aws.detective.AccessDeniedException/ErrorCode :cognitect.aws.detective/ErrorCode)

(s/def
  :cognitect.aws.detective.AccessDeniedException/ErrorCodeReason
  :cognitect.aws.detective/ErrorCodeReason)

(s/def
  :cognitect.aws.detective.AccessDeniedException/SubErrorCode
  :cognitect.aws.detective/ErrorCode)

(s/def
  :cognitect.aws.detective.AccessDeniedException/SubErrorCodeReason
  :cognitect.aws.detective/ErrorCodeReason)

(s/def :cognitect.aws.detective.UnprocessedAccount/AccountId :cognitect.aws.detective/AccountId)

(s/def
  :cognitect.aws.detective.UnprocessedAccount/Reason
  :cognitect.aws.detective/UnprocessedReason)

(s/def :cognitect.aws.detective.ListMembersRequest/GraphArn :cognitect.aws.detective/GraphArn)

(s/def
  :cognitect.aws.detective.ListMembersRequest/NextToken
  :cognitect.aws.detective/PaginationToken)

(s/def
  :cognitect.aws.detective.ListMembersRequest/MaxResults
  :cognitect.aws.detective/MemberResultsLimit)

(s/def :cognitect.aws.detective.UnprocessedGraph/GraphArn :cognitect.aws.detective/GraphArn)

(s/def :cognitect.aws.detective.UnprocessedGraph/Reason :cognitect.aws.detective/UnprocessedReason)

(s/def
  :cognitect.aws.detective.DescribeOrganizationConfigurationResponse/AutoEnable
  :cognitect.aws.detective/Boolean)

(s/def
  :cognitect.aws.detective.DescribeOrganizationConfigurationRequest/GraphArn
  :cognitect.aws.detective/GraphArn)

(s/def
  :cognitect.aws.detective.UpdateOrganizationConfigurationRequest/GraphArn
  :cognitect.aws.detective/GraphArn)

(s/def
  :cognitect.aws.detective.UpdateOrganizationConfigurationRequest/AutoEnable
  :cognitect.aws.detective/Boolean)

(s/def
  :cognitect.aws.detective.ListDatasourcePackagesRequest/GraphArn
  :cognitect.aws.detective/GraphArn)

(s/def
  :cognitect.aws.detective.ListDatasourcePackagesRequest/NextToken
  :cognitect.aws.detective/PaginationToken)

(s/def
  :cognitect.aws.detective.ListDatasourcePackagesRequest/MaxResults
  :cognitect.aws.detective/MemberResultsLimit)

(s/def :cognitect.aws.detective.FlaggedIpAddressDetail/IpAddress :cognitect.aws.detective/IpAddress)

(s/def :cognitect.aws.detective.FlaggedIpAddressDetail/Reason :cognitect.aws.detective/Reason)

(s/def :cognitect.aws.detective.GetMembersRequest/GraphArn :cognitect.aws.detective/GraphArn)

(s/def :cognitect.aws.detective.GetMembersRequest/AccountIds :cognitect.aws.detective/AccountIdList)

(s/def :cognitect.aws.detective.CreateGraphResponse/GraphArn :cognitect.aws.detective/GraphArn)

(s/def
  :cognitect.aws.detective.DatasourcePackageIngestDetail/DatasourcePackageIngestState
  :cognitect.aws.detective/DatasourcePackageIngestState)

(s/def
  :cognitect.aws.detective.DatasourcePackageIngestDetail/LastIngestStateChange
  :cognitect.aws.detective/LastIngestStateChangeDates)

(s/def :cognitect.aws.detective.CreateMembersRequest/GraphArn :cognitect.aws.detective/GraphArn)

(s/def :cognitect.aws.detective.CreateMembersRequest/Message :cognitect.aws.detective/EmailMessage)

(s/def
  :cognitect.aws.detective.CreateMembersRequest/DisableEmailNotification
  :cognitect.aws.detective/Boolean)

(s/def :cognitect.aws.detective.CreateMembersRequest/Accounts :cognitect.aws.detective/AccountList)

(s/def :cognitect.aws.detective.CreateGraphRequest/Tags :cognitect.aws.detective/TagMap)

(s/def
  :cognitect.aws.detective.ListInvitationsResponse/Invitations
  :cognitect.aws.detective/MemberDetailList)

(s/def
  :cognitect.aws.detective.ListInvitationsResponse/NextToken
  :cognitect.aws.detective/PaginationToken)

(s/def
  :cognitect.aws.detective.ListOrganizationAdminAccountsResponse/Administrators
  :cognitect.aws.detective/AdministratorList)

(s/def
  :cognitect.aws.detective.ListOrganizationAdminAccountsResponse/NextToken
  :cognitect.aws.detective/PaginationToken)

(s/def :cognitect.aws.detective.GetInvestigationResponse/Severity :cognitect.aws.detective/Severity)

(s/def
  :cognitect.aws.detective.GetInvestigationResponse/ScopeEndTime
  :cognitect.aws.detective/Timestamp)

(s/def
  :cognitect.aws.detective.GetInvestigationResponse/EntityArn
  :cognitect.aws.detective/EntityArn)

(s/def
  :cognitect.aws.detective.GetInvestigationResponse/EntityType
  :cognitect.aws.detective/EntityType)

(s/def :cognitect.aws.detective.GetInvestigationResponse/GraphArn :cognitect.aws.detective/GraphArn)

(s/def
  :cognitect.aws.detective.GetInvestigationResponse/ScopeStartTime
  :cognitect.aws.detective/Timestamp)

(s/def
  :cognitect.aws.detective.GetInvestigationResponse/InvestigationId
  :cognitect.aws.detective/InvestigationId)

(s/def :cognitect.aws.detective.GetInvestigationResponse/State :cognitect.aws.detective/State)

(s/def
  :cognitect.aws.detective.GetInvestigationResponse/CreatedTime
  :cognitect.aws.detective/Timestamp)

(s/def :cognitect.aws.detective.GetInvestigationResponse/Status :cognitect.aws.detective/Status)

(s/def :cognitect.aws.detective.ListIndicatorsRequest/GraphArn :cognitect.aws.detective/GraphArn)

(s/def
  :cognitect.aws.detective.ListIndicatorsRequest/InvestigationId
  :cognitect.aws.detective/InvestigationId)

(s/def
  :cognitect.aws.detective.ListIndicatorsRequest/IndicatorType
  :cognitect.aws.detective/IndicatorType)

(s/def
  :cognitect.aws.detective.ListIndicatorsRequest/NextToken
  :cognitect.aws.detective/AiPaginationToken)

(s/def
  :cognitect.aws.detective.ListIndicatorsRequest/MaxResults
  :cognitect.aws.detective/MaxResults)

(s/def
  :cognitect.aws.detective.StartMonitoringMemberRequest/GraphArn
  :cognitect.aws.detective/GraphArn)

(s/def
  :cognitect.aws.detective.StartMonitoringMemberRequest/AccountId
  :cognitect.aws.detective/AccountId)

(s/def
  :cognitect.aws.detective.UpdateInvestigationStateRequest/GraphArn
  :cognitect.aws.detective/GraphArn)

(s/def
  :cognitect.aws.detective.UpdateInvestigationStateRequest/InvestigationId
  :cognitect.aws.detective/InvestigationId)

(s/def
  :cognitect.aws.detective.UpdateInvestigationStateRequest/State
  :cognitect.aws.detective/State)

(s/def :cognitect.aws.detective.RelatedFindingGroupDetail/Id :cognitect.aws.detective/Id)

(s/def :cognitect.aws.detective.DateFilter/StartInclusive :cognitect.aws.detective/Timestamp)

(s/def :cognitect.aws.detective.DateFilter/EndInclusive :cognitect.aws.detective/Timestamp)

(s/def :cognitect.aws.detective.Administrator/AccountId :cognitect.aws.detective/AccountId)

(s/def :cognitect.aws.detective.Administrator/GraphArn :cognitect.aws.detective/GraphArn)

(s/def :cognitect.aws.detective.Administrator/DelegationTime :cognitect.aws.detective/Timestamp)

(s/def :cognitect.aws.detective.DeleteMembersRequest/GraphArn :cognitect.aws.detective/GraphArn)

(s/def
  :cognitect.aws.detective.DeleteMembersRequest/AccountIds
  :cognitect.aws.detective/AccountIdList)

(s/def
  :cognitect.aws.detective.CreateMembersResponse/Members
  :cognitect.aws.detective/MemberDetailList)

(s/def
  :cognitect.aws.detective.CreateMembersResponse/UnprocessedAccounts
  :cognitect.aws.detective/UnprocessedAccountList)

(s/def
  :cognitect.aws.detective.DisassociateMembershipRequest/GraphArn
  :cognitect.aws.detective/GraphArn)

(s/def
  :cognitect.aws.detective.IndicatorDetail/TTPsObservedDetail
  :cognitect.aws.detective/TTPsObservedDetail)

(s/def
  :cognitect.aws.detective.IndicatorDetail/ImpossibleTravelDetail
  :cognitect.aws.detective/ImpossibleTravelDetail)

(s/def
  :cognitect.aws.detective.IndicatorDetail/FlaggedIpAddressDetail
  :cognitect.aws.detective/FlaggedIpAddressDetail)

(s/def
  :cognitect.aws.detective.IndicatorDetail/NewGeolocationDetail
  :cognitect.aws.detective/NewGeolocationDetail)

(s/def :cognitect.aws.detective.IndicatorDetail/NewAsoDetail :cognitect.aws.detective/NewAsoDetail)

(s/def
  :cognitect.aws.detective.IndicatorDetail/NewUserAgentDetail
  :cognitect.aws.detective/NewUserAgentDetail)

(s/def
  :cognitect.aws.detective.IndicatorDetail/RelatedFindingDetail
  :cognitect.aws.detective/RelatedFindingDetail)

(s/def
  :cognitect.aws.detective.IndicatorDetail/RelatedFindingGroupDetail
  :cognitect.aws.detective/RelatedFindingGroupDetail)

(s/def
  :cognitect.aws.detective.ListTagsForResourceRequest/ResourceArn
  :cognitect.aws.detective/GraphArn)

(s/def
  :cognitect.aws.detective.GetMembersResponse/MemberDetails
  :cognitect.aws.detective/MemberDetailList)

(s/def
  :cognitect.aws.detective.GetMembersResponse/UnprocessedAccounts
  :cognitect.aws.detective/UnprocessedAccountList)

(s/def
  :cognitect.aws.detective.BatchGetMembershipDatasourcesResponse/MembershipDatasources
  :cognitect.aws.detective/MembershipDatasourcesList)

(s/def
  :cognitect.aws.detective.BatchGetMembershipDatasourcesResponse/UnprocessedGraphs
  :cognitect.aws.detective/UnprocessedGraphList)

(s/def
  :cognitect.aws.detective.ListInvitationsRequest/NextToken
  :cognitect.aws.detective/PaginationToken)

(s/def
  :cognitect.aws.detective.ListInvitationsRequest/MaxResults
  :cognitect.aws.detective/MemberResultsLimit)

(s/def :cognitect.aws.detective.TTPsObservedDetail/Tactic :cognitect.aws.detective/Tactic)

(s/def :cognitect.aws.detective.TTPsObservedDetail/Technique :cognitect.aws.detective/Technique)

(s/def :cognitect.aws.detective.TTPsObservedDetail/Procedure :cognitect.aws.detective/Procedure)

(s/def :cognitect.aws.detective.TTPsObservedDetail/IpAddress :cognitect.aws.detective/IpAddress)

(s/def :cognitect.aws.detective.TTPsObservedDetail/APIName :cognitect.aws.detective/APIName)

(s/def
  :cognitect.aws.detective.TTPsObservedDetail/APISuccessCount
  :cognitect.aws.detective/APISuccessCount)

(s/def
  :cognitect.aws.detective.TTPsObservedDetail/APIFailureCount
  :cognitect.aws.detective/APIFailureCount)

(s/def :cognitect.aws.detective.NewGeolocationDetail/Location :cognitect.aws.detective/Location)

(s/def :cognitect.aws.detective.NewGeolocationDetail/IpAddress :cognitect.aws.detective/IpAddress)

(s/def
  :cognitect.aws.detective.NewGeolocationDetail/IsNewForEntireAccount
  :cognitect.aws.detective/IsNewForEntireAccount)

(s/def :cognitect.aws.detective.NewUserAgentDetail/UserAgent :cognitect.aws.detective/UserAgent)

(s/def
  :cognitect.aws.detective.NewUserAgentDetail/IsNewForEntireAccount
  :cognitect.aws.detective/IsNewForEntireAccount)

(s/def
  :cognitect.aws.detective.ListInvestigationsRequest/GraphArn
  :cognitect.aws.detective/GraphArn)

(s/def
  :cognitect.aws.detective.ListInvestigationsRequest/NextToken
  :cognitect.aws.detective/AiPaginationToken)

(s/def
  :cognitect.aws.detective.ListInvestigationsRequest/MaxResults
  :cognitect.aws.detective/MaxResults)

(s/def
  :cognitect.aws.detective.ListInvestigationsRequest/FilterCriteria
  :cognitect.aws.detective/FilterCriteria)

(s/def
  :cognitect.aws.detective.ListInvestigationsRequest/SortCriteria
  :cognitect.aws.detective/SortCriteria)

(s/def :cognitect.aws.detective.Account/AccountId :cognitect.aws.detective/AccountId)

(s/def :cognitect.aws.detective.Account/EmailAddress :cognitect.aws.detective/EmailAddress)

(s/def :cognitect.aws.detective.DeleteGraphRequest/GraphArn :cognitect.aws.detective/GraphArn)

(s/def :cognitect.aws.detective.NewAsoDetail/Aso :cognitect.aws.detective/Aso)

(s/def
  :cognitect.aws.detective.NewAsoDetail/IsNewForEntireAccount
  :cognitect.aws.detective/IsNewForEntireAccount)

(s/def
  :cognitect.aws.detective.ListDatasourcePackagesResponse/DatasourcePackages
  :cognitect.aws.detective/DatasourcePackageIngestDetails)

(s/def
  :cognitect.aws.detective.ListDatasourcePackagesResponse/NextToken
  :cognitect.aws.detective/PaginationToken)

(s/def :cognitect.aws.detective.GetInvestigationRequest/GraphArn :cognitect.aws.detective/GraphArn)

(s/def
  :cognitect.aws.detective.GetInvestigationRequest/InvestigationId
  :cognitect.aws.detective/InvestigationId)

(s/def
  :cognitect.aws.detective.StartInvestigationResponse/InvestigationId
  :cognitect.aws.detective/InvestigationId)

(s/def :cognitect.aws.detective.MembershipDatasources/AccountId :cognitect.aws.detective/AccountId)

(s/def :cognitect.aws.detective.MembershipDatasources/GraphArn :cognitect.aws.detective/GraphArn)

(s/def
  :cognitect.aws.detective.MembershipDatasources/DatasourcePackageIngestHistory
  :cognitect.aws.detective/DatasourcePackageIngestHistory)

(s/def :cognitect.aws.detective.RejectInvitationRequest/GraphArn :cognitect.aws.detective/GraphArn)

(s/def
  :cognitect.aws.detective.ListOrganizationAdminAccountsRequest/NextToken
  :cognitect.aws.detective/PaginationToken)

(s/def
  :cognitect.aws.detective.ListOrganizationAdminAccountsRequest/MaxResults
  :cognitect.aws.detective/MemberResultsLimit)

(s/def
  :cognitect.aws.detective.MemberDetail/VolumeUsageUpdatedTime
  :cognitect.aws.detective/Timestamp)

(s/def :cognitect.aws.detective.MemberDetail/AccountId :cognitect.aws.detective/AccountId)

(s/def
  :cognitect.aws.detective.MemberDetail/VolumeUsageByDatasourcePackage
  :cognitect.aws.detective/VolumeUsageByDatasourcePackage)

(s/def
  :cognitect.aws.detective.MemberDetail/PercentOfGraphUtilizationUpdatedTime
  :cognitect.aws.detective/Timestamp)

(s/def :cognitect.aws.detective.MemberDetail/EmailAddress :cognitect.aws.detective/EmailAddress)

(s/def :cognitect.aws.detective.MemberDetail/GraphArn :cognitect.aws.detective/GraphArn)

(s/def :cognitect.aws.detective.MemberDetail/InvitationType :cognitect.aws.detective/InvitationType)

(s/def :cognitect.aws.detective.MemberDetail/InvitedTime :cognitect.aws.detective/Timestamp)

(s/def :cognitect.aws.detective.MemberDetail/VolumeUsageInBytes :cognitect.aws.detective/ByteValue)

(s/def
  :cognitect.aws.detective.MemberDetail/PercentOfGraphUtilization
  :cognitect.aws.detective/Percentage)

(s/def :cognitect.aws.detective.MemberDetail/AdministratorId :cognitect.aws.detective/AccountId)

(s/def
  :cognitect.aws.detective.MemberDetail/DisabledReason
  :cognitect.aws.detective/MemberDisabledReason)

(s/def :cognitect.aws.detective.MemberDetail/UpdatedTime :cognitect.aws.detective/Timestamp)

(s/def
  :cognitect.aws.detective.MemberDetail/DatasourcePackageIngestStates
  :cognitect.aws.detective/DatasourcePackageIngestStates)

(s/def :cognitect.aws.detective.MemberDetail/MasterId :cognitect.aws.detective/AccountId)

(s/def :cognitect.aws.detective.MemberDetail/Status :cognitect.aws.detective/MemberStatus)

(s/def :cognitect.aws.detective.ListIndicatorsResponse/GraphArn :cognitect.aws.detective/GraphArn)

(s/def
  :cognitect.aws.detective.ListIndicatorsResponse/InvestigationId
  :cognitect.aws.detective/InvestigationId)

(s/def
  :cognitect.aws.detective.ListIndicatorsResponse/NextToken
  :cognitect.aws.detective/AiPaginationToken)

(s/def
  :cognitect.aws.detective.ListIndicatorsResponse/Indicators
  :cognitect.aws.detective/Indicators)

(s/def :cognitect.aws.detective.UntagResourceRequest/ResourceArn :cognitect.aws.detective/GraphArn)

(s/def :cognitect.aws.detective.UntagResourceRequest/TagKeys :cognitect.aws.detective/TagKeyList)

(s/def :cognitect.aws.detective.SortCriteria/Field :cognitect.aws.detective/Field)

(s/def :cognitect.aws.detective.SortCriteria/SortOrder :cognitect.aws.detective/SortOrder)

(s/def :cognitect.aws.detective.Graph/Arn :cognitect.aws.detective/GraphArn)

(s/def :cognitect.aws.detective.Graph/CreatedTime :cognitect.aws.detective/Timestamp)

(s/def
  :cognitect.aws.detective.InvestigationDetail/InvestigationId
  :cognitect.aws.detective/InvestigationId)

(s/def :cognitect.aws.detective.InvestigationDetail/Severity :cognitect.aws.detective/Severity)

(s/def :cognitect.aws.detective.InvestigationDetail/Status :cognitect.aws.detective/Status)

(s/def :cognitect.aws.detective.InvestigationDetail/State :cognitect.aws.detective/State)

(s/def :cognitect.aws.detective.InvestigationDetail/CreatedTime :cognitect.aws.detective/Timestamp)

(s/def :cognitect.aws.detective.InvestigationDetail/EntityArn :cognitect.aws.detective/EntityArn)

(s/def :cognitect.aws.detective.InvestigationDetail/EntityType :cognitect.aws.detective/EntityType)

(s/def :cognitect.aws.detective.StringFilter/Value :cognitect.aws.detective/Value)

(s/def :cognitect.aws.detective.Indicator/IndicatorType :cognitect.aws.detective/IndicatorType)

(s/def :cognitect.aws.detective.Indicator/IndicatorDetail :cognitect.aws.detective/IndicatorDetail)

(s/def
  :cognitect.aws.detective.DeleteMembersResponse/AccountIds
  :cognitect.aws.detective/AccountIdList)

(s/def
  :cognitect.aws.detective.DeleteMembersResponse/UnprocessedAccounts
  :cognitect.aws.detective/UnprocessedAccountList)

(s/def :cognitect.aws.detective.RelatedFindingDetail/Arn :cognitect.aws.detective/EntityArn)

(s/def :cognitect.aws.detective.RelatedFindingDetail/Type :cognitect.aws.detective/Type)

(s/def :cognitect.aws.detective.RelatedFindingDetail/IpAddress :cognitect.aws.detective/IpAddress)

(s/def
  :cognitect.aws.detective.ListInvestigationsResponse/InvestigationDetails
  :cognitect.aws.detective/InvestigationDetails)

(s/def
  :cognitect.aws.detective.ListInvestigationsResponse/NextToken
  :cognitect.aws.detective/AiPaginationToken)

(s/def :cognitect.aws.detective.ListTagsForResourceResponse/Tags :cognitect.aws.detective/TagMap)

(s/def
  :cognitect.aws.detective.DatasourcePackageUsageInfo/VolumeUsageInBytes
  :cognitect.aws.detective/ByteValue)

(s/def
  :cognitect.aws.detective.DatasourcePackageUsageInfo/VolumeUsageUpdateTime
  :cognitect.aws.detective/Timestamp)

(s/def
  :cognitect.aws.detective.BatchGetGraphMemberDatasourcesRequest/GraphArn
  :cognitect.aws.detective/GraphArn)

(s/def
  :cognitect.aws.detective.BatchGetGraphMemberDatasourcesRequest/AccountIds
  :cognitect.aws.detective/AccountIdExtendedList)

(s/def :cognitect.aws.detective.AcceptInvitationRequest/GraphArn :cognitect.aws.detective/GraphArn)

(s/def
  :cognitect.aws.detective.TooManyRequestsException/Message
  :cognitect.aws.detective/ErrorMessage)

(s/def
  :cognitect.aws.detective.BatchGetMembershipDatasourcesRequest/GraphArns
  :cognitect.aws.detective/GraphArnList)

(s/def
  :cognitect.aws.detective.ResourceNotFoundException/Message
  :cognitect.aws.detective/ErrorMessage)

(s/def :cognitect.aws.detective.ConflictException/Message :cognitect.aws.detective/ErrorMessage)

