;; Copyright (c) Cognitect, Inc.
;; All rights reserved.

(ns cognitect.aws.fsx.specs
  (:require [clojure.spec.alpha :as s] [clojure.spec.gen.alpha :as gen]))

(s/def :cognitect.aws/client map?)

(s/def :core.async/channel any?)

(s/def
  :cognitect.aws.fsx/OpenZFSDataCompressionType
  (s/spec string? :gen #(s/gen #{"LZ4" "NONE" "ZSTD"})))

(s/def :cognitect.aws.fsx/EventType (s/spec string? :gen #(s/gen #{"NEW" "CHANGED" "DELETED"})))

(s/def
  :cognitect.aws.fsx/AdminPassword
  (s/spec
    #(re-matches (re-pattern "^[^\\u0000\\u0085\\u2028\\u2029\\r\\n]{8,50}$") %)
    :gen
    #(gen/string)))

(s/def
  :cognitect.aws.fsx/SelfManagedActiveDirectoryConfiguration
  (s/keys
    :req-un
    [:cognitect.aws.fsx.SelfManagedActiveDirectoryConfiguration/DomainName
     :cognitect.aws.fsx.SelfManagedActiveDirectoryConfiguration/UserName
     :cognitect.aws.fsx.SelfManagedActiveDirectoryConfiguration/Password
     :cognitect.aws.fsx.SelfManagedActiveDirectoryConfiguration/DnsIps]
    :opt-un
    [:cognitect.aws.fsx.SelfManagedActiveDirectoryConfiguration/FileSystemAdministratorsGroup
     :cognitect.aws.fsx.SelfManagedActiveDirectoryConfiguration/OrganizationalUnitDistinguishedName]))

(s/def
  :cognitect.aws.fsx/BackupId
  (s/spec #(re-matches (re-pattern "^(backup-[0-9a-f]{8,})$") %) :gen #(gen/string)))

(s/def :cognitect.aws.fsx/SubnetIds (s/coll-of :cognitect.aws.fsx/SubnetId :max-count 50))

(s/def
  :cognitect.aws.fsx/TagResourceRequest
  (s/keys
    :req-un
    [:cognitect.aws.fsx.TagResourceRequest/ResourceARN :cognitect.aws.fsx.TagResourceRequest/Tags]))

(s/def
  :cognitect.aws.fsx/DeleteVolumeOntapConfiguration
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.DeleteVolumeOntapConfiguration/SkipFinalBackup
     :cognitect.aws.fsx.DeleteVolumeOntapConfiguration/FinalBackupTags]))

(s/def
  :cognitect.aws.fsx/TaskId
  (s/spec #(re-matches (re-pattern "^(task-[0-9a-f]{17,})$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.fsx/StorageVirtualMachineSubtype
  (s/spec string? :gen #(s/gen #{"DP_DESTINATION" "DEFAULT" "SYNC_DESTINATION" "SYNC_SOURCE"})))

(s/def :cognitect.aws.fsx/EndTime inst?)

(s/def
  :cognitect.aws.fsx/DescribeBackupsResponse
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.DescribeBackupsResponse/NextToken
     :cognitect.aws.fsx.DescribeBackupsResponse/Backups]))

(s/def :cognitect.aws.fsx/ResourceType (s/spec string? :gen #(s/gen #{"FILE_SYSTEM" "VOLUME"})))

(s/def
  :cognitect.aws.fsx/DescribeFileSystemsRequest
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.DescribeFileSystemsRequest/FileSystemIds
     :cognitect.aws.fsx.DescribeFileSystemsRequest/NextToken
     :cognitect.aws.fsx.DescribeFileSystemsRequest/MaxResults]))

(s/def
  :cognitect.aws.fsx/DataRepositoryTaskPaths
  (s/coll-of :cognitect.aws.fsx/DataRepositoryTaskPath :max-count 100))

(s/def :cognitect.aws.fsx/FileSystemIds (s/coll-of :cognitect.aws.fsx/FileSystemId :max-count 50))

(s/def
  :cognitect.aws.fsx/CreateFileSystemFromBackupResponse
  (s/keys :opt-un [:cognitect.aws.fsx.CreateFileSystemFromBackupResponse/FileSystem]))

(s/def
  :cognitect.aws.fsx/S3DataRepositoryConfiguration
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.S3DataRepositoryConfiguration/AutoImportPolicy
     :cognitect.aws.fsx.S3DataRepositoryConfiguration/AutoExportPolicy]))

(s/def
  :cognitect.aws.fsx/CreateSvmActiveDirectoryConfiguration
  (s/keys
    :req-un
    [:cognitect.aws.fsx.CreateSvmActiveDirectoryConfiguration/NetBiosName]
    :opt-un
    [:cognitect.aws.fsx.CreateSvmActiveDirectoryConfiguration/SelfManagedActiveDirectoryConfiguration]))

(s/def
  :cognitect.aws.fsx/CreateVolumeResponse
  (s/keys :opt-un [:cognitect.aws.fsx.CreateVolumeResponse/Volume]))

(s/def
  :cognitect.aws.fsx/OpenZFSCreateRootVolumeConfiguration
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.OpenZFSCreateRootVolumeConfiguration/DataCompressionType
     :cognitect.aws.fsx.OpenZFSCreateRootVolumeConfiguration/ReadOnly
     :cognitect.aws.fsx.OpenZFSCreateRootVolumeConfiguration/CopyTagsToSnapshots
     :cognitect.aws.fsx.OpenZFSCreateRootVolumeConfiguration/NfsExports
     :cognitect.aws.fsx.OpenZFSCreateRootVolumeConfiguration/RecordSizeKiB
     :cognitect.aws.fsx.OpenZFSCreateRootVolumeConfiguration/UserAndGroupQuotas]))

(s/def
  :cognitect.aws.fsx/DeleteFileSystemOpenZFSOption
  (s/spec string? :gen #(s/gen #{"DELETE_CHILD_VOLUMES_AND_SNAPSHOTS"})))

(s/def
  :cognitect.aws.fsx/Tag
  (s/keys :req-un [:cognitect.aws.fsx.Tag/Key :cognitect.aws.fsx.Tag/Value]))

(s/def
  :cognitect.aws.fsx/CreateStorageVirtualMachineResponse
  (s/keys :opt-un [:cognitect.aws.fsx.CreateStorageVirtualMachineResponse/StorageVirtualMachine]))

(s/def
  :cognitect.aws.fsx/DNSName
  (s/spec #(re-matches (re-pattern "^(fsi?-[0-9a-f]{8,}\\..{4,253})$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.fsx/SelfManagedActiveDirectoryConfigurationUpdates
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.SelfManagedActiveDirectoryConfigurationUpdates/Password
     :cognitect.aws.fsx.SelfManagedActiveDirectoryConfigurationUpdates/UserName
     :cognitect.aws.fsx.SelfManagedActiveDirectoryConfigurationUpdates/DnsIps]))

(s/def
  :cognitect.aws.fsx/VolumeLifecycle
  (s/spec
    string?
    :gen
    #(s/gen #{"MISCONFIGURED" "AVAILABLE" "PENDING" "DELETING" "CREATING" "FAILED" "CREATED"})))

(s/def
  :cognitect.aws.fsx/AutomaticBackupRetentionDays
  (s/spec (s/and int? #(<= 0 % 90)) :gen #(gen/choose 0 90)))

(s/def
  :cognitect.aws.fsx/VolumePath
  (s/spec #(re-matches (re-pattern "^[A-za-z0-9\\_\\.\\:\\-\\/]*$") %) :gen #(gen/string)))

(s/def :cognitect.aws.fsx/ReportFormat (s/spec string? :gen #(s/gen #{"REPORT_CSV_20191124"})))

(s/def :cognitect.aws.fsx/SecurityStyle (s/spec string? :gen #(s/gen #{"MIXED" "UNIX" "NTFS"})))

(s/def :cognitect.aws.fsx/TagResourceResponse (s/keys))

(s/def
  :cognitect.aws.fsx/IntegerRecordSizeKiB
  (s/spec (s/and int? #(<= 4 % 1024)) :gen #(gen/choose 4 1024)))

(s/def
  :cognitect.aws.fsx/DescribeDataRepositoryTasksRequest
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.DescribeDataRepositoryTasksRequest/TaskIds
     :cognitect.aws.fsx.DescribeDataRepositoryTasksRequest/NextToken
     :cognitect.aws.fsx.DescribeDataRepositoryTasksRequest/Filters
     :cognitect.aws.fsx.DescribeDataRepositoryTasksRequest/MaxResults]))

(s/def
  :cognitect.aws.fsx/CreateBackupResponse
  (s/keys :opt-un [:cognitect.aws.fsx.CreateBackupResponse/Backup]))

(s/def
  :cognitect.aws.fsx/UpdateStorageVirtualMachineResponse
  (s/keys :opt-un [:cognitect.aws.fsx.UpdateStorageVirtualMachineResponse/StorageVirtualMachine]))

(s/def
  :cognitect.aws.fsx/CreateVolumeRequest
  (s/keys
    :req-un
    [:cognitect.aws.fsx.CreateVolumeRequest/VolumeType :cognitect.aws.fsx.CreateVolumeRequest/Name]
    :opt-un
    [:cognitect.aws.fsx.CreateVolumeRequest/ClientRequestToken
     :cognitect.aws.fsx.CreateVolumeRequest/Tags
     :cognitect.aws.fsx.CreateVolumeRequest/OntapConfiguration
     :cognitect.aws.fsx.CreateVolumeRequest/OpenZFSConfiguration]))

(s/def :cognitect.aws.fsx/FilterValues (s/coll-of :cognitect.aws.fsx/FilterValue :max-count 20))

(s/def
  :cognitect.aws.fsx/UpdateStorageVirtualMachineRequest
  (s/keys
    :req-un
    [:cognitect.aws.fsx.UpdateStorageVirtualMachineRequest/StorageVirtualMachineId]
    :opt-un
    [:cognitect.aws.fsx.UpdateStorageVirtualMachineRequest/ClientRequestToken
     :cognitect.aws.fsx.UpdateStorageVirtualMachineRequest/ActiveDirectoryConfiguration
     :cognitect.aws.fsx.UpdateStorageVirtualMachineRequest/SvmAdminPassword]))

(s/def
  :cognitect.aws.fsx/IpAddressRange
  (s/spec
    #(re-matches (re-pattern "^[^\\u0000\\u0085\\u2028\\u2029\\r\\n]{9,17}$") %)
    :gen
    #(gen/string)))

(s/def
  :cognitect.aws.fsx/ProgressPercent
  (s/spec (s/and int? #(<= 0 % 100)) :gen #(gen/choose 0 100)))

(s/def
  :cognitect.aws.fsx/Filter
  (s/keys :opt-un [:cognitect.aws.fsx.Filter/Values :cognitect.aws.fsx.Filter/Name]))

(s/def :cognitect.aws.fsx/SnapshotIds (s/coll-of :cognitect.aws.fsx/SnapshotId :max-count 50))

(s/def
  :cognitect.aws.fsx/CreateFileSystemRequest
  (s/keys
    :req-un
    [:cognitect.aws.fsx.CreateFileSystemRequest/FileSystemType
     :cognitect.aws.fsx.CreateFileSystemRequest/StorageCapacity
     :cognitect.aws.fsx.CreateFileSystemRequest/SubnetIds]
    :opt-un
    [:cognitect.aws.fsx.CreateFileSystemRequest/ClientRequestToken
     :cognitect.aws.fsx.CreateFileSystemRequest/SecurityGroupIds
     :cognitect.aws.fsx.CreateFileSystemRequest/FileSystemTypeVersion
     :cognitect.aws.fsx.CreateFileSystemRequest/Tags
     :cognitect.aws.fsx.CreateFileSystemRequest/OntapConfiguration
     :cognitect.aws.fsx.CreateFileSystemRequest/WindowsConfiguration
     :cognitect.aws.fsx.CreateFileSystemRequest/KmsKeyId
     :cognitect.aws.fsx.CreateFileSystemRequest/StorageType
     :cognitect.aws.fsx.CreateFileSystemRequest/LustreConfiguration
     :cognitect.aws.fsx.CreateFileSystemRequest/OpenZFSConfiguration]))

(s/def
  :cognitect.aws.fsx/IpAddress
  (s/spec
    #(re-matches
      (re-pattern
        "^(([0-9]|[1-9][0-9]|1[0-9]{2}|2[0-4][0-9]|25[0-5])\\.){3}([0-9]|[1-9][0-9]|1[0-9]{2}|2[0-4][0-9]|25[0-5])$")
      %)
    :gen
    #(gen/string)))

(s/def
  :cognitect.aws.fsx/AssociateFileSystemAliasesRequest
  (s/keys
    :req-un
    [:cognitect.aws.fsx.AssociateFileSystemAliasesRequest/FileSystemId
     :cognitect.aws.fsx.AssociateFileSystemAliasesRequest/Aliases]
    :opt-un
    [:cognitect.aws.fsx.AssociateFileSystemAliasesRequest/ClientRequestToken]))

(s/def
  :cognitect.aws.fsx/StorageCapacity
  (s/spec (s/and int? #(<= 0 % 2147483647)) :gen #(gen/choose 0 2147483647)))

(s/def
  :cognitect.aws.fsx/DirectoryPassword
  (s/spec #(re-matches (re-pattern "^.{1,256}$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.fsx/OpenZFSNfsExports
  (s/coll-of :cognitect.aws.fsx/OpenZFSNfsExport :max-count 1))

(s/def
  :cognitect.aws.fsx/VolumeFilter
  (s/keys :opt-un [:cognitect.aws.fsx.VolumeFilter/Values :cognitect.aws.fsx.VolumeFilter/Name]))

(s/def
  :cognitect.aws.fsx/UpdateSnapshotResponse
  (s/keys :opt-un [:cognitect.aws.fsx.UpdateSnapshotResponse/Snapshot]))

(s/def
  :cognitect.aws.fsx/LifecycleTransitionReason
  (s/keys :opt-un [:cognitect.aws.fsx.LifecycleTransitionReason/Message]))

(s/def
  :cognitect.aws.fsx/OrganizationalUnitDistinguishedName
  (s/spec
    #(re-matches (re-pattern "^[^\\u0000\\u0085\\u2028\\u2029\\r\\n]{1,2000}$") %)
    :gen
    #(gen/string)))

(s/def
  :cognitect.aws.fsx/ErrorMessage
  (s/spec
    (s/and string? #(<= 1 (count %) 256))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 1 256) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.fsx/StorageVirtualMachineLifecycle
  (s/spec
    string?
    :gen
    #(s/gen #{"MISCONFIGURED" "PENDING" "DELETING" "CREATING" "FAILED" "CREATED"})))

(s/def
  :cognitect.aws.fsx/UpdateFileSystemWindowsConfiguration
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.UpdateFileSystemWindowsConfiguration/ThroughputCapacity
     :cognitect.aws.fsx.UpdateFileSystemWindowsConfiguration/DailyAutomaticBackupStartTime
     :cognitect.aws.fsx.UpdateFileSystemWindowsConfiguration/WeeklyMaintenanceStartTime
     :cognitect.aws.fsx.UpdateFileSystemWindowsConfiguration/SelfManagedActiveDirectoryConfiguration
     :cognitect.aws.fsx.UpdateFileSystemWindowsConfiguration/AuditLogConfiguration
     :cognitect.aws.fsx.UpdateFileSystemWindowsConfiguration/AutomaticBackupRetentionDays]))

(s/def :cognitect.aws.fsx/EventTypes (s/coll-of :cognitect.aws.fsx/EventType :max-count 3))

(s/def
  :cognitect.aws.fsx/ActiveDirectoryBackupAttributes
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.ActiveDirectoryBackupAttributes/ActiveDirectoryId
     :cognitect.aws.fsx.ActiveDirectoryBackupAttributes/DomainName
     :cognitect.aws.fsx.ActiveDirectoryBackupAttributes/ResourceARN]))

(s/def
  :cognitect.aws.fsx/LustreLogCreateConfiguration
  (s/keys
    :req-un
    [:cognitect.aws.fsx.LustreLogCreateConfiguration/Level]
    :opt-un
    [:cognitect.aws.fsx.LustreLogCreateConfiguration/Destination]))

(s/def :cognitect.aws.fsx/LastUpdatedTime inst?)

(s/def
  :cognitect.aws.fsx/OpenZFSClientConfiguration
  (s/keys
    :req-un
    [:cognitect.aws.fsx.OpenZFSClientConfiguration/Clients
     :cognitect.aws.fsx.OpenZFSClientConfiguration/Options]))

(s/def
  :cognitect.aws.fsx/VolumeName
  (s/spec
    #(re-matches (re-pattern "^[^\\u0000\\u0085\\u2028\\u2029\\r\\n]{1,203}$") %)
    :gen
    #(gen/string)))

(s/def :cognitect.aws.fsx/OntapVolumeType (s/spec string? :gen #(s/gen #{"DP" "LS" "RW"})))

(s/def
  :cognitect.aws.fsx/CreateSnapshotRequest
  (s/keys
    :req-un
    [:cognitect.aws.fsx.CreateSnapshotRequest/Name
     :cognitect.aws.fsx.CreateSnapshotRequest/VolumeId]
    :opt-un
    [:cognitect.aws.fsx.CreateSnapshotRequest/ClientRequestToken
     :cognitect.aws.fsx.CreateSnapshotRequest/Tags]))

(s/def
  :cognitect.aws.fsx/DirectoryId
  (s/spec #(re-matches (re-pattern "^d-[0-9a-f]{10}$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.fsx/StorageVirtualMachineIds
  (s/coll-of :cognitect.aws.fsx/StorageVirtualMachineId :max-count 50))

(s/def
  :cognitect.aws.fsx/DescribeFileSystemAliasesResponse
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.DescribeFileSystemAliasesResponse/Aliases
     :cognitect.aws.fsx.DescribeFileSystemAliasesResponse/NextToken]))

(s/def
  :cognitect.aws.fsx/LustreDeploymentType
  (s/spec string? :gen #(s/gen #{"PERSISTENT_1" "SCRATCH_2" "PERSISTENT_2" "SCRATCH_1"})))

(s/def
  :cognitect.aws.fsx/RouteTableId
  (s/spec #(re-matches (re-pattern "^(rtb-[0-9a-f]{8,})$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.fsx/DescribeStorageVirtualMachinesResponse
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.DescribeStorageVirtualMachinesResponse/NextToken
     :cognitect.aws.fsx.DescribeStorageVirtualMachinesResponse/StorageVirtualMachines]))

(s/def
  :cognitect.aws.fsx/OntapVolumeConfiguration
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.OntapVolumeConfiguration/FlexCacheEndpointType
     :cognitect.aws.fsx.OntapVolumeConfiguration/SizeInMegabytes
     :cognitect.aws.fsx.OntapVolumeConfiguration/SecurityStyle
     :cognitect.aws.fsx.OntapVolumeConfiguration/OntapVolumeType
     :cognitect.aws.fsx.OntapVolumeConfiguration/JunctionPath
     :cognitect.aws.fsx.OntapVolumeConfiguration/StorageVirtualMachineRoot
     :cognitect.aws.fsx.OntapVolumeConfiguration/TieringPolicy
     :cognitect.aws.fsx.OntapVolumeConfiguration/StorageEfficiencyEnabled
     :cognitect.aws.fsx.OntapVolumeConfiguration/StorageVirtualMachineId
     :cognitect.aws.fsx.OntapVolumeConfiguration/UUID]))

(s/def :cognitect.aws.fsx/Tags (s/coll-of :cognitect.aws.fsx/Tag :min-count 1 :max-count 50))

(s/def
  :cognitect.aws.fsx/OpenZFSUserOrGroupQuota
  (s/keys
    :req-un
    [:cognitect.aws.fsx.OpenZFSUserOrGroupQuota/Type
     :cognitect.aws.fsx.OpenZFSUserOrGroupQuota/Id
     :cognitect.aws.fsx.OpenZFSUserOrGroupQuota/StorageCapacityQuotaGiB]))

(s/def
  :cognitect.aws.fsx/DisassociateFileSystemAliasesResponse
  (s/keys :opt-un [:cognitect.aws.fsx.DisassociateFileSystemAliasesResponse/Aliases]))

(s/def :cognitect.aws.fsx/ReadOnly boolean?)

(s/def
  :cognitect.aws.fsx/UpdateFileSystemOpenZFSConfiguration
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.UpdateFileSystemOpenZFSConfiguration/DiskIopsConfiguration
     :cognitect.aws.fsx.UpdateFileSystemOpenZFSConfiguration/CopyTagsToBackups
     :cognitect.aws.fsx.UpdateFileSystemOpenZFSConfiguration/ThroughputCapacity
     :cognitect.aws.fsx.UpdateFileSystemOpenZFSConfiguration/DailyAutomaticBackupStartTime
     :cognitect.aws.fsx.UpdateFileSystemOpenZFSConfiguration/CopyTagsToVolumes
     :cognitect.aws.fsx.UpdateFileSystemOpenZFSConfiguration/WeeklyMaintenanceStartTime
     :cognitect.aws.fsx.UpdateFileSystemOpenZFSConfiguration/AutomaticBackupRetentionDays]))

(s/def
  :cognitect.aws.fsx/AssociateFileSystemAliasesResponse
  (s/keys :opt-un [:cognitect.aws.fsx.AssociateFileSystemAliasesResponse/Aliases]))

(s/def
  :cognitect.aws.fsx/StorageVirtualMachine
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.StorageVirtualMachine/Endpoints
     :cognitect.aws.fsx.StorageVirtualMachine/Tags
     :cognitect.aws.fsx.StorageVirtualMachine/RootVolumeSecurityStyle
     :cognitect.aws.fsx.StorageVirtualMachine/FileSystemId
     :cognitect.aws.fsx.StorageVirtualMachine/LifecycleTransitionReason
     :cognitect.aws.fsx.StorageVirtualMachine/CreationTime
     :cognitect.aws.fsx.StorageVirtualMachine/ActiveDirectoryConfiguration
     :cognitect.aws.fsx.StorageVirtualMachine/Subtype
     :cognitect.aws.fsx.StorageVirtualMachine/Lifecycle
     :cognitect.aws.fsx.StorageVirtualMachine/Name
     :cognitect.aws.fsx.StorageVirtualMachine/ResourceARN
     :cognitect.aws.fsx.StorageVirtualMachine/StorageVirtualMachineId
     :cognitect.aws.fsx.StorageVirtualMachine/UUID]))

(s/def
  :cognitect.aws.fsx/WindowsAuditLogConfiguration
  (s/keys
    :req-un
    [:cognitect.aws.fsx.WindowsAuditLogConfiguration/FileAccessAuditLogLevel
     :cognitect.aws.fsx.WindowsAuditLogConfiguration/FileShareAccessAuditLogLevel]
    :opt-un
    [:cognitect.aws.fsx.WindowsAuditLogConfiguration/AuditLogDestination]))

(s/def
  :cognitect.aws.fsx/CopyBackupResponse
  (s/keys :opt-un [:cognitect.aws.fsx.CopyBackupResponse/Backup]))

(s/def
  :cognitect.aws.fsx/DeleteFileSystemOpenZFSConfiguration
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.DeleteFileSystemOpenZFSConfiguration/SkipFinalBackup
     :cognitect.aws.fsx.DeleteFileSystemOpenZFSConfiguration/FinalBackupTags
     :cognitect.aws.fsx.DeleteFileSystemOpenZFSConfiguration/Options]))

(s/def
  :cognitect.aws.fsx/SnapshotFilterName
  (s/spec string? :gen #(s/gen #{"volume-id" "file-system-id"})))

(s/def
  :cognitect.aws.fsx/AdministrativeAction
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.AdministrativeAction/RequestTime
     :cognitect.aws.fsx.AdministrativeAction/Status
     :cognitect.aws.fsx.AdministrativeAction/TargetSnapshotValues
     :cognitect.aws.fsx.AdministrativeAction/TargetVolumeValues
     :cognitect.aws.fsx.AdministrativeAction/FailureDetails
     :cognitect.aws.fsx.AdministrativeAction/ProgressPercent
     :cognitect.aws.fsx.AdministrativeAction/AdministrativeActionType
     :cognitect.aws.fsx.AdministrativeAction/TargetFileSystemValues]))

(s/def
  :cognitect.aws.fsx/DescribeDataRepositoryAssociationsRequest
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.DescribeDataRepositoryAssociationsRequest/NextToken
     :cognitect.aws.fsx.DescribeDataRepositoryAssociationsRequest/Filters
     :cognitect.aws.fsx.DescribeDataRepositoryAssociationsRequest/MaxResults
     :cognitect.aws.fsx.DescribeDataRepositoryAssociationsRequest/AssociationIds]))

(s/def
  :cognitect.aws.fsx/Backup
  (s/keys
    :req-un
    [:cognitect.aws.fsx.Backup/BackupId
     :cognitect.aws.fsx.Backup/Lifecycle
     :cognitect.aws.fsx.Backup/Type
     :cognitect.aws.fsx.Backup/CreationTime
     :cognitect.aws.fsx.Backup/FileSystem]
    :opt-un
    [:cognitect.aws.fsx.Backup/Volume
     :cognitect.aws.fsx.Backup/OwnerId
     :cognitect.aws.fsx.Backup/SourceBackupRegion
     :cognitect.aws.fsx.Backup/Tags
     :cognitect.aws.fsx.Backup/DirectoryInformation
     :cognitect.aws.fsx.Backup/FailureDetails
     :cognitect.aws.fsx.Backup/KmsKeyId
     :cognitect.aws.fsx.Backup/ResourceType
     :cognitect.aws.fsx.Backup/ProgressPercent
     :cognitect.aws.fsx.Backup/SourceBackupId
     :cognitect.aws.fsx.Backup/ResourceARN]))

(s/def
  :cognitect.aws.fsx/DeleteVolumeOntapResponse
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.DeleteVolumeOntapResponse/FinalBackupId
     :cognitect.aws.fsx.DeleteVolumeOntapResponse/FinalBackupTags]))

(s/def :cognitect.aws.fsx/DeleteDataInFileSystem boolean?)

(s/def
  :cognitect.aws.fsx/DataRepositoryAssociationIds
  (s/coll-of :cognitect.aws.fsx/DataRepositoryAssociationId :max-count 50))

(s/def
  :cognitect.aws.fsx/DeleteStorageVirtualMachineResponse
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.DeleteStorageVirtualMachineResponse/Lifecycle
     :cognitect.aws.fsx.DeleteStorageVirtualMachineResponse/StorageVirtualMachineId]))

(s/def
  :cognitect.aws.fsx/CreateFileSystemFromBackupRequest
  (s/keys
    :req-un
    [:cognitect.aws.fsx.CreateFileSystemFromBackupRequest/BackupId
     :cognitect.aws.fsx.CreateFileSystemFromBackupRequest/SubnetIds]
    :opt-un
    [:cognitect.aws.fsx.CreateFileSystemFromBackupRequest/ClientRequestToken
     :cognitect.aws.fsx.CreateFileSystemFromBackupRequest/SecurityGroupIds
     :cognitect.aws.fsx.CreateFileSystemFromBackupRequest/FileSystemTypeVersion
     :cognitect.aws.fsx.CreateFileSystemFromBackupRequest/Tags
     :cognitect.aws.fsx.CreateFileSystemFromBackupRequest/WindowsConfiguration
     :cognitect.aws.fsx.CreateFileSystemFromBackupRequest/KmsKeyId
     :cognitect.aws.fsx.CreateFileSystemFromBackupRequest/StorageType
     :cognitect.aws.fsx.CreateFileSystemFromBackupRequest/LustreConfiguration
     :cognitect.aws.fsx.CreateFileSystemFromBackupRequest/OpenZFSConfiguration]))

(s/def :cognitect.aws.fsx/DnsIps (s/coll-of :cognitect.aws.fsx/IpAddress :min-count 1 :max-count 3))

(s/def
  :cognitect.aws.fsx/WindowsAccessAuditLogLevel
  (s/spec string? :gen #(s/gen #{"SUCCESS_AND_FAILURE" "DISABLED" "SUCCESS_ONLY" "FAILURE_ONLY"})))

(s/def
  :cognitect.aws.fsx/CreateDataRepositoryTaskRequest
  (s/keys
    :req-un
    [:cognitect.aws.fsx.CreateDataRepositoryTaskRequest/Type
     :cognitect.aws.fsx.CreateDataRepositoryTaskRequest/FileSystemId
     :cognitect.aws.fsx.CreateDataRepositoryTaskRequest/Report]
    :opt-un
    [:cognitect.aws.fsx.CreateDataRepositoryTaskRequest/ClientRequestToken
     :cognitect.aws.fsx.CreateDataRepositoryTaskRequest/Tags
     :cognitect.aws.fsx.CreateDataRepositoryTaskRequest/Paths]))

(s/def
  :cognitect.aws.fsx/SnapshotFilter
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.SnapshotFilter/Values :cognitect.aws.fsx.SnapshotFilter/Name]))

(s/def
  :cognitect.aws.fsx/FileSystemFailureDetails
  (s/keys :opt-un [:cognitect.aws.fsx.FileSystemFailureDetails/Message]))

(s/def
  :cognitect.aws.fsx/CreateFileSystemWindowsConfiguration
  (s/keys
    :req-un
    [:cognitect.aws.fsx.CreateFileSystemWindowsConfiguration/ThroughputCapacity]
    :opt-un
    [:cognitect.aws.fsx.CreateFileSystemWindowsConfiguration/CopyTagsToBackups
     :cognitect.aws.fsx.CreateFileSystemWindowsConfiguration/DeploymentType
     :cognitect.aws.fsx.CreateFileSystemWindowsConfiguration/Aliases
     :cognitect.aws.fsx.CreateFileSystemWindowsConfiguration/ActiveDirectoryId
     :cognitect.aws.fsx.CreateFileSystemWindowsConfiguration/DailyAutomaticBackupStartTime
     :cognitect.aws.fsx.CreateFileSystemWindowsConfiguration/WeeklyMaintenanceStartTime
     :cognitect.aws.fsx.CreateFileSystemWindowsConfiguration/SelfManagedActiveDirectoryConfiguration
     :cognitect.aws.fsx.CreateFileSystemWindowsConfiguration/AuditLogConfiguration
     :cognitect.aws.fsx.CreateFileSystemWindowsConfiguration/AutomaticBackupRetentionDays
     :cognitect.aws.fsx.CreateFileSystemWindowsConfiguration/PreferredSubnetId]))

(s/def
  :cognitect.aws.fsx/UpdateFileSystemResponse
  (s/keys :opt-un [:cognitect.aws.fsx.UpdateFileSystemResponse/FileSystem]))

(s/def :cognitect.aws.fsx/Aliases (s/coll-of :cognitect.aws.fsx/Alias :max-count 50))

(s/def
  :cognitect.aws.fsx/WeeklyTime
  (s/spec #(re-matches (re-pattern "^[1-7]:([01]\\d|2[0-3]):?([0-5]\\d)$") %) :gen #(gen/string)))

(s/def :cognitect.aws.fsx/VolumeType (s/spec string? :gen #(s/gen #{"OPENZFS" "ONTAP"})))

(s/def
  :cognitect.aws.fsx/Snapshot
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.Snapshot/AdministrativeActions
     :cognitect.aws.fsx.Snapshot/Tags
     :cognitect.aws.fsx.Snapshot/LifecycleTransitionReason
     :cognitect.aws.fsx.Snapshot/CreationTime
     :cognitect.aws.fsx.Snapshot/VolumeId
     :cognitect.aws.fsx.Snapshot/Lifecycle
     :cognitect.aws.fsx.Snapshot/Name
     :cognitect.aws.fsx.Snapshot/ResourceARN
     :cognitect.aws.fsx.Snapshot/SnapshotId]))

(s/def
  :cognitect.aws.fsx/OpenZFSOriginSnapshotConfiguration
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.OpenZFSOriginSnapshotConfiguration/SnapshotARN
     :cognitect.aws.fsx.OpenZFSOriginSnapshotConfiguration/CopyStrategy]))

(s/def
  :cognitect.aws.fsx/DeleteBackupRequest
  (s/keys
    :req-un
    [:cognitect.aws.fsx.DeleteBackupRequest/BackupId]
    :opt-un
    [:cognitect.aws.fsx.DeleteBackupRequest/ClientRequestToken]))

(s/def
  :cognitect.aws.fsx/WindowsAuditLogCreateConfiguration
  (s/keys
    :req-un
    [:cognitect.aws.fsx.WindowsAuditLogCreateConfiguration/FileAccessAuditLogLevel
     :cognitect.aws.fsx.WindowsAuditLogCreateConfiguration/FileShareAccessAuditLogLevel]
    :opt-un
    [:cognitect.aws.fsx.WindowsAuditLogCreateConfiguration/AuditLogDestination]))

(s/def
  :cognitect.aws.fsx/UpdateDataRepositoryAssociationResponse
  (s/keys :opt-un [:cognitect.aws.fsx.UpdateDataRepositoryAssociationResponse/Association]))

(s/def
  :cognitect.aws.fsx/OpenZFSVolumeConfiguration
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.OpenZFSVolumeConfiguration/ParentVolumeId
     :cognitect.aws.fsx.OpenZFSVolumeConfiguration/VolumePath
     :cognitect.aws.fsx.OpenZFSVolumeConfiguration/DataCompressionType
     :cognitect.aws.fsx.OpenZFSVolumeConfiguration/OriginSnapshot
     :cognitect.aws.fsx.OpenZFSVolumeConfiguration/StorageCapacityQuotaGiB
     :cognitect.aws.fsx.OpenZFSVolumeConfiguration/ReadOnly
     :cognitect.aws.fsx.OpenZFSVolumeConfiguration/CopyTagsToSnapshots
     :cognitect.aws.fsx.OpenZFSVolumeConfiguration/NfsExports
     :cognitect.aws.fsx.OpenZFSVolumeConfiguration/StorageCapacityReservationGiB
     :cognitect.aws.fsx.OpenZFSVolumeConfiguration/RecordSizeKiB
     :cognitect.aws.fsx.OpenZFSVolumeConfiguration/UserAndGroupQuotas]))

(s/def
  :cognitect.aws.fsx/StorageVirtualMachineId
  (s/spec #(re-matches (re-pattern "^(svm-[0-9a-f]{17,})$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.fsx/DeleteStorageVirtualMachineRequest
  (s/keys
    :req-un
    [:cognitect.aws.fsx.DeleteStorageVirtualMachineRequest/StorageVirtualMachineId]
    :opt-un
    [:cognitect.aws.fsx.DeleteStorageVirtualMachineRequest/ClientRequestToken]))

(s/def
  :cognitect.aws.fsx/DeleteFileSystemLustreResponse
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.DeleteFileSystemLustreResponse/FinalBackupId
     :cognitect.aws.fsx.DeleteFileSystemLustreResponse/FinalBackupTags]))

(s/def :cognitect.aws.fsx/DriveCacheType (s/spec string? :gen #(s/gen #{"NONE" "READ"})))

(s/def
  :cognitect.aws.fsx/CancelDataRepositoryTaskResponse
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.CancelDataRepositoryTaskResponse/TaskId
     :cognitect.aws.fsx.CancelDataRepositoryTaskResponse/Lifecycle]))

(s/def
  :cognitect.aws.fsx/DeleteSnapshotRequest
  (s/keys
    :req-un
    [:cognitect.aws.fsx.DeleteSnapshotRequest/SnapshotId]
    :opt-un
    [:cognitect.aws.fsx.DeleteSnapshotRequest/ClientRequestToken]))

(s/def :cognitect.aws.fsx/FileSystems (s/coll-of :cognitect.aws.fsx/FileSystem :max-count 50))

(s/def
  :cognitect.aws.fsx/LimitedMaxResults
  (s/spec (s/and int? #(<= 1 % 25)) :gen #(gen/choose 1 25)))

(s/def
  :cognitect.aws.fsx/MegabytesPerSecond
  (s/spec (s/and int? #(<= 8 % 4096)) :gen #(gen/choose 8 4096)))

(s/def
  :cognitect.aws.fsx/DeleteFileSystemOpenZFSResponse
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.DeleteFileSystemOpenZFSResponse/FinalBackupId
     :cognitect.aws.fsx.DeleteFileSystemOpenZFSResponse/FinalBackupTags]))

(s/def
  :cognitect.aws.fsx/CreateOpenZFSOriginSnapshotConfiguration
  (s/keys
    :req-un
    [:cognitect.aws.fsx.CreateOpenZFSOriginSnapshotConfiguration/SnapshotARN
     :cognitect.aws.fsx.CreateOpenZFSOriginSnapshotConfiguration/CopyStrategy]))

(s/def
  :cognitect.aws.fsx/FileSystemMaintenanceOperations
  (s/coll-of :cognitect.aws.fsx/FileSystemMaintenanceOperation :max-count 20))

(s/def
  :cognitect.aws.fsx/FileSystemTypeVersion
  (s/spec #(re-matches (re-pattern "^[0-9](.[0-9]*)*$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.fsx/RestoreOpenZFSVolumeOptions
  (s/coll-of :cognitect.aws.fsx/RestoreOpenZFSVolumeOption :max-count 2))

(s/def
  :cognitect.aws.fsx/NextToken
  (s/spec
    #(re-matches
      (re-pattern "^(?:[A-Za-z0-9+\\/]{4})*(?:[A-Za-z0-9+\\/]{2}==|[A-Za-z0-9+\\/]{3}=)?$")
      %)
    :gen
    #(gen/string)))

(s/def
  :cognitect.aws.fsx/StorageVirtualMachineFilters
  (s/coll-of :cognitect.aws.fsx/StorageVirtualMachineFilter :max-count 1))

(s/def
  :cognitect.aws.fsx/CreateDataRepositoryAssociationRequest
  (s/keys
    :req-un
    [:cognitect.aws.fsx.CreateDataRepositoryAssociationRequest/FileSystemId
     :cognitect.aws.fsx.CreateDataRepositoryAssociationRequest/FileSystemPath
     :cognitect.aws.fsx.CreateDataRepositoryAssociationRequest/DataRepositoryPath]
    :opt-un
    [:cognitect.aws.fsx.CreateDataRepositoryAssociationRequest/ClientRequestToken
     :cognitect.aws.fsx.CreateDataRepositoryAssociationRequest/Tags
     :cognitect.aws.fsx.CreateDataRepositoryAssociationRequest/BatchImportMetaDataOnCreate
     :cognitect.aws.fsx.CreateDataRepositoryAssociationRequest/S3
     :cognitect.aws.fsx.CreateDataRepositoryAssociationRequest/ImportedFileChunkSize]))

(s/def
  :cognitect.aws.fsx/DataRepositoryAssociationId
  (s/spec #(re-matches (re-pattern "^(dra-[0-9a-f]{8,})$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.fsx/DeleteFileSystemResponse
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.DeleteFileSystemResponse/OpenZFSResponse
     :cognitect.aws.fsx.DeleteFileSystemResponse/LustreResponse
     :cognitect.aws.fsx.DeleteFileSystemResponse/FileSystemId
     :cognitect.aws.fsx.DeleteFileSystemResponse/WindowsResponse
     :cognitect.aws.fsx.DeleteFileSystemResponse/Lifecycle]))

(s/def
  :cognitect.aws.fsx/DeleteFileSystemOpenZFSOptions
  (s/coll-of :cognitect.aws.fsx/DeleteFileSystemOpenZFSOption :max-count 1))

(s/def
  :cognitect.aws.fsx/CreateFileSystemOntapConfiguration
  (s/keys
    :req-un
    [:cognitect.aws.fsx.CreateFileSystemOntapConfiguration/DeploymentType
     :cognitect.aws.fsx.CreateFileSystemOntapConfiguration/ThroughputCapacity]
    :opt-un
    [:cognitect.aws.fsx.CreateFileSystemOntapConfiguration/DiskIopsConfiguration
     :cognitect.aws.fsx.CreateFileSystemOntapConfiguration/FsxAdminPassword
     :cognitect.aws.fsx.CreateFileSystemOntapConfiguration/DailyAutomaticBackupStartTime
     :cognitect.aws.fsx.CreateFileSystemOntapConfiguration/WeeklyMaintenanceStartTime
     :cognitect.aws.fsx.CreateFileSystemOntapConfiguration/AutomaticBackupRetentionDays
     :cognitect.aws.fsx.CreateFileSystemOntapConfiguration/PreferredSubnetId
     :cognitect.aws.fsx.CreateFileSystemOntapConfiguration/EndpointIpAddressRange
     :cognitect.aws.fsx.CreateFileSystemOntapConfiguration/RouteTableIds]))

(s/def
  :cognitect.aws.fsx/CreateVolumeFromBackupResponse
  (s/keys :opt-un [:cognitect.aws.fsx.CreateVolumeFromBackupResponse/Volume]))

(s/def
  :cognitect.aws.fsx/CreateStorageVirtualMachineRequest
  (s/keys
    :req-un
    [:cognitect.aws.fsx.CreateStorageVirtualMachineRequest/FileSystemId
     :cognitect.aws.fsx.CreateStorageVirtualMachineRequest/Name]
    :opt-un
    [:cognitect.aws.fsx.CreateStorageVirtualMachineRequest/ClientRequestToken
     :cognitect.aws.fsx.CreateStorageVirtualMachineRequest/Tags
     :cognitect.aws.fsx.CreateStorageVirtualMachineRequest/RootVolumeSecurityStyle
     :cognitect.aws.fsx.CreateStorageVirtualMachineRequest/ActiveDirectoryConfiguration
     :cognitect.aws.fsx.CreateStorageVirtualMachineRequest/SvmAdminPassword]))

(s/def
  :cognitect.aws.fsx/UpdateFileSystemRequest
  (s/keys
    :req-un
    [:cognitect.aws.fsx.UpdateFileSystemRequest/FileSystemId]
    :opt-un
    [:cognitect.aws.fsx.UpdateFileSystemRequest/ClientRequestToken
     :cognitect.aws.fsx.UpdateFileSystemRequest/OntapConfiguration
     :cognitect.aws.fsx.UpdateFileSystemRequest/WindowsConfiguration
     :cognitect.aws.fsx.UpdateFileSystemRequest/StorageCapacity
     :cognitect.aws.fsx.UpdateFileSystemRequest/LustreConfiguration
     :cognitect.aws.fsx.UpdateFileSystemRequest/OpenZFSConfiguration]))

(s/def
  :cognitect.aws.fsx/DirectoryUserName
  (s/spec
    #(re-matches (re-pattern "^[^\\u0000\\u0085\\u2028\\u2029\\r\\n]{1,256}$") %)
    :gen
    #(gen/string)))

(s/def
  :cognitect.aws.fsx/GeneralARN
  (s/spec
    #(re-matches
      (re-pattern "^arn:[^:]{1,63}:[^:]{0,63}:[^:]{0,63}:(?:|\\d{12}):[^/].{0,1023}$")
      %)
    :gen
    #(gen/string)))

(s/def
  :cognitect.aws.fsx/TagValue
  (s/spec #(re-matches (re-pattern "^([\\p{L}\\p{Z}\\p{N}_.:/=+\\-@]*)$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.fsx/BackupFailureDetails
  (s/keys :opt-un [:cognitect.aws.fsx.BackupFailureDetails/Message]))

(s/def
  :cognitect.aws.fsx/IntegerNoMaxFromNegativeOne
  (s/spec (s/and int? #(<= -1 % 2147483647)) :gen #(gen/choose -1 2147483647)))

(s/def
  :cognitect.aws.fsx/DeleteFileSystemWindowsResponse
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.DeleteFileSystemWindowsResponse/FinalBackupId
     :cognitect.aws.fsx.DeleteFileSystemWindowsResponse/FinalBackupTags]))

(s/def
  :cognitect.aws.fsx/VolumeFilterName
  (s/spec string? :gen #(s/gen #{"file-system-id" "storage-virtual-machine-id"})))

(s/def
  :cognitect.aws.fsx/OntapFileSystemConfiguration
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.OntapFileSystemConfiguration/DiskIopsConfiguration
     :cognitect.aws.fsx.OntapFileSystemConfiguration/DeploymentType
     :cognitect.aws.fsx.OntapFileSystemConfiguration/Endpoints
     :cognitect.aws.fsx.OntapFileSystemConfiguration/ThroughputCapacity
     :cognitect.aws.fsx.OntapFileSystemConfiguration/DailyAutomaticBackupStartTime
     :cognitect.aws.fsx.OntapFileSystemConfiguration/WeeklyMaintenanceStartTime
     :cognitect.aws.fsx.OntapFileSystemConfiguration/AutomaticBackupRetentionDays
     :cognitect.aws.fsx.OntapFileSystemConfiguration/PreferredSubnetId
     :cognitect.aws.fsx.OntapFileSystemConfiguration/EndpointIpAddressRange
     :cognitect.aws.fsx.OntapFileSystemConfiguration/RouteTableIds]))

(s/def
  :cognitect.aws.fsx/DataRepositoryConfiguration
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.DataRepositoryConfiguration/ExportPath
     :cognitect.aws.fsx.DataRepositoryConfiguration/ImportPath
     :cognitect.aws.fsx.DataRepositoryConfiguration/AutoImportPolicy
     :cognitect.aws.fsx.DataRepositoryConfiguration/FailureDetails
     :cognitect.aws.fsx.DataRepositoryConfiguration/ImportedFileChunkSize
     :cognitect.aws.fsx.DataRepositoryConfiguration/Lifecycle]))

(s/def
  :cognitect.aws.fsx/Volume
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.Volume/AdministrativeActions
     :cognitect.aws.fsx.Volume/Tags
     :cognitect.aws.fsx.Volume/OntapConfiguration
     :cognitect.aws.fsx.Volume/FileSystemId
     :cognitect.aws.fsx.Volume/LifecycleTransitionReason
     :cognitect.aws.fsx.Volume/CreationTime
     :cognitect.aws.fsx.Volume/VolumeType
     :cognitect.aws.fsx.Volume/VolumeId
     :cognitect.aws.fsx.Volume/Lifecycle
     :cognitect.aws.fsx.Volume/Name
     :cognitect.aws.fsx.Volume/OpenZFSConfiguration
     :cognitect.aws.fsx.Volume/ResourceARN]))

(s/def
  :cognitect.aws.fsx/StorageVirtualMachineFilterName
  (s/spec string? :gen #(s/gen #{"file-system-id"})))

(s/def
  :cognitect.aws.fsx/DeleteVolumeResponse
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.DeleteVolumeResponse/OntapResponse
     :cognitect.aws.fsx.DeleteVolumeResponse/VolumeId
     :cognitect.aws.fsx.DeleteVolumeResponse/Lifecycle]))

(s/def
  :cognitect.aws.fsx/ClientRequestToken
  (s/spec #(re-matches (re-pattern "[A-za-z0-9_.-]{0,63}$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.fsx/LustreFileSystemMountName
  (s/spec #(re-matches (re-pattern "^([A-Za-z0-9_-]{1,8})$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.fsx/SnapshotFilters
  (s/coll-of :cognitect.aws.fsx/SnapshotFilter :max-count 2))

(s/def
  :cognitect.aws.fsx/ListTagsForResourceRequest
  (s/keys
    :req-un
    [:cognitect.aws.fsx.ListTagsForResourceRequest/ResourceARN]
    :opt-un
    [:cognitect.aws.fsx.ListTagsForResourceRequest/NextToken
     :cognitect.aws.fsx.ListTagsForResourceRequest/MaxResults]))

(s/def
  :cognitect.aws.fsx/SvmEndpoint
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.SvmEndpoint/DNSName :cognitect.aws.fsx.SvmEndpoint/IpAddresses]))

(s/def
  :cognitect.aws.fsx/DataRepositoryTaskPath
  (s/spec
    #(re-matches (re-pattern "^[^\\u0000\\u0085\\u2028\\u2029\\r\\n]{0,4096}$") %)
    :gen
    #(gen/string)))

(s/def
  :cognitect.aws.fsx/FileSystemEndpoint
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.FileSystemEndpoint/DNSName
     :cognitect.aws.fsx.FileSystemEndpoint/IpAddresses]))

(s/def
  :cognitect.aws.fsx/DataRepositoryTaskFilterValue
  (s/spec #(re-matches (re-pattern "^[0-9a-zA-Z\\*\\.\\\\/\\?\\-\\_]*$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.fsx/OpenZFSNfsExport
  (s/keys :req-un [:cognitect.aws.fsx.OpenZFSNfsExport/ClientConfigurations]))

(s/def
  :cognitect.aws.fsx/SnapshotName
  (s/spec #(re-matches (re-pattern "^[a-zA-Z0-9_:.-]{1,203}$") %) :gen #(gen/string)))

(s/def :cognitect.aws.fsx/Flag boolean?)

(s/def
  :cognitect.aws.fsx/DiskIopsConfigurationMode
  (s/spec string? :gen #(s/gen #{"USER_PROVISIONED" "AUTOMATIC"})))

(s/def :cognitect.aws.fsx/UntagResourceResponse (s/keys))

(s/def
  :cognitect.aws.fsx/ReleaseFileSystemNfsV3LocksResponse
  (s/keys :opt-un [:cognitect.aws.fsx.ReleaseFileSystemNfsV3LocksResponse/FileSystem]))

(s/def
  :cognitect.aws.fsx/FileSystemType
  (s/spec string? :gen #(s/gen #{"OPENZFS" "WINDOWS" "LUSTRE" "ONTAP"})))

(s/def
  :cognitect.aws.fsx/SecurityGroupId
  (s/spec #(re-matches (re-pattern "^(sg-[0-9a-f]{8,})$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.fsx/DeleteOpenZFSVolumeOptions
  (s/coll-of :cognitect.aws.fsx/DeleteOpenZFSVolumeOption :max-count 1))

(s/def
  :cognitect.aws.fsx/UpdateOpenZFSVolumeConfiguration
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.UpdateOpenZFSVolumeConfiguration/DataCompressionType
     :cognitect.aws.fsx.UpdateOpenZFSVolumeConfiguration/StorageCapacityQuotaGiB
     :cognitect.aws.fsx.UpdateOpenZFSVolumeConfiguration/ReadOnly
     :cognitect.aws.fsx.UpdateOpenZFSVolumeConfiguration/NfsExports
     :cognitect.aws.fsx.UpdateOpenZFSVolumeConfiguration/StorageCapacityReservationGiB
     :cognitect.aws.fsx.UpdateOpenZFSVolumeConfiguration/RecordSizeKiB
     :cognitect.aws.fsx.UpdateOpenZFSVolumeConfiguration/UserAndGroupQuotas]))

(s/def :cognitect.aws.fsx/Backups (s/coll-of :cognitect.aws.fsx/Backup :max-count 50))

(s/def
  :cognitect.aws.fsx/DescribeDataRepositoryTasksResponse
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.DescribeDataRepositoryTasksResponse/DataRepositoryTasks
     :cognitect.aws.fsx.DescribeDataRepositoryTasksResponse/NextToken]))

(s/def
  :cognitect.aws.fsx/DescribeDataRepositoryAssociationsResponse
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.DescribeDataRepositoryAssociationsResponse/NextToken
     :cognitect.aws.fsx.DescribeDataRepositoryAssociationsResponse/Associations]))

(s/def
  :cognitect.aws.fsx/OpenZFSNfsExportOption
  (s/spec #(re-matches (re-pattern "^[ -~]{1,128}$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.fsx/SnapshotLifecycle
  (s/spec string? :gen #(s/gen #{"AVAILABLE" "PENDING" "DELETING" "CREATING"})))

(s/def
  :cognitect.aws.fsx/StorageVirtualMachineRootVolumeSecurityStyle
  (s/spec string? :gen #(s/gen #{"MIXED" "UNIX" "NTFS"})))

(s/def
  :cognitect.aws.fsx/LustreAccessAuditLogLevel
  (s/spec string? :gen #(s/gen #{"DISABLED" "ERROR_ONLY" "WARN_ERROR" "WARN_ONLY"})))

(s/def :cognitect.aws.fsx/StartTime inst?)

(s/def
  :cognitect.aws.fsx/AdministrativeActions
  (s/coll-of :cognitect.aws.fsx/AdministrativeAction :max-count 50))

(s/def
  :cognitect.aws.fsx/FailedCount
  (s/spec int? :gen #(gen/choose Long/MIN_VALUE Long/MAX_VALUE)))

(s/def :cognitect.aws.fsx/OntapDeploymentType (s/spec string? :gen #(s/gen #{"MULTI_AZ_1"})))

(s/def
  :cognitect.aws.fsx/CreateDataRepositoryAssociationResponse
  (s/keys :opt-un [:cognitect.aws.fsx.CreateDataRepositoryAssociationResponse/Association]))

(s/def
  :cognitect.aws.fsx/SubnetId
  (s/spec #(re-matches (re-pattern "^(subnet-[0-9a-f]{8,})$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.fsx/ResourceARN
  (s/spec
    #(re-matches
      (re-pattern
        "^arn:(?=[^:]+:fsx:[^:]+:\\d{12}:)((|(?=[a-z0-9-.]{1,63})(?!\\d{1,3}(\\.\\d{1,3}){3})(?![^:]*-{2})(?![^:]*-\\.)(?![^:]*\\.-)[a-z0-9].*(?<!-)):){4}(?!/).{0,1024}$")
      %)
    :gen
    #(gen/string)))

(s/def
  :cognitect.aws.fsx/Region
  (s/spec #(re-matches (re-pattern "^[a-z0-9-]{1,20}$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.fsx/AliasLifecycle
  (s/spec
    string?
    :gen
    #(s/gen #{"AVAILABLE" "CREATE_FAILED" "DELETING" "CREATING" "DELETE_FAILED"})))

(s/def
  :cognitect.aws.fsx/KmsKeyId
  (s/spec #(re-matches (re-pattern "^.{1,2048}$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.fsx/BackupType
  (s/spec string? :gen #(s/gen #{"AWS_BACKUP" "AUTOMATIC" "USER_INITIATED"})))

(s/def
  :cognitect.aws.fsx/CreateFileSystemOpenZFSConfiguration
  (s/keys
    :req-un
    [:cognitect.aws.fsx.CreateFileSystemOpenZFSConfiguration/DeploymentType
     :cognitect.aws.fsx.CreateFileSystemOpenZFSConfiguration/ThroughputCapacity]
    :opt-un
    [:cognitect.aws.fsx.CreateFileSystemOpenZFSConfiguration/DiskIopsConfiguration
     :cognitect.aws.fsx.CreateFileSystemOpenZFSConfiguration/CopyTagsToBackups
     :cognitect.aws.fsx.CreateFileSystemOpenZFSConfiguration/DailyAutomaticBackupStartTime
     :cognitect.aws.fsx.CreateFileSystemOpenZFSConfiguration/CopyTagsToVolumes
     :cognitect.aws.fsx.CreateFileSystemOpenZFSConfiguration/WeeklyMaintenanceStartTime
     :cognitect.aws.fsx.CreateFileSystemOpenZFSConfiguration/RootVolumeConfiguration
     :cognitect.aws.fsx.CreateFileSystemOpenZFSConfiguration/AutomaticBackupRetentionDays]))

(s/def :cognitect.aws.fsx/RequestTime inst?)

(s/def
  :cognitect.aws.fsx/StorageVirtualMachineFilterValue
  (s/spec #(re-matches (re-pattern "^[0-9a-zA-Z\\*\\.\\\\/\\?\\-\\_]*$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.fsx/DataRepositoryTaskFilterValues
  (s/coll-of :cognitect.aws.fsx/DataRepositoryTaskFilterValue :max-count 20))

(s/def
  :cognitect.aws.fsx/VolumeFilterValue
  (s/spec #(re-matches (re-pattern "^[0-9a-zA-Z\\*\\.\\\\/\\?\\-\\_]*$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.fsx/TieringPolicyName
  (s/spec string? :gen #(s/gen #{"SNAPSHOT_ONLY" "AUTO" "NONE" "ALL"})))

(s/def
  :cognitect.aws.fsx/FileSystem
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.FileSystem/SubnetIds
     :cognitect.aws.fsx.FileSystem/FileSystemTypeVersion
     :cognitect.aws.fsx.FileSystem/OwnerId
     :cognitect.aws.fsx.FileSystem/AdministrativeActions
     :cognitect.aws.fsx.FileSystem/Tags
     :cognitect.aws.fsx.FileSystem/OntapConfiguration
     :cognitect.aws.fsx.FileSystem/WindowsConfiguration
     :cognitect.aws.fsx.FileSystem/DNSName
     :cognitect.aws.fsx.FileSystem/StorageCapacity
     :cognitect.aws.fsx.FileSystem/FileSystemId
     :cognitect.aws.fsx.FileSystem/VpcId
     :cognitect.aws.fsx.FileSystem/FailureDetails
     :cognitect.aws.fsx.FileSystem/KmsKeyId
     :cognitect.aws.fsx.FileSystem/FileSystemType
     :cognitect.aws.fsx.FileSystem/NetworkInterfaceIds
     :cognitect.aws.fsx.FileSystem/StorageType
     :cognitect.aws.fsx.FileSystem/LustreConfiguration
     :cognitect.aws.fsx.FileSystem/CreationTime
     :cognitect.aws.fsx.FileSystem/Lifecycle
     :cognitect.aws.fsx.FileSystem/OpenZFSConfiguration
     :cognitect.aws.fsx.FileSystem/ResourceARN]))

(s/def
  :cognitect.aws.fsx/DescribeSnapshotsResponse
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.DescribeSnapshotsResponse/Snapshots
     :cognitect.aws.fsx.DescribeSnapshotsResponse/NextToken]))

(s/def
  :cognitect.aws.fsx/DataRepositoryAssociation
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.DataRepositoryAssociation/AssociationId
     :cognitect.aws.fsx.DataRepositoryAssociation/Tags
     :cognitect.aws.fsx.DataRepositoryAssociation/BatchImportMetaDataOnCreate
     :cognitect.aws.fsx.DataRepositoryAssociation/FileSystemId
     :cognitect.aws.fsx.DataRepositoryAssociation/FailureDetails
     :cognitect.aws.fsx.DataRepositoryAssociation/DataRepositoryPath
     :cognitect.aws.fsx.DataRepositoryAssociation/CreationTime
     :cognitect.aws.fsx.DataRepositoryAssociation/FileSystemPath
     :cognitect.aws.fsx.DataRepositoryAssociation/S3
     :cognitect.aws.fsx.DataRepositoryAssociation/ImportedFileChunkSize
     :cognitect.aws.fsx.DataRepositoryAssociation/Lifecycle
     :cognitect.aws.fsx.DataRepositoryAssociation/ResourceARN]))

(s/def
  :cognitect.aws.fsx/DataRepositoryAssociations
  (s/coll-of :cognitect.aws.fsx/DataRepositoryAssociation :max-count 100))

(s/def
  :cognitect.aws.fsx/FileSystemLifecycle
  (s/spec
    string?
    :gen
    #(s/gen #{"MISCONFIGURED" "AVAILABLE" "UPDATING" "DELETING" "CREATING" "FAILED"})))

(s/def :cognitect.aws.fsx/OpenZFSCopyStrategy (s/spec string? :gen #(s/gen #{"CLONE" "FULL_COPY"})))

(s/def
  :cognitect.aws.fsx/OpenZFSNfsExportOptions
  (s/coll-of :cognitect.aws.fsx/OpenZFSNfsExportOption :min-count 1 :max-count 20))

(s/def
  :cognitect.aws.fsx/SourceBackupId
  (s/spec #(re-matches (re-pattern "^(backup-[0-9a-f]{8,})$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.fsx/UpdateFileSystemOntapConfiguration
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.UpdateFileSystemOntapConfiguration/DiskIopsConfiguration
     :cognitect.aws.fsx.UpdateFileSystemOntapConfiguration/FsxAdminPassword
     :cognitect.aws.fsx.UpdateFileSystemOntapConfiguration/ThroughputCapacity
     :cognitect.aws.fsx.UpdateFileSystemOntapConfiguration/DailyAutomaticBackupStartTime
     :cognitect.aws.fsx.UpdateFileSystemOntapConfiguration/WeeklyMaintenanceStartTime
     :cognitect.aws.fsx.UpdateFileSystemOntapConfiguration/AutomaticBackupRetentionDays]))

(s/def
  :cognitect.aws.fsx/DataRepositoryTasks
  (s/coll-of :cognitect.aws.fsx/DataRepositoryTask :max-count 50))

(s/def
  :cognitect.aws.fsx/AlternateDNSName
  (s/spec
    #(re-matches (re-pattern "^[^\\u0000\\u0085\\u2028\\u2029\\r\\n]{4,253}$") %)
    :gen
    #(gen/string)))

(s/def
  :cognitect.aws.fsx/DailyTime
  (s/spec #(re-matches (re-pattern "^([01]\\d|2[0-3]):?([0-5]\\d)$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.fsx/SucceededCount
  (s/spec int? :gen #(gen/choose Long/MIN_VALUE Long/MAX_VALUE)))

(s/def
  :cognitect.aws.fsx/Namespace
  (s/spec
    #(re-matches (re-pattern "^[^\\u0000\\u0085\\u2028\\u2029\\r\\n]{1,4096}$") %)
    :gen
    #(gen/string)))

(s/def
  :cognitect.aws.fsx/DataRepositoryTaskFilter
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.DataRepositoryTaskFilter/Values
     :cognitect.aws.fsx.DataRepositoryTaskFilter/Name]))

(s/def
  :cognitect.aws.fsx/DataRepositoryTaskType
  (s/spec string? :gen #(s/gen #{"IMPORT_METADATA_FROM_REPOSITORY" "EXPORT_TO_REPOSITORY"})))

(s/def
  :cognitect.aws.fsx/FileSystemEndpoints
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.FileSystemEndpoints/Management
     :cognitect.aws.fsx.FileSystemEndpoints/Intercluster]))

(s/def
  :cognitect.aws.fsx/FileSystemId
  (s/spec #(re-matches (re-pattern "^(fs-[0-9a-f]{8,})$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.fsx/DescribeStorageVirtualMachinesRequest
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.DescribeStorageVirtualMachinesRequest/NextToken
     :cognitect.aws.fsx.DescribeStorageVirtualMachinesRequest/StorageVirtualMachineIds
     :cognitect.aws.fsx.DescribeStorageVirtualMachinesRequest/Filters
     :cognitect.aws.fsx.DescribeStorageVirtualMachinesRequest/MaxResults]))

(s/def :cognitect.aws.fsx/CreationTime inst?)

(s/def :cognitect.aws.fsx/BackupIds (s/coll-of :cognitect.aws.fsx/BackupId :max-count 50))

(s/def
  :cognitect.aws.fsx/UpdateFileSystemLustreConfiguration
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.UpdateFileSystemLustreConfiguration/DataCompressionType
     :cognitect.aws.fsx.UpdateFileSystemLustreConfiguration/LogConfiguration
     :cognitect.aws.fsx.UpdateFileSystemLustreConfiguration/AutoImportPolicy
     :cognitect.aws.fsx.UpdateFileSystemLustreConfiguration/DailyAutomaticBackupStartTime
     :cognitect.aws.fsx.UpdateFileSystemLustreConfiguration/WeeklyMaintenanceStartTime
     :cognitect.aws.fsx.UpdateFileSystemLustreConfiguration/AutomaticBackupRetentionDays]))

(s/def
  :cognitect.aws.fsx/AutoImportPolicy
  (s/keys :opt-un [:cognitect.aws.fsx.AutoImportPolicy/Events]))

(s/def
  :cognitect.aws.fsx/BackupLifecycle
  (s/spec
    string?
    :gen
    #(s/gen #{"COPYING" "AVAILABLE" "TRANSFERRING" "PENDING" "CREATING" "FAILED" "DELETED"})))

(s/def
  :cognitect.aws.fsx/DescribeFileSystemsResponse
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.DescribeFileSystemsResponse/NextToken
     :cognitect.aws.fsx.DescribeFileSystemsResponse/FileSystems]))

(s/def :cognitect.aws.fsx/Snapshots (s/coll-of :cognitect.aws.fsx/Snapshot :max-count 50))

(s/def
  :cognitect.aws.fsx/StorageVirtualMachines
  (s/coll-of :cognitect.aws.fsx/StorageVirtualMachine :max-count 50))

(s/def :cognitect.aws.fsx/OpenZFSDeploymentType (s/spec string? :gen #(s/gen #{"SINGLE_AZ_1"})))

(s/def
  :cognitect.aws.fsx/AdministrativeActionType
  (s/spec
    string?
    :gen
    #(s/gen
      #{"SNAPSHOT_UPDATE" "FILE_SYSTEM_ALIAS_DISASSOCIATION" "STORAGE_OPTIMIZATION"
        "FILE_SYSTEM_UPDATE" "FILE_SYSTEM_ALIAS_ASSOCIATION" "VOLUME_UPDATE"
        "RELEASE_NFS_V3_LOCKS"})))

(s/def
  :cognitect.aws.fsx/DeleteFileSystemWindowsConfiguration
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.DeleteFileSystemWindowsConfiguration/SkipFinalBackup
     :cognitect.aws.fsx.DeleteFileSystemWindowsConfiguration/FinalBackupTags]))

(s/def
  :cognitect.aws.fsx/DeleteFileSystemLustreConfiguration
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.DeleteFileSystemLustreConfiguration/SkipFinalBackup
     :cognitect.aws.fsx.DeleteFileSystemLustreConfiguration/FinalBackupTags]))

(s/def
  :cognitect.aws.fsx/RestoreOpenZFSVolumeOption
  (s/spec string? :gen #(s/gen #{"DELETE_INTERMEDIATE_SNAPSHOTS" "DELETE_CLONED_VOLUMES"})))

(s/def
  :cognitect.aws.fsx/WindowsFileSystemConfiguration
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.WindowsFileSystemConfiguration/CopyTagsToBackups
     :cognitect.aws.fsx.WindowsFileSystemConfiguration/DeploymentType
     :cognitect.aws.fsx.WindowsFileSystemConfiguration/PreferredFileServerIp
     :cognitect.aws.fsx.WindowsFileSystemConfiguration/Aliases
     :cognitect.aws.fsx.WindowsFileSystemConfiguration/RemoteAdministrationEndpoint
     :cognitect.aws.fsx.WindowsFileSystemConfiguration/ActiveDirectoryId
     :cognitect.aws.fsx.WindowsFileSystemConfiguration/ThroughputCapacity
     :cognitect.aws.fsx.WindowsFileSystemConfiguration/DailyAutomaticBackupStartTime
     :cognitect.aws.fsx.WindowsFileSystemConfiguration/WeeklyMaintenanceStartTime
     :cognitect.aws.fsx.WindowsFileSystemConfiguration/SelfManagedActiveDirectoryConfiguration
     :cognitect.aws.fsx.WindowsFileSystemConfiguration/AuditLogConfiguration
     :cognitect.aws.fsx.WindowsFileSystemConfiguration/AutomaticBackupRetentionDays
     :cognitect.aws.fsx.WindowsFileSystemConfiguration/MaintenanceOperationsInProgress
     :cognitect.aws.fsx.WindowsFileSystemConfiguration/PreferredSubnetId]))

(s/def
  :cognitect.aws.fsx/CreateOntapVolumeConfiguration
  (s/keys
    :req-un
    [:cognitect.aws.fsx.CreateOntapVolumeConfiguration/JunctionPath
     :cognitect.aws.fsx.CreateOntapVolumeConfiguration/SizeInMegabytes
     :cognitect.aws.fsx.CreateOntapVolumeConfiguration/StorageEfficiencyEnabled
     :cognitect.aws.fsx.CreateOntapVolumeConfiguration/StorageVirtualMachineId]
    :opt-un
    [:cognitect.aws.fsx.CreateOntapVolumeConfiguration/SecurityStyle
     :cognitect.aws.fsx.CreateOntapVolumeConfiguration/TieringPolicy]))

(s/def
  :cognitect.aws.fsx/OpenZFSUserAndGroupQuotas
  (s/coll-of :cognitect.aws.fsx/OpenZFSUserOrGroupQuota :max-count 100))

(s/def
  :cognitect.aws.fsx/DataRepositoryTaskFilters
  (s/coll-of :cognitect.aws.fsx/DataRepositoryTaskFilter :max-count 3))

(s/def :cognitect.aws.fsx/VolumeFilters (s/coll-of :cognitect.aws.fsx/VolumeFilter :max-count 2))

(s/def
  :cognitect.aws.fsx/CoolingPeriod
  (s/spec (s/and int? #(<= 2 % 183)) :gen #(gen/choose 2 183)))

(s/def
  :cognitect.aws.fsx/FileSystemMaintenanceOperation
  (s/spec string? :gen #(s/gen #{"BACKING_UP" "PATCHING"})))

(s/def
  :cognitect.aws.fsx/VolumeId
  (s/spec #(re-matches (re-pattern "^(fsvol-[0-9a-f]{17,})$") %) :gen #(gen/string)))

(s/def :cognitect.aws.fsx/BatchImportMetaDataOnCreate boolean?)

(s/def
  :cognitect.aws.fsx/DataRepositoryLifecycle
  (s/spec
    string?
    :gen
    #(s/gen #{"MISCONFIGURED" "AVAILABLE" "UPDATING" "DELETING" "CREATING" "FAILED"})))

(s/def
  :cognitect.aws.fsx/CreateVolumeFromBackupRequest
  (s/keys
    :req-un
    [:cognitect.aws.fsx.CreateVolumeFromBackupRequest/BackupId
     :cognitect.aws.fsx.CreateVolumeFromBackupRequest/Name]
    :opt-un
    [:cognitect.aws.fsx.CreateVolumeFromBackupRequest/ClientRequestToken
     :cognitect.aws.fsx.CreateVolumeFromBackupRequest/Tags
     :cognitect.aws.fsx.CreateVolumeFromBackupRequest/OntapConfiguration]))

(s/def :cognitect.aws.fsx/Volumes (s/coll-of :cognitect.aws.fsx/Volume :max-count 50))

(s/def
  :cognitect.aws.fsx/MaxResults
  (s/spec (s/and int? #(<= 1 % 2147483647)) :gen #(gen/choose 1 2147483647)))

(s/def
  :cognitect.aws.fsx/UntagResourceRequest
  (s/keys
    :req-un
    [:cognitect.aws.fsx.UntagResourceRequest/ResourceARN
     :cognitect.aws.fsx.UntagResourceRequest/TagKeys]))

(s/def
  :cognitect.aws.fsx/TagKey
  (s/spec #(re-matches (re-pattern "^([\\p{L}\\p{Z}\\p{N}_.:/=+\\-@]*)$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.fsx/AlternateDNSNames
  (s/coll-of :cognitect.aws.fsx/AlternateDNSName :max-count 50))

(s/def
  :cognitect.aws.fsx/OntapEndpointIpAddresses
  (s/coll-of :cognitect.aws.fsx/IpAddress :min-count 1 :max-count 2))

(s/def
  :cognitect.aws.fsx/DataRepositoryTaskFailureDetails
  (s/keys :opt-un [:cognitect.aws.fsx.DataRepositoryTaskFailureDetails/Message]))

(s/def
  :cognitect.aws.fsx/OpenZFSFileSystemConfiguration
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.OpenZFSFileSystemConfiguration/DiskIopsConfiguration
     :cognitect.aws.fsx.OpenZFSFileSystemConfiguration/CopyTagsToBackups
     :cognitect.aws.fsx.OpenZFSFileSystemConfiguration/DeploymentType
     :cognitect.aws.fsx.OpenZFSFileSystemConfiguration/RootVolumeId
     :cognitect.aws.fsx.OpenZFSFileSystemConfiguration/ThroughputCapacity
     :cognitect.aws.fsx.OpenZFSFileSystemConfiguration/DailyAutomaticBackupStartTime
     :cognitect.aws.fsx.OpenZFSFileSystemConfiguration/CopyTagsToVolumes
     :cognitect.aws.fsx.OpenZFSFileSystemConfiguration/WeeklyMaintenanceStartTime
     :cognitect.aws.fsx.OpenZFSFileSystemConfiguration/AutomaticBackupRetentionDays]))

(s/def
  :cognitect.aws.fsx/DeleteVolumeRequest
  (s/keys
    :req-un
    [:cognitect.aws.fsx.DeleteVolumeRequest/VolumeId]
    :opt-un
    [:cognitect.aws.fsx.DeleteVolumeRequest/ClientRequestToken
     :cognitect.aws.fsx.DeleteVolumeRequest/OntapConfiguration
     :cognitect.aws.fsx.DeleteVolumeRequest/OpenZFSConfiguration]))

(s/def
  :cognitect.aws.fsx/UpdateSnapshotRequest
  (s/keys
    :req-un
    [:cognitect.aws.fsx.UpdateSnapshotRequest/Name
     :cognitect.aws.fsx.UpdateSnapshotRequest/SnapshotId]
    :opt-un
    [:cognitect.aws.fsx.UpdateSnapshotRequest/ClientRequestToken]))

(s/def :cognitect.aws.fsx/DataCompressionType (s/spec string? :gen #(s/gen #{"LZ4" "NONE"})))

(s/def
  :cognitect.aws.fsx/DeleteSnapshotResponse
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.DeleteSnapshotResponse/Lifecycle
     :cognitect.aws.fsx.DeleteSnapshotResponse/SnapshotId]))

(s/def
  :cognitect.aws.fsx/WindowsDeploymentType
  (s/spec string? :gen #(s/gen #{"MULTI_AZ_1" "SINGLE_AZ_2" "SINGLE_AZ_1"})))

(s/def
  :cognitect.aws.fsx/DeleteVolumeOpenZFSConfiguration
  (s/keys :opt-un [:cognitect.aws.fsx.DeleteVolumeOpenZFSConfiguration/Options]))

(s/def
  :cognitect.aws.fsx/DataRepositoryFailureDetails
  (s/keys :opt-un [:cognitect.aws.fsx.DataRepositoryFailureDetails/Message]))

(s/def
  :cognitect.aws.fsx/CompletionReport
  (s/keys
    :req-un
    [:cognitect.aws.fsx.CompletionReport/Enabled]
    :opt-un
    [:cognitect.aws.fsx.CompletionReport/Path
     :cognitect.aws.fsx.CompletionReport/Format
     :cognitect.aws.fsx.CompletionReport/Scope]))

(s/def :cognitect.aws.fsx/ReportScope (s/spec string? :gen #(s/gen #{"FAILED_FILES_ONLY"})))

(s/def
  :cognitect.aws.fsx/DataRepositoryTaskStatus
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.DataRepositoryTaskStatus/FailedCount
     :cognitect.aws.fsx.DataRepositoryTaskStatus/SucceededCount
     :cognitect.aws.fsx.DataRepositoryTaskStatus/TotalCount
     :cognitect.aws.fsx.DataRepositoryTaskStatus/LastUpdatedTime]))

(s/def
  :cognitect.aws.fsx/CreateDataRepositoryTaskResponse
  (s/keys :opt-un [:cognitect.aws.fsx.CreateDataRepositoryTaskResponse/DataRepositoryTask]))

(s/def
  :cognitect.aws.fsx/DescribeSnapshotsRequest
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.DescribeSnapshotsRequest/NextToken
     :cognitect.aws.fsx.DescribeSnapshotsRequest/Filters
     :cognitect.aws.fsx.DescribeSnapshotsRequest/MaxResults
     :cognitect.aws.fsx.DescribeSnapshotsRequest/SnapshotIds]))

(s/def
  :cognitect.aws.fsx/VolumeFilterValues
  (s/coll-of :cognitect.aws.fsx/VolumeFilterValue :max-count 20))

(s/def
  :cognitect.aws.fsx/DescribeBackupsRequest
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.DescribeBackupsRequest/BackupIds
     :cognitect.aws.fsx.DescribeBackupsRequest/NextToken
     :cognitect.aws.fsx.DescribeBackupsRequest/Filters
     :cognitect.aws.fsx.DescribeBackupsRequest/MaxResults]))

(s/def
  :cognitect.aws.fsx/DataRepositoryTaskFilterName
  (s/spec
    string?
    :gen
    #(s/gen #{"file-system-id" "task-lifecycle" "data-repository-association-id"})))

(s/def
  :cognitect.aws.fsx/SnapshotFilterValue
  (s/spec #(re-matches (re-pattern "^[0-9a-zA-Z\\*\\.\\\\/\\?\\-\\_]*$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.fsx/CreateSnapshotResponse
  (s/keys :opt-un [:cognitect.aws.fsx.CreateSnapshotResponse/Snapshot]))

(s/def :cognitect.aws.fsx/OpenZFSQuotaType (s/spec string? :gen #(s/gen #{"USER" "GROUP"})))

(s/def
  :cognitect.aws.fsx/FilterName
  (s/spec
    string?
    :gen
    #(s/gen
      #{"volume-id" "file-system-id" "data-repository-type" "file-system-type" "backup-type"})))

(s/def
  :cognitect.aws.fsx/NetBiosAlias
  (s/spec
    #(re-matches (re-pattern "^[^\\u0000\\u0085\\u2028\\u2029\\r\\n]{1,255}$") %)
    :gen
    #(gen/string)))

(s/def
  :cognitect.aws.fsx/UpdateVolumeResponse
  (s/keys :opt-un [:cognitect.aws.fsx.UpdateVolumeResponse/Volume]))

(s/def
  :cognitect.aws.fsx/VolumeCapacity
  (s/spec (s/and int? #(<= 0 % 2147483647)) :gen #(gen/choose 0 2147483647)))

(s/def
  :cognitect.aws.fsx/StorageVirtualMachineName
  (s/spec
    #(re-matches (re-pattern "^[^\\u0000\\u0085\\u2028\\u2029\\r\\n]{1,47}$") %)
    :gen
    #(gen/string)))

(s/def
  :cognitect.aws.fsx/UUID
  (s/spec
    #(re-matches (re-pattern "^[^\\u0000\\u0085\\u2028\\u2029\\r\\n]{1,36}$") %)
    :gen
    #(gen/string)))

(s/def
  :cognitect.aws.fsx/NetworkInterfaceIds
  (s/coll-of :cognitect.aws.fsx/NetworkInterfaceId :max-count 50))

(s/def
  :cognitect.aws.fsx/ArchivePath
  (s/spec
    #(re-matches (re-pattern "^[^\\u0000\\u0085\\u2028\\u2029\\r\\n]{3,4357}$") %)
    :gen
    #(gen/string)))

(s/def
  :cognitect.aws.fsx/DeleteOpenZFSVolumeOption
  (s/spec string? :gen #(s/gen #{"DELETE_CHILD_VOLUMES_AND_SNAPSHOTS"})))

(s/def
  :cognitect.aws.fsx/IntegerNoMax
  (s/spec (s/and int? #(<= 0 % 2147483647)) :gen #(gen/choose 0 2147483647)))

(s/def
  :cognitect.aws.fsx/LustreLogConfiguration
  (s/keys
    :req-un
    [:cognitect.aws.fsx.LustreLogConfiguration/Level]
    :opt-un
    [:cognitect.aws.fsx.LustreLogConfiguration/Destination]))

(s/def
  :cognitect.aws.fsx/CancelDataRepositoryTaskRequest
  (s/keys :req-un [:cognitect.aws.fsx.CancelDataRepositoryTaskRequest/TaskId]))

(s/def
  :cognitect.aws.fsx/PerUnitStorageThroughput
  (s/spec (s/and int? #(<= 12 % 1000)) :gen #(gen/choose 12 1000)))

(s/def
  :cognitect.aws.fsx/CopyBackupRequest
  (s/keys
    :req-un
    [:cognitect.aws.fsx.CopyBackupRequest/SourceBackupId]
    :opt-un
    [:cognitect.aws.fsx.CopyBackupRequest/ClientRequestToken
     :cognitect.aws.fsx.CopyBackupRequest/SourceRegion
     :cognitect.aws.fsx.CopyBackupRequest/Tags
     :cognitect.aws.fsx.CopyBackupRequest/CopyTags
     :cognitect.aws.fsx.CopyBackupRequest/KmsKeyId]))

(s/def
  :cognitect.aws.fsx/UpdateSvmActiveDirectoryConfiguration
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.UpdateSvmActiveDirectoryConfiguration/SelfManagedActiveDirectoryConfiguration]))

(s/def :cognitect.aws.fsx/Filters (s/coll-of :cognitect.aws.fsx/Filter :max-count 10))

(s/def
  :cognitect.aws.fsx/RestoreVolumeFromSnapshotRequest
  (s/keys
    :req-un
    [:cognitect.aws.fsx.RestoreVolumeFromSnapshotRequest/VolumeId
     :cognitect.aws.fsx.RestoreVolumeFromSnapshotRequest/SnapshotId]
    :opt-un
    [:cognitect.aws.fsx.RestoreVolumeFromSnapshotRequest/ClientRequestToken
     :cognitect.aws.fsx.RestoreVolumeFromSnapshotRequest/Options]))

(s/def
  :cognitect.aws.fsx/StorageVirtualMachineFilter
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.StorageVirtualMachineFilter/Values
     :cognitect.aws.fsx.StorageVirtualMachineFilter/Name]))

(s/def
  :cognitect.aws.fsx/DescribeVolumesResponse
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.DescribeVolumesResponse/NextToken
     :cognitect.aws.fsx.DescribeVolumesResponse/Volumes]))

(s/def
  :cognitect.aws.fsx/Alias
  (s/keys :opt-un [:cognitect.aws.fsx.Alias/Lifecycle :cognitect.aws.fsx.Alias/Name]))

(s/def
  :cognitect.aws.fsx/AutoExportPolicy
  (s/keys :opt-un [:cognitect.aws.fsx.AutoExportPolicy/Events]))

(s/def
  :cognitect.aws.fsx/DataRepositoryTaskLifecycle
  (s/spec
    string?
    :gen
    #(s/gen #{"CANCELED" "PENDING" "EXECUTING" "SUCCEEDED" "FAILED" "CANCELING"})))

(s/def
  :cognitect.aws.fsx/DescribeFileSystemAliasesRequest
  (s/keys
    :req-un
    [:cognitect.aws.fsx.DescribeFileSystemAliasesRequest/FileSystemId]
    :opt-un
    [:cognitect.aws.fsx.DescribeFileSystemAliasesRequest/ClientRequestToken
     :cognitect.aws.fsx.DescribeFileSystemAliasesRequest/NextToken
     :cognitect.aws.fsx.DescribeFileSystemAliasesRequest/MaxResults]))

(s/def :cognitect.aws.fsx/RouteTableIds (s/coll-of :cognitect.aws.fsx/RouteTableId :max-count 50))

(s/def :cognitect.aws.fsx/TaskIds (s/coll-of :cognitect.aws.fsx/TaskId :max-count 50))

(s/def
  :cognitect.aws.fsx/SecurityGroupIds
  (s/coll-of :cognitect.aws.fsx/SecurityGroupId :max-count 50))

(s/def
  :cognitect.aws.fsx/CreateFileSystemLustreConfiguration
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.CreateFileSystemLustreConfiguration/CopyTagsToBackups
     :cognitect.aws.fsx.CreateFileSystemLustreConfiguration/DataCompressionType
     :cognitect.aws.fsx.CreateFileSystemLustreConfiguration/DeploymentType
     :cognitect.aws.fsx.CreateFileSystemLustreConfiguration/DriveCacheType
     :cognitect.aws.fsx.CreateFileSystemLustreConfiguration/ExportPath
     :cognitect.aws.fsx.CreateFileSystemLustreConfiguration/ImportPath
     :cognitect.aws.fsx.CreateFileSystemLustreConfiguration/LogConfiguration
     :cognitect.aws.fsx.CreateFileSystemLustreConfiguration/PerUnitStorageThroughput
     :cognitect.aws.fsx.CreateFileSystemLustreConfiguration/AutoImportPolicy
     :cognitect.aws.fsx.CreateFileSystemLustreConfiguration/DailyAutomaticBackupStartTime
     :cognitect.aws.fsx.CreateFileSystemLustreConfiguration/WeeklyMaintenanceStartTime
     :cognitect.aws.fsx.CreateFileSystemLustreConfiguration/AutomaticBackupRetentionDays
     :cognitect.aws.fsx.CreateFileSystemLustreConfiguration/ImportedFileChunkSize]))

(s/def :cognitect.aws.fsx/TagKeys (s/coll-of :cognitect.aws.fsx/TagKey :min-count 1 :max-count 50))

(s/def
  :cognitect.aws.fsx/SvmActiveDirectoryConfiguration
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.SvmActiveDirectoryConfiguration/NetBiosName
     :cognitect.aws.fsx.SvmActiveDirectoryConfiguration/SelfManagedActiveDirectoryConfiguration]))

(s/def
  :cognitect.aws.fsx/DisassociateFileSystemAliasesRequest
  (s/keys
    :req-un
    [:cognitect.aws.fsx.DisassociateFileSystemAliasesRequest/FileSystemId
     :cognitect.aws.fsx.DisassociateFileSystemAliasesRequest/Aliases]
    :opt-un
    [:cognitect.aws.fsx.DisassociateFileSystemAliasesRequest/ClientRequestToken]))

(s/def
  :cognitect.aws.fsx/DeleteFileSystemRequest
  (s/keys
    :req-un
    [:cognitect.aws.fsx.DeleteFileSystemRequest/FileSystemId]
    :opt-un
    [:cognitect.aws.fsx.DeleteFileSystemRequest/ClientRequestToken
     :cognitect.aws.fsx.DeleteFileSystemRequest/WindowsConfiguration
     :cognitect.aws.fsx.DeleteFileSystemRequest/LustreConfiguration
     :cognitect.aws.fsx.DeleteFileSystemRequest/OpenZFSConfiguration]))

(s/def
  :cognitect.aws.fsx/FileSystemAdministratorsGroupName
  (s/spec
    #(re-matches (re-pattern "^[^\\u0000\\u0085\\u2028\\u2029\\r\\n]{1,256}$") %)
    :gen
    #(gen/string)))

(s/def
  :cognitect.aws.fsx/CreateFileSystemResponse
  (s/keys :opt-un [:cognitect.aws.fsx.CreateFileSystemResponse/FileSystem]))

(s/def
  :cognitect.aws.fsx/SnapshotFilterValues
  (s/coll-of :cognitect.aws.fsx/SnapshotFilterValue :max-count 20))

(s/def
  :cognitect.aws.fsx/OpenZFSClients
  (s/spec #(re-matches (re-pattern "^[ -~]{1,128}$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.fsx/DeleteDataRepositoryAssociationRequest
  (s/keys
    :req-un
    [:cognitect.aws.fsx.DeleteDataRepositoryAssociationRequest/AssociationId
     :cognitect.aws.fsx.DeleteDataRepositoryAssociationRequest/DeleteDataInFileSystem]
    :opt-un
    [:cognitect.aws.fsx.DeleteDataRepositoryAssociationRequest/ClientRequestToken]))

(s/def
  :cognitect.aws.fsx/TieringPolicy
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.TieringPolicy/Name :cognitect.aws.fsx.TieringPolicy/CoolingPeriod]))

(s/def
  :cognitect.aws.fsx/StorageVirtualMachineFilterValues
  (s/coll-of :cognitect.aws.fsx/StorageVirtualMachineFilterValue :max-count 20))

(s/def
  :cognitect.aws.fsx/AdministrativeActionFailureDetails
  (s/keys :opt-un [:cognitect.aws.fsx.AdministrativeActionFailureDetails/Message]))

(s/def
  :cognitect.aws.fsx/Megabytes
  (s/spec (s/and int? #(<= 1 % 512000)) :gen #(gen/choose 1 512000)))

(s/def
  :cognitect.aws.fsx/FilterValue
  (s/spec #(re-matches (re-pattern "^[0-9a-zA-Z\\*\\.\\\\/\\?\\-\\_]*$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.fsx/UpdateVolumeRequest
  (s/keys
    :req-un
    [:cognitect.aws.fsx.UpdateVolumeRequest/VolumeId]
    :opt-un
    [:cognitect.aws.fsx.UpdateVolumeRequest/ClientRequestToken
     :cognitect.aws.fsx.UpdateVolumeRequest/OntapConfiguration
     :cognitect.aws.fsx.UpdateVolumeRequest/Name
     :cognitect.aws.fsx.UpdateVolumeRequest/OpenZFSConfiguration]))

(s/def
  :cognitect.aws.fsx/UpdateOntapVolumeConfiguration
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.UpdateOntapVolumeConfiguration/SizeInMegabytes
     :cognitect.aws.fsx.UpdateOntapVolumeConfiguration/SecurityStyle
     :cognitect.aws.fsx.UpdateOntapVolumeConfiguration/JunctionPath
     :cognitect.aws.fsx.UpdateOntapVolumeConfiguration/TieringPolicy
     :cognitect.aws.fsx.UpdateOntapVolumeConfiguration/StorageEfficiencyEnabled]))

(s/def
  :cognitect.aws.fsx/ListTagsForResourceResponse
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.ListTagsForResourceResponse/Tags
     :cognitect.aws.fsx.ListTagsForResourceResponse/NextToken]))

(s/def
  :cognitect.aws.fsx/ActiveDirectoryFullyQualifiedName
  (s/spec
    #(re-matches (re-pattern "^[^\\u0000\\u0085\\u2028\\u2029\\r\\n]{1,255}$") %)
    :gen
    #(gen/string)))

(s/def
  :cognitect.aws.fsx/DiskIopsConfiguration
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.DiskIopsConfiguration/Mode :cognitect.aws.fsx.DiskIopsConfiguration/Iops]))

(s/def
  :cognitect.aws.fsx/AWSAccountId
  (s/spec #(re-matches (re-pattern "^\\d{12}$") %) :gen #(gen/string)))

(s/def :cognitect.aws.fsx/VolumeIds (s/coll-of :cognitect.aws.fsx/VolumeId :max-count 50))

(s/def
  :cognitect.aws.fsx/DataRepositoryTask
  (s/keys
    :req-un
    [:cognitect.aws.fsx.DataRepositoryTask/TaskId
     :cognitect.aws.fsx.DataRepositoryTask/Lifecycle
     :cognitect.aws.fsx.DataRepositoryTask/Type
     :cognitect.aws.fsx.DataRepositoryTask/CreationTime
     :cognitect.aws.fsx.DataRepositoryTask/FileSystemId]
    :opt-un
    [:cognitect.aws.fsx.DataRepositoryTask/EndTime
     :cognitect.aws.fsx.DataRepositoryTask/StartTime
     :cognitect.aws.fsx.DataRepositoryTask/Tags
     :cognitect.aws.fsx.DataRepositoryTask/Status
     :cognitect.aws.fsx.DataRepositoryTask/Report
     :cognitect.aws.fsx.DataRepositoryTask/FailureDetails
     :cognitect.aws.fsx.DataRepositoryTask/Paths
     :cognitect.aws.fsx.DataRepositoryTask/ResourceARN]))

(s/def
  :cognitect.aws.fsx/DeleteDataRepositoryAssociationResponse
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.DeleteDataRepositoryAssociationResponse/AssociationId
     :cognitect.aws.fsx.DeleteDataRepositoryAssociationResponse/DeleteDataInFileSystem
     :cognitect.aws.fsx.DeleteDataRepositoryAssociationResponse/Lifecycle]))

(s/def
  :cognitect.aws.fsx/UpdateDataRepositoryAssociationRequest
  (s/keys
    :req-un
    [:cognitect.aws.fsx.UpdateDataRepositoryAssociationRequest/AssociationId]
    :opt-un
    [:cognitect.aws.fsx.UpdateDataRepositoryAssociationRequest/ClientRequestToken
     :cognitect.aws.fsx.UpdateDataRepositoryAssociationRequest/S3
     :cognitect.aws.fsx.UpdateDataRepositoryAssociationRequest/ImportedFileChunkSize]))

(s/def :cognitect.aws.fsx/TotalCount (s/spec int? :gen #(gen/choose Long/MIN_VALUE Long/MAX_VALUE)))

(s/def
  :cognitect.aws.fsx/ReleaseFileSystemNfsV3LocksRequest
  (s/keys
    :req-un
    [:cognitect.aws.fsx.ReleaseFileSystemNfsV3LocksRequest/FileSystemId]
    :opt-un
    [:cognitect.aws.fsx.ReleaseFileSystemNfsV3LocksRequest/ClientRequestToken]))

(s/def
  :cognitect.aws.fsx/CreateOpenZFSVolumeConfiguration
  (s/keys
    :req-un
    [:cognitect.aws.fsx.CreateOpenZFSVolumeConfiguration/ParentVolumeId]
    :opt-un
    [:cognitect.aws.fsx.CreateOpenZFSVolumeConfiguration/DataCompressionType
     :cognitect.aws.fsx.CreateOpenZFSVolumeConfiguration/OriginSnapshot
     :cognitect.aws.fsx.CreateOpenZFSVolumeConfiguration/StorageCapacityQuotaGiB
     :cognitect.aws.fsx.CreateOpenZFSVolumeConfiguration/ReadOnly
     :cognitect.aws.fsx.CreateOpenZFSVolumeConfiguration/CopyTagsToSnapshots
     :cognitect.aws.fsx.CreateOpenZFSVolumeConfiguration/NfsExports
     :cognitect.aws.fsx.CreateOpenZFSVolumeConfiguration/StorageCapacityReservationGiB
     :cognitect.aws.fsx.CreateOpenZFSVolumeConfiguration/RecordSizeKiB
     :cognitect.aws.fsx.CreateOpenZFSVolumeConfiguration/UserAndGroupQuotas]))

(s/def :cognitect.aws.fsx/Iops (s/spec (s/and int? #(<= 0 % 160000)) :gen #(gen/choose 0 160000)))

(s/def
  :cognitect.aws.fsx/OpenZFSClientConfigurations
  (s/coll-of :cognitect.aws.fsx/OpenZFSClientConfiguration :max-count 25))

(s/def
  :cognitect.aws.fsx/AutoImportPolicyType
  (s/spec string? :gen #(s/gen #{"NEW" "NONE" "NEW_CHANGED_DELETED" "NEW_CHANGED"})))

(s/def
  :cognitect.aws.fsx/FlexCacheEndpointType
  (s/spec string? :gen #(s/gen #{"ORIGIN" "NONE" "CACHE"})))

(s/def
  :cognitect.aws.fsx/VpcId
  (s/spec #(re-matches (re-pattern "^(vpc-[0-9a-f]{8,})$") %) :gen #(gen/string)))

(s/def :cognitect.aws.fsx/StorageType (s/spec string? :gen #(s/gen #{"HDD" "SSD"})))

(s/def
  :cognitect.aws.fsx/SnapshotId
  (s/spec #(re-matches (re-pattern "^((fs)?volsnap-[0-9a-f]{8,})$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.fsx/CreateBackupRequest
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.CreateBackupRequest/ClientRequestToken
     :cognitect.aws.fsx.CreateBackupRequest/Tags
     :cognitect.aws.fsx.CreateBackupRequest/FileSystemId
     :cognitect.aws.fsx.CreateBackupRequest/VolumeId]))

(s/def
  :cognitect.aws.fsx/RestoreVolumeFromSnapshotResponse
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.RestoreVolumeFromSnapshotResponse/VolumeId
     :cognitect.aws.fsx.RestoreVolumeFromSnapshotResponse/Lifecycle]))

(s/def
  :cognitect.aws.fsx/SelfManagedActiveDirectoryAttributes
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.SelfManagedActiveDirectoryAttributes/UserName
     :cognitect.aws.fsx.SelfManagedActiveDirectoryAttributes/DnsIps
     :cognitect.aws.fsx.SelfManagedActiveDirectoryAttributes/DomainName
     :cognitect.aws.fsx.SelfManagedActiveDirectoryAttributes/FileSystemAdministratorsGroup
     :cognitect.aws.fsx.SelfManagedActiveDirectoryAttributes/OrganizationalUnitDistinguishedName]))

(s/def
  :cognitect.aws.fsx/SvmEndpoints
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.SvmEndpoints/Nfs
     :cognitect.aws.fsx.SvmEndpoints/Management
     :cognitect.aws.fsx.SvmEndpoints/Iscsi
     :cognitect.aws.fsx.SvmEndpoints/Smb]))

(s/def
  :cognitect.aws.fsx/JunctionPath
  (s/spec
    #(re-matches (re-pattern "^[^\\u0000\\u0085\\u2028\\u2029\\r\\n]{1,255}$") %)
    :gen
    #(gen/string)))

(s/def
  :cognitect.aws.fsx/DeleteBackupResponse
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.DeleteBackupResponse/BackupId
     :cognitect.aws.fsx.DeleteBackupResponse/Lifecycle]))

(s/def
  :cognitect.aws.fsx/DescribeVolumesRequest
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.DescribeVolumesRequest/VolumeIds
     :cognitect.aws.fsx.DescribeVolumesRequest/NextToken
     :cognitect.aws.fsx.DescribeVolumesRequest/Filters
     :cognitect.aws.fsx.DescribeVolumesRequest/MaxResults]))

(s/def
  :cognitect.aws.fsx/NetworkInterfaceId
  (s/spec #(re-matches (re-pattern "^(eni-[0-9a-f]{8,})$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.fsx/LustreFileSystemConfiguration
  (s/keys
    :opt-un
    [:cognitect.aws.fsx.LustreFileSystemConfiguration/CopyTagsToBackups
     :cognitect.aws.fsx.LustreFileSystemConfiguration/DataCompressionType
     :cognitect.aws.fsx.LustreFileSystemConfiguration/DeploymentType
     :cognitect.aws.fsx.LustreFileSystemConfiguration/DriveCacheType
     :cognitect.aws.fsx.LustreFileSystemConfiguration/LogConfiguration
     :cognitect.aws.fsx.LustreFileSystemConfiguration/PerUnitStorageThroughput
     :cognitect.aws.fsx.LustreFileSystemConfiguration/DataRepositoryConfiguration
     :cognitect.aws.fsx.LustreFileSystemConfiguration/DailyAutomaticBackupStartTime
     :cognitect.aws.fsx.LustreFileSystemConfiguration/WeeklyMaintenanceStartTime
     :cognitect.aws.fsx.LustreFileSystemConfiguration/AutomaticBackupRetentionDays
     :cognitect.aws.fsx.LustreFileSystemConfiguration/MountName]))

(s/def
  :cognitect.aws.fsx/Status
  (s/spec
    string?
    :gen
    #(s/gen #{"UPDATED_OPTIMIZING" "COMPLETED" "IN_PROGRESS" "PENDING" "FAILED"})))

(s/def
  :cognitect.aws.fsx.SelfManagedActiveDirectoryConfiguration/DomainName
  :cognitect.aws.fsx/ActiveDirectoryFullyQualifiedName)

(s/def
  :cognitect.aws.fsx.SelfManagedActiveDirectoryConfiguration/OrganizationalUnitDistinguishedName
  :cognitect.aws.fsx/OrganizationalUnitDistinguishedName)

(s/def
  :cognitect.aws.fsx.SelfManagedActiveDirectoryConfiguration/FileSystemAdministratorsGroup
  :cognitect.aws.fsx/FileSystemAdministratorsGroupName)

(s/def
  :cognitect.aws.fsx.SelfManagedActiveDirectoryConfiguration/UserName
  :cognitect.aws.fsx/DirectoryUserName)

(s/def
  :cognitect.aws.fsx.SelfManagedActiveDirectoryConfiguration/Password
  :cognitect.aws.fsx/DirectoryPassword)

(s/def :cognitect.aws.fsx.SelfManagedActiveDirectoryConfiguration/DnsIps :cognitect.aws.fsx/DnsIps)

(s/def :cognitect.aws.fsx.TagResourceRequest/ResourceARN :cognitect.aws.fsx/ResourceARN)

(s/def :cognitect.aws.fsx.TagResourceRequest/Tags :cognitect.aws.fsx/Tags)

(s/def :cognitect.aws.fsx.DeleteVolumeOntapConfiguration/SkipFinalBackup :cognitect.aws.fsx/Flag)

(s/def :cognitect.aws.fsx.DeleteVolumeOntapConfiguration/FinalBackupTags :cognitect.aws.fsx/Tags)

(s/def :cognitect.aws.fsx.DescribeBackupsResponse/Backups :cognitect.aws.fsx/Backups)

(s/def :cognitect.aws.fsx.DescribeBackupsResponse/NextToken :cognitect.aws.fsx/NextToken)

(s/def :cognitect.aws.fsx.DescribeFileSystemsRequest/FileSystemIds :cognitect.aws.fsx/FileSystemIds)

(s/def :cognitect.aws.fsx.DescribeFileSystemsRequest/MaxResults :cognitect.aws.fsx/MaxResults)

(s/def :cognitect.aws.fsx.DescribeFileSystemsRequest/NextToken :cognitect.aws.fsx/NextToken)

(s/def
  :cognitect.aws.fsx.CreateFileSystemFromBackupResponse/FileSystem
  :cognitect.aws.fsx/FileSystem)

(s/def
  :cognitect.aws.fsx.S3DataRepositoryConfiguration/AutoImportPolicy
  :cognitect.aws.fsx/AutoImportPolicy)

(s/def
  :cognitect.aws.fsx.S3DataRepositoryConfiguration/AutoExportPolicy
  :cognitect.aws.fsx/AutoExportPolicy)

(s/def
  :cognitect.aws.fsx.CreateSvmActiveDirectoryConfiguration/NetBiosName
  :cognitect.aws.fsx/NetBiosAlias)

(s/def
  :cognitect.aws.fsx.CreateSvmActiveDirectoryConfiguration/SelfManagedActiveDirectoryConfiguration
  :cognitect.aws.fsx/SelfManagedActiveDirectoryConfiguration)

(s/def :cognitect.aws.fsx.CreateVolumeResponse/Volume :cognitect.aws.fsx/Volume)

(s/def
  :cognitect.aws.fsx.OpenZFSCreateRootVolumeConfiguration/RecordSizeKiB
  :cognitect.aws.fsx/IntegerRecordSizeKiB)

(s/def
  :cognitect.aws.fsx.OpenZFSCreateRootVolumeConfiguration/DataCompressionType
  :cognitect.aws.fsx/OpenZFSDataCompressionType)

(s/def
  :cognitect.aws.fsx.OpenZFSCreateRootVolumeConfiguration/NfsExports
  :cognitect.aws.fsx/OpenZFSNfsExports)

(s/def
  :cognitect.aws.fsx.OpenZFSCreateRootVolumeConfiguration/UserAndGroupQuotas
  :cognitect.aws.fsx/OpenZFSUserAndGroupQuotas)

(s/def
  :cognitect.aws.fsx.OpenZFSCreateRootVolumeConfiguration/CopyTagsToSnapshots
  :cognitect.aws.fsx/Flag)

(s/def :cognitect.aws.fsx.OpenZFSCreateRootVolumeConfiguration/ReadOnly :cognitect.aws.fsx/ReadOnly)

(s/def :cognitect.aws.fsx.Tag/Key :cognitect.aws.fsx/TagKey)

(s/def :cognitect.aws.fsx.Tag/Value :cognitect.aws.fsx/TagValue)

(s/def
  :cognitect.aws.fsx.CreateStorageVirtualMachineResponse/StorageVirtualMachine
  :cognitect.aws.fsx/StorageVirtualMachine)

(s/def
  :cognitect.aws.fsx.SelfManagedActiveDirectoryConfigurationUpdates/UserName
  :cognitect.aws.fsx/DirectoryUserName)

(s/def
  :cognitect.aws.fsx.SelfManagedActiveDirectoryConfigurationUpdates/Password
  :cognitect.aws.fsx/DirectoryPassword)

(s/def
  :cognitect.aws.fsx.SelfManagedActiveDirectoryConfigurationUpdates/DnsIps
  :cognitect.aws.fsx/DnsIps)

(s/def :cognitect.aws.fsx.DescribeDataRepositoryTasksRequest/TaskIds :cognitect.aws.fsx/TaskIds)

(s/def
  :cognitect.aws.fsx.DescribeDataRepositoryTasksRequest/Filters
  :cognitect.aws.fsx/DataRepositoryTaskFilters)

(s/def
  :cognitect.aws.fsx.DescribeDataRepositoryTasksRequest/MaxResults
  :cognitect.aws.fsx/MaxResults)

(s/def :cognitect.aws.fsx.DescribeDataRepositoryTasksRequest/NextToken :cognitect.aws.fsx/NextToken)

(s/def :cognitect.aws.fsx.CreateBackupResponse/Backup :cognitect.aws.fsx/Backup)

(s/def
  :cognitect.aws.fsx.UpdateStorageVirtualMachineResponse/StorageVirtualMachine
  :cognitect.aws.fsx/StorageVirtualMachine)

(s/def
  :cognitect.aws.fsx.CreateVolumeRequest/ClientRequestToken
  :cognitect.aws.fsx/ClientRequestToken)

(s/def :cognitect.aws.fsx.CreateVolumeRequest/VolumeType :cognitect.aws.fsx/VolumeType)

(s/def :cognitect.aws.fsx.CreateVolumeRequest/Name :cognitect.aws.fsx/VolumeName)

(s/def
  :cognitect.aws.fsx.CreateVolumeRequest/OntapConfiguration
  :cognitect.aws.fsx/CreateOntapVolumeConfiguration)

(s/def :cognitect.aws.fsx.CreateVolumeRequest/Tags :cognitect.aws.fsx/Tags)

(s/def
  :cognitect.aws.fsx.CreateVolumeRequest/OpenZFSConfiguration
  :cognitect.aws.fsx/CreateOpenZFSVolumeConfiguration)

(s/def
  :cognitect.aws.fsx.UpdateStorageVirtualMachineRequest/ActiveDirectoryConfiguration
  :cognitect.aws.fsx/UpdateSvmActiveDirectoryConfiguration)

(s/def
  :cognitect.aws.fsx.UpdateStorageVirtualMachineRequest/ClientRequestToken
  :cognitect.aws.fsx/ClientRequestToken)

(s/def
  :cognitect.aws.fsx.UpdateStorageVirtualMachineRequest/StorageVirtualMachineId
  :cognitect.aws.fsx/StorageVirtualMachineId)

(s/def
  :cognitect.aws.fsx.UpdateStorageVirtualMachineRequest/SvmAdminPassword
  :cognitect.aws.fsx/AdminPassword)

(s/def :cognitect.aws.fsx.Filter/Name :cognitect.aws.fsx/FilterName)

(s/def :cognitect.aws.fsx.Filter/Values :cognitect.aws.fsx/FilterValues)

(s/def
  :cognitect.aws.fsx.CreateFileSystemRequest/LustreConfiguration
  :cognitect.aws.fsx/CreateFileSystemLustreConfiguration)

(s/def :cognitect.aws.fsx.CreateFileSystemRequest/SubnetIds :cognitect.aws.fsx/SubnetIds)

(s/def
  :cognitect.aws.fsx.CreateFileSystemRequest/StorageCapacity
  :cognitect.aws.fsx/StorageCapacity)

(s/def
  :cognitect.aws.fsx.CreateFileSystemRequest/WindowsConfiguration
  :cognitect.aws.fsx/CreateFileSystemWindowsConfiguration)

(s/def :cognitect.aws.fsx.CreateFileSystemRequest/Tags :cognitect.aws.fsx/Tags)

(s/def
  :cognitect.aws.fsx.CreateFileSystemRequest/FileSystemTypeVersion
  :cognitect.aws.fsx/FileSystemTypeVersion)

(s/def
  :cognitect.aws.fsx.CreateFileSystemRequest/ClientRequestToken
  :cognitect.aws.fsx/ClientRequestToken)

(s/def :cognitect.aws.fsx.CreateFileSystemRequest/FileSystemType :cognitect.aws.fsx/FileSystemType)

(s/def :cognitect.aws.fsx.CreateFileSystemRequest/KmsKeyId :cognitect.aws.fsx/KmsKeyId)

(s/def
  :cognitect.aws.fsx.CreateFileSystemRequest/OpenZFSConfiguration
  :cognitect.aws.fsx/CreateFileSystemOpenZFSConfiguration)

(s/def
  :cognitect.aws.fsx.CreateFileSystemRequest/OntapConfiguration
  :cognitect.aws.fsx/CreateFileSystemOntapConfiguration)

(s/def
  :cognitect.aws.fsx.CreateFileSystemRequest/SecurityGroupIds
  :cognitect.aws.fsx/SecurityGroupIds)

(s/def :cognitect.aws.fsx.CreateFileSystemRequest/StorageType :cognitect.aws.fsx/StorageType)

(s/def
  :cognitect.aws.fsx.AssociateFileSystemAliasesRequest/ClientRequestToken
  :cognitect.aws.fsx/ClientRequestToken)

(s/def
  :cognitect.aws.fsx.AssociateFileSystemAliasesRequest/FileSystemId
  :cognitect.aws.fsx/FileSystemId)

(s/def
  :cognitect.aws.fsx.AssociateFileSystemAliasesRequest/Aliases
  :cognitect.aws.fsx/AlternateDNSNames)

(s/def :cognitect.aws.fsx.VolumeFilter/Name :cognitect.aws.fsx/VolumeFilterName)

(s/def :cognitect.aws.fsx.VolumeFilter/Values :cognitect.aws.fsx/VolumeFilterValues)

(s/def :cognitect.aws.fsx.UpdateSnapshotResponse/Snapshot :cognitect.aws.fsx/Snapshot)

(s/def :cognitect.aws.fsx.LifecycleTransitionReason/Message :cognitect.aws.fsx/ErrorMessage)

(s/def
  :cognitect.aws.fsx.UpdateFileSystemWindowsConfiguration/WeeklyMaintenanceStartTime
  :cognitect.aws.fsx/WeeklyTime)

(s/def
  :cognitect.aws.fsx.UpdateFileSystemWindowsConfiguration/DailyAutomaticBackupStartTime
  :cognitect.aws.fsx/DailyTime)

(s/def
  :cognitect.aws.fsx.UpdateFileSystemWindowsConfiguration/AutomaticBackupRetentionDays
  :cognitect.aws.fsx/AutomaticBackupRetentionDays)

(s/def
  :cognitect.aws.fsx.UpdateFileSystemWindowsConfiguration/ThroughputCapacity
  :cognitect.aws.fsx/MegabytesPerSecond)

(s/def
  :cognitect.aws.fsx.UpdateFileSystemWindowsConfiguration/SelfManagedActiveDirectoryConfiguration
  :cognitect.aws.fsx/SelfManagedActiveDirectoryConfigurationUpdates)

(s/def
  :cognitect.aws.fsx.UpdateFileSystemWindowsConfiguration/AuditLogConfiguration
  :cognitect.aws.fsx/WindowsAuditLogCreateConfiguration)

(s/def
  :cognitect.aws.fsx.ActiveDirectoryBackupAttributes/DomainName
  :cognitect.aws.fsx/ActiveDirectoryFullyQualifiedName)

(s/def
  :cognitect.aws.fsx.ActiveDirectoryBackupAttributes/ActiveDirectoryId
  :cognitect.aws.fsx/DirectoryId)

(s/def
  :cognitect.aws.fsx.ActiveDirectoryBackupAttributes/ResourceARN
  :cognitect.aws.fsx/ResourceARN)

(s/def
  :cognitect.aws.fsx.LustreLogCreateConfiguration/Level
  :cognitect.aws.fsx/LustreAccessAuditLogLevel)

(s/def :cognitect.aws.fsx.LustreLogCreateConfiguration/Destination :cognitect.aws.fsx/GeneralARN)

(s/def :cognitect.aws.fsx.OpenZFSClientConfiguration/Clients :cognitect.aws.fsx/OpenZFSClients)

(s/def
  :cognitect.aws.fsx.OpenZFSClientConfiguration/Options
  :cognitect.aws.fsx/OpenZFSNfsExportOptions)

(s/def
  :cognitect.aws.fsx.CreateSnapshotRequest/ClientRequestToken
  :cognitect.aws.fsx/ClientRequestToken)

(s/def :cognitect.aws.fsx.CreateSnapshotRequest/Name :cognitect.aws.fsx/SnapshotName)

(s/def :cognitect.aws.fsx.CreateSnapshotRequest/VolumeId :cognitect.aws.fsx/VolumeId)

(s/def :cognitect.aws.fsx.CreateSnapshotRequest/Tags :cognitect.aws.fsx/Tags)

(s/def :cognitect.aws.fsx.DescribeFileSystemAliasesResponse/Aliases :cognitect.aws.fsx/Aliases)

(s/def :cognitect.aws.fsx.DescribeFileSystemAliasesResponse/NextToken :cognitect.aws.fsx/NextToken)

(s/def
  :cognitect.aws.fsx.DescribeStorageVirtualMachinesResponse/StorageVirtualMachines
  :cognitect.aws.fsx/StorageVirtualMachines)

(s/def
  :cognitect.aws.fsx.DescribeStorageVirtualMachinesResponse/NextToken
  :cognitect.aws.fsx/NextToken)

(s/def
  :cognitect.aws.fsx.OntapVolumeConfiguration/SizeInMegabytes
  :cognitect.aws.fsx/VolumeCapacity)

(s/def :cognitect.aws.fsx.OntapVolumeConfiguration/SecurityStyle :cognitect.aws.fsx/SecurityStyle)

(s/def
  :cognitect.aws.fsx.OntapVolumeConfiguration/OntapVolumeType
  :cognitect.aws.fsx/OntapVolumeType)

(s/def
  :cognitect.aws.fsx.OntapVolumeConfiguration/StorageVirtualMachineId
  :cognitect.aws.fsx/StorageVirtualMachineId)

(s/def :cognitect.aws.fsx.OntapVolumeConfiguration/UUID :cognitect.aws.fsx/UUID)

(s/def :cognitect.aws.fsx.OntapVolumeConfiguration/StorageEfficiencyEnabled :cognitect.aws.fsx/Flag)

(s/def
  :cognitect.aws.fsx.OntapVolumeConfiguration/StorageVirtualMachineRoot
  :cognitect.aws.fsx/Flag)

(s/def :cognitect.aws.fsx.OntapVolumeConfiguration/TieringPolicy :cognitect.aws.fsx/TieringPolicy)

(s/def
  :cognitect.aws.fsx.OntapVolumeConfiguration/FlexCacheEndpointType
  :cognitect.aws.fsx/FlexCacheEndpointType)

(s/def :cognitect.aws.fsx.OntapVolumeConfiguration/JunctionPath :cognitect.aws.fsx/JunctionPath)

(s/def :cognitect.aws.fsx.OpenZFSUserOrGroupQuota/Type :cognitect.aws.fsx/OpenZFSQuotaType)

(s/def :cognitect.aws.fsx.OpenZFSUserOrGroupQuota/Id :cognitect.aws.fsx/IntegerNoMax)

(s/def
  :cognitect.aws.fsx.OpenZFSUserOrGroupQuota/StorageCapacityQuotaGiB
  :cognitect.aws.fsx/IntegerNoMax)

(s/def :cognitect.aws.fsx.DisassociateFileSystemAliasesResponse/Aliases :cognitect.aws.fsx/Aliases)

(s/def
  :cognitect.aws.fsx.UpdateFileSystemOpenZFSConfiguration/AutomaticBackupRetentionDays
  :cognitect.aws.fsx/AutomaticBackupRetentionDays)

(s/def
  :cognitect.aws.fsx.UpdateFileSystemOpenZFSConfiguration/CopyTagsToBackups
  :cognitect.aws.fsx/Flag)

(s/def
  :cognitect.aws.fsx.UpdateFileSystemOpenZFSConfiguration/CopyTagsToVolumes
  :cognitect.aws.fsx/Flag)

(s/def
  :cognitect.aws.fsx.UpdateFileSystemOpenZFSConfiguration/DailyAutomaticBackupStartTime
  :cognitect.aws.fsx/DailyTime)

(s/def
  :cognitect.aws.fsx.UpdateFileSystemOpenZFSConfiguration/ThroughputCapacity
  :cognitect.aws.fsx/MegabytesPerSecond)

(s/def
  :cognitect.aws.fsx.UpdateFileSystemOpenZFSConfiguration/WeeklyMaintenanceStartTime
  :cognitect.aws.fsx/WeeklyTime)

(s/def
  :cognitect.aws.fsx.UpdateFileSystemOpenZFSConfiguration/DiskIopsConfiguration
  :cognitect.aws.fsx/DiskIopsConfiguration)

(s/def :cognitect.aws.fsx.AssociateFileSystemAliasesResponse/Aliases :cognitect.aws.fsx/Aliases)

(s/def
  :cognitect.aws.fsx.StorageVirtualMachine/ActiveDirectoryConfiguration
  :cognitect.aws.fsx/SvmActiveDirectoryConfiguration)

(s/def
  :cognitect.aws.fsx.StorageVirtualMachine/LifecycleTransitionReason
  :cognitect.aws.fsx/LifecycleTransitionReason)

(s/def
  :cognitect.aws.fsx.StorageVirtualMachine/Subtype
  :cognitect.aws.fsx/StorageVirtualMachineSubtype)

(s/def
  :cognitect.aws.fsx.StorageVirtualMachine/RootVolumeSecurityStyle
  :cognitect.aws.fsx/StorageVirtualMachineRootVolumeSecurityStyle)

(s/def :cognitect.aws.fsx.StorageVirtualMachine/Tags :cognitect.aws.fsx/Tags)

(s/def
  :cognitect.aws.fsx.StorageVirtualMachine/StorageVirtualMachineId
  :cognitect.aws.fsx/StorageVirtualMachineId)

(s/def :cognitect.aws.fsx.StorageVirtualMachine/ResourceARN :cognitect.aws.fsx/ResourceARN)

(s/def :cognitect.aws.fsx.StorageVirtualMachine/FileSystemId :cognitect.aws.fsx/FileSystemId)

(s/def :cognitect.aws.fsx.StorageVirtualMachine/CreationTime :cognitect.aws.fsx/CreationTime)

(s/def :cognitect.aws.fsx.StorageVirtualMachine/Name :cognitect.aws.fsx/StorageVirtualMachineName)

(s/def :cognitect.aws.fsx.StorageVirtualMachine/Endpoints :cognitect.aws.fsx/SvmEndpoints)

(s/def :cognitect.aws.fsx.StorageVirtualMachine/UUID :cognitect.aws.fsx/UUID)

(s/def
  :cognitect.aws.fsx.StorageVirtualMachine/Lifecycle
  :cognitect.aws.fsx/StorageVirtualMachineLifecycle)

(s/def
  :cognitect.aws.fsx.WindowsAuditLogConfiguration/FileAccessAuditLogLevel
  :cognitect.aws.fsx/WindowsAccessAuditLogLevel)

(s/def
  :cognitect.aws.fsx.WindowsAuditLogConfiguration/FileShareAccessAuditLogLevel
  :cognitect.aws.fsx/WindowsAccessAuditLogLevel)

(s/def
  :cognitect.aws.fsx.WindowsAuditLogConfiguration/AuditLogDestination
  :cognitect.aws.fsx/GeneralARN)

(s/def :cognitect.aws.fsx.CopyBackupResponse/Backup :cognitect.aws.fsx/Backup)

(s/def
  :cognitect.aws.fsx.DeleteFileSystemOpenZFSConfiguration/SkipFinalBackup
  :cognitect.aws.fsx/Flag)

(s/def
  :cognitect.aws.fsx.DeleteFileSystemOpenZFSConfiguration/FinalBackupTags
  :cognitect.aws.fsx/Tags)

(s/def
  :cognitect.aws.fsx.DeleteFileSystemOpenZFSConfiguration/Options
  :cognitect.aws.fsx/DeleteFileSystemOpenZFSOptions)

(s/def
  :cognitect.aws.fsx.AdministrativeAction/AdministrativeActionType
  :cognitect.aws.fsx/AdministrativeActionType)

(s/def :cognitect.aws.fsx.AdministrativeAction/ProgressPercent :cognitect.aws.fsx/ProgressPercent)

(s/def :cognitect.aws.fsx.AdministrativeAction/RequestTime :cognitect.aws.fsx/RequestTime)

(s/def :cognitect.aws.fsx.AdministrativeAction/Status :cognitect.aws.fsx/Status)

(s/def :cognitect.aws.fsx.AdministrativeAction/TargetFileSystemValues :cognitect.aws.fsx/FileSystem)

(s/def
  :cognitect.aws.fsx.AdministrativeAction/FailureDetails
  :cognitect.aws.fsx/AdministrativeActionFailureDetails)

(s/def :cognitect.aws.fsx.AdministrativeAction/TargetVolumeValues :cognitect.aws.fsx/Volume)

(s/def :cognitect.aws.fsx.AdministrativeAction/TargetSnapshotValues :cognitect.aws.fsx/Snapshot)

(s/def
  :cognitect.aws.fsx.DescribeDataRepositoryAssociationsRequest/AssociationIds
  :cognitect.aws.fsx/DataRepositoryAssociationIds)

(s/def
  :cognitect.aws.fsx.DescribeDataRepositoryAssociationsRequest/Filters
  :cognitect.aws.fsx/Filters)

(s/def
  :cognitect.aws.fsx.DescribeDataRepositoryAssociationsRequest/MaxResults
  :cognitect.aws.fsx/LimitedMaxResults)

(s/def
  :cognitect.aws.fsx.DescribeDataRepositoryAssociationsRequest/NextToken
  :cognitect.aws.fsx/NextToken)

(s/def :cognitect.aws.fsx.Backup/BackupId :cognitect.aws.fsx/BackupId)

(s/def :cognitect.aws.fsx.Backup/ResourceType :cognitect.aws.fsx/ResourceType)

(s/def :cognitect.aws.fsx.Backup/ProgressPercent :cognitect.aws.fsx/ProgressPercent)

(s/def :cognitect.aws.fsx.Backup/Tags :cognitect.aws.fsx/Tags)

(s/def :cognitect.aws.fsx.Backup/Volume :cognitect.aws.fsx/Volume)

(s/def
  :cognitect.aws.fsx.Backup/DirectoryInformation
  :cognitect.aws.fsx/ActiveDirectoryBackupAttributes)

(s/def :cognitect.aws.fsx.Backup/ResourceARN :cognitect.aws.fsx/ResourceARN)

(s/def :cognitect.aws.fsx.Backup/KmsKeyId :cognitect.aws.fsx/KmsKeyId)

(s/def :cognitect.aws.fsx.Backup/FailureDetails :cognitect.aws.fsx/BackupFailureDetails)

(s/def :cognitect.aws.fsx.Backup/OwnerId :cognitect.aws.fsx/AWSAccountId)

(s/def :cognitect.aws.fsx.Backup/FileSystem :cognitect.aws.fsx/FileSystem)

(s/def :cognitect.aws.fsx.Backup/SourceBackupId :cognitect.aws.fsx/BackupId)

(s/def :cognitect.aws.fsx.Backup/CreationTime :cognitect.aws.fsx/CreationTime)

(s/def :cognitect.aws.fsx.Backup/SourceBackupRegion :cognitect.aws.fsx/Region)

(s/def :cognitect.aws.fsx.Backup/Type :cognitect.aws.fsx/BackupType)

(s/def :cognitect.aws.fsx.Backup/Lifecycle :cognitect.aws.fsx/BackupLifecycle)

(s/def :cognitect.aws.fsx.DeleteVolumeOntapResponse/FinalBackupId :cognitect.aws.fsx/BackupId)

(s/def :cognitect.aws.fsx.DeleteVolumeOntapResponse/FinalBackupTags :cognitect.aws.fsx/Tags)

(s/def
  :cognitect.aws.fsx.DeleteStorageVirtualMachineResponse/StorageVirtualMachineId
  :cognitect.aws.fsx/StorageVirtualMachineId)

(s/def
  :cognitect.aws.fsx.DeleteStorageVirtualMachineResponse/Lifecycle
  :cognitect.aws.fsx/StorageVirtualMachineLifecycle)

(s/def
  :cognitect.aws.fsx.CreateFileSystemFromBackupRequest/LustreConfiguration
  :cognitect.aws.fsx/CreateFileSystemLustreConfiguration)

(s/def :cognitect.aws.fsx.CreateFileSystemFromBackupRequest/BackupId :cognitect.aws.fsx/BackupId)

(s/def :cognitect.aws.fsx.CreateFileSystemFromBackupRequest/SubnetIds :cognitect.aws.fsx/SubnetIds)

(s/def
  :cognitect.aws.fsx.CreateFileSystemFromBackupRequest/WindowsConfiguration
  :cognitect.aws.fsx/CreateFileSystemWindowsConfiguration)

(s/def :cognitect.aws.fsx.CreateFileSystemFromBackupRequest/Tags :cognitect.aws.fsx/Tags)

(s/def
  :cognitect.aws.fsx.CreateFileSystemFromBackupRequest/FileSystemTypeVersion
  :cognitect.aws.fsx/FileSystemTypeVersion)

(s/def
  :cognitect.aws.fsx.CreateFileSystemFromBackupRequest/ClientRequestToken
  :cognitect.aws.fsx/ClientRequestToken)

(s/def :cognitect.aws.fsx.CreateFileSystemFromBackupRequest/KmsKeyId :cognitect.aws.fsx/KmsKeyId)

(s/def
  :cognitect.aws.fsx.CreateFileSystemFromBackupRequest/OpenZFSConfiguration
  :cognitect.aws.fsx/CreateFileSystemOpenZFSConfiguration)

(s/def
  :cognitect.aws.fsx.CreateFileSystemFromBackupRequest/SecurityGroupIds
  :cognitect.aws.fsx/SecurityGroupIds)

(s/def
  :cognitect.aws.fsx.CreateFileSystemFromBackupRequest/StorageType
  :cognitect.aws.fsx/StorageType)

(s/def
  :cognitect.aws.fsx.CreateDataRepositoryTaskRequest/Type
  :cognitect.aws.fsx/DataRepositoryTaskType)

(s/def
  :cognitect.aws.fsx.CreateDataRepositoryTaskRequest/Paths
  :cognitect.aws.fsx/DataRepositoryTaskPaths)

(s/def
  :cognitect.aws.fsx.CreateDataRepositoryTaskRequest/FileSystemId
  :cognitect.aws.fsx/FileSystemId)

(s/def
  :cognitect.aws.fsx.CreateDataRepositoryTaskRequest/Report
  :cognitect.aws.fsx/CompletionReport)

(s/def
  :cognitect.aws.fsx.CreateDataRepositoryTaskRequest/ClientRequestToken
  :cognitect.aws.fsx/ClientRequestToken)

(s/def :cognitect.aws.fsx.CreateDataRepositoryTaskRequest/Tags :cognitect.aws.fsx/Tags)

(s/def :cognitect.aws.fsx.SnapshotFilter/Name :cognitect.aws.fsx/SnapshotFilterName)

(s/def :cognitect.aws.fsx.SnapshotFilter/Values :cognitect.aws.fsx/SnapshotFilterValues)

(s/def :cognitect.aws.fsx.FileSystemFailureDetails/Message :cognitect.aws.fsx/ErrorMessage)

(s/def
  :cognitect.aws.fsx.CreateFileSystemWindowsConfiguration/SelfManagedActiveDirectoryConfiguration
  :cognitect.aws.fsx/SelfManagedActiveDirectoryConfiguration)

(s/def
  :cognitect.aws.fsx.CreateFileSystemWindowsConfiguration/AuditLogConfiguration
  :cognitect.aws.fsx/WindowsAuditLogCreateConfiguration)

(s/def
  :cognitect.aws.fsx.CreateFileSystemWindowsConfiguration/AutomaticBackupRetentionDays
  :cognitect.aws.fsx/AutomaticBackupRetentionDays)

(s/def
  :cognitect.aws.fsx.CreateFileSystemWindowsConfiguration/PreferredSubnetId
  :cognitect.aws.fsx/SubnetId)

(s/def
  :cognitect.aws.fsx.CreateFileSystemWindowsConfiguration/DeploymentType
  :cognitect.aws.fsx/WindowsDeploymentType)

(s/def
  :cognitect.aws.fsx.CreateFileSystemWindowsConfiguration/Aliases
  :cognitect.aws.fsx/AlternateDNSNames)

(s/def
  :cognitect.aws.fsx.CreateFileSystemWindowsConfiguration/ThroughputCapacity
  :cognitect.aws.fsx/MegabytesPerSecond)

(s/def
  :cognitect.aws.fsx.CreateFileSystemWindowsConfiguration/DailyAutomaticBackupStartTime
  :cognitect.aws.fsx/DailyTime)

(s/def
  :cognitect.aws.fsx.CreateFileSystemWindowsConfiguration/WeeklyMaintenanceStartTime
  :cognitect.aws.fsx/WeeklyTime)

(s/def
  :cognitect.aws.fsx.CreateFileSystemWindowsConfiguration/CopyTagsToBackups
  :cognitect.aws.fsx/Flag)

(s/def
  :cognitect.aws.fsx.CreateFileSystemWindowsConfiguration/ActiveDirectoryId
  :cognitect.aws.fsx/DirectoryId)

(s/def :cognitect.aws.fsx.UpdateFileSystemResponse/FileSystem :cognitect.aws.fsx/FileSystem)

(s/def
  :cognitect.aws.fsx.Snapshot/LifecycleTransitionReason
  :cognitect.aws.fsx/LifecycleTransitionReason)

(s/def :cognitect.aws.fsx.Snapshot/Tags :cognitect.aws.fsx/Tags)

(s/def :cognitect.aws.fsx.Snapshot/AdministrativeActions :cognitect.aws.fsx/AdministrativeActions)

(s/def :cognitect.aws.fsx.Snapshot/ResourceARN :cognitect.aws.fsx/ResourceARN)

(s/def :cognitect.aws.fsx.Snapshot/CreationTime :cognitect.aws.fsx/CreationTime)

(s/def :cognitect.aws.fsx.Snapshot/VolumeId :cognitect.aws.fsx/VolumeId)

(s/def :cognitect.aws.fsx.Snapshot/Name :cognitect.aws.fsx/SnapshotName)

(s/def :cognitect.aws.fsx.Snapshot/Lifecycle :cognitect.aws.fsx/SnapshotLifecycle)

(s/def :cognitect.aws.fsx.Snapshot/SnapshotId :cognitect.aws.fsx/SnapshotId)

(s/def
  :cognitect.aws.fsx.OpenZFSOriginSnapshotConfiguration/SnapshotARN
  :cognitect.aws.fsx/ResourceARN)

(s/def
  :cognitect.aws.fsx.OpenZFSOriginSnapshotConfiguration/CopyStrategy
  :cognitect.aws.fsx/OpenZFSCopyStrategy)

(s/def :cognitect.aws.fsx.DeleteBackupRequest/BackupId :cognitect.aws.fsx/BackupId)

(s/def
  :cognitect.aws.fsx.DeleteBackupRequest/ClientRequestToken
  :cognitect.aws.fsx/ClientRequestToken)

(s/def
  :cognitect.aws.fsx.WindowsAuditLogCreateConfiguration/FileAccessAuditLogLevel
  :cognitect.aws.fsx/WindowsAccessAuditLogLevel)

(s/def
  :cognitect.aws.fsx.WindowsAuditLogCreateConfiguration/FileShareAccessAuditLogLevel
  :cognitect.aws.fsx/WindowsAccessAuditLogLevel)

(s/def
  :cognitect.aws.fsx.WindowsAuditLogCreateConfiguration/AuditLogDestination
  :cognitect.aws.fsx/GeneralARN)

(s/def
  :cognitect.aws.fsx.UpdateDataRepositoryAssociationResponse/Association
  :cognitect.aws.fsx/DataRepositoryAssociation)

(s/def :cognitect.aws.fsx.OpenZFSVolumeConfiguration/VolumePath :cognitect.aws.fsx/VolumePath)

(s/def
  :cognitect.aws.fsx.OpenZFSVolumeConfiguration/StorageCapacityReservationGiB
  :cognitect.aws.fsx/IntegerNoMax)

(s/def :cognitect.aws.fsx.OpenZFSVolumeConfiguration/ParentVolumeId :cognitect.aws.fsx/VolumeId)

(s/def :cognitect.aws.fsx.OpenZFSVolumeConfiguration/ReadOnly :cognitect.aws.fsx/ReadOnly)

(s/def
  :cognitect.aws.fsx.OpenZFSVolumeConfiguration/StorageCapacityQuotaGiB
  :cognitect.aws.fsx/IntegerNoMax)

(s/def :cognitect.aws.fsx.OpenZFSVolumeConfiguration/CopyTagsToSnapshots :cognitect.aws.fsx/Flag)

(s/def
  :cognitect.aws.fsx.OpenZFSVolumeConfiguration/NfsExports
  :cognitect.aws.fsx/OpenZFSNfsExports)

(s/def
  :cognitect.aws.fsx.OpenZFSVolumeConfiguration/OriginSnapshot
  :cognitect.aws.fsx/OpenZFSOriginSnapshotConfiguration)

(s/def
  :cognitect.aws.fsx.OpenZFSVolumeConfiguration/UserAndGroupQuotas
  :cognitect.aws.fsx/OpenZFSUserAndGroupQuotas)

(s/def
  :cognitect.aws.fsx.OpenZFSVolumeConfiguration/RecordSizeKiB
  :cognitect.aws.fsx/IntegerRecordSizeKiB)

(s/def
  :cognitect.aws.fsx.OpenZFSVolumeConfiguration/DataCompressionType
  :cognitect.aws.fsx/OpenZFSDataCompressionType)

(s/def
  :cognitect.aws.fsx.DeleteStorageVirtualMachineRequest/ClientRequestToken
  :cognitect.aws.fsx/ClientRequestToken)

(s/def
  :cognitect.aws.fsx.DeleteStorageVirtualMachineRequest/StorageVirtualMachineId
  :cognitect.aws.fsx/StorageVirtualMachineId)

(s/def :cognitect.aws.fsx.DeleteFileSystemLustreResponse/FinalBackupId :cognitect.aws.fsx/BackupId)

(s/def :cognitect.aws.fsx.DeleteFileSystemLustreResponse/FinalBackupTags :cognitect.aws.fsx/Tags)

(s/def
  :cognitect.aws.fsx.CancelDataRepositoryTaskResponse/Lifecycle
  :cognitect.aws.fsx/DataRepositoryTaskLifecycle)

(s/def :cognitect.aws.fsx.CancelDataRepositoryTaskResponse/TaskId :cognitect.aws.fsx/TaskId)

(s/def
  :cognitect.aws.fsx.DeleteSnapshotRequest/ClientRequestToken
  :cognitect.aws.fsx/ClientRequestToken)

(s/def :cognitect.aws.fsx.DeleteSnapshotRequest/SnapshotId :cognitect.aws.fsx/SnapshotId)

(s/def :cognitect.aws.fsx.DeleteFileSystemOpenZFSResponse/FinalBackupId :cognitect.aws.fsx/BackupId)

(s/def :cognitect.aws.fsx.DeleteFileSystemOpenZFSResponse/FinalBackupTags :cognitect.aws.fsx/Tags)

(s/def
  :cognitect.aws.fsx.CreateOpenZFSOriginSnapshotConfiguration/SnapshotARN
  :cognitect.aws.fsx/ResourceARN)

(s/def
  :cognitect.aws.fsx.CreateOpenZFSOriginSnapshotConfiguration/CopyStrategy
  :cognitect.aws.fsx/OpenZFSCopyStrategy)

(s/def
  :cognitect.aws.fsx.CreateDataRepositoryAssociationRequest/FileSystemId
  :cognitect.aws.fsx/FileSystemId)

(s/def
  :cognitect.aws.fsx.CreateDataRepositoryAssociationRequest/FileSystemPath
  :cognitect.aws.fsx/Namespace)

(s/def
  :cognitect.aws.fsx.CreateDataRepositoryAssociationRequest/DataRepositoryPath
  :cognitect.aws.fsx/ArchivePath)

(s/def
  :cognitect.aws.fsx.CreateDataRepositoryAssociationRequest/BatchImportMetaDataOnCreate
  :cognitect.aws.fsx/BatchImportMetaDataOnCreate)

(s/def
  :cognitect.aws.fsx.CreateDataRepositoryAssociationRequest/ImportedFileChunkSize
  :cognitect.aws.fsx/Megabytes)

(s/def
  :cognitect.aws.fsx.CreateDataRepositoryAssociationRequest/S3
  :cognitect.aws.fsx/S3DataRepositoryConfiguration)

(s/def
  :cognitect.aws.fsx.CreateDataRepositoryAssociationRequest/ClientRequestToken
  :cognitect.aws.fsx/ClientRequestToken)

(s/def :cognitect.aws.fsx.CreateDataRepositoryAssociationRequest/Tags :cognitect.aws.fsx/Tags)

(s/def :cognitect.aws.fsx.DeleteFileSystemResponse/FileSystemId :cognitect.aws.fsx/FileSystemId)

(s/def :cognitect.aws.fsx.DeleteFileSystemResponse/Lifecycle :cognitect.aws.fsx/FileSystemLifecycle)

(s/def
  :cognitect.aws.fsx.DeleteFileSystemResponse/WindowsResponse
  :cognitect.aws.fsx/DeleteFileSystemWindowsResponse)

(s/def
  :cognitect.aws.fsx.DeleteFileSystemResponse/LustreResponse
  :cognitect.aws.fsx/DeleteFileSystemLustreResponse)

(s/def
  :cognitect.aws.fsx.DeleteFileSystemResponse/OpenZFSResponse
  :cognitect.aws.fsx/DeleteFileSystemOpenZFSResponse)

(s/def
  :cognitect.aws.fsx.CreateFileSystemOntapConfiguration/AutomaticBackupRetentionDays
  :cognitect.aws.fsx/AutomaticBackupRetentionDays)

(s/def
  :cognitect.aws.fsx.CreateFileSystemOntapConfiguration/PreferredSubnetId
  :cognitect.aws.fsx/SubnetId)

(s/def
  :cognitect.aws.fsx.CreateFileSystemOntapConfiguration/DeploymentType
  :cognitect.aws.fsx/OntapDeploymentType)

(s/def
  :cognitect.aws.fsx.CreateFileSystemOntapConfiguration/EndpointIpAddressRange
  :cognitect.aws.fsx/IpAddressRange)

(s/def
  :cognitect.aws.fsx.CreateFileSystemOntapConfiguration/ThroughputCapacity
  :cognitect.aws.fsx/MegabytesPerSecond)

(s/def
  :cognitect.aws.fsx.CreateFileSystemOntapConfiguration/DailyAutomaticBackupStartTime
  :cognitect.aws.fsx/DailyTime)

(s/def
  :cognitect.aws.fsx.CreateFileSystemOntapConfiguration/FsxAdminPassword
  :cognitect.aws.fsx/AdminPassword)

(s/def
  :cognitect.aws.fsx.CreateFileSystemOntapConfiguration/WeeklyMaintenanceStartTime
  :cognitect.aws.fsx/WeeklyTime)

(s/def
  :cognitect.aws.fsx.CreateFileSystemOntapConfiguration/RouteTableIds
  :cognitect.aws.fsx/RouteTableIds)

(s/def
  :cognitect.aws.fsx.CreateFileSystemOntapConfiguration/DiskIopsConfiguration
  :cognitect.aws.fsx/DiskIopsConfiguration)

(s/def :cognitect.aws.fsx.CreateVolumeFromBackupResponse/Volume :cognitect.aws.fsx/Volume)

(s/def
  :cognitect.aws.fsx.CreateStorageVirtualMachineRequest/ActiveDirectoryConfiguration
  :cognitect.aws.fsx/CreateSvmActiveDirectoryConfiguration)

(s/def
  :cognitect.aws.fsx.CreateStorageVirtualMachineRequest/ClientRequestToken
  :cognitect.aws.fsx/ClientRequestToken)

(s/def
  :cognitect.aws.fsx.CreateStorageVirtualMachineRequest/FileSystemId
  :cognitect.aws.fsx/FileSystemId)

(s/def
  :cognitect.aws.fsx.CreateStorageVirtualMachineRequest/Name
  :cognitect.aws.fsx/StorageVirtualMachineName)

(s/def
  :cognitect.aws.fsx.CreateStorageVirtualMachineRequest/SvmAdminPassword
  :cognitect.aws.fsx/AdminPassword)

(s/def :cognitect.aws.fsx.CreateStorageVirtualMachineRequest/Tags :cognitect.aws.fsx/Tags)

(s/def
  :cognitect.aws.fsx.CreateStorageVirtualMachineRequest/RootVolumeSecurityStyle
  :cognitect.aws.fsx/StorageVirtualMachineRootVolumeSecurityStyle)

(s/def :cognitect.aws.fsx.UpdateFileSystemRequest/FileSystemId :cognitect.aws.fsx/FileSystemId)

(s/def
  :cognitect.aws.fsx.UpdateFileSystemRequest/ClientRequestToken
  :cognitect.aws.fsx/ClientRequestToken)

(s/def
  :cognitect.aws.fsx.UpdateFileSystemRequest/StorageCapacity
  :cognitect.aws.fsx/StorageCapacity)

(s/def
  :cognitect.aws.fsx.UpdateFileSystemRequest/WindowsConfiguration
  :cognitect.aws.fsx/UpdateFileSystemWindowsConfiguration)

(s/def
  :cognitect.aws.fsx.UpdateFileSystemRequest/LustreConfiguration
  :cognitect.aws.fsx/UpdateFileSystemLustreConfiguration)

(s/def
  :cognitect.aws.fsx.UpdateFileSystemRequest/OntapConfiguration
  :cognitect.aws.fsx/UpdateFileSystemOntapConfiguration)

(s/def
  :cognitect.aws.fsx.UpdateFileSystemRequest/OpenZFSConfiguration
  :cognitect.aws.fsx/UpdateFileSystemOpenZFSConfiguration)

(s/def :cognitect.aws.fsx.BackupFailureDetails/Message :cognitect.aws.fsx/ErrorMessage)

(s/def :cognitect.aws.fsx.DeleteFileSystemWindowsResponse/FinalBackupId :cognitect.aws.fsx/BackupId)

(s/def :cognitect.aws.fsx.DeleteFileSystemWindowsResponse/FinalBackupTags :cognitect.aws.fsx/Tags)

(s/def
  :cognitect.aws.fsx.OntapFileSystemConfiguration/AutomaticBackupRetentionDays
  :cognitect.aws.fsx/AutomaticBackupRetentionDays)

(s/def
  :cognitect.aws.fsx.OntapFileSystemConfiguration/PreferredSubnetId
  :cognitect.aws.fsx/SubnetId)

(s/def
  :cognitect.aws.fsx.OntapFileSystemConfiguration/DeploymentType
  :cognitect.aws.fsx/OntapDeploymentType)

(s/def
  :cognitect.aws.fsx.OntapFileSystemConfiguration/EndpointIpAddressRange
  :cognitect.aws.fsx/IpAddressRange)

(s/def
  :cognitect.aws.fsx.OntapFileSystemConfiguration/ThroughputCapacity
  :cognitect.aws.fsx/MegabytesPerSecond)

(s/def
  :cognitect.aws.fsx.OntapFileSystemConfiguration/DailyAutomaticBackupStartTime
  :cognitect.aws.fsx/DailyTime)

(s/def
  :cognitect.aws.fsx.OntapFileSystemConfiguration/WeeklyMaintenanceStartTime
  :cognitect.aws.fsx/WeeklyTime)

(s/def
  :cognitect.aws.fsx.OntapFileSystemConfiguration/Endpoints
  :cognitect.aws.fsx/FileSystemEndpoints)

(s/def
  :cognitect.aws.fsx.OntapFileSystemConfiguration/RouteTableIds
  :cognitect.aws.fsx/RouteTableIds)

(s/def
  :cognitect.aws.fsx.OntapFileSystemConfiguration/DiskIopsConfiguration
  :cognitect.aws.fsx/DiskIopsConfiguration)

(s/def
  :cognitect.aws.fsx.DataRepositoryConfiguration/Lifecycle
  :cognitect.aws.fsx/DataRepositoryLifecycle)

(s/def :cognitect.aws.fsx.DataRepositoryConfiguration/ImportPath :cognitect.aws.fsx/ArchivePath)

(s/def :cognitect.aws.fsx.DataRepositoryConfiguration/ExportPath :cognitect.aws.fsx/ArchivePath)

(s/def
  :cognitect.aws.fsx.DataRepositoryConfiguration/ImportedFileChunkSize
  :cognitect.aws.fsx/Megabytes)

(s/def
  :cognitect.aws.fsx.DataRepositoryConfiguration/AutoImportPolicy
  :cognitect.aws.fsx/AutoImportPolicyType)

(s/def
  :cognitect.aws.fsx.DataRepositoryConfiguration/FailureDetails
  :cognitect.aws.fsx/DataRepositoryFailureDetails)

(s/def
  :cognitect.aws.fsx.Volume/LifecycleTransitionReason
  :cognitect.aws.fsx/LifecycleTransitionReason)

(s/def :cognitect.aws.fsx.Volume/Tags :cognitect.aws.fsx/Tags)

(s/def :cognitect.aws.fsx.Volume/VolumeType :cognitect.aws.fsx/VolumeType)

(s/def :cognitect.aws.fsx.Volume/AdministrativeActions :cognitect.aws.fsx/AdministrativeActions)

(s/def :cognitect.aws.fsx.Volume/ResourceARN :cognitect.aws.fsx/ResourceARN)

(s/def :cognitect.aws.fsx.Volume/FileSystemId :cognitect.aws.fsx/FileSystemId)

(s/def :cognitect.aws.fsx.Volume/CreationTime :cognitect.aws.fsx/CreationTime)

(s/def :cognitect.aws.fsx.Volume/VolumeId :cognitect.aws.fsx/VolumeId)

(s/def :cognitect.aws.fsx.Volume/OpenZFSConfiguration :cognitect.aws.fsx/OpenZFSVolumeConfiguration)

(s/def :cognitect.aws.fsx.Volume/Name :cognitect.aws.fsx/VolumeName)

(s/def :cognitect.aws.fsx.Volume/Lifecycle :cognitect.aws.fsx/VolumeLifecycle)

(s/def :cognitect.aws.fsx.Volume/OntapConfiguration :cognitect.aws.fsx/OntapVolumeConfiguration)

(s/def :cognitect.aws.fsx.DeleteVolumeResponse/VolumeId :cognitect.aws.fsx/VolumeId)

(s/def :cognitect.aws.fsx.DeleteVolumeResponse/Lifecycle :cognitect.aws.fsx/VolumeLifecycle)

(s/def
  :cognitect.aws.fsx.DeleteVolumeResponse/OntapResponse
  :cognitect.aws.fsx/DeleteVolumeOntapResponse)

(s/def :cognitect.aws.fsx.ListTagsForResourceRequest/ResourceARN :cognitect.aws.fsx/ResourceARN)

(s/def :cognitect.aws.fsx.ListTagsForResourceRequest/MaxResults :cognitect.aws.fsx/MaxResults)

(s/def :cognitect.aws.fsx.ListTagsForResourceRequest/NextToken :cognitect.aws.fsx/NextToken)

(s/def :cognitect.aws.fsx.SvmEndpoint/DNSName :cognitect.aws.fsx/DNSName)

(s/def :cognitect.aws.fsx.SvmEndpoint/IpAddresses :cognitect.aws.fsx/OntapEndpointIpAddresses)

(s/def :cognitect.aws.fsx.FileSystemEndpoint/DNSName :cognitect.aws.fsx/DNSName)

(s/def
  :cognitect.aws.fsx.FileSystemEndpoint/IpAddresses
  :cognitect.aws.fsx/OntapEndpointIpAddresses)

(s/def
  :cognitect.aws.fsx.OpenZFSNfsExport/ClientConfigurations
  :cognitect.aws.fsx/OpenZFSClientConfigurations)

(s/def
  :cognitect.aws.fsx.ReleaseFileSystemNfsV3LocksResponse/FileSystem
  :cognitect.aws.fsx/FileSystem)

(s/def
  :cognitect.aws.fsx.UpdateOpenZFSVolumeConfiguration/StorageCapacityReservationGiB
  :cognitect.aws.fsx/IntegerNoMaxFromNegativeOne)

(s/def
  :cognitect.aws.fsx.UpdateOpenZFSVolumeConfiguration/StorageCapacityQuotaGiB
  :cognitect.aws.fsx/IntegerNoMaxFromNegativeOne)

(s/def
  :cognitect.aws.fsx.UpdateOpenZFSVolumeConfiguration/RecordSizeKiB
  :cognitect.aws.fsx/IntegerRecordSizeKiB)

(s/def
  :cognitect.aws.fsx.UpdateOpenZFSVolumeConfiguration/DataCompressionType
  :cognitect.aws.fsx/OpenZFSDataCompressionType)

(s/def
  :cognitect.aws.fsx.UpdateOpenZFSVolumeConfiguration/NfsExports
  :cognitect.aws.fsx/OpenZFSNfsExports)

(s/def
  :cognitect.aws.fsx.UpdateOpenZFSVolumeConfiguration/UserAndGroupQuotas
  :cognitect.aws.fsx/OpenZFSUserAndGroupQuotas)

(s/def :cognitect.aws.fsx.UpdateOpenZFSVolumeConfiguration/ReadOnly :cognitect.aws.fsx/ReadOnly)

(s/def
  :cognitect.aws.fsx.DescribeDataRepositoryTasksResponse/DataRepositoryTasks
  :cognitect.aws.fsx/DataRepositoryTasks)

(s/def
  :cognitect.aws.fsx.DescribeDataRepositoryTasksResponse/NextToken
  :cognitect.aws.fsx/NextToken)

(s/def
  :cognitect.aws.fsx.DescribeDataRepositoryAssociationsResponse/Associations
  :cognitect.aws.fsx/DataRepositoryAssociations)

(s/def
  :cognitect.aws.fsx.DescribeDataRepositoryAssociationsResponse/NextToken
  :cognitect.aws.fsx/NextToken)

(s/def
  :cognitect.aws.fsx.CreateDataRepositoryAssociationResponse/Association
  :cognitect.aws.fsx/DataRepositoryAssociation)

(s/def
  :cognitect.aws.fsx.CreateFileSystemOpenZFSConfiguration/AutomaticBackupRetentionDays
  :cognitect.aws.fsx/AutomaticBackupRetentionDays)

(s/def
  :cognitect.aws.fsx.CreateFileSystemOpenZFSConfiguration/DeploymentType
  :cognitect.aws.fsx/OpenZFSDeploymentType)

(s/def
  :cognitect.aws.fsx.CreateFileSystemOpenZFSConfiguration/ThroughputCapacity
  :cognitect.aws.fsx/MegabytesPerSecond)

(s/def
  :cognitect.aws.fsx.CreateFileSystemOpenZFSConfiguration/DailyAutomaticBackupStartTime
  :cognitect.aws.fsx/DailyTime)

(s/def
  :cognitect.aws.fsx.CreateFileSystemOpenZFSConfiguration/RootVolumeConfiguration
  :cognitect.aws.fsx/OpenZFSCreateRootVolumeConfiguration)

(s/def
  :cognitect.aws.fsx.CreateFileSystemOpenZFSConfiguration/CopyTagsToVolumes
  :cognitect.aws.fsx/Flag)

(s/def
  :cognitect.aws.fsx.CreateFileSystemOpenZFSConfiguration/WeeklyMaintenanceStartTime
  :cognitect.aws.fsx/WeeklyTime)

(s/def
  :cognitect.aws.fsx.CreateFileSystemOpenZFSConfiguration/DiskIopsConfiguration
  :cognitect.aws.fsx/DiskIopsConfiguration)

(s/def
  :cognitect.aws.fsx.CreateFileSystemOpenZFSConfiguration/CopyTagsToBackups
  :cognitect.aws.fsx/Flag)

(s/def
  :cognitect.aws.fsx.FileSystem/LustreConfiguration
  :cognitect.aws.fsx/LustreFileSystemConfiguration)

(s/def :cognitect.aws.fsx.FileSystem/SubnetIds :cognitect.aws.fsx/SubnetIds)

(s/def :cognitect.aws.fsx.FileSystem/DNSName :cognitect.aws.fsx/DNSName)

(s/def :cognitect.aws.fsx.FileSystem/StorageCapacity :cognitect.aws.fsx/StorageCapacity)

(s/def
  :cognitect.aws.fsx.FileSystem/WindowsConfiguration
  :cognitect.aws.fsx/WindowsFileSystemConfiguration)

(s/def :cognitect.aws.fsx.FileSystem/Tags :cognitect.aws.fsx/Tags)

(s/def :cognitect.aws.fsx.FileSystem/FileSystemTypeVersion :cognitect.aws.fsx/FileSystemTypeVersion)

(s/def :cognitect.aws.fsx.FileSystem/FileSystemType :cognitect.aws.fsx/FileSystemType)

(s/def :cognitect.aws.fsx.FileSystem/AdministrativeActions :cognitect.aws.fsx/AdministrativeActions)

(s/def :cognitect.aws.fsx.FileSystem/ResourceARN :cognitect.aws.fsx/ResourceARN)

(s/def :cognitect.aws.fsx.FileSystem/KmsKeyId :cognitect.aws.fsx/KmsKeyId)

(s/def :cognitect.aws.fsx.FileSystem/FailureDetails :cognitect.aws.fsx/FileSystemFailureDetails)

(s/def :cognitect.aws.fsx.FileSystem/OwnerId :cognitect.aws.fsx/AWSAccountId)

(s/def :cognitect.aws.fsx.FileSystem/FileSystemId :cognitect.aws.fsx/FileSystemId)

(s/def :cognitect.aws.fsx.FileSystem/CreationTime :cognitect.aws.fsx/CreationTime)

(s/def
  :cognitect.aws.fsx.FileSystem/OpenZFSConfiguration
  :cognitect.aws.fsx/OpenZFSFileSystemConfiguration)

(s/def :cognitect.aws.fsx.FileSystem/Lifecycle :cognitect.aws.fsx/FileSystemLifecycle)

(s/def :cognitect.aws.fsx.FileSystem/NetworkInterfaceIds :cognitect.aws.fsx/NetworkInterfaceIds)

(s/def
  :cognitect.aws.fsx.FileSystem/OntapConfiguration
  :cognitect.aws.fsx/OntapFileSystemConfiguration)

(s/def :cognitect.aws.fsx.FileSystem/VpcId :cognitect.aws.fsx/VpcId)

(s/def :cognitect.aws.fsx.FileSystem/StorageType :cognitect.aws.fsx/StorageType)

(s/def :cognitect.aws.fsx.DescribeSnapshotsResponse/Snapshots :cognitect.aws.fsx/Snapshots)

(s/def :cognitect.aws.fsx.DescribeSnapshotsResponse/NextToken :cognitect.aws.fsx/NextToken)

(s/def
  :cognitect.aws.fsx.DataRepositoryAssociation/AssociationId
  :cognitect.aws.fsx/DataRepositoryAssociationId)

(s/def :cognitect.aws.fsx.DataRepositoryAssociation/Tags :cognitect.aws.fsx/Tags)

(s/def :cognitect.aws.fsx.DataRepositoryAssociation/ResourceARN :cognitect.aws.fsx/ResourceARN)

(s/def
  :cognitect.aws.fsx.DataRepositoryAssociation/FailureDetails
  :cognitect.aws.fsx/DataRepositoryFailureDetails)

(s/def :cognitect.aws.fsx.DataRepositoryAssociation/FileSystemId :cognitect.aws.fsx/FileSystemId)

(s/def :cognitect.aws.fsx.DataRepositoryAssociation/CreationTime :cognitect.aws.fsx/CreationTime)

(s/def
  :cognitect.aws.fsx.DataRepositoryAssociation/BatchImportMetaDataOnCreate
  :cognitect.aws.fsx/BatchImportMetaDataOnCreate)

(s/def
  :cognitect.aws.fsx.DataRepositoryAssociation/DataRepositoryPath
  :cognitect.aws.fsx/ArchivePath)

(s/def
  :cognitect.aws.fsx.DataRepositoryAssociation/S3
  :cognitect.aws.fsx/S3DataRepositoryConfiguration)

(s/def
  :cognitect.aws.fsx.DataRepositoryAssociation/Lifecycle
  :cognitect.aws.fsx/DataRepositoryLifecycle)

(s/def
  :cognitect.aws.fsx.DataRepositoryAssociation/ImportedFileChunkSize
  :cognitect.aws.fsx/Megabytes)

(s/def :cognitect.aws.fsx.DataRepositoryAssociation/FileSystemPath :cognitect.aws.fsx/Namespace)

(s/def
  :cognitect.aws.fsx.UpdateFileSystemOntapConfiguration/AutomaticBackupRetentionDays
  :cognitect.aws.fsx/AutomaticBackupRetentionDays)

(s/def
  :cognitect.aws.fsx.UpdateFileSystemOntapConfiguration/DailyAutomaticBackupStartTime
  :cognitect.aws.fsx/DailyTime)

(s/def
  :cognitect.aws.fsx.UpdateFileSystemOntapConfiguration/FsxAdminPassword
  :cognitect.aws.fsx/AdminPassword)

(s/def
  :cognitect.aws.fsx.UpdateFileSystemOntapConfiguration/WeeklyMaintenanceStartTime
  :cognitect.aws.fsx/WeeklyTime)

(s/def
  :cognitect.aws.fsx.UpdateFileSystemOntapConfiguration/DiskIopsConfiguration
  :cognitect.aws.fsx/DiskIopsConfiguration)

(s/def
  :cognitect.aws.fsx.UpdateFileSystemOntapConfiguration/ThroughputCapacity
  :cognitect.aws.fsx/MegabytesPerSecond)

(s/def
  :cognitect.aws.fsx.DataRepositoryTaskFilter/Name
  :cognitect.aws.fsx/DataRepositoryTaskFilterName)

(s/def
  :cognitect.aws.fsx.DataRepositoryTaskFilter/Values
  :cognitect.aws.fsx/DataRepositoryTaskFilterValues)

(s/def :cognitect.aws.fsx.FileSystemEndpoints/Intercluster :cognitect.aws.fsx/FileSystemEndpoint)

(s/def :cognitect.aws.fsx.FileSystemEndpoints/Management :cognitect.aws.fsx/FileSystemEndpoint)

(s/def
  :cognitect.aws.fsx.DescribeStorageVirtualMachinesRequest/StorageVirtualMachineIds
  :cognitect.aws.fsx/StorageVirtualMachineIds)

(s/def
  :cognitect.aws.fsx.DescribeStorageVirtualMachinesRequest/Filters
  :cognitect.aws.fsx/StorageVirtualMachineFilters)

(s/def
  :cognitect.aws.fsx.DescribeStorageVirtualMachinesRequest/MaxResults
  :cognitect.aws.fsx/MaxResults)

(s/def
  :cognitect.aws.fsx.DescribeStorageVirtualMachinesRequest/NextToken
  :cognitect.aws.fsx/NextToken)

(s/def
  :cognitect.aws.fsx.UpdateFileSystemLustreConfiguration/WeeklyMaintenanceStartTime
  :cognitect.aws.fsx/WeeklyTime)

(s/def
  :cognitect.aws.fsx.UpdateFileSystemLustreConfiguration/DailyAutomaticBackupStartTime
  :cognitect.aws.fsx/DailyTime)

(s/def
  :cognitect.aws.fsx.UpdateFileSystemLustreConfiguration/AutomaticBackupRetentionDays
  :cognitect.aws.fsx/AutomaticBackupRetentionDays)

(s/def
  :cognitect.aws.fsx.UpdateFileSystemLustreConfiguration/AutoImportPolicy
  :cognitect.aws.fsx/AutoImportPolicyType)

(s/def
  :cognitect.aws.fsx.UpdateFileSystemLustreConfiguration/DataCompressionType
  :cognitect.aws.fsx/DataCompressionType)

(s/def
  :cognitect.aws.fsx.UpdateFileSystemLustreConfiguration/LogConfiguration
  :cognitect.aws.fsx/LustreLogCreateConfiguration)

(s/def :cognitect.aws.fsx.AutoImportPolicy/Events :cognitect.aws.fsx/EventTypes)

(s/def :cognitect.aws.fsx.DescribeFileSystemsResponse/FileSystems :cognitect.aws.fsx/FileSystems)

(s/def :cognitect.aws.fsx.DescribeFileSystemsResponse/NextToken :cognitect.aws.fsx/NextToken)

(s/def
  :cognitect.aws.fsx.DeleteFileSystemWindowsConfiguration/SkipFinalBackup
  :cognitect.aws.fsx/Flag)

(s/def
  :cognitect.aws.fsx.DeleteFileSystemWindowsConfiguration/FinalBackupTags
  :cognitect.aws.fsx/Tags)

(s/def
  :cognitect.aws.fsx.DeleteFileSystemLustreConfiguration/SkipFinalBackup
  :cognitect.aws.fsx/Flag)

(s/def
  :cognitect.aws.fsx.DeleteFileSystemLustreConfiguration/FinalBackupTags
  :cognitect.aws.fsx/Tags)

(s/def
  :cognitect.aws.fsx.WindowsFileSystemConfiguration/SelfManagedActiveDirectoryConfiguration
  :cognitect.aws.fsx/SelfManagedActiveDirectoryAttributes)

(s/def
  :cognitect.aws.fsx.WindowsFileSystemConfiguration/AuditLogConfiguration
  :cognitect.aws.fsx/WindowsAuditLogConfiguration)

(s/def
  :cognitect.aws.fsx.WindowsFileSystemConfiguration/AutomaticBackupRetentionDays
  :cognitect.aws.fsx/AutomaticBackupRetentionDays)

(s/def
  :cognitect.aws.fsx.WindowsFileSystemConfiguration/MaintenanceOperationsInProgress
  :cognitect.aws.fsx/FileSystemMaintenanceOperations)

(s/def
  :cognitect.aws.fsx.WindowsFileSystemConfiguration/PreferredSubnetId
  :cognitect.aws.fsx/SubnetId)

(s/def
  :cognitect.aws.fsx.WindowsFileSystemConfiguration/DeploymentType
  :cognitect.aws.fsx/WindowsDeploymentType)

(s/def :cognitect.aws.fsx.WindowsFileSystemConfiguration/Aliases :cognitect.aws.fsx/Aliases)

(s/def
  :cognitect.aws.fsx.WindowsFileSystemConfiguration/ThroughputCapacity
  :cognitect.aws.fsx/MegabytesPerSecond)

(s/def
  :cognitect.aws.fsx.WindowsFileSystemConfiguration/DailyAutomaticBackupStartTime
  :cognitect.aws.fsx/DailyTime)

(s/def
  :cognitect.aws.fsx.WindowsFileSystemConfiguration/WeeklyMaintenanceStartTime
  :cognitect.aws.fsx/WeeklyTime)

(s/def
  :cognitect.aws.fsx.WindowsFileSystemConfiguration/PreferredFileServerIp
  :cognitect.aws.fsx/IpAddress)

(s/def
  :cognitect.aws.fsx.WindowsFileSystemConfiguration/RemoteAdministrationEndpoint
  :cognitect.aws.fsx/DNSName)

(s/def :cognitect.aws.fsx.WindowsFileSystemConfiguration/CopyTagsToBackups :cognitect.aws.fsx/Flag)

(s/def
  :cognitect.aws.fsx.WindowsFileSystemConfiguration/ActiveDirectoryId
  :cognitect.aws.fsx/DirectoryId)

(s/def
  :cognitect.aws.fsx.CreateOntapVolumeConfiguration/JunctionPath
  :cognitect.aws.fsx/JunctionPath)

(s/def
  :cognitect.aws.fsx.CreateOntapVolumeConfiguration/SecurityStyle
  :cognitect.aws.fsx/SecurityStyle)

(s/def
  :cognitect.aws.fsx.CreateOntapVolumeConfiguration/SizeInMegabytes
  :cognitect.aws.fsx/VolumeCapacity)

(s/def
  :cognitect.aws.fsx.CreateOntapVolumeConfiguration/StorageEfficiencyEnabled
  :cognitect.aws.fsx/Flag)

(s/def
  :cognitect.aws.fsx.CreateOntapVolumeConfiguration/StorageVirtualMachineId
  :cognitect.aws.fsx/StorageVirtualMachineId)

(s/def
  :cognitect.aws.fsx.CreateOntapVolumeConfiguration/TieringPolicy
  :cognitect.aws.fsx/TieringPolicy)

(s/def :cognitect.aws.fsx.CreateVolumeFromBackupRequest/BackupId :cognitect.aws.fsx/BackupId)

(s/def
  :cognitect.aws.fsx.CreateVolumeFromBackupRequest/ClientRequestToken
  :cognitect.aws.fsx/ClientRequestToken)

(s/def :cognitect.aws.fsx.CreateVolumeFromBackupRequest/Name :cognitect.aws.fsx/VolumeName)

(s/def
  :cognitect.aws.fsx.CreateVolumeFromBackupRequest/OntapConfiguration
  :cognitect.aws.fsx/CreateOntapVolumeConfiguration)

(s/def :cognitect.aws.fsx.CreateVolumeFromBackupRequest/Tags :cognitect.aws.fsx/Tags)

(s/def :cognitect.aws.fsx.UntagResourceRequest/ResourceARN :cognitect.aws.fsx/ResourceARN)

(s/def :cognitect.aws.fsx.UntagResourceRequest/TagKeys :cognitect.aws.fsx/TagKeys)

(s/def :cognitect.aws.fsx.DataRepositoryTaskFailureDetails/Message :cognitect.aws.fsx/ErrorMessage)

(s/def
  :cognitect.aws.fsx.OpenZFSFileSystemConfiguration/AutomaticBackupRetentionDays
  :cognitect.aws.fsx/AutomaticBackupRetentionDays)

(s/def
  :cognitect.aws.fsx.OpenZFSFileSystemConfiguration/DeploymentType
  :cognitect.aws.fsx/OpenZFSDeploymentType)

(s/def :cognitect.aws.fsx.OpenZFSFileSystemConfiguration/RootVolumeId :cognitect.aws.fsx/VolumeId)

(s/def
  :cognitect.aws.fsx.OpenZFSFileSystemConfiguration/ThroughputCapacity
  :cognitect.aws.fsx/MegabytesPerSecond)

(s/def
  :cognitect.aws.fsx.OpenZFSFileSystemConfiguration/DailyAutomaticBackupStartTime
  :cognitect.aws.fsx/DailyTime)

(s/def :cognitect.aws.fsx.OpenZFSFileSystemConfiguration/CopyTagsToVolumes :cognitect.aws.fsx/Flag)

(s/def
  :cognitect.aws.fsx.OpenZFSFileSystemConfiguration/WeeklyMaintenanceStartTime
  :cognitect.aws.fsx/WeeklyTime)

(s/def
  :cognitect.aws.fsx.OpenZFSFileSystemConfiguration/DiskIopsConfiguration
  :cognitect.aws.fsx/DiskIopsConfiguration)

(s/def :cognitect.aws.fsx.OpenZFSFileSystemConfiguration/CopyTagsToBackups :cognitect.aws.fsx/Flag)

(s/def
  :cognitect.aws.fsx.DeleteVolumeRequest/ClientRequestToken
  :cognitect.aws.fsx/ClientRequestToken)

(s/def :cognitect.aws.fsx.DeleteVolumeRequest/VolumeId :cognitect.aws.fsx/VolumeId)

(s/def
  :cognitect.aws.fsx.DeleteVolumeRequest/OntapConfiguration
  :cognitect.aws.fsx/DeleteVolumeOntapConfiguration)

(s/def
  :cognitect.aws.fsx.DeleteVolumeRequest/OpenZFSConfiguration
  :cognitect.aws.fsx/DeleteVolumeOpenZFSConfiguration)

(s/def
  :cognitect.aws.fsx.UpdateSnapshotRequest/ClientRequestToken
  :cognitect.aws.fsx/ClientRequestToken)

(s/def :cognitect.aws.fsx.UpdateSnapshotRequest/Name :cognitect.aws.fsx/SnapshotName)

(s/def :cognitect.aws.fsx.UpdateSnapshotRequest/SnapshotId :cognitect.aws.fsx/SnapshotId)

(s/def :cognitect.aws.fsx.DeleteSnapshotResponse/SnapshotId :cognitect.aws.fsx/SnapshotId)

(s/def :cognitect.aws.fsx.DeleteSnapshotResponse/Lifecycle :cognitect.aws.fsx/SnapshotLifecycle)

(s/def
  :cognitect.aws.fsx.DeleteVolumeOpenZFSConfiguration/Options
  :cognitect.aws.fsx/DeleteOpenZFSVolumeOptions)

(s/def :cognitect.aws.fsx.DataRepositoryFailureDetails/Message :cognitect.aws.fsx/ErrorMessage)

(s/def :cognitect.aws.fsx.CompletionReport/Enabled :cognitect.aws.fsx/Flag)

(s/def :cognitect.aws.fsx.CompletionReport/Path :cognitect.aws.fsx/ArchivePath)

(s/def :cognitect.aws.fsx.CompletionReport/Format :cognitect.aws.fsx/ReportFormat)

(s/def :cognitect.aws.fsx.CompletionReport/Scope :cognitect.aws.fsx/ReportScope)

(s/def :cognitect.aws.fsx.DataRepositoryTaskStatus/TotalCount :cognitect.aws.fsx/TotalCount)

(s/def :cognitect.aws.fsx.DataRepositoryTaskStatus/SucceededCount :cognitect.aws.fsx/SucceededCount)

(s/def :cognitect.aws.fsx.DataRepositoryTaskStatus/FailedCount :cognitect.aws.fsx/FailedCount)

(s/def
  :cognitect.aws.fsx.DataRepositoryTaskStatus/LastUpdatedTime
  :cognitect.aws.fsx/LastUpdatedTime)

(s/def
  :cognitect.aws.fsx.CreateDataRepositoryTaskResponse/DataRepositoryTask
  :cognitect.aws.fsx/DataRepositoryTask)

(s/def :cognitect.aws.fsx.DescribeSnapshotsRequest/SnapshotIds :cognitect.aws.fsx/SnapshotIds)

(s/def :cognitect.aws.fsx.DescribeSnapshotsRequest/Filters :cognitect.aws.fsx/SnapshotFilters)

(s/def :cognitect.aws.fsx.DescribeSnapshotsRequest/MaxResults :cognitect.aws.fsx/MaxResults)

(s/def :cognitect.aws.fsx.DescribeSnapshotsRequest/NextToken :cognitect.aws.fsx/NextToken)

(s/def :cognitect.aws.fsx.DescribeBackupsRequest/BackupIds :cognitect.aws.fsx/BackupIds)

(s/def :cognitect.aws.fsx.DescribeBackupsRequest/Filters :cognitect.aws.fsx/Filters)

(s/def :cognitect.aws.fsx.DescribeBackupsRequest/MaxResults :cognitect.aws.fsx/MaxResults)

(s/def :cognitect.aws.fsx.DescribeBackupsRequest/NextToken :cognitect.aws.fsx/NextToken)

(s/def :cognitect.aws.fsx.CreateSnapshotResponse/Snapshot :cognitect.aws.fsx/Snapshot)

(s/def :cognitect.aws.fsx.UpdateVolumeResponse/Volume :cognitect.aws.fsx/Volume)

(s/def :cognitect.aws.fsx.LustreLogConfiguration/Level :cognitect.aws.fsx/LustreAccessAuditLogLevel)

(s/def :cognitect.aws.fsx.LustreLogConfiguration/Destination :cognitect.aws.fsx/GeneralARN)

(s/def :cognitect.aws.fsx.CancelDataRepositoryTaskRequest/TaskId :cognitect.aws.fsx/TaskId)

(s/def
  :cognitect.aws.fsx.CopyBackupRequest/ClientRequestToken
  :cognitect.aws.fsx/ClientRequestToken)

(s/def :cognitect.aws.fsx.CopyBackupRequest/SourceBackupId :cognitect.aws.fsx/SourceBackupId)

(s/def :cognitect.aws.fsx.CopyBackupRequest/SourceRegion :cognitect.aws.fsx/Region)

(s/def :cognitect.aws.fsx.CopyBackupRequest/KmsKeyId :cognitect.aws.fsx/KmsKeyId)

(s/def :cognitect.aws.fsx.CopyBackupRequest/CopyTags :cognitect.aws.fsx/Flag)

(s/def :cognitect.aws.fsx.CopyBackupRequest/Tags :cognitect.aws.fsx/Tags)

(s/def
  :cognitect.aws.fsx.UpdateSvmActiveDirectoryConfiguration/SelfManagedActiveDirectoryConfiguration
  :cognitect.aws.fsx/SelfManagedActiveDirectoryConfigurationUpdates)

(s/def
  :cognitect.aws.fsx.RestoreVolumeFromSnapshotRequest/ClientRequestToken
  :cognitect.aws.fsx/ClientRequestToken)

(s/def :cognitect.aws.fsx.RestoreVolumeFromSnapshotRequest/VolumeId :cognitect.aws.fsx/VolumeId)

(s/def :cognitect.aws.fsx.RestoreVolumeFromSnapshotRequest/SnapshotId :cognitect.aws.fsx/SnapshotId)

(s/def
  :cognitect.aws.fsx.RestoreVolumeFromSnapshotRequest/Options
  :cognitect.aws.fsx/RestoreOpenZFSVolumeOptions)

(s/def
  :cognitect.aws.fsx.StorageVirtualMachineFilter/Name
  :cognitect.aws.fsx/StorageVirtualMachineFilterName)

(s/def
  :cognitect.aws.fsx.StorageVirtualMachineFilter/Values
  :cognitect.aws.fsx/StorageVirtualMachineFilterValues)

(s/def :cognitect.aws.fsx.DescribeVolumesResponse/Volumes :cognitect.aws.fsx/Volumes)

(s/def :cognitect.aws.fsx.DescribeVolumesResponse/NextToken :cognitect.aws.fsx/NextToken)

(s/def :cognitect.aws.fsx.Alias/Name :cognitect.aws.fsx/AlternateDNSName)

(s/def :cognitect.aws.fsx.Alias/Lifecycle :cognitect.aws.fsx/AliasLifecycle)

(s/def :cognitect.aws.fsx.AutoExportPolicy/Events :cognitect.aws.fsx/EventTypes)

(s/def
  :cognitect.aws.fsx.DescribeFileSystemAliasesRequest/ClientRequestToken
  :cognitect.aws.fsx/ClientRequestToken)

(s/def
  :cognitect.aws.fsx.DescribeFileSystemAliasesRequest/FileSystemId
  :cognitect.aws.fsx/FileSystemId)

(s/def :cognitect.aws.fsx.DescribeFileSystemAliasesRequest/MaxResults :cognitect.aws.fsx/MaxResults)

(s/def :cognitect.aws.fsx.DescribeFileSystemAliasesRequest/NextToken :cognitect.aws.fsx/NextToken)

(s/def
  :cognitect.aws.fsx.CreateFileSystemLustreConfiguration/ExportPath
  :cognitect.aws.fsx/ArchivePath)

(s/def
  :cognitect.aws.fsx.CreateFileSystemLustreConfiguration/AutomaticBackupRetentionDays
  :cognitect.aws.fsx/AutomaticBackupRetentionDays)

(s/def
  :cognitect.aws.fsx.CreateFileSystemLustreConfiguration/LogConfiguration
  :cognitect.aws.fsx/LustreLogCreateConfiguration)

(s/def
  :cognitect.aws.fsx.CreateFileSystemLustreConfiguration/DeploymentType
  :cognitect.aws.fsx/LustreDeploymentType)

(s/def
  :cognitect.aws.fsx.CreateFileSystemLustreConfiguration/DriveCacheType
  :cognitect.aws.fsx/DriveCacheType)

(s/def
  :cognitect.aws.fsx.CreateFileSystemLustreConfiguration/DailyAutomaticBackupStartTime
  :cognitect.aws.fsx/DailyTime)

(s/def
  :cognitect.aws.fsx.CreateFileSystemLustreConfiguration/AutoImportPolicy
  :cognitect.aws.fsx/AutoImportPolicyType)

(s/def
  :cognitect.aws.fsx.CreateFileSystemLustreConfiguration/WeeklyMaintenanceStartTime
  :cognitect.aws.fsx/WeeklyTime)

(s/def
  :cognitect.aws.fsx.CreateFileSystemLustreConfiguration/DataCompressionType
  :cognitect.aws.fsx/DataCompressionType)

(s/def
  :cognitect.aws.fsx.CreateFileSystemLustreConfiguration/PerUnitStorageThroughput
  :cognitect.aws.fsx/PerUnitStorageThroughput)

(s/def
  :cognitect.aws.fsx.CreateFileSystemLustreConfiguration/ImportedFileChunkSize
  :cognitect.aws.fsx/Megabytes)

(s/def
  :cognitect.aws.fsx.CreateFileSystemLustreConfiguration/ImportPath
  :cognitect.aws.fsx/ArchivePath)

(s/def
  :cognitect.aws.fsx.CreateFileSystemLustreConfiguration/CopyTagsToBackups
  :cognitect.aws.fsx/Flag)

(s/def
  :cognitect.aws.fsx.SvmActiveDirectoryConfiguration/NetBiosName
  :cognitect.aws.fsx/NetBiosAlias)

(s/def
  :cognitect.aws.fsx.SvmActiveDirectoryConfiguration/SelfManagedActiveDirectoryConfiguration
  :cognitect.aws.fsx/SelfManagedActiveDirectoryAttributes)

(s/def
  :cognitect.aws.fsx.DisassociateFileSystemAliasesRequest/ClientRequestToken
  :cognitect.aws.fsx/ClientRequestToken)

(s/def
  :cognitect.aws.fsx.DisassociateFileSystemAliasesRequest/FileSystemId
  :cognitect.aws.fsx/FileSystemId)

(s/def
  :cognitect.aws.fsx.DisassociateFileSystemAliasesRequest/Aliases
  :cognitect.aws.fsx/AlternateDNSNames)

(s/def :cognitect.aws.fsx.DeleteFileSystemRequest/FileSystemId :cognitect.aws.fsx/FileSystemId)

(s/def
  :cognitect.aws.fsx.DeleteFileSystemRequest/ClientRequestToken
  :cognitect.aws.fsx/ClientRequestToken)

(s/def
  :cognitect.aws.fsx.DeleteFileSystemRequest/WindowsConfiguration
  :cognitect.aws.fsx/DeleteFileSystemWindowsConfiguration)

(s/def
  :cognitect.aws.fsx.DeleteFileSystemRequest/LustreConfiguration
  :cognitect.aws.fsx/DeleteFileSystemLustreConfiguration)

(s/def
  :cognitect.aws.fsx.DeleteFileSystemRequest/OpenZFSConfiguration
  :cognitect.aws.fsx/DeleteFileSystemOpenZFSConfiguration)

(s/def :cognitect.aws.fsx.CreateFileSystemResponse/FileSystem :cognitect.aws.fsx/FileSystem)

(s/def
  :cognitect.aws.fsx.DeleteDataRepositoryAssociationRequest/AssociationId
  :cognitect.aws.fsx/DataRepositoryAssociationId)

(s/def
  :cognitect.aws.fsx.DeleteDataRepositoryAssociationRequest/ClientRequestToken
  :cognitect.aws.fsx/ClientRequestToken)

(s/def
  :cognitect.aws.fsx.DeleteDataRepositoryAssociationRequest/DeleteDataInFileSystem
  :cognitect.aws.fsx/DeleteDataInFileSystem)

(s/def :cognitect.aws.fsx.TieringPolicy/CoolingPeriod :cognitect.aws.fsx/CoolingPeriod)

(s/def :cognitect.aws.fsx.TieringPolicy/Name :cognitect.aws.fsx/TieringPolicyName)

(s/def
  :cognitect.aws.fsx.AdministrativeActionFailureDetails/Message
  :cognitect.aws.fsx/ErrorMessage)

(s/def
  :cognitect.aws.fsx.UpdateVolumeRequest/ClientRequestToken
  :cognitect.aws.fsx/ClientRequestToken)

(s/def :cognitect.aws.fsx.UpdateVolumeRequest/VolumeId :cognitect.aws.fsx/VolumeId)

(s/def
  :cognitect.aws.fsx.UpdateVolumeRequest/OntapConfiguration
  :cognitect.aws.fsx/UpdateOntapVolumeConfiguration)

(s/def :cognitect.aws.fsx.UpdateVolumeRequest/Name :cognitect.aws.fsx/VolumeName)

(s/def
  :cognitect.aws.fsx.UpdateVolumeRequest/OpenZFSConfiguration
  :cognitect.aws.fsx/UpdateOpenZFSVolumeConfiguration)

(s/def
  :cognitect.aws.fsx.UpdateOntapVolumeConfiguration/JunctionPath
  :cognitect.aws.fsx/JunctionPath)

(s/def
  :cognitect.aws.fsx.UpdateOntapVolumeConfiguration/SecurityStyle
  :cognitect.aws.fsx/SecurityStyle)

(s/def
  :cognitect.aws.fsx.UpdateOntapVolumeConfiguration/SizeInMegabytes
  :cognitect.aws.fsx/VolumeCapacity)

(s/def
  :cognitect.aws.fsx.UpdateOntapVolumeConfiguration/StorageEfficiencyEnabled
  :cognitect.aws.fsx/Flag)

(s/def
  :cognitect.aws.fsx.UpdateOntapVolumeConfiguration/TieringPolicy
  :cognitect.aws.fsx/TieringPolicy)

(s/def :cognitect.aws.fsx.ListTagsForResourceResponse/Tags :cognitect.aws.fsx/Tags)

(s/def :cognitect.aws.fsx.ListTagsForResourceResponse/NextToken :cognitect.aws.fsx/NextToken)

(s/def :cognitect.aws.fsx.DiskIopsConfiguration/Mode :cognitect.aws.fsx/DiskIopsConfigurationMode)

(s/def :cognitect.aws.fsx.DiskIopsConfiguration/Iops :cognitect.aws.fsx/Iops)

(s/def :cognitect.aws.fsx.DataRepositoryTask/TaskId :cognitect.aws.fsx/TaskId)

(s/def :cognitect.aws.fsx.DataRepositoryTask/EndTime :cognitect.aws.fsx/EndTime)

(s/def :cognitect.aws.fsx.DataRepositoryTask/Paths :cognitect.aws.fsx/DataRepositoryTaskPaths)

(s/def :cognitect.aws.fsx.DataRepositoryTask/Report :cognitect.aws.fsx/CompletionReport)

(s/def :cognitect.aws.fsx.DataRepositoryTask/Tags :cognitect.aws.fsx/Tags)

(s/def :cognitect.aws.fsx.DataRepositoryTask/StartTime :cognitect.aws.fsx/StartTime)

(s/def :cognitect.aws.fsx.DataRepositoryTask/ResourceARN :cognitect.aws.fsx/ResourceARN)

(s/def
  :cognitect.aws.fsx.DataRepositoryTask/FailureDetails
  :cognitect.aws.fsx/DataRepositoryTaskFailureDetails)

(s/def :cognitect.aws.fsx.DataRepositoryTask/FileSystemId :cognitect.aws.fsx/FileSystemId)

(s/def :cognitect.aws.fsx.DataRepositoryTask/CreationTime :cognitect.aws.fsx/CreationTime)

(s/def :cognitect.aws.fsx.DataRepositoryTask/Type :cognitect.aws.fsx/DataRepositoryTaskType)

(s/def
  :cognitect.aws.fsx.DataRepositoryTask/Lifecycle
  :cognitect.aws.fsx/DataRepositoryTaskLifecycle)

(s/def :cognitect.aws.fsx.DataRepositoryTask/Status :cognitect.aws.fsx/DataRepositoryTaskStatus)

(s/def
  :cognitect.aws.fsx.DeleteDataRepositoryAssociationResponse/AssociationId
  :cognitect.aws.fsx/DataRepositoryAssociationId)

(s/def
  :cognitect.aws.fsx.DeleteDataRepositoryAssociationResponse/Lifecycle
  :cognitect.aws.fsx/DataRepositoryLifecycle)

(s/def
  :cognitect.aws.fsx.DeleteDataRepositoryAssociationResponse/DeleteDataInFileSystem
  :cognitect.aws.fsx/DeleteDataInFileSystem)

(s/def
  :cognitect.aws.fsx.UpdateDataRepositoryAssociationRequest/AssociationId
  :cognitect.aws.fsx/DataRepositoryAssociationId)

(s/def
  :cognitect.aws.fsx.UpdateDataRepositoryAssociationRequest/ClientRequestToken
  :cognitect.aws.fsx/ClientRequestToken)

(s/def
  :cognitect.aws.fsx.UpdateDataRepositoryAssociationRequest/ImportedFileChunkSize
  :cognitect.aws.fsx/Megabytes)

(s/def
  :cognitect.aws.fsx.UpdateDataRepositoryAssociationRequest/S3
  :cognitect.aws.fsx/S3DataRepositoryConfiguration)

(s/def
  :cognitect.aws.fsx.ReleaseFileSystemNfsV3LocksRequest/FileSystemId
  :cognitect.aws.fsx/FileSystemId)

(s/def
  :cognitect.aws.fsx.ReleaseFileSystemNfsV3LocksRequest/ClientRequestToken
  :cognitect.aws.fsx/ClientRequestToken)

(s/def
  :cognitect.aws.fsx.CreateOpenZFSVolumeConfiguration/StorageCapacityReservationGiB
  :cognitect.aws.fsx/IntegerNoMaxFromNegativeOne)

(s/def
  :cognitect.aws.fsx.CreateOpenZFSVolumeConfiguration/ParentVolumeId
  :cognitect.aws.fsx/VolumeId)

(s/def :cognitect.aws.fsx.CreateOpenZFSVolumeConfiguration/ReadOnly :cognitect.aws.fsx/ReadOnly)

(s/def
  :cognitect.aws.fsx.CreateOpenZFSVolumeConfiguration/StorageCapacityQuotaGiB
  :cognitect.aws.fsx/IntegerNoMaxFromNegativeOne)

(s/def
  :cognitect.aws.fsx.CreateOpenZFSVolumeConfiguration/CopyTagsToSnapshots
  :cognitect.aws.fsx/Flag)

(s/def
  :cognitect.aws.fsx.CreateOpenZFSVolumeConfiguration/NfsExports
  :cognitect.aws.fsx/OpenZFSNfsExports)

(s/def
  :cognitect.aws.fsx.CreateOpenZFSVolumeConfiguration/OriginSnapshot
  :cognitect.aws.fsx/CreateOpenZFSOriginSnapshotConfiguration)

(s/def
  :cognitect.aws.fsx.CreateOpenZFSVolumeConfiguration/UserAndGroupQuotas
  :cognitect.aws.fsx/OpenZFSUserAndGroupQuotas)

(s/def
  :cognitect.aws.fsx.CreateOpenZFSVolumeConfiguration/RecordSizeKiB
  :cognitect.aws.fsx/IntegerRecordSizeKiB)

(s/def
  :cognitect.aws.fsx.CreateOpenZFSVolumeConfiguration/DataCompressionType
  :cognitect.aws.fsx/OpenZFSDataCompressionType)

(s/def :cognitect.aws.fsx.CreateBackupRequest/FileSystemId :cognitect.aws.fsx/FileSystemId)

(s/def
  :cognitect.aws.fsx.CreateBackupRequest/ClientRequestToken
  :cognitect.aws.fsx/ClientRequestToken)

(s/def :cognitect.aws.fsx.CreateBackupRequest/Tags :cognitect.aws.fsx/Tags)

(s/def :cognitect.aws.fsx.CreateBackupRequest/VolumeId :cognitect.aws.fsx/VolumeId)

(s/def :cognitect.aws.fsx.RestoreVolumeFromSnapshotResponse/VolumeId :cognitect.aws.fsx/VolumeId)

(s/def
  :cognitect.aws.fsx.RestoreVolumeFromSnapshotResponse/Lifecycle
  :cognitect.aws.fsx/VolumeLifecycle)

(s/def
  :cognitect.aws.fsx.SelfManagedActiveDirectoryAttributes/DomainName
  :cognitect.aws.fsx/ActiveDirectoryFullyQualifiedName)

(s/def
  :cognitect.aws.fsx.SelfManagedActiveDirectoryAttributes/OrganizationalUnitDistinguishedName
  :cognitect.aws.fsx/OrganizationalUnitDistinguishedName)

(s/def
  :cognitect.aws.fsx.SelfManagedActiveDirectoryAttributes/FileSystemAdministratorsGroup
  :cognitect.aws.fsx/FileSystemAdministratorsGroupName)

(s/def
  :cognitect.aws.fsx.SelfManagedActiveDirectoryAttributes/UserName
  :cognitect.aws.fsx/DirectoryUserName)

(s/def :cognitect.aws.fsx.SelfManagedActiveDirectoryAttributes/DnsIps :cognitect.aws.fsx/DnsIps)

(s/def :cognitect.aws.fsx.SvmEndpoints/Iscsi :cognitect.aws.fsx/SvmEndpoint)

(s/def :cognitect.aws.fsx.SvmEndpoints/Management :cognitect.aws.fsx/SvmEndpoint)

(s/def :cognitect.aws.fsx.SvmEndpoints/Nfs :cognitect.aws.fsx/SvmEndpoint)

(s/def :cognitect.aws.fsx.SvmEndpoints/Smb :cognitect.aws.fsx/SvmEndpoint)

(s/def :cognitect.aws.fsx.DeleteBackupResponse/BackupId :cognitect.aws.fsx/BackupId)

(s/def :cognitect.aws.fsx.DeleteBackupResponse/Lifecycle :cognitect.aws.fsx/BackupLifecycle)

(s/def :cognitect.aws.fsx.DescribeVolumesRequest/VolumeIds :cognitect.aws.fsx/VolumeIds)

(s/def :cognitect.aws.fsx.DescribeVolumesRequest/Filters :cognitect.aws.fsx/VolumeFilters)

(s/def :cognitect.aws.fsx.DescribeVolumesRequest/MaxResults :cognitect.aws.fsx/MaxResults)

(s/def :cognitect.aws.fsx.DescribeVolumesRequest/NextToken :cognitect.aws.fsx/NextToken)

(s/def
  :cognitect.aws.fsx.LustreFileSystemConfiguration/AutomaticBackupRetentionDays
  :cognitect.aws.fsx/AutomaticBackupRetentionDays)

(s/def
  :cognitect.aws.fsx.LustreFileSystemConfiguration/LogConfiguration
  :cognitect.aws.fsx/LustreLogConfiguration)

(s/def
  :cognitect.aws.fsx.LustreFileSystemConfiguration/DeploymentType
  :cognitect.aws.fsx/LustreDeploymentType)

(s/def
  :cognitect.aws.fsx.LustreFileSystemConfiguration/MountName
  :cognitect.aws.fsx/LustreFileSystemMountName)

(s/def
  :cognitect.aws.fsx.LustreFileSystemConfiguration/DriveCacheType
  :cognitect.aws.fsx/DriveCacheType)

(s/def
  :cognitect.aws.fsx.LustreFileSystemConfiguration/DailyAutomaticBackupStartTime
  :cognitect.aws.fsx/DailyTime)

(s/def
  :cognitect.aws.fsx.LustreFileSystemConfiguration/DataRepositoryConfiguration
  :cognitect.aws.fsx/DataRepositoryConfiguration)

(s/def
  :cognitect.aws.fsx.LustreFileSystemConfiguration/WeeklyMaintenanceStartTime
  :cognitect.aws.fsx/WeeklyTime)

(s/def
  :cognitect.aws.fsx.LustreFileSystemConfiguration/DataCompressionType
  :cognitect.aws.fsx/DataCompressionType)

(s/def
  :cognitect.aws.fsx.LustreFileSystemConfiguration/PerUnitStorageThroughput
  :cognitect.aws.fsx/PerUnitStorageThroughput)

(s/def :cognitect.aws.fsx.LustreFileSystemConfiguration/CopyTagsToBackups :cognitect.aws.fsx/Flag)

