;; Copyright (c) Cognitect, Inc.
;; All rights reserved.

(ns cognitect.aws.guardduty.specs
  (:require [clojure.spec.alpha :as s] [clojure.spec.gen.alpha :as gen]))

(s/def :cognitect.aws/client map?)

(s/def :core.async/channel any?)

(s/def :cognitect.aws.guardduty/DeleteThreatIntelSetResponse (s/keys))

(s/def
  :cognitect.aws.guardduty/TrustedEntitySetStatus
  (s/spec
    string?
    :gen
    #(s/gen #{"DEACTIVATING" "ACTIVATING" "ERROR" "INACTIVE" "ACTIVE" "DELETE_PENDING" "DELETED"})))

(s/def
  :cognitect.aws.guardduty/FindingType
  (s/spec
    (s/and string? #(<= 1 (count %) 50))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 1 50) #(gen/vector (gen/char-alpha) %))))))

(s/def :cognitect.aws.guardduty/Actors (s/coll-of :cognitect.aws.guardduty/Actor :max-count 400))

(s/def
  :cognitect.aws.guardduty/DataSourceFreeTrial
  (s/keys :opt-un [:cognitect.aws.guardduty.DataSourceFreeTrial/FreeTrialDaysRemaining]))

(s/def
  :cognitect.aws.guardduty/UsageFeature
  (s/spec
    string?
    :gen
    #(s/gen
      #{"EBS_MALWARE_PROTECTION"
        "EKS_RUNTIME_MONITORING"
        "LAMBDA_NETWORK_LOGS"
        "RDS_DBI_PROTECTION_PROVISIONED"
        "RDS_LOGIN_EVENTS"
        "S3_DATA_EVENTS"
        "EKS_AUDIT_LOGS"
        "EC2_RUNTIME_MONITORING"
        "RDS_DBI_PROTECTION_SERVERLESS"
        "FLOW_LOGS"
        "FARGATE_RUNTIME_MONITORING"
        "CLOUD_TRAIL"
        "DNS_LOGS"})))

(s/def
  :cognitect.aws.guardduty/IndicatorValues
  (s/coll-of :cognitect.aws.guardduty/IndicatorValueString :min-count 1 :max-count 400))

(s/def
  :cognitect.aws.guardduty/OrganizationFeatureStatistics
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.OrganizationFeatureStatistics/EnabledAccountsCount
     :cognitect.aws.guardduty.OrganizationFeatureStatistics/AdditionalConfiguration
     :cognitect.aws.guardduty.OrganizationFeatureStatistics/Name]))

(s/def :cognitect.aws.guardduty/SubnetIds (s/coll-of :cognitect.aws.guardduty/String))

(s/def
  :cognitect.aws.guardduty/CreateIPSetResponse
  (s/keys :req-un [:cognitect.aws.guardduty.CreateIPSetResponse/IpSetId]))

(s/def :cognitect.aws.guardduty/ScanResult (s/spec string? :gen #(s/gen #{"INFECTED" "CLEAN"})))

(s/def
  :cognitect.aws.guardduty/GetDetectorResponse
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.GetDetectorResponse/ServiceRole
     :cognitect.aws.guardduty.GetDetectorResponse/Status]
    :opt-un
    [:cognitect.aws.guardduty.GetDetectorResponse/Tags
     :cognitect.aws.guardduty.GetDetectorResponse/DataSources
     :cognitect.aws.guardduty.GetDetectorResponse/CreatedAt
     :cognitect.aws.guardduty.GetDetectorResponse/UpdatedAt
     :cognitect.aws.guardduty.GetDetectorResponse/FindingPublishingFrequency
     :cognitect.aws.guardduty.GetDetectorResponse/Features]))

(s/def :cognitect.aws.guardduty/DataSourceList (s/coll-of :cognitect.aws.guardduty/DataSource))

(s/def
  :cognitect.aws.guardduty/FilterDescription
  (s/spec
    (s/and string? #(<= 0 (count %) 512))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 0 512) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.guardduty/SignalDescription
  (s/spec
    (s/and string? #(>= 2000 (count %)))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 0 2000) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.guardduty/SecurityContext
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.SecurityContext/Privileged
     :cognitect.aws.guardduty.SecurityContext/AllowPrivilegeEscalation]))

(s/def :cognitect.aws.guardduty/FlagsList (s/coll-of :cognitect.aws.guardduty/String))

(s/def :cognitect.aws.guardduty/MemoryRegionsList (s/coll-of :cognitect.aws.guardduty/String))

(s/def
  :cognitect.aws.guardduty/Service
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.Service/MalwareScanDetails
     :cognitect.aws.guardduty.Service/Count
     :cognitect.aws.guardduty.Service/RuntimeDetails
     :cognitect.aws.guardduty.Service/Evidence
     :cognitect.aws.guardduty.Service/EventFirstSeen
     :cognitect.aws.guardduty.Service/AdditionalInfo
     :cognitect.aws.guardduty.Service/EbsVolumeScanDetails
     :cognitect.aws.guardduty.Service/EventLastSeen
     :cognitect.aws.guardduty.Service/FeatureName
     :cognitect.aws.guardduty.Service/DetectorId
     :cognitect.aws.guardduty.Service/Detection
     :cognitect.aws.guardduty.Service/ServiceName
     :cognitect.aws.guardduty.Service/ResourceRole
     :cognitect.aws.guardduty.Service/UserFeedback
     :cognitect.aws.guardduty.Service/Action
     :cognitect.aws.guardduty.Service/Archived]))

(s/def :cognitect.aws.guardduty/ActorIds (s/coll-of :cognitect.aws.guardduty/String :max-count 400))

(s/def
  :cognitect.aws.guardduty/TagResourceRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.TagResourceRequest/ResourceArn
     :cognitect.aws.guardduty.TagResourceRequest/Tags]))

(s/def
  :cognitect.aws.guardduty/ManagementType
  (s/spec string? :gen #(s/gen #{"DISABLED" "AUTO_MANAGED" "MANUAL"})))

(s/def
  :cognitect.aws.guardduty/ListCoverageResponse
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.ListCoverageResponse/Resources]
    :opt-un
    [:cognitect.aws.guardduty.ListCoverageResponse/NextToken]))

(s/def :cognitect.aws.guardduty/DisassociateFromMasterAccountResponse (s/keys))

(s/def :cognitect.aws.guardduty/ResourceType (s/spec string? :gen #(s/gen #{"EC2" "ECS" "EKS"})))

(s/def
  :cognitect.aws.guardduty/UpdatePublishingDestinationRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.UpdatePublishingDestinationRequest/DetectorId
     :cognitect.aws.guardduty.UpdatePublishingDestinationRequest/DestinationId]
    :opt-un
    [:cognitect.aws.guardduty.UpdatePublishingDestinationRequest/DestinationProperties]))

(s/def
  :cognitect.aws.guardduty/DescribeMalwareScansResponse
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.DescribeMalwareScansResponse/Scans]
    :opt-un
    [:cognitect.aws.guardduty.DescribeMalwareScansResponse/NextToken]))

(s/def
  :cognitect.aws.guardduty/ActorProcess
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.ActorProcess/Name :cognitect.aws.guardduty.ActorProcess/Path]
    :opt-un
    [:cognitect.aws.guardduty.ActorProcess/Sha256]))

(s/def :cognitect.aws.guardduty/Neq (s/coll-of :cognitect.aws.guardduty/String))

(s/def
  :cognitect.aws.guardduty/VolumeDetail
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.VolumeDetail/VolumeArn
     :cognitect.aws.guardduty.VolumeDetail/VolumeSizeInGB
     :cognitect.aws.guardduty.VolumeDetail/KmsKeyArn
     :cognitect.aws.guardduty.VolumeDetail/SnapshotArn
     :cognitect.aws.guardduty.VolumeDetail/EncryptionType
     :cognitect.aws.guardduty.VolumeDetail/VolumeType
     :cognitect.aws.guardduty.VolumeDetail/DeviceName]))

(s/def
  :cognitect.aws.guardduty/StartMalwareScanResponse
  (s/keys :opt-un [:cognitect.aws.guardduty.StartMalwareScanResponse/ScanId]))

(s/def
  :cognitect.aws.guardduty/HighestSeverityThreatDetails
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.HighestSeverityThreatDetails/Severity
     :cognitect.aws.guardduty.HighestSeverityThreatDetails/Count
     :cognitect.aws.guardduty.HighestSeverityThreatDetails/ThreatName]))

(s/def :cognitect.aws.guardduty/UnarchiveFindingsResponse (s/keys))

(s/def
  :cognitect.aws.guardduty/UsageCriteria
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.UsageCriteria/DataSources
     :cognitect.aws.guardduty.UsageCriteria/Resources
     :cognitect.aws.guardduty.UsageCriteria/AccountIds
     :cognitect.aws.guardduty.UsageCriteria/Features]))

(s/def
  :cognitect.aws.guardduty/AdditionalSequenceTypes
  (s/coll-of :cognitect.aws.guardduty/FindingType))

(s/def
  :cognitect.aws.guardduty/DestinationProperties
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.DestinationProperties/KmsKeyArn
     :cognitect.aws.guardduty.DestinationProperties/DestinationArn]))

(s/def
  :cognitect.aws.guardduty/CreateTrustedEntitySetResponse
  (s/keys :req-un [:cognitect.aws.guardduty.CreateTrustedEntitySetResponse/TrustedEntitySetId]))

(s/def
  :cognitect.aws.guardduty/PublishingStatus
  (s/spec
    string?
    :gen
    #(s/gen
      #{"UNABLE_TO_PUBLISH_FIX_DESTINATION_PROPERTY"
        "PUBLISHING"
        "STOPPED"
        "PENDING_VERIFICATION"})))

(s/def
  :cognitect.aws.guardduty/Ec2Instance
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.Ec2Instance/Platform
     :cognitect.aws.guardduty.Ec2Instance/InstanceType
     :cognitect.aws.guardduty.Ec2Instance/ImageDescription
     :cognitect.aws.guardduty.Ec2Instance/AvailabilityZone
     :cognitect.aws.guardduty.Ec2Instance/Ec2NetworkInterfaceUids
     :cognitect.aws.guardduty.Ec2Instance/InstanceState
     :cognitect.aws.guardduty.Ec2Instance/ProductCodes
     :cognitect.aws.guardduty.Ec2Instance/OutpostArn
     :cognitect.aws.guardduty.Ec2Instance/IamInstanceProfile]))

(s/def
  :cognitect.aws.guardduty/AccountDetail
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.AccountDetail/AccountId
     :cognitect.aws.guardduty.AccountDetail/Email]))

(s/def
  :cognitect.aws.guardduty/GetRemainingFreeTrialDaysResponse
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.GetRemainingFreeTrialDaysResponse/Accounts
     :cognitect.aws.guardduty.GetRemainingFreeTrialDaysResponse/UnprocessedAccounts]))

(s/def
  :cognitect.aws.guardduty/ThreatIntelligenceDetails
  (s/coll-of :cognitect.aws.guardduty/ThreatIntelligenceDetail))

(s/def :cognitect.aws.guardduty/Sources (s/coll-of :cognitect.aws.guardduty/String))

(s/def
  :cognitect.aws.guardduty/CreateThreatIntelSetResponse
  (s/keys :req-un [:cognitect.aws.guardduty.CreateThreatIntelSetResponse/ThreatIntelSetId]))

(s/def
  :cognitect.aws.guardduty/LocalPortDetails
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.LocalPortDetails/PortName
     :cognitect.aws.guardduty.LocalPortDetails/Port]))

(s/def
  :cognitect.aws.guardduty/FilterCriteria
  (s/keys :opt-un [:cognitect.aws.guardduty.FilterCriteria/FilterCriterion]))

(s/def
  :cognitect.aws.guardduty/DisassociateMembersResponse
  (s/keys :req-un [:cognitect.aws.guardduty.DisassociateMembersResponse/UnprocessedAccounts]))

(s/def
  :cognitect.aws.guardduty/OrganizationFeatureConfigurationResult
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.OrganizationFeatureConfigurationResult/AutoEnable
     :cognitect.aws.guardduty.OrganizationFeatureConfigurationResult/AdditionalConfiguration
     :cognitect.aws.guardduty.OrganizationFeatureConfigurationResult/Name]))

(s/def
  :cognitect.aws.guardduty/DeclineInvitationsResponse
  (s/keys :req-un [:cognitect.aws.guardduty.DeclineInvitationsResponse/UnprocessedAccounts]))

(s/def
  :cognitect.aws.guardduty/Members
  (s/coll-of :cognitect.aws.guardduty/Member :min-count 0 :max-count 50))

(s/def
  :cognitect.aws.guardduty/FargateDetails
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.FargateDetails/ManagementType
     :cognitect.aws.guardduty.FargateDetails/Issues]))

(s/def
  :cognitect.aws.guardduty/Resources
  (s/coll-of :cognitect.aws.guardduty/ResourceV2 :max-count 400))

(s/def
  :cognitect.aws.guardduty/GetMalwareProtectionPlanResponse
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.GetMalwareProtectionPlanResponse/Actions
     :cognitect.aws.guardduty.GetMalwareProtectionPlanResponse/StatusReasons
     :cognitect.aws.guardduty.GetMalwareProtectionPlanResponse/Tags
     :cognitect.aws.guardduty.GetMalwareProtectionPlanResponse/Status
     :cognitect.aws.guardduty.GetMalwareProtectionPlanResponse/Role
     :cognitect.aws.guardduty.GetMalwareProtectionPlanResponse/Arn
     :cognitect.aws.guardduty.GetMalwareProtectionPlanResponse/CreatedAt
     :cognitect.aws.guardduty.GetMalwareProtectionPlanResponse/ProtectedResource]))

(s/def
  :cognitect.aws.guardduty/GetMasterAccountRequest
  (s/keys :req-un [:cognitect.aws.guardduty.GetMasterAccountRequest/DetectorId]))

(s/def
  :cognitect.aws.guardduty/Tag
  (s/keys :opt-un [:cognitect.aws.guardduty.Tag/Key :cognitect.aws.guardduty.Tag/Value]))

(s/def
  :cognitect.aws.guardduty/ResourceStatistics
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.ResourceStatistics/LastGeneratedAt
     :cognitect.aws.guardduty.ResourceStatistics/ResourceId
     :cognitect.aws.guardduty.ResourceStatistics/TotalFindings
     :cognitect.aws.guardduty.ResourceStatistics/AccountId
     :cognitect.aws.guardduty.ResourceStatistics/ResourceType]))

(s/def :cognitect.aws.guardduty/Eq (s/coll-of :cognitect.aws.guardduty/String))

(s/def
  :cognitect.aws.guardduty/AwsApiCallAction
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.AwsApiCallAction/AffectedResources
     :cognitect.aws.guardduty.AwsApiCallAction/Api
     :cognitect.aws.guardduty.AwsApiCallAction/CallerType
     :cognitect.aws.guardduty.AwsApiCallAction/ErrorCode
     :cognitect.aws.guardduty.AwsApiCallAction/RemoteIpDetails
     :cognitect.aws.guardduty.AwsApiCallAction/UserAgent
     :cognitect.aws.guardduty.AwsApiCallAction/RemoteAccountDetails
     :cognitect.aws.guardduty.AwsApiCallAction/DomainDetails
     :cognitect.aws.guardduty.AwsApiCallAction/ServiceName]))

(s/def
  :cognitect.aws.guardduty/GetThreatEntitySetResponse
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.GetThreatEntitySetResponse/Name
     :cognitect.aws.guardduty.GetThreatEntitySetResponse/Format
     :cognitect.aws.guardduty.GetThreatEntitySetResponse/Location
     :cognitect.aws.guardduty.GetThreatEntitySetResponse/Status]
    :opt-un
    [:cognitect.aws.guardduty.GetThreatEntitySetResponse/Tags
     :cognitect.aws.guardduty.GetThreatEntitySetResponse/ExpectedBucketOwner
     :cognitect.aws.guardduty.GetThreatEntitySetResponse/CreatedAt
     :cognitect.aws.guardduty.GetThreatEntitySetResponse/ErrorDetails
     :cognitect.aws.guardduty.GetThreatEntitySetResponse/UpdatedAt]))

(s/def
  :cognitect.aws.guardduty/UsageTopAccountResult
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.UsageTopAccountResult/Total
     :cognitect.aws.guardduty.UsageTopAccountResult/AccountId]))

(s/def
  :cognitect.aws.guardduty/ScanResultDetails
  (s/keys :opt-un [:cognitect.aws.guardduty.ScanResultDetails/ScanResult]))

(s/def
  :cognitect.aws.guardduty/Signals
  (s/coll-of :cognitect.aws.guardduty/Signal :min-count 1 :max-count 100))

(s/def :cognitect.aws.guardduty/Groups (s/coll-of :cognitect.aws.guardduty/String))

(s/def
  :cognitect.aws.guardduty/ContainerImageUid
  (s/spec
    (s/and string? #(<= 1 (count %) 1024))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 1 1024) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.guardduty/OrganizationAdditionalConfigurationResults
  (s/coll-of :cognitect.aws.guardduty/OrganizationAdditionalConfigurationResult))

(s/def
  :cognitect.aws.guardduty/ScanEc2InstanceWithFindings
  (s/keys :opt-un [:cognitect.aws.guardduty.ScanEc2InstanceWithFindings/EbsVolumes]))

(s/def
  :cognitect.aws.guardduty/TagMap
  (s/map-of
    :cognitect.aws.guardduty/TagKey
    :cognitect.aws.guardduty/TagValue
    :min-count
    1
    :max-count
    200))

(s/def
  :cognitect.aws.guardduty/GroupedByAccount
  (s/coll-of :cognitect.aws.guardduty/AccountStatistics))

(s/def :cognitect.aws.guardduty/DeleteIPSetResponse (s/keys))

(s/def
  :cognitect.aws.guardduty/Evidence
  (s/keys :opt-un [:cognitect.aws.guardduty.Evidence/ThreatIntelligenceDetails]))

(s/def
  :cognitect.aws.guardduty/UsageAccountResult
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.UsageAccountResult/Total
     :cognitect.aws.guardduty.UsageAccountResult/AccountId]))

(s/def
  :cognitect.aws.guardduty/FindingPublishingFrequency
  (s/spec string? :gen #(s/gen #{"FIFTEEN_MINUTES" "SIX_HOURS" "ONE_HOUR"})))

(s/def
  :cognitect.aws.guardduty/ThreatIntelSetFormat
  (s/spec string? :gen #(s/gen #{"STIX" "PROOF_POINT" "TXT" "ALIEN_VAULT" "OTX_CSV" "FIRE_EYE"})))

(s/def :cognitect.aws.guardduty/GroupedByDate (s/coll-of :cognitect.aws.guardduty/DateStatistics))

(s/def
  :cognitect.aws.guardduty/DeleteInvitationsRequest
  (s/keys :req-un [:cognitect.aws.guardduty.DeleteInvitationsRequest/AccountIds]))

(s/def
  :cognitect.aws.guardduty/MalwareProtectionPlanStatusReasonsList
  (s/coll-of :cognitect.aws.guardduty/MalwareProtectionPlanStatusReason :min-count 0 :max-count 50))

(s/def :cognitect.aws.guardduty/S3ObjectDetails (s/coll-of :cognitect.aws.guardduty/S3ObjectDetail))

(s/def
  :cognitect.aws.guardduty/ListMembersResponse
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.ListMembersResponse/Members
     :cognitect.aws.guardduty.ListMembersResponse/NextToken]))

(s/def :cognitect.aws.guardduty/TagResourceResponse (s/keys))

(s/def
  :cognitect.aws.guardduty/TrustedEntitySetIds
  (s/coll-of :cognitect.aws.guardduty/String :min-count 0 :max-count 50))

(s/def
  :cognitect.aws.guardduty/ServiceAdditionalInfo
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.ServiceAdditionalInfo/Type
     :cognitect.aws.guardduty.ServiceAdditionalInfo/Value]))

(s/def
  :cognitect.aws.guardduty/GroupByType
  (s/spec string? :gen #(s/gen #{"DATE" "SEVERITY" "ACCOUNT" "RESOURCE" "FINDING_TYPE"})))

(s/def
  :cognitect.aws.guardduty/DetectorAdditionalConfigurations
  (s/coll-of :cognitect.aws.guardduty/DetectorAdditionalConfiguration))

(s/def
  :cognitect.aws.guardduty/GetFindingsResponse
  (s/keys :req-un [:cognitect.aws.guardduty.GetFindingsResponse/Findings]))

(s/def
  :cognitect.aws.guardduty/DeleteTrustedEntitySetRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.DeleteTrustedEntitySetRequest/DetectorId
     :cognitect.aws.guardduty.DeleteTrustedEntitySetRequest/TrustedEntitySetId]))

(s/def
  :cognitect.aws.guardduty/CriterionKey
  (s/spec
    string?
    :gen
    #(s/gen
      #{"SCAN_TYPE"
        "SCAN_ID"
        "EC2_INSTANCE_ARN"
        "SCAN_START_TIME"
        "ACCOUNT_ID"
        "GUARDDUTY_FINDING_ID"
        "SCAN_STATUS"})))

(s/def
  :cognitect.aws.guardduty/DescribeMalwareScansRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.DescribeMalwareScansRequest/DetectorId]
    :opt-un
    [:cognitect.aws.guardduty.DescribeMalwareScansRequest/NextToken
     :cognitect.aws.guardduty.DescribeMalwareScansRequest/FilterCriteria
     :cognitect.aws.guardduty.DescribeMalwareScansRequest/MaxResults
     :cognitect.aws.guardduty.DescribeMalwareScansRequest/SortCriteria]))

(s/def :cognitect.aws.guardduty/UpdatePublishingDestinationResponse (s/keys))

(s/def
  :cognitect.aws.guardduty/OrganizationMalwareProtectionConfigurationResult
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.OrganizationMalwareProtectionConfigurationResult/ScanEc2InstanceWithFindings]))

(s/def
  :cognitect.aws.guardduty/GetTrustedEntitySetResponse
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.GetTrustedEntitySetResponse/Name
     :cognitect.aws.guardduty.GetTrustedEntitySetResponse/Format
     :cognitect.aws.guardduty.GetTrustedEntitySetResponse/Location
     :cognitect.aws.guardduty.GetTrustedEntitySetResponse/Status]
    :opt-un
    [:cognitect.aws.guardduty.GetTrustedEntitySetResponse/Tags
     :cognitect.aws.guardduty.GetTrustedEntitySetResponse/ExpectedBucketOwner
     :cognitect.aws.guardduty.GetTrustedEntitySetResponse/CreatedAt
     :cognitect.aws.guardduty.GetTrustedEntitySetResponse/ErrorDetails
     :cognitect.aws.guardduty.GetTrustedEntitySetResponse/UpdatedAt]))

(s/def
  :cognitect.aws.guardduty/InviteMembersResponse
  (s/keys :req-un [:cognitect.aws.guardduty.InviteMembersResponse/UnprocessedAccounts]))

(s/def
  :cognitect.aws.guardduty/KubernetesRoleBindingDetails
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.KubernetesRoleBindingDetails/Uid
     :cognitect.aws.guardduty.KubernetesRoleBindingDetails/Name
     :cognitect.aws.guardduty.KubernetesRoleBindingDetails/RoleRefKind
     :cognitect.aws.guardduty.KubernetesRoleBindingDetails/Kind
     :cognitect.aws.guardduty.KubernetesRoleBindingDetails/RoleRefName]))

(s/def
  :cognitect.aws.guardduty/KubernetesAuditLogsConfigurationResult
  (s/keys :req-un [:cognitect.aws.guardduty.KubernetesAuditLogsConfigurationResult/Status]))

(s/def
  :cognitect.aws.guardduty/FlowLogsConfigurationResult
  (s/keys :req-un [:cognitect.aws.guardduty.FlowLogsConfigurationResult/Status]))

(s/def
  :cognitect.aws.guardduty/Scan
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.Scan/ResourceDetails
     :cognitect.aws.guardduty.Scan/ScanStatus
     :cognitect.aws.guardduty.Scan/ScanType
     :cognitect.aws.guardduty.Scan/ScanEndTime
     :cognitect.aws.guardduty.Scan/ScanId
     :cognitect.aws.guardduty.Scan/AdminDetectorId
     :cognitect.aws.guardduty.Scan/FileCount
     :cognitect.aws.guardduty.Scan/ScanStartTime
     :cognitect.aws.guardduty.Scan/FailureReason
     :cognitect.aws.guardduty.Scan/AccountId
     :cognitect.aws.guardduty.Scan/AttachedVolumes
     :cognitect.aws.guardduty.Scan/DetectorId
     :cognitect.aws.guardduty.Scan/ScanResultDetails
     :cognitect.aws.guardduty.Scan/TriggerDetails
     :cognitect.aws.guardduty.Scan/TotalBytes]))

(s/def
  :cognitect.aws.guardduty/BucketPolicy
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.BucketPolicy/AllowsPublicReadAccess
     :cognitect.aws.guardduty.BucketPolicy/AllowsPublicWriteAccess]))

(s/def
  :cognitect.aws.guardduty/BlockPublicAccess
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.BlockPublicAccess/IgnorePublicAcls
     :cognitect.aws.guardduty.BlockPublicAccess/BlockPublicPolicy
     :cognitect.aws.guardduty.BlockPublicAccess/RestrictPublicBuckets
     :cognitect.aws.guardduty.BlockPublicAccess/BlockPublicAcls]))

(s/def
  :cognitect.aws.guardduty/DetectorFeatureResult
  (s/spec
    string?
    :gen
    #(s/gen
      #{"EBS_MALWARE_PROTECTION"
        "EKS_RUNTIME_MONITORING"
        "LAMBDA_NETWORK_LOGS"
        "RUNTIME_MONITORING"
        "RDS_LOGIN_EVENTS"
        "S3_DATA_EVENTS"
        "EKS_AUDIT_LOGS"
        "FLOW_LOGS"
        "CLOUD_TRAIL"
        "DNS_LOGS"})))

(s/def
  :cognitect.aws.guardduty/VpcConfig
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.VpcConfig/SubnetIds
     :cognitect.aws.guardduty.VpcConfig/SecurityGroups
     :cognitect.aws.guardduty.VpcConfig/VpcId]))

(s/def :cognitect.aws.guardduty/Lineage (s/coll-of :cognitect.aws.guardduty/LineageObject))

(s/def
  :cognitect.aws.guardduty/MalwareProtectionConfigurationResult
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.MalwareProtectionConfigurationResult/ServiceRole
     :cognitect.aws.guardduty.MalwareProtectionConfigurationResult/ScanEc2InstanceWithFindings]))

(s/def
  :cognitect.aws.guardduty/MemberDataSourceConfiguration
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.MemberDataSourceConfiguration/AccountId]
    :opt-un
    [:cognitect.aws.guardduty.MemberDataSourceConfiguration/DataSources
     :cognitect.aws.guardduty.MemberDataSourceConfiguration/Features]))

(s/def
  :cognitect.aws.guardduty/MalwareProtectionDataSourceFreeTrial
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.MalwareProtectionDataSourceFreeTrial/ScanEc2InstanceWithFindings]))

(s/def :cognitect.aws.guardduty/UpdateMalwareScanSettingsResponse (s/keys))

(s/def
  :cognitect.aws.guardduty/OrganizationAdditionalConfigurations
  (s/coll-of :cognitect.aws.guardduty/OrganizationAdditionalConfiguration))

(s/def
  :cognitect.aws.guardduty/GetFindingsStatisticsRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.GetFindingsStatisticsRequest/DetectorId]
    :opt-un
    [:cognitect.aws.guardduty.GetFindingsStatisticsRequest/GroupBy
     :cognitect.aws.guardduty.GetFindingsStatisticsRequest/OrderBy
     :cognitect.aws.guardduty.GetFindingsStatisticsRequest/MaxResults
     :cognitect.aws.guardduty.GetFindingsStatisticsRequest/FindingStatisticTypes
     :cognitect.aws.guardduty.GetFindingsStatisticsRequest/FindingCriteria]))

(s/def
  :cognitect.aws.guardduty/EnableOrganizationAdminAccountRequest
  (s/keys :req-un [:cognitect.aws.guardduty.EnableOrganizationAdminAccountRequest/AdminAccountId]))

(s/def
  :cognitect.aws.guardduty/AccountFreeTrialInfos
  (s/coll-of :cognitect.aws.guardduty/AccountFreeTrialInfo))

(s/def
  :cognitect.aws.guardduty/MalwareScanDetails
  (s/keys :opt-un [:cognitect.aws.guardduty.MalwareScanDetails/Threats]))

(s/def
  :cognitect.aws.guardduty/InstanceArn
  (s/spec
    #(re-matches
      (re-pattern
        "^arn:(aws|aws-cn|aws-us-gov):[a-z]+:[a-z]+(-[0-9]+|-[a-z]+)+:([0-9]{12}):[a-z\\-]+\\/[a-zA-Z0-9]*$")
      %)
    :gen
    #(gen/string)))

(s/def
  :cognitect.aws.guardduty/PositiveLong
  (s/spec (s/and int? #(<= 0 %)) :gen #(gen/choose 0 Long/MAX_VALUE)))

(s/def
  :cognitect.aws.guardduty/UpdateDetectorRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.UpdateDetectorRequest/DetectorId]
    :opt-un
    [:cognitect.aws.guardduty.UpdateDetectorRequest/Enable
     :cognitect.aws.guardduty.UpdateDetectorRequest/DataSources
     :cognitect.aws.guardduty.UpdateDetectorRequest/FindingPublishingFrequency
     :cognitect.aws.guardduty.UpdateDetectorRequest/Features]))

(s/def
  :cognitect.aws.guardduty/TrustedEntitySetFormat
  (s/spec string? :gen #(s/gen #{"STIX" "PROOF_POINT" "TXT" "ALIEN_VAULT" "OTX_CSV" "FIRE_EYE"})))

(s/def
  :cognitect.aws.guardduty/CoverageFilterCriteria
  (s/keys :opt-un [:cognitect.aws.guardduty.CoverageFilterCriteria/FilterCriterion]))

(s/def
  :cognitect.aws.guardduty/FeatureStatus
  (s/spec string? :gen #(s/gen #{"DISABLED" "ENABLED"})))

(s/def
  :cognitect.aws.guardduty/ProcessSha256
  (s/spec
    (s/and string? #(<= 0 (count %) 1024))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 0 1024) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.guardduty/Invitation
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.Invitation/InvitationId
     :cognitect.aws.guardduty.Invitation/RelationshipStatus
     :cognitect.aws.guardduty.Invitation/AccountId
     :cognitect.aws.guardduty.Invitation/InvitedAt]))

(s/def
  :cognitect.aws.guardduty/OrganizationFeatureStatisticsAdditionalConfiguration
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.OrganizationFeatureStatisticsAdditionalConfiguration/EnabledAccountsCount
     :cognitect.aws.guardduty.OrganizationFeatureStatisticsAdditionalConfiguration/Name]))

(s/def
  :cognitect.aws.guardduty/Container
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.Container/VolumeMounts
     :cognitect.aws.guardduty.Container/ContainerRuntime
     :cognitect.aws.guardduty.Container/SecurityContext
     :cognitect.aws.guardduty.Container/Image
     :cognitect.aws.guardduty.Container/Name
     :cognitect.aws.guardduty.Container/Id
     :cognitect.aws.guardduty.Container/ImagePrefix]))

(s/def
  :cognitect.aws.guardduty/UpdateMalwareProtectionPlanRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.UpdateMalwareProtectionPlanRequest/MalwareProtectionPlanId]
    :opt-un
    [:cognitect.aws.guardduty.UpdateMalwareProtectionPlanRequest/Actions
     :cognitect.aws.guardduty.UpdateMalwareProtectionPlanRequest/Role
     :cognitect.aws.guardduty.UpdateMalwareProtectionPlanRequest/ProtectedResource]))

(s/def
  :cognitect.aws.guardduty/ThreatEntitySetStatus
  (s/spec
    string?
    :gen
    #(s/gen #{"DEACTIVATING" "ACTIVATING" "ERROR" "INACTIVE" "ACTIVE" "DELETE_PENDING" "DELETED"})))

(s/def
  :cognitect.aws.guardduty/AccessDeniedException
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.AccessDeniedException/Type
     :cognitect.aws.guardduty.AccessDeniedException/Message]))

(s/def
  :cognitect.aws.guardduty/ScanStatus
  (s/spec string? :gen #(s/gen #{"COMPLETED" "FAILED" "RUNNING" "SKIPPED"})))

(s/def
  :cognitect.aws.guardduty/DescribePublishingDestinationRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.DescribePublishingDestinationRequest/DetectorId
     :cognitect.aws.guardduty.DescribePublishingDestinationRequest/DestinationId]))

(s/def
  :cognitect.aws.guardduty/OrganizationAdditionalConfiguration
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.OrganizationAdditionalConfiguration/AutoEnable
     :cognitect.aws.guardduty.OrganizationAdditionalConfiguration/Name]))

(s/def
  :cognitect.aws.guardduty/UnprocessedAccount
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.UnprocessedAccount/AccountId
     :cognitect.aws.guardduty.UnprocessedAccount/Result]))

(s/def :cognitect.aws.guardduty/ObservationTexts (s/coll-of :cognitect.aws.guardduty/String))

(s/def
  :cognitect.aws.guardduty/Member
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.Member/AccountId
     :cognitect.aws.guardduty.Member/MasterId
     :cognitect.aws.guardduty.Member/Email
     :cognitect.aws.guardduty.Member/RelationshipStatus
     :cognitect.aws.guardduty.Member/UpdatedAt]
    :opt-un
    [:cognitect.aws.guardduty.Member/DetectorId
     :cognitect.aws.guardduty.Member/AdministratorId
     :cognitect.aws.guardduty.Member/InvitedAt]))

(s/def
  :cognitect.aws.guardduty/Observations
  (s/keys :opt-un [:cognitect.aws.guardduty.Observations/Text]))

(s/def :cognitect.aws.guardduty/ProfileType (s/spec string? :gen #(s/gen #{"FREQUENCY"})))

(s/def
  :cognitect.aws.guardduty/RuntimeDetails
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.RuntimeDetails/Context
     :cognitect.aws.guardduty.RuntimeDetails/Process]))

(s/def
  :cognitect.aws.guardduty/GetMasterAccountResponse
  (s/keys :req-un [:cognitect.aws.guardduty.GetMasterAccountResponse/Master]))

(s/def :cognitect.aws.guardduty/ArchiveFindingsResponse (s/keys))

(s/def
  :cognitect.aws.guardduty/CoverageStatisticsType
  (s/spec string? :gen #(s/gen #{"COUNT_BY_RESOURCE_TYPE" "COUNT_BY_COVERAGE_STATUS"})))

(s/def
  :cognitect.aws.guardduty/GetCoverageStatisticsRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.GetCoverageStatisticsRequest/DetectorId
     :cognitect.aws.guardduty.GetCoverageStatisticsRequest/StatisticsType]
    :opt-un
    [:cognitect.aws.guardduty.GetCoverageStatisticsRequest/FilterCriteria]))

(s/def
  :cognitect.aws.guardduty/AccountId
  (s/spec
    (s/and string? #(<= 12 (count %) 12))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 12 12) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.guardduty/ListMembersRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.ListMembersRequest/DetectorId]
    :opt-un
    [:cognitect.aws.guardduty.ListMembersRequest/NextToken
     :cognitect.aws.guardduty.ListMembersRequest/MaxResults
     :cognitect.aws.guardduty.ListMembersRequest/OnlyAssociated]))

(s/def
  :cognitect.aws.guardduty/ContainerUid
  (s/spec
    (s/and string? #(<= 0 (count %) 256))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 0 256) #(gen/vector (gen/char-alpha) %))))))

(s/def :cognitect.aws.guardduty/S3ObjectUids (s/coll-of :cognitect.aws.guardduty/String))

(s/def
  :cognitect.aws.guardduty/AnomalyUnusualBehaviorFeature
  (s/map-of :cognitect.aws.guardduty/String :cognitect.aws.guardduty/AnomalyObject))

(s/def
  :cognitect.aws.guardduty/ProductCode
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.ProductCode/ProductType :cognitect.aws.guardduty.ProductCode/Code]))

(s/def :cognitect.aws.guardduty/Feedback (s/spec string? :gen #(s/gen #{"USEFUL" "NOT_USEFUL"})))

(s/def
  :cognitect.aws.guardduty/DescribeOrganizationConfigurationResponse
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.DescribeOrganizationConfigurationResponse/MemberAccountLimitReached]
    :opt-un
    [:cognitect.aws.guardduty.DescribeOrganizationConfigurationResponse/AutoEnable
     :cognitect.aws.guardduty.DescribeOrganizationConfigurationResponse/NextToken
     :cognitect.aws.guardduty.DescribeOrganizationConfigurationResponse/DataSources
     :cognitect.aws.guardduty.DescribeOrganizationConfigurationResponse/AutoEnableOrganizationMembers
     :cognitect.aws.guardduty.DescribeOrganizationConfigurationResponse/Features]))

(s/def
  :cognitect.aws.guardduty/ListFiltersResponse
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.ListFiltersResponse/FilterNames]
    :opt-un
    [:cognitect.aws.guardduty.ListFiltersResponse/NextToken]))

(s/def
  :cognitect.aws.guardduty/UpdateThreatEntitySetRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.UpdateThreatEntitySetRequest/DetectorId
     :cognitect.aws.guardduty.UpdateThreatEntitySetRequest/ThreatEntitySetId]
    :opt-un
    [:cognitect.aws.guardduty.UpdateThreatEntitySetRequest/Location
     :cognitect.aws.guardduty.UpdateThreatEntitySetRequest/ExpectedBucketOwner
     :cognitect.aws.guardduty.UpdateThreatEntitySetRequest/Name
     :cognitect.aws.guardduty.UpdateThreatEntitySetRequest/Activate]))

(s/def
  :cognitect.aws.guardduty/DescribeOrganizationConfigurationRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.DescribeOrganizationConfigurationRequest/DetectorId]
    :opt-un
    [:cognitect.aws.guardduty.DescribeOrganizationConfigurationRequest/NextToken
     :cognitect.aws.guardduty.DescribeOrganizationConfigurationRequest/MaxResults]))

(s/def
  :cognitect.aws.guardduty/UpdateOrganizationConfigurationRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.UpdateOrganizationConfigurationRequest/DetectorId]
    :opt-un
    [:cognitect.aws.guardduty.UpdateOrganizationConfigurationRequest/AutoEnable
     :cognitect.aws.guardduty.UpdateOrganizationConfigurationRequest/DataSources
     :cognitect.aws.guardduty.UpdateOrganizationConfigurationRequest/AutoEnableOrganizationMembers
     :cognitect.aws.guardduty.UpdateOrganizationConfigurationRequest/Features]))

(s/def
  :cognitect.aws.guardduty/DetectorAdditionalConfiguration
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.DetectorAdditionalConfiguration/Status
     :cognitect.aws.guardduty.DetectorAdditionalConfiguration/Name]))

(s/def
  :cognitect.aws.guardduty/ThreatIntelSetIds
  (s/coll-of :cognitect.aws.guardduty/String :min-count 0 :max-count 50))

(s/def
  :cognitect.aws.guardduty/GetOrganizationStatisticsResponse
  (s/keys :opt-un [:cognitect.aws.guardduty.GetOrganizationStatisticsResponse/OrganizationDetails]))

(s/def
  :cognitect.aws.guardduty/AnomalyObject
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.AnomalyObject/ProfileSubtype
     :cognitect.aws.guardduty.AnomalyObject/Observations
     :cognitect.aws.guardduty.AnomalyObject/ProfileType]))

(s/def
  :cognitect.aws.guardduty/ResourceUids
  (s/coll-of :cognitect.aws.guardduty/String :max-count 400))

(s/def
  :cognitect.aws.guardduty/ListDetectorsResponse
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.ListDetectorsResponse/DetectorIds]
    :opt-un
    [:cognitect.aws.guardduty.ListDetectorsResponse/NextToken]))

(s/def
  :cognitect.aws.guardduty/ListCoverageRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.ListCoverageRequest/DetectorId]
    :opt-un
    [:cognitect.aws.guardduty.ListCoverageRequest/NextToken
     :cognitect.aws.guardduty.ListCoverageRequest/FilterCriteria
     :cognitect.aws.guardduty.ListCoverageRequest/MaxResults
     :cognitect.aws.guardduty.ListCoverageRequest/SortCriteria]))

(s/def
  :cognitect.aws.guardduty/SecurityGroup
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.SecurityGroup/GroupName
     :cognitect.aws.guardduty.SecurityGroup/GroupId]))

(s/def
  :cognitect.aws.guardduty/DetectorFeatureConfigurationResult
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.DetectorFeatureConfigurationResult/Status
     :cognitect.aws.guardduty.DetectorFeatureConfigurationResult/AdditionalConfiguration
     :cognitect.aws.guardduty.DetectorFeatureConfigurationResult/UpdatedAt
     :cognitect.aws.guardduty.DetectorFeatureConfigurationResult/Name]))

(s/def :cognitect.aws.guardduty/Owner (s/keys :opt-un [:cognitect.aws.guardduty.Owner/Id]))

(s/def
  :cognitect.aws.guardduty/NetworkConnection
  (s/keys :req-un [:cognitect.aws.guardduty.NetworkConnection/Direction]))

(s/def
  :cognitect.aws.guardduty/Destination
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.Destination/DestinationId
     :cognitect.aws.guardduty.Destination/DestinationType
     :cognitect.aws.guardduty.Destination/Status]))

(s/def
  :cognitect.aws.guardduty/DeclineInvitationsRequest
  (s/keys :req-un [:cognitect.aws.guardduty.DeclineInvitationsRequest/AccountIds]))

(s/def :cognitect.aws.guardduty/NotEquals (s/coll-of :cognitect.aws.guardduty/String))

(s/def
  :cognitect.aws.guardduty/Sequence
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.Sequence/Uid
     :cognitect.aws.guardduty.Sequence/Description
     :cognitect.aws.guardduty.Sequence/Signals]
    :opt-un
    [:cognitect.aws.guardduty.Sequence/Endpoints
     :cognitect.aws.guardduty.Sequence/AdditionalSequenceTypes
     :cognitect.aws.guardduty.Sequence/Resources
     :cognitect.aws.guardduty.Sequence/Actors
     :cognitect.aws.guardduty.Sequence/SequenceIndicators]))

(s/def :cognitect.aws.guardduty/OrderBy (s/spec string? :gen #(s/gen #{"DESC" "ASC"})))

(s/def :cognitect.aws.guardduty/Tags (s/coll-of :cognitect.aws.guardduty/Tag))

(s/def
  :cognitect.aws.guardduty/LoginAttribute
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.LoginAttribute/Application
     :cognitect.aws.guardduty.LoginAttribute/User
     :cognitect.aws.guardduty.LoginAttribute/SuccessfulLoginAttempts
     :cognitect.aws.guardduty.LoginAttribute/FailedLoginAttempts]))

(s/def
  :cognitect.aws.guardduty/UsageTopAccountsResultList
  (s/coll-of :cognitect.aws.guardduty/UsageTopAccountsResult))

(s/def :cognitect.aws.guardduty/HostPath (s/keys :opt-un [:cognitect.aws.guardduty.HostPath/Path]))

(s/def
  :cognitect.aws.guardduty/FindingStatisticTypes
  (s/coll-of :cognitect.aws.guardduty/FindingStatisticType :min-count 0 :max-count 10))

(s/def
  :cognitect.aws.guardduty/ResourceV2
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.ResourceV2/Uid :cognitect.aws.guardduty.ResourceV2/ResourceType]
    :opt-un
    [:cognitect.aws.guardduty.ResourceV2/CloudPartition
     :cognitect.aws.guardduty.ResourceV2/Tags
     :cognitect.aws.guardduty.ResourceV2/AccountId
     :cognitect.aws.guardduty.ResourceV2/Data
     :cognitect.aws.guardduty.ResourceV2/Service
     :cognitect.aws.guardduty.ResourceV2/Name
     :cognitect.aws.guardduty.ResourceV2/Region]))

(s/def
  :cognitect.aws.guardduty/MalwareProtectionPlanTaggingActionStatus
  (s/spec string? :gen #(s/gen #{"DISABLED" "ENABLED"})))

(s/def
  :cognitect.aws.guardduty/S3LogsConfigurationResult
  (s/keys :req-un [:cognitect.aws.guardduty.S3LogsConfigurationResult/Status]))

(s/def
  :cognitect.aws.guardduty/EcsTaskDetails
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.EcsTaskDetails/StartedBy
     :cognitect.aws.guardduty.EcsTaskDetails/Containers
     :cognitect.aws.guardduty.EcsTaskDetails/Tags
     :cognitect.aws.guardduty.EcsTaskDetails/Arn
     :cognitect.aws.guardduty.EcsTaskDetails/Version
     :cognitect.aws.guardduty.EcsTaskDetails/LaunchType
     :cognitect.aws.guardduty.EcsTaskDetails/DefinitionArn
     :cognitect.aws.guardduty.EcsTaskDetails/TaskCreatedAt
     :cognitect.aws.guardduty.EcsTaskDetails/Group
     :cognitect.aws.guardduty.EcsTaskDetails/Volumes
     :cognitect.aws.guardduty.EcsTaskDetails/StartedAt]))

(s/def
  :cognitect.aws.guardduty/OrganizationKubernetesConfigurationResult
  (s/keys :req-un [:cognitect.aws.guardduty.OrganizationKubernetesConfigurationResult/AuditLogs]))

(s/def
  :cognitect.aws.guardduty/ProfileSubtype
  (s/spec string? :gen #(s/gen #{"FREQUENT" "INFREQUENT" "RARE" "UNSEEN"})))

(s/def
  :cognitect.aws.guardduty/OrgFeature
  (s/spec
    string?
    :gen
    #(s/gen
      #{"EBS_MALWARE_PROTECTION"
        "EKS_RUNTIME_MONITORING"
        "LAMBDA_NETWORK_LOGS"
        "RUNTIME_MONITORING"
        "RDS_LOGIN_EVENTS"
        "S3_DATA_EVENTS"
        "EKS_AUDIT_LOGS"})))

(s/def
  :cognitect.aws.guardduty/GetRemainingFreeTrialDaysRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.GetRemainingFreeTrialDaysRequest/DetectorId]
    :opt-un
    [:cognitect.aws.guardduty.GetRemainingFreeTrialDaysRequest/AccountIds]))

(s/def
  :cognitect.aws.guardduty/IntegerValueWithMax
  (s/spec (s/and int? #(<= 1 % 50)) :gen #(gen/choose 1 50)))

(s/def
  :cognitect.aws.guardduty/GetMembersRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.GetMembersRequest/DetectorId
     :cognitect.aws.guardduty.GetMembersRequest/AccountIds]))

(s/def
  :cognitect.aws.guardduty/Email
  (s/spec
    (s/and string? #(<= 6 (count %) 64))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 6 64) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.guardduty/GetMalwareScanSettingsResponse
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.GetMalwareScanSettingsResponse/EbsSnapshotPreservation
     :cognitect.aws.guardduty.GetMalwareScanSettingsResponse/ScanResourceCriteria]))

(s/def
  :cognitect.aws.guardduty/EbsVolumeScanDetails
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.EbsVolumeScanDetails/ScanType
     :cognitect.aws.guardduty.EbsVolumeScanDetails/ScanId
     :cognitect.aws.guardduty.EbsVolumeScanDetails/TriggerFindingId
     :cognitect.aws.guardduty.EbsVolumeScanDetails/ScanStartedAt
     :cognitect.aws.guardduty.EbsVolumeScanDetails/ScanDetections
     :cognitect.aws.guardduty.EbsVolumeScanDetails/ScanCompletedAt
     :cognitect.aws.guardduty.EbsVolumeScanDetails/Sources]))

(s/def
  :cognitect.aws.guardduty/DeleteIPSetRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.DeleteIPSetRequest/DetectorId
     :cognitect.aws.guardduty.DeleteIPSetRequest/IpSetId]))

(s/def
  :cognitect.aws.guardduty/NetworkInterface
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.NetworkInterface/PublicDnsName
     :cognitect.aws.guardduty.NetworkInterface/PublicIp
     :cognitect.aws.guardduty.NetworkInterface/Ipv6Addresses
     :cognitect.aws.guardduty.NetworkInterface/SubnetId
     :cognitect.aws.guardduty.NetworkInterface/SecurityGroups
     :cognitect.aws.guardduty.NetworkInterface/NetworkInterfaceId
     :cognitect.aws.guardduty.NetworkInterface/PrivateDnsName
     :cognitect.aws.guardduty.NetworkInterface/VpcId
     :cognitect.aws.guardduty.NetworkInterface/PrivateIpAddresses
     :cognitect.aws.guardduty.NetworkInterface/PrivateIpAddress]))

(s/def :cognitect.aws.guardduty/DestinationType (s/spec string? :gen #(s/gen #{"S3"})))

(s/def
  :cognitect.aws.guardduty/UsageStatistics
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.UsageStatistics/SumByAccount
     :cognitect.aws.guardduty.UsageStatistics/TopResources
     :cognitect.aws.guardduty.UsageStatistics/SumByFeature
     :cognitect.aws.guardduty.UsageStatistics/TopAccountsByFeature
     :cognitect.aws.guardduty.UsageStatistics/SumByResource
     :cognitect.aws.guardduty.UsageStatistics/SumByDataSource]))

(s/def :cognitect.aws.guardduty/Ec2NetworkInterfaceUids (s/coll-of :cognitect.aws.guardduty/String))

(s/def
  :cognitect.aws.guardduty/AnomalyProfiles
  (s/map-of :cognitect.aws.guardduty/String :cognitect.aws.guardduty/AnomalyProfileFeatures))

(s/def
  :cognitect.aws.guardduty/DetectorAdditionalConfigurationResult
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.DetectorAdditionalConfigurationResult/Status
     :cognitect.aws.guardduty.DetectorAdditionalConfigurationResult/UpdatedAt
     :cognitect.aws.guardduty.DetectorAdditionalConfigurationResult/Name]))

(s/def
  :cognitect.aws.guardduty/UsageResourceResult
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.UsageResourceResult/Total
     :cognitect.aws.guardduty.UsageResourceResult/Resource]))

(s/def
  :cognitect.aws.guardduty/UpdateFilterResponse
  (s/keys :req-un [:cognitect.aws.guardduty.UpdateFilterResponse/Name]))

(s/def
  :cognitect.aws.guardduty/Anomaly
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.Anomaly/Profiles :cognitect.aws.guardduty.Anomaly/Unusual]))

(s/def :cognitect.aws.guardduty/City (s/keys :opt-un [:cognitect.aws.guardduty.City/CityName]))

(s/def
  :cognitect.aws.guardduty/DeleteThreatIntelSetRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.DeleteThreatIntelSetRequest/DetectorId
     :cognitect.aws.guardduty.DeleteThreatIntelSetRequest/ThreatIntelSetId]))

(s/def
  :cognitect.aws.guardduty/StartMonitoringMembersRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.StartMonitoringMembersRequest/DetectorId
     :cognitect.aws.guardduty.StartMonitoringMembersRequest/AccountIds]))

(s/def
  :cognitect.aws.guardduty/PrivateIpAddresses
  (s/coll-of :cognitect.aws.guardduty/PrivateIpAddressDetails))

(s/def
  :cognitect.aws.guardduty/StopMonitoringMembersRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.StopMonitoringMembersRequest/DetectorId
     :cognitect.aws.guardduty.StopMonitoringMembersRequest/AccountIds]))

(s/def
  :cognitect.aws.guardduty/DisassociateMembersRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.DisassociateMembersRequest/DetectorId
     :cognitect.aws.guardduty.DisassociateMembersRequest/AccountIds]))

(s/def
  :cognitect.aws.guardduty/GetMalwareProtectionPlanRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.GetMalwareProtectionPlanRequest/MalwareProtectionPlanId]))

(s/def
  :cognitect.aws.guardduty/CreatePublishingDestinationRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.CreatePublishingDestinationRequest/DetectorId
     :cognitect.aws.guardduty.CreatePublishingDestinationRequest/DestinationType
     :cognitect.aws.guardduty.CreatePublishingDestinationRequest/DestinationProperties]
    :opt-un
    [:cognitect.aws.guardduty.CreatePublishingDestinationRequest/ClientToken]))

(s/def
  :cognitect.aws.guardduty/LocalIpDetails
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.LocalIpDetails/IpAddressV6
     :cognitect.aws.guardduty.LocalIpDetails/IpAddressV4]))

(s/def
  :cognitect.aws.guardduty/MemberFeaturesConfigurationsResults
  (s/coll-of :cognitect.aws.guardduty/MemberFeaturesConfigurationResult))

(s/def
  :cognitect.aws.guardduty/AccessControlList
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.AccessControlList/AllowsPublicReadAccess
     :cognitect.aws.guardduty.AccessControlList/AllowsPublicWriteAccess]))

(s/def :cognitect.aws.guardduty/Containers (s/coll-of :cognitect.aws.guardduty/Container))

(s/def
  :cognitect.aws.guardduty/TriggerDetails
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.TriggerDetails/GuardDutyFindingId
     :cognitect.aws.guardduty.TriggerDetails/Description]))

(s/def :cognitect.aws.guardduty/DeleteThreatEntitySetResponse (s/keys))

(s/def
  :cognitect.aws.guardduty/ScanFilePath
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.ScanFilePath/VolumeArn
     :cognitect.aws.guardduty.ScanFilePath/Hash
     :cognitect.aws.guardduty.ScanFilePath/FilePath
     :cognitect.aws.guardduty.ScanFilePath/FileName]))

(s/def
  :cognitect.aws.guardduty/CreateMembersRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.CreateMembersRequest/DetectorId
     :cognitect.aws.guardduty.CreateMembersRequest/AccountDetails]))

(s/def
  :cognitect.aws.guardduty/Ec2NetworkInterface
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.Ec2NetworkInterface/PublicIp
     :cognitect.aws.guardduty.Ec2NetworkInterface/Ipv6Addresses
     :cognitect.aws.guardduty.Ec2NetworkInterface/SecurityGroups
     :cognitect.aws.guardduty.Ec2NetworkInterface/VpcId
     :cognitect.aws.guardduty.Ec2NetworkInterface/PrivateIpAddresses
     :cognitect.aws.guardduty.Ec2NetworkInterface/SubNetId]))

(s/def
  :cognitect.aws.guardduty/GetThreatEntitySetRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.GetThreatEntitySetRequest/DetectorId
     :cognitect.aws.guardduty.GetThreatEntitySetRequest/ThreatEntitySetId]))

(s/def
  :cognitect.aws.guardduty/ScanConditionPair
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.ScanConditionPair/Key]
    :opt-un
    [:cognitect.aws.guardduty.ScanConditionPair/Value]))

(s/def
  :cognitect.aws.guardduty/GetInvitationsCountResponse
  (s/keys :opt-un [:cognitect.aws.guardduty.GetInvitationsCountResponse/InvitationsCount]))

(s/def
  :cognitect.aws.guardduty/PublicAccessConfiguration
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.PublicAccessConfiguration/PublicPolicyAccess
     :cognitect.aws.guardduty.PublicAccessConfiguration/PublicBucketRestrictBehavior
     :cognitect.aws.guardduty.PublicAccessConfiguration/PublicAclIgnoreBehavior
     :cognitect.aws.guardduty.PublicAccessConfiguration/PublicAclAccess]))

(s/def :cognitect.aws.guardduty/Destinations (s/coll-of :cognitect.aws.guardduty/Destination))

(s/def
  :cognitect.aws.guardduty/CreateS3BucketResource
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.CreateS3BucketResource/ObjectPrefixes
     :cognitect.aws.guardduty.CreateS3BucketResource/BucketName]))

(s/def
  :cognitect.aws.guardduty/PublicAccessStatus
  (s/spec string? :gen #(s/gen #{"BLOCKED" "ALLOWED"})))

(s/def
  :cognitect.aws.guardduty/AgentDetails
  (s/keys :opt-un [:cognitect.aws.guardduty.AgentDetails/Version]))

(s/def
  :cognitect.aws.guardduty/GroupedByFindingType
  (s/coll-of :cognitect.aws.guardduty/FindingTypeStatistics))

(s/def
  :cognitect.aws.guardduty/FilterRank
  (s/spec (s/and int? #(<= 1 % 100)) :gen #(gen/choose 1 100)))

(s/def :cognitect.aws.guardduty/Boolean boolean?)

(s/def
  :cognitect.aws.guardduty/UsageDataSourceResult
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.UsageDataSourceResult/Total
     :cognitect.aws.guardduty.UsageDataSourceResult/DataSource]))

(s/def
  :cognitect.aws.guardduty/AutoEnableMembers
  (s/spec string? :gen #(s/gen #{"NEW" "NONE" "ALL"})))

(s/def
  :cognitect.aws.guardduty/FindingIds
  (s/coll-of :cognitect.aws.guardduty/FindingId :min-count 0 :max-count 50))

(s/def
  :cognitect.aws.guardduty/GetFindingsStatisticsResponse
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.GetFindingsStatisticsResponse/FindingStatistics]
    :opt-un
    [:cognitect.aws.guardduty.GetFindingsStatisticsResponse/NextToken]))

(s/def
  :cognitect.aws.guardduty/CreateFilterRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.CreateFilterRequest/DetectorId
     :cognitect.aws.guardduty.CreateFilterRequest/Name
     :cognitect.aws.guardduty.CreateFilterRequest/FindingCriteria]
    :opt-un
    [:cognitect.aws.guardduty.CreateFilterRequest/Tags
     :cognitect.aws.guardduty.CreateFilterRequest/Description
     :cognitect.aws.guardduty.CreateFilterRequest/Rank
     :cognitect.aws.guardduty.CreateFilterRequest/Action
     :cognitect.aws.guardduty.CreateFilterRequest/ClientToken]))

(s/def
  :cognitect.aws.guardduty/MalwareProtectionPlanStatus
  (s/spec string? :gen #(s/gen #{"WARNING" "ERROR" "ACTIVE"})))

(s/def
  :cognitect.aws.guardduty/DetectorFeatureConfigurations
  (s/coll-of :cognitect.aws.guardduty/DetectorFeatureConfiguration))

(s/def
  :cognitect.aws.guardduty/AffectedResources
  (s/map-of :cognitect.aws.guardduty/String :cognitect.aws.guardduty/String))

(s/def
  :cognitect.aws.guardduty/OrganizationMalwareProtectionConfiguration
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.OrganizationMalwareProtectionConfiguration/ScanEc2InstanceWithFindings]))

(s/def
  :cognitect.aws.guardduty/CreateSampleFindingsRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.CreateSampleFindingsRequest/DetectorId]
    :opt-un
    [:cognitect.aws.guardduty.CreateSampleFindingsRequest/FindingTypes]))

(s/def :cognitect.aws.guardduty/ScanThreatNames (s/coll-of :cognitect.aws.guardduty/ScanThreatName))

(s/def
  :cognitect.aws.guardduty/DetectorFeatureConfigurationsResults
  (s/coll-of :cognitect.aws.guardduty/DetectorFeatureConfigurationResult))

(s/def :cognitect.aws.guardduty/EnableOrganizationAdminAccountResponse (s/keys))

(s/def
  :cognitect.aws.guardduty/ListInvitationsResponse
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.ListInvitationsResponse/NextToken
     :cognitect.aws.guardduty.ListInvitationsResponse/Invitations]))

(s/def
  :cognitect.aws.guardduty/GetThreatIntelSetRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.GetThreatIntelSetRequest/DetectorId
     :cognitect.aws.guardduty.GetThreatIntelSetRequest/ThreatIntelSetId]))

(s/def
  :cognitect.aws.guardduty/ListOrganizationAdminAccountsResponse
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.ListOrganizationAdminAccountsResponse/NextToken
     :cognitect.aws.guardduty.ListOrganizationAdminAccountsResponse/AdminAccounts]))

(s/def
  :cognitect.aws.guardduty/IpSetFormat
  (s/spec string? :gen #(s/gen #{"STIX" "PROOF_POINT" "TXT" "ALIEN_VAULT" "OTX_CSV" "FIRE_EYE"})))

(s/def
  :cognitect.aws.guardduty/Invitations
  (s/coll-of :cognitect.aws.guardduty/Invitation :min-count 0 :max-count 50))

(s/def :cognitect.aws.guardduty/DeleteDetectorResponse (s/keys))

(s/def
  :cognitect.aws.guardduty/RdsLoginAttemptAction
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.RdsLoginAttemptAction/RemoteIpDetails
     :cognitect.aws.guardduty.RdsLoginAttemptAction/LoginAttributes]))

(s/def
  :cognitect.aws.guardduty/FilterCriterionList
  (s/coll-of :cognitect.aws.guardduty/FilterCriterion :min-count 0 :max-count 1))

(s/def
  :cognitect.aws.guardduty/Ec2InstanceUid
  (s/spec
    (s/and string? #(<= 0 (count %) 256))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 0 256) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.guardduty/IndicatorTitle
  (s/spec
    (s/and string? #(<= 1 (count %) 256))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 1 256) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.guardduty/CoverageEksClusterDetails
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.CoverageEksClusterDetails/CoveredNodes
     :cognitect.aws.guardduty.CoverageEksClusterDetails/AddonDetails
     :cognitect.aws.guardduty.CoverageEksClusterDetails/CompatibleNodes
     :cognitect.aws.guardduty.CoverageEksClusterDetails/ManagementType
     :cognitect.aws.guardduty.CoverageEksClusterDetails/ClusterName]))

(s/def
  :cognitect.aws.guardduty/PortProbeDetails
  (s/coll-of :cognitect.aws.guardduty/PortProbeDetail))

(s/def
  :cognitect.aws.guardduty/GetFilterRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.GetFilterRequest/DetectorId
     :cognitect.aws.guardduty.GetFilterRequest/FilterName]))

(s/def :cognitect.aws.guardduty/UpdateDetectorResponse (s/keys))

(s/def
  :cognitect.aws.guardduty/ScanCondition
  (s/keys :req-un [:cognitect.aws.guardduty.ScanCondition/MapEquals]))

(s/def
  :cognitect.aws.guardduty/S3Bucket
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.S3Bucket/AccountPublicAccess
     :cognitect.aws.guardduty.S3Bucket/PublicReadAccess
     :cognitect.aws.guardduty.S3Bucket/OwnerId
     :cognitect.aws.guardduty.S3Bucket/EffectivePermission
     :cognitect.aws.guardduty.S3Bucket/EncryptionKeyArn
     :cognitect.aws.guardduty.S3Bucket/EncryptionType
     :cognitect.aws.guardduty.S3Bucket/BucketPublicAccess
     :cognitect.aws.guardduty.S3Bucket/S3ObjectUids
     :cognitect.aws.guardduty.S3Bucket/PublicWriteAccess
     :cognitect.aws.guardduty.S3Bucket/CreatedAt]))

(s/def
  :cognitect.aws.guardduty/OrganizationS3LogsConfiguration
  (s/keys :req-un [:cognitect.aws.guardduty.OrganizationS3LogsConfiguration/AutoEnable]))

(s/def
  :cognitect.aws.guardduty/CreateProtectedResource
  (s/keys :opt-un [:cognitect.aws.guardduty.CreateProtectedResource/S3Bucket]))

(s/def
  :cognitect.aws.guardduty/OrganizationEbsVolumes
  (s/keys :opt-un [:cognitect.aws.guardduty.OrganizationEbsVolumes/AutoEnable]))

(s/def
  :cognitect.aws.guardduty/RdsDbInstanceDetails
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.RdsDbInstanceDetails/DbClusterIdentifier
     :cognitect.aws.guardduty.RdsDbInstanceDetails/Tags
     :cognitect.aws.guardduty.RdsDbInstanceDetails/DbInstanceIdentifier
     :cognitect.aws.guardduty.RdsDbInstanceDetails/DbInstanceArn
     :cognitect.aws.guardduty.RdsDbInstanceDetails/EngineVersion
     :cognitect.aws.guardduty.RdsDbInstanceDetails/Engine]))

(s/def
  :cognitect.aws.guardduty/SequenceDescription
  (s/spec
    (s/and string? #(>= 4096 (count %)))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 0 4096) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.guardduty/CoverageStatisticsTypeList
  (s/coll-of :cognitect.aws.guardduty/CoverageStatisticsType))

(s/def
  :cognitect.aws.guardduty/CreateTrustedEntitySetRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.CreateTrustedEntitySetRequest/DetectorId
     :cognitect.aws.guardduty.CreateTrustedEntitySetRequest/Name
     :cognitect.aws.guardduty.CreateTrustedEntitySetRequest/Format
     :cognitect.aws.guardduty.CreateTrustedEntitySetRequest/Location
     :cognitect.aws.guardduty.CreateTrustedEntitySetRequest/Activate]
    :opt-un
    [:cognitect.aws.guardduty.CreateTrustedEntitySetRequest/Tags
     :cognitect.aws.guardduty.CreateTrustedEntitySetRequest/ExpectedBucketOwner
     :cognitect.aws.guardduty.CreateTrustedEntitySetRequest/ClientToken]))

(s/def
  :cognitect.aws.guardduty/MalwareProtectionPlanActions
  (s/keys :opt-un [:cognitect.aws.guardduty.MalwareProtectionPlanActions/Tagging]))

(s/def
  :cognitect.aws.guardduty/GetThreatIntelSetResponse
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.GetThreatIntelSetResponse/Name
     :cognitect.aws.guardduty.GetThreatIntelSetResponse/Format
     :cognitect.aws.guardduty.GetThreatIntelSetResponse/Location
     :cognitect.aws.guardduty.GetThreatIntelSetResponse/Status]
    :opt-un
    [:cognitect.aws.guardduty.GetThreatIntelSetResponse/Tags
     :cognitect.aws.guardduty.GetThreatIntelSetResponse/ExpectedBucketOwner]))

(s/def
  :cognitect.aws.guardduty/CreateFilterResponse
  (s/keys :req-un [:cognitect.aws.guardduty.CreateFilterResponse/Name]))

(s/def
  :cognitect.aws.guardduty/MemberFeaturesConfigurationResult
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.MemberFeaturesConfigurationResult/Status
     :cognitect.aws.guardduty.MemberFeaturesConfigurationResult/AdditionalConfiguration
     :cognitect.aws.guardduty.MemberFeaturesConfigurationResult/UpdatedAt
     :cognitect.aws.guardduty.MemberFeaturesConfigurationResult/Name]))

(s/def
  :cognitect.aws.guardduty/Ec2InstanceUids
  (s/coll-of :cognitect.aws.guardduty/Ec2InstanceUid :min-count 0 :max-count 25))

(s/def
  :cognitect.aws.guardduty/OrganizationScanEc2InstanceWithFindings
  (s/keys :opt-un [:cognitect.aws.guardduty.OrganizationScanEc2InstanceWithFindings/EbsVolumes]))

(s/def
  :cognitect.aws.guardduty/ScanType
  (s/spec string? :gen #(s/gen #{"ON_DEMAND" "GUARDDUTY_INITIATED"})))

(s/def
  :cognitect.aws.guardduty/ResourceDetails
  (s/keys :opt-un [:cognitect.aws.guardduty.ResourceDetails/InstanceArn]))

(s/def
  :cognitect.aws.guardduty/Behavior
  (s/map-of :cognitect.aws.guardduty/String :cognitect.aws.guardduty/AnomalyUnusualBehaviorFeature))

(s/def
  :cognitect.aws.guardduty/CoverageFilterCriterion
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.CoverageFilterCriterion/FilterCondition
     :cognitect.aws.guardduty.CoverageFilterCriterion/CriterionKey]))

(s/def
  :cognitect.aws.guardduty/KubernetesConfiguration
  (s/keys :req-un [:cognitect.aws.guardduty.KubernetesConfiguration/AuditLogs]))

(s/def
  :cognitect.aws.guardduty/Administrator
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.Administrator/InvitationId
     :cognitect.aws.guardduty.Administrator/RelationshipStatus
     :cognitect.aws.guardduty.Administrator/AccountId
     :cognitect.aws.guardduty.Administrator/InvitedAt]))

(s/def
  :cognitect.aws.guardduty/ListTrustedEntitySetsRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.ListTrustedEntitySetsRequest/DetectorId]
    :opt-un
    [:cognitect.aws.guardduty.ListTrustedEntitySetsRequest/NextToken
     :cognitect.aws.guardduty.ListTrustedEntitySetsRequest/MaxResults]))

(s/def :cognitect.aws.guardduty/ProductCodes (s/coll-of :cognitect.aws.guardduty/ProductCode))

(s/def
  :cognitect.aws.guardduty/EksCluster
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.EksCluster/Ec2InstanceUids
     :cognitect.aws.guardduty.EksCluster/Status
     :cognitect.aws.guardduty.EksCluster/VpcId
     :cognitect.aws.guardduty.EksCluster/Arn
     :cognitect.aws.guardduty.EksCluster/CreatedAt]))

(s/def
  :cognitect.aws.guardduty/Integer
  (s/spec int? :gen #(gen/choose Long/MIN_VALUE Long/MAX_VALUE)))

(s/def
  :cognitect.aws.guardduty/DNSLogsConfigurationResult
  (s/keys :req-un [:cognitect.aws.guardduty.DNSLogsConfigurationResult/Status]))

(s/def
  :cognitect.aws.guardduty/MalwareProtectionConfiguration
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.MalwareProtectionConfiguration/ScanEc2InstanceWithFindings]))

(s/def
  :cognitect.aws.guardduty/DeleteMembersRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.DeleteMembersRequest/DetectorId
     :cognitect.aws.guardduty.DeleteMembersRequest/AccountIds]))

(s/def
  :cognitect.aws.guardduty/ListPublishingDestinationsResponse
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.ListPublishingDestinationsResponse/Destinations]
    :opt-un
    [:cognitect.aws.guardduty.ListPublishingDestinationsResponse/NextToken]))

(s/def
  :cognitect.aws.guardduty/DeletePublishingDestinationRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.DeletePublishingDestinationRequest/DetectorId
     :cognitect.aws.guardduty.DeletePublishingDestinationRequest/DestinationId]))

(s/def :cognitect.aws.guardduty/Scans (s/coll-of :cognitect.aws.guardduty/Scan))

(s/def
  :cognitect.aws.guardduty/DataSourceConfigurationsResult
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.DataSourceConfigurationsResult/CloudTrail
     :cognitect.aws.guardduty.DataSourceConfigurationsResult/DNSLogs
     :cognitect.aws.guardduty.DataSourceConfigurationsResult/FlowLogs
     :cognitect.aws.guardduty.DataSourceConfigurationsResult/S3Logs]
    :opt-un
    [:cognitect.aws.guardduty.DataSourceConfigurationsResult/MalwareProtection
     :cognitect.aws.guardduty.DataSourceConfigurationsResult/Kubernetes]))

(s/def
  :cognitect.aws.guardduty/CoverageSortCriteria
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.CoverageSortCriteria/AttributeName
     :cognitect.aws.guardduty.CoverageSortCriteria/OrderBy]))

(s/def
  :cognitect.aws.guardduty/NonEmptyString
  (s/spec
    (s/and string? #(<= 1 (count %) 200))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 1 200) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.guardduty/OrganizationAdditionalConfigurationResult
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.OrganizationAdditionalConfigurationResult/AutoEnable
     :cognitect.aws.guardduty.OrganizationAdditionalConfigurationResult/Name]))

(s/def
  :cognitect.aws.guardduty/AccountLevelPermissions
  (s/keys :opt-un [:cognitect.aws.guardduty.AccountLevelPermissions/BlockPublicAccess]))

(s/def
  :cognitect.aws.guardduty/RdsLimitlessDbDetails
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.RdsLimitlessDbDetails/DbShardGroupIdentifier
     :cognitect.aws.guardduty.RdsLimitlessDbDetails/DbClusterIdentifier
     :cognitect.aws.guardduty.RdsLimitlessDbDetails/DbShardGroupResourceId
     :cognitect.aws.guardduty.RdsLimitlessDbDetails/Tags
     :cognitect.aws.guardduty.RdsLimitlessDbDetails/EngineVersion
     :cognitect.aws.guardduty.RdsLimitlessDbDetails/DbShardGroupArn
     :cognitect.aws.guardduty.RdsLimitlessDbDetails/Engine]))

(s/def
  :cognitect.aws.guardduty/CountBySeverity
  (s/map-of :cognitect.aws.guardduty/String :cognitect.aws.guardduty/Integer))

(s/def
  :cognitect.aws.guardduty/KubernetesDataSourceFreeTrial
  (s/keys :opt-un [:cognitect.aws.guardduty.KubernetesDataSourceFreeTrial/AuditLogs]))

(s/def
  :cognitect.aws.guardduty/TagValue
  (s/spec
    (s/and string? #(>= 256 (count %)))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 0 256) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.guardduty/ItemPath
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.ItemPath/Hash :cognitect.aws.guardduty.ItemPath/NestedItemPath]))

(s/def
  :cognitect.aws.guardduty/ListFindingsResponse
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.ListFindingsResponse/FindingIds]
    :opt-un
    [:cognitect.aws.guardduty.ListFindingsResponse/NextToken]))

(s/def
  :cognitect.aws.guardduty/ThreatEntitySetIds
  (s/coll-of :cognitect.aws.guardduty/String :min-count 0 :max-count 50))

(s/def :cognitect.aws.guardduty/MfaStatus (s/spec string? :gen #(s/gen #{"DISABLED" "ENABLED"})))

(s/def
  :cognitect.aws.guardduty/RemoteAccountDetails
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.RemoteAccountDetails/AccountId
     :cognitect.aws.guardduty.RemoteAccountDetails/Affiliated]))

(s/def
  :cognitect.aws.guardduty/ListTrustedEntitySetsResponse
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.ListTrustedEntitySetsResponse/TrustedEntitySetIds]
    :opt-un
    [:cognitect.aws.guardduty.ListTrustedEntitySetsResponse/NextToken]))

(s/def
  :cognitect.aws.guardduty/GetDetectorRequest
  (s/keys :req-un [:cognitect.aws.guardduty.GetDetectorRequest/DetectorId]))

(s/def
  :cognitect.aws.guardduty/FilterNames
  (s/coll-of :cognitect.aws.guardduty/FilterName :min-count 0 :max-count 50))

(s/def
  :cognitect.aws.guardduty/CreateIPSetRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.CreateIPSetRequest/DetectorId
     :cognitect.aws.guardduty.CreateIPSetRequest/Name
     :cognitect.aws.guardduty.CreateIPSetRequest/Format
     :cognitect.aws.guardduty.CreateIPSetRequest/Location
     :cognitect.aws.guardduty.CreateIPSetRequest/Activate]
    :opt-un
    [:cognitect.aws.guardduty.CreateIPSetRequest/Tags
     :cognitect.aws.guardduty.CreateIPSetRequest/ExpectedBucketOwner
     :cognitect.aws.guardduty.CreateIPSetRequest/ClientToken]))

(s/def
  :cognitect.aws.guardduty/CreateMembersResponse
  (s/keys :req-un [:cognitect.aws.guardduty.CreateMembersResponse/UnprocessedAccounts]))

(s/def
  :cognitect.aws.guardduty/KubernetesWorkloadDetails
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.KubernetesWorkloadDetails/HostNetwork
     :cognitect.aws.guardduty.KubernetesWorkloadDetails/Type
     :cognitect.aws.guardduty.KubernetesWorkloadDetails/Containers
     :cognitect.aws.guardduty.KubernetesWorkloadDetails/HostPID
     :cognitect.aws.guardduty.KubernetesWorkloadDetails/Uid
     :cognitect.aws.guardduty.KubernetesWorkloadDetails/ServiceAccountName
     :cognitect.aws.guardduty.KubernetesWorkloadDetails/Namespace
     :cognitect.aws.guardduty.KubernetesWorkloadDetails/HostIPC
     :cognitect.aws.guardduty.KubernetesWorkloadDetails/Name
     :cognitect.aws.guardduty.KubernetesWorkloadDetails/Volumes]))

(s/def
  :cognitect.aws.guardduty/InstanceDetails
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.InstanceDetails/InstanceId
     :cognitect.aws.guardduty.InstanceDetails/Platform
     :cognitect.aws.guardduty.InstanceDetails/ImageId
     :cognitect.aws.guardduty.InstanceDetails/InstanceType
     :cognitect.aws.guardduty.InstanceDetails/ImageDescription
     :cognitect.aws.guardduty.InstanceDetails/Tags
     :cognitect.aws.guardduty.InstanceDetails/AvailabilityZone
     :cognitect.aws.guardduty.InstanceDetails/InstanceState
     :cognitect.aws.guardduty.InstanceDetails/ProductCodes
     :cognitect.aws.guardduty.InstanceDetails/LaunchTime
     :cognitect.aws.guardduty.InstanceDetails/OutpostArn
     :cognitect.aws.guardduty.InstanceDetails/IamInstanceProfile
     :cognitect.aws.guardduty.InstanceDetails/NetworkInterfaces]))

(s/def
  :cognitect.aws.guardduty/ScanResourceCriteria
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.ScanResourceCriteria/Exclude
     :cognitect.aws.guardduty.ScanResourceCriteria/Include]))

(s/def
  :cognitect.aws.guardduty/Volume
  (s/keys :opt-un [:cognitect.aws.guardduty.Volume/Name :cognitect.aws.guardduty.Volume/HostPath]))

(s/def
  :cognitect.aws.guardduty/ListDetectorsRequest
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.ListDetectorsRequest/NextToken
     :cognitect.aws.guardduty.ListDetectorsRequest/MaxResults]))

(s/def
  :cognitect.aws.guardduty/StartMonitoringMembersResponse
  (s/keys :req-un [:cognitect.aws.guardduty.StartMonitoringMembersResponse/UnprocessedAccounts]))

(s/def
  :cognitect.aws.guardduty/ThreatsDetectedItemCount
  (s/keys :opt-un [:cognitect.aws.guardduty.ThreatsDetectedItemCount/Files]))

(s/def
  :cognitect.aws.guardduty/FindingTypes
  (s/coll-of :cognitect.aws.guardduty/FindingType :min-count 0 :max-count 50))

(s/def
  :cognitect.aws.guardduty/UnprocessedAccounts
  (s/coll-of :cognitect.aws.guardduty/UnprocessedAccount :min-count 0 :max-count 50))

(s/def
  :cognitect.aws.guardduty/FindingResourceType
  (s/spec
    string?
    :gen
    #(s/gen
      #{"S3_OBJECT"
        "KUBERNETES_WORKLOAD"
        "S3_BUCKET"
        "EC2_INSTANCE"
        "EC2_NETWORK_INTERFACE"
        "ACCESS_KEY"
        "CONTAINER"
        "EKS_CLUSTER"})))

(s/def
  :cognitect.aws.guardduty/FreeTrialFeatureConfigurationsResults
  (s/coll-of :cognitect.aws.guardduty/FreeTrialFeatureConfigurationResult))

(s/def
  :cognitect.aws.guardduty/KubernetesPermissionCheckedDetails
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.KubernetesPermissionCheckedDetails/Namespace
     :cognitect.aws.guardduty.KubernetesPermissionCheckedDetails/Allowed
     :cognitect.aws.guardduty.KubernetesPermissionCheckedDetails/Verb
     :cognitect.aws.guardduty.KubernetesPermissionCheckedDetails/Resource]))

(s/def
  :cognitect.aws.guardduty/StartMalwareScanRequest
  (s/keys :req-un [:cognitect.aws.guardduty.StartMalwareScanRequest/ResourceArn]))

(s/def
  :cognitect.aws.guardduty/GetFilterResponse
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.GetFilterResponse/Name
     :cognitect.aws.guardduty.GetFilterResponse/Action
     :cognitect.aws.guardduty.GetFilterResponse/FindingCriteria]
    :opt-un
    [:cognitect.aws.guardduty.GetFilterResponse/Tags
     :cognitect.aws.guardduty.GetFilterResponse/Description
     :cognitect.aws.guardduty.GetFilterResponse/Rank]))

(s/def
  :cognitect.aws.guardduty/UpdateProtectedResource
  (s/keys :opt-un [:cognitect.aws.guardduty.UpdateProtectedResource/S3Bucket]))

(s/def
  :cognitect.aws.guardduty/MemberFeaturesConfigurations
  (s/coll-of :cognitect.aws.guardduty/MemberFeaturesConfiguration))

(s/def
  :cognitect.aws.guardduty/DeleteThreatEntitySetRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.DeleteThreatEntitySetRequest/DetectorId
     :cognitect.aws.guardduty.DeleteThreatEntitySetRequest/ThreatEntitySetId]))

(s/def :cognitect.aws.guardduty/AcceptInvitationResponse (s/keys))

(s/def
  :cognitect.aws.guardduty/ListTagsForResourceRequest
  (s/keys :req-un [:cognitect.aws.guardduty.ListTagsForResourceRequest/ResourceArn]))

(s/def
  :cognitect.aws.guardduty/ListFiltersRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.ListFiltersRequest/DetectorId]
    :opt-un
    [:cognitect.aws.guardduty.ListFiltersRequest/NextToken
     :cognitect.aws.guardduty.ListFiltersRequest/MaxResults]))

(s/def
  :cognitect.aws.guardduty/ThreatEntitySetFormat
  (s/spec string? :gen #(s/gen #{"STIX" "PROOF_POINT" "TXT" "ALIEN_VAULT" "OTX_CSV" "FIRE_EYE"})))

(s/def
  :cognitect.aws.guardduty/ThreatIntelligenceDetail
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.ThreatIntelligenceDetail/ThreatNames
     :cognitect.aws.guardduty.ThreatIntelligenceDetail/ThreatListName
     :cognitect.aws.guardduty.ThreatIntelligenceDetail/ThreatFileSha256]))

(s/def
  :cognitect.aws.guardduty/GroupedByResource
  (s/coll-of :cognitect.aws.guardduty/ResourceStatistics))

(s/def
  :cognitect.aws.guardduty/S3ObjectDetail
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.S3ObjectDetail/VersionId
     :cognitect.aws.guardduty.S3ObjectDetail/Hash
     :cognitect.aws.guardduty.S3ObjectDetail/ETag
     :cognitect.aws.guardduty.S3ObjectDetail/Key
     :cognitect.aws.guardduty.S3ObjectDetail/ObjectArn]))

(s/def
  :cognitect.aws.guardduty/ScanThreatName
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.ScanThreatName/Severity
     :cognitect.aws.guardduty.ScanThreatName/FilePaths
     :cognitect.aws.guardduty.ScanThreatName/ItemCount
     :cognitect.aws.guardduty.ScanThreatName/Name]))

(s/def :cognitect.aws.guardduty/AcceptAdministratorInvitationResponse (s/keys))

(s/def
  :cognitect.aws.guardduty/OrganizationFeatureConfiguration
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.OrganizationFeatureConfiguration/AutoEnable
     :cognitect.aws.guardduty.OrganizationFeatureConfiguration/AdditionalConfiguration
     :cognitect.aws.guardduty.OrganizationFeatureConfiguration/Name]))

(s/def
  :cognitect.aws.guardduty/GetFindingsRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.GetFindingsRequest/DetectorId
     :cognitect.aws.guardduty.GetFindingsRequest/FindingIds]
    :opt-un
    [:cognitect.aws.guardduty.GetFindingsRequest/SortCriteria]))

(s/def
  :cognitect.aws.guardduty/GetMembersResponse
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.GetMembersResponse/Members
     :cognitect.aws.guardduty.GetMembersResponse/UnprocessedAccounts]))

(s/def
  :cognitect.aws.guardduty/BucketLevelPermissions
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.BucketLevelPermissions/AccessControlList
     :cognitect.aws.guardduty.BucketLevelPermissions/BlockPublicAccess
     :cognitect.aws.guardduty.BucketLevelPermissions/BucketPolicy]))

(s/def
  :cognitect.aws.guardduty/UpdateMemberDetectorsResponse
  (s/keys :req-un [:cognitect.aws.guardduty.UpdateMemberDetectorsResponse/UnprocessedAccounts]))

(s/def
  :cognitect.aws.guardduty/CreateDetectorRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.CreateDetectorRequest/Enable]
    :opt-un
    [:cognitect.aws.guardduty.CreateDetectorRequest/Tags
     :cognitect.aws.guardduty.CreateDetectorRequest/DataSources
     :cognitect.aws.guardduty.CreateDetectorRequest/FindingPublishingFrequency
     :cognitect.aws.guardduty.CreateDetectorRequest/Features
     :cognitect.aws.guardduty.CreateDetectorRequest/ClientToken]))

(s/def
  :cognitect.aws.guardduty/MaxResults100
  (s/spec (s/and int? #(<= 1 % 100)) :gen #(gen/choose 1 100)))

(s/def
  :cognitect.aws.guardduty/ThreatIntelSetStatus
  (s/spec
    string?
    :gen
    #(s/gen #{"DEACTIVATING" "ACTIVATING" "ERROR" "INACTIVE" "ACTIVE" "DELETE_PENDING" "DELETED"})))

(s/def :cognitect.aws.guardduty/VolumeMounts (s/coll-of :cognitect.aws.guardduty/VolumeMount))

(s/def :cognitect.aws.guardduty/UntagResourceResponse (s/keys))

(s/def
  :cognitect.aws.guardduty/S3LogsConfiguration
  (s/keys :req-un [:cognitect.aws.guardduty.S3LogsConfiguration/Enable]))

(s/def
  :cognitect.aws.guardduty/ListMalwareProtectionPlansResponse
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.ListMalwareProtectionPlansResponse/MalwareProtectionPlans
     :cognitect.aws.guardduty.ListMalwareProtectionPlansResponse/NextToken]))

(s/def
  :cognitect.aws.guardduty/PublicAclIgnoreBehavior
  (s/spec string? :gen #(s/gen #{"IGNORED" "NOT_IGNORED"})))

(s/def
  :cognitect.aws.guardduty/UsageFeatureResult
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.UsageFeatureResult/Feature
     :cognitect.aws.guardduty.UsageFeatureResult/Total]))

(s/def
  :cognitect.aws.guardduty/UsageFeatureResultList
  (s/coll-of :cognitect.aws.guardduty/UsageFeatureResult))

(s/def
  :cognitect.aws.guardduty/AddonDetails
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.AddonDetails/AddonVersion
     :cognitect.aws.guardduty.AddonDetails/AddonStatus]))

(s/def
  :cognitect.aws.guardduty/ListInvitationsRequest
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.ListInvitationsRequest/NextToken
     :cognitect.aws.guardduty.ListInvitationsRequest/MaxResults]))

(s/def
  :cognitect.aws.guardduty/IndicatorType
  (s/spec
    string?
    :gen
    #(s/gen
      #{"ATTACK_TACTIC"
        "UNUSUAL_ASN_FOR_USER"
        "MALICIOUS_DOMAIN"
        "CRYPTOMINING_DOMAIN"
        "ATTACK_TECHNIQUE"
        "SUSPICIOUS_PROCESS"
        "UNUSUAL_API_FOR_ACCOUNT"
        "TOR_IP"
        "SUSPICIOUS_USER_AGENT"
        "HIGH_RISK_API"
        "CRYPTOMINING_PROCESS"
        "CRYPTOMINING_IP"
        "UNUSUAL_ASN_FOR_ACCOUNT"
        "SUSPICIOUS_NETWORK"
        "MALICIOUS_IP"
        "MALICIOUS_PROCESS"})))

(s/def
  :cognitect.aws.guardduty/UpdateFilterRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.UpdateFilterRequest/DetectorId
     :cognitect.aws.guardduty.UpdateFilterRequest/FilterName]
    :opt-un
    [:cognitect.aws.guardduty.UpdateFilterRequest/Description
     :cognitect.aws.guardduty.UpdateFilterRequest/Rank
     :cognitect.aws.guardduty.UpdateFilterRequest/FindingCriteria
     :cognitect.aws.guardduty.UpdateFilterRequest/Action]))

(s/def
  :cognitect.aws.guardduty/User
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.User/Name
     :cognitect.aws.guardduty.User/Uid
     :cognitect.aws.guardduty.User/Type]
    :opt-un
    [:cognitect.aws.guardduty.User/CredentialUid :cognitect.aws.guardduty.User/Account]))

(s/def
  :cognitect.aws.guardduty/EcsClusterDetails
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.EcsClusterDetails/RegisteredContainerInstancesCount
     :cognitect.aws.guardduty.EcsClusterDetails/TaskDetails
     :cognitect.aws.guardduty.EcsClusterDetails/Tags
     :cognitect.aws.guardduty.EcsClusterDetails/RunningTasksCount
     :cognitect.aws.guardduty.EcsClusterDetails/Status
     :cognitect.aws.guardduty.EcsClusterDetails/Arn
     :cognitect.aws.guardduty.EcsClusterDetails/ActiveServicesCount
     :cognitect.aws.guardduty.EcsClusterDetails/Name]))

(s/def
  :cognitect.aws.guardduty/OrganizationFeaturesConfigurationsResults
  (s/coll-of :cognitect.aws.guardduty/OrganizationFeatureConfigurationResult))

(s/def
  :cognitect.aws.guardduty/OrganizationDataSourceConfigurations
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.OrganizationDataSourceConfigurations/S3Logs
     :cognitect.aws.guardduty.OrganizationDataSourceConfigurations/MalwareProtection
     :cognitect.aws.guardduty.OrganizationDataSourceConfigurations/Kubernetes]))

(s/def
  :cognitect.aws.guardduty/FindingStatisticType
  (s/spec string? :gen #(s/gen #{"COUNT_BY_SEVERITY"})))

(s/def :cognitect.aws.guardduty/S3BucketDetails (s/coll-of :cognitect.aws.guardduty/S3BucketDetail))

(s/def
  :cognitect.aws.guardduty/AccountDetails
  (s/coll-of :cognitect.aws.guardduty/AccountDetail :min-count 1 :max-count 50))

(s/def
  :cognitect.aws.guardduty/FindingCriteria
  (s/keys :opt-un [:cognitect.aws.guardduty.FindingCriteria/Criterion]))

(s/def
  :cognitect.aws.guardduty/AnomalyProfileFeatureObjects
  (s/coll-of :cognitect.aws.guardduty/AnomalyObject))

(s/def
  :cognitect.aws.guardduty/InternalServerErrorException
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.InternalServerErrorException/Type
     :cognitect.aws.guardduty.InternalServerErrorException/Message]))

(s/def
  :cognitect.aws.guardduty/ListThreatEntitySetsResponse
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.ListThreatEntitySetsResponse/ThreatEntitySetIds]
    :opt-un
    [:cognitect.aws.guardduty.ListThreatEntitySetsResponse/NextToken]))

(s/def
  :cognitect.aws.guardduty/IpSetIds
  (s/coll-of :cognitect.aws.guardduty/String :min-count 0 :max-count 50))

(s/def
  :cognitect.aws.guardduty/RdsDbUserDetails
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.RdsDbUserDetails/Application
     :cognitect.aws.guardduty.RdsDbUserDetails/AuthMethod
     :cognitect.aws.guardduty.RdsDbUserDetails/User
     :cognitect.aws.guardduty.RdsDbUserDetails/Ssl
     :cognitect.aws.guardduty.RdsDbUserDetails/Database]))

(s/def
  :cognitect.aws.guardduty/DisassociateFromMasterAccountRequest
  (s/keys :req-un [:cognitect.aws.guardduty.DisassociateFromMasterAccountRequest/DetectorId]))

(s/def :cognitect.aws.guardduty/MapEquals (s/coll-of :cognitect.aws.guardduty/ScanConditionPair))

(s/def
  :cognitect.aws.guardduty/PublicBucketRestrictBehavior
  (s/spec string? :gen #(s/gen #{"NOT_RESTRICTED" "RESTRICTED"})))

(s/def
  :cognitect.aws.guardduty/InviteMembersRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.InviteMembersRequest/DetectorId
     :cognitect.aws.guardduty.InviteMembersRequest/AccountIds]
    :opt-un
    [:cognitect.aws.guardduty.InviteMembersRequest/DisableEmailNotification
     :cognitect.aws.guardduty.InviteMembersRequest/Message]))

(s/def
  :cognitect.aws.guardduty/KubernetesResourcesTypes
  (s/spec
    string?
    :gen
    #(s/gen
      #{"DAEMONSETS"
        "DEPLOYMENTS"
        "REPLICATIONCONTROLLERS"
        "PODS"
        "JOBS"
        "CRONJOBS"
        "REPLICASETS"
        "STATEFULSETS"})))

(s/def
  :cognitect.aws.guardduty/CreateDetectorResponse
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.CreateDetectorResponse/UnprocessedDataSources
     :cognitect.aws.guardduty.CreateDetectorResponse/DetectorId]))

(s/def :cognitect.aws.guardduty/UpdateThreatIntelSetResponse (s/keys))

(s/def
  :cognitect.aws.guardduty/FindingId
  (s/spec
    (s/and string? #(<= 1 (count %) 300))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 1 300) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.guardduty/UpdateS3BucketResource
  (s/keys :opt-un [:cognitect.aws.guardduty.UpdateS3BucketResource/ObjectPrefixes]))

(s/def
  :cognitect.aws.guardduty/OrganizationDetails
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.OrganizationDetails/OrganizationStatistics
     :cognitect.aws.guardduty.OrganizationDetails/UpdatedAt]))

(s/def
  :cognitect.aws.guardduty/RuntimeContext
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.RuntimeContext/IanaProtocolNumber
     :cognitect.aws.guardduty.RuntimeContext/ModuleSha256
     :cognitect.aws.guardduty.RuntimeContext/AddressFamily
     :cognitect.aws.guardduty.RuntimeContext/LibraryPath
     :cognitect.aws.guardduty.RuntimeContext/MemoryRegions
     :cognitect.aws.guardduty.RuntimeContext/ToolCategory
     :cognitect.aws.guardduty.RuntimeContext/ModifyingProcess
     :cognitect.aws.guardduty.RuntimeContext/ThreatFilePath
     :cognitect.aws.guardduty.RuntimeContext/ShellHistoryFilePath
     :cognitect.aws.guardduty.RuntimeContext/MountTarget
     :cognitect.aws.guardduty.RuntimeContext/TargetProcess
     :cognitect.aws.guardduty.RuntimeContext/ToolName
     :cognitect.aws.guardduty.RuntimeContext/LdPreloadValue
     :cognitect.aws.guardduty.RuntimeContext/RuncBinaryPath
     :cognitect.aws.guardduty.RuntimeContext/ReleaseAgentPath
     :cognitect.aws.guardduty.RuntimeContext/CommandLineExample
     :cognitect.aws.guardduty.RuntimeContext/Flags
     :cognitect.aws.guardduty.RuntimeContext/ModifiedAt
     :cognitect.aws.guardduty.RuntimeContext/ModuleFilePath
     :cognitect.aws.guardduty.RuntimeContext/FileSystemType
     :cognitect.aws.guardduty.RuntimeContext/MountSource
     :cognitect.aws.guardduty.RuntimeContext/ModuleName
     :cognitect.aws.guardduty.RuntimeContext/ServiceName
     :cognitect.aws.guardduty.RuntimeContext/ScriptPath
     :cognitect.aws.guardduty.RuntimeContext/SocketPath]))

(s/def
  :cognitect.aws.guardduty/CoverageResourceDetails
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.CoverageResourceDetails/Ec2InstanceDetails
     :cognitect.aws.guardduty.CoverageResourceDetails/EksClusterDetails
     :cognitect.aws.guardduty.CoverageResourceDetails/EcsClusterDetails
     :cognitect.aws.guardduty.CoverageResourceDetails/ResourceType]))

(s/def
  :cognitect.aws.guardduty/ScannedItemCount
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.ScannedItemCount/Files
     :cognitect.aws.guardduty.ScannedItemCount/TotalGb
     :cognitect.aws.guardduty.ScannedItemCount/Volumes]))

(s/def
  :cognitect.aws.guardduty/Actor
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.Actor/Id]
    :opt-un
    [:cognitect.aws.guardduty.Actor/User
     :cognitect.aws.guardduty.Actor/Process
     :cognitect.aws.guardduty.Actor/Session]))

(s/def
  :cognitect.aws.guardduty/ScanEc2InstanceWithFindingsResult
  (s/keys :opt-un [:cognitect.aws.guardduty.ScanEc2InstanceWithFindingsResult/EbsVolumes]))

(s/def
  :cognitect.aws.guardduty/UsageAccountResultList
  (s/coll-of :cognitect.aws.guardduty/UsageAccountResult))

(s/def
  :cognitect.aws.guardduty/KubernetesAuditLogsConfiguration
  (s/keys :req-un [:cognitect.aws.guardduty.KubernetesAuditLogsConfiguration/Enable]))

(s/def
  :cognitect.aws.guardduty/RemotePortDetails
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.RemotePortDetails/PortName
     :cognitect.aws.guardduty.RemotePortDetails/Port]))

(s/def
  :cognitect.aws.guardduty/Total
  (s/keys :opt-un [:cognitect.aws.guardduty.Total/Unit :cognitect.aws.guardduty.Total/Amount]))

(s/def
  :cognitect.aws.guardduty/ContainerFindingResource
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.ContainerFindingResource/Image]
    :opt-un
    [:cognitect.aws.guardduty.ContainerFindingResource/ImageUid]))

(s/def
  :cognitect.aws.guardduty/KubernetesApiCallAction
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.KubernetesApiCallAction/RequestUri
     :cognitect.aws.guardduty.KubernetesApiCallAction/StatusCode
     :cognitect.aws.guardduty.KubernetesApiCallAction/Parameters
     :cognitect.aws.guardduty.KubernetesApiCallAction/RemoteIpDetails
     :cognitect.aws.guardduty.KubernetesApiCallAction/UserAgent
     :cognitect.aws.guardduty.KubernetesApiCallAction/Namespace
     :cognitect.aws.guardduty.KubernetesApiCallAction/ResourceName
     :cognitect.aws.guardduty.KubernetesApiCallAction/Verb
     :cognitect.aws.guardduty.KubernetesApiCallAction/Resource
     :cognitect.aws.guardduty.KubernetesApiCallAction/Subresource
     :cognitect.aws.guardduty.KubernetesApiCallAction/SourceIps]))

(s/def
  :cognitect.aws.guardduty/FreeTrialFeatureConfigurationResult
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.FreeTrialFeatureConfigurationResult/FreeTrialDaysRemaining
     :cognitect.aws.guardduty.FreeTrialFeatureConfigurationResult/Name]))

(s/def
  :cognitect.aws.guardduty/S3Object
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.S3Object/VersionId
     :cognitect.aws.guardduty.S3Object/ETag
     :cognitect.aws.guardduty.S3Object/Key]))

(s/def
  :cognitect.aws.guardduty/Signal
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.Signal/Uid
     :cognitect.aws.guardduty.Signal/Type
     :cognitect.aws.guardduty.Signal/Name
     :cognitect.aws.guardduty.Signal/CreatedAt
     :cognitect.aws.guardduty.Signal/UpdatedAt
     :cognitect.aws.guardduty.Signal/FirstSeenAt
     :cognitect.aws.guardduty.Signal/LastSeenAt
     :cognitect.aws.guardduty.Signal/Count]
    :opt-un
    [:cognitect.aws.guardduty.Signal/Severity
     :cognitect.aws.guardduty.Signal/SignalIndicators
     :cognitect.aws.guardduty.Signal/ResourceUids
     :cognitect.aws.guardduty.Signal/EndpointIds
     :cognitect.aws.guardduty.Signal/ActorIds
     :cognitect.aws.guardduty.Signal/Description]))

(s/def :cognitect.aws.guardduty/SecurityGroups (s/coll-of :cognitect.aws.guardduty/SecurityGroup))

(s/def
  :cognitect.aws.guardduty/MemberFeaturesConfiguration
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.MemberFeaturesConfiguration/Status
     :cognitect.aws.guardduty.MemberFeaturesConfiguration/AdditionalConfiguration
     :cognitect.aws.guardduty.MemberFeaturesConfiguration/Name]))

(s/def
  :cognitect.aws.guardduty/PublicAccess
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.PublicAccess/PermissionConfiguration
     :cognitect.aws.guardduty.PublicAccess/EffectivePermission]))

(s/def
  :cognitect.aws.guardduty/Account
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.Account/Uid]
    :opt-un
    [:cognitect.aws.guardduty.Account/Name]))

(s/def :cognitect.aws.guardduty/ItemPaths (s/coll-of :cognitect.aws.guardduty/ItemPath))

(s/def
  :cognitect.aws.guardduty/MemberDataSourceConfigurations
  (s/coll-of :cognitect.aws.guardduty/MemberDataSourceConfiguration :min-count 1 :max-count 50))

(s/def
  :cognitect.aws.guardduty/GetUsageStatisticsResponse
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.GetUsageStatisticsResponse/NextToken
     :cognitect.aws.guardduty.GetUsageStatisticsResponse/UsageStatistics]))

(s/def
  :cognitect.aws.guardduty/UnprocessedDataSourcesResult
  (s/keys :opt-un [:cognitect.aws.guardduty.UnprocessedDataSourcesResult/MalwareProtection]))

(s/def
  :cognitect.aws.guardduty/OrganizationKubernetesAuditLogsConfiguration
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.OrganizationKubernetesAuditLogsConfiguration/AutoEnable]))

(s/def
  :cognitect.aws.guardduty/OrganizationFeatureStatisticsAdditionalConfigurations
  (s/coll-of :cognitect.aws.guardduty/OrganizationFeatureStatisticsAdditionalConfiguration))

(s/def
  :cognitect.aws.guardduty/CountByCoverageStatus
  (s/map-of :cognitect.aws.guardduty/CoverageStatus :cognitect.aws.guardduty/Long))

(s/def
  :cognitect.aws.guardduty/Criterion
  (s/map-of :cognitect.aws.guardduty/String :cognitect.aws.guardduty/Condition))

(s/def
  :cognitect.aws.guardduty/OrganizationKubernetesAuditLogsConfigurationResult
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.OrganizationKubernetesAuditLogsConfigurationResult/AutoEnable]))

(s/def
  :cognitect.aws.guardduty/AccountStatistics
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.AccountStatistics/LastGeneratedAt
     :cognitect.aws.guardduty.AccountStatistics/TotalFindings
     :cognitect.aws.guardduty.AccountStatistics/AccountId]))

(s/def
  :cognitect.aws.guardduty/CreatePublishingDestinationResponse
  (s/keys :req-un [:cognitect.aws.guardduty.CreatePublishingDestinationResponse/DestinationId]))

(s/def
  :cognitect.aws.guardduty/S3BucketDetail
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.S3BucketDetail/Type
     :cognitect.aws.guardduty.S3BucketDetail/Tags
     :cognitect.aws.guardduty.S3BucketDetail/DefaultServerSideEncryption
     :cognitect.aws.guardduty.S3BucketDetail/S3ObjectDetails
     :cognitect.aws.guardduty.S3BucketDetail/Arn
     :cognitect.aws.guardduty.S3BucketDetail/CreatedAt
     :cognitect.aws.guardduty.S3BucketDetail/Owner
     :cognitect.aws.guardduty.S3BucketDetail/PublicAccess
     :cognitect.aws.guardduty.S3BucketDetail/Name]))

(s/def
  :cognitect.aws.guardduty/AnomalyProfileFeatures
  (s/map-of :cognitect.aws.guardduty/String :cognitect.aws.guardduty/AnomalyProfileFeatureObjects))

(s/def
  :cognitect.aws.guardduty/GetTrustedEntitySetRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.GetTrustedEntitySetRequest/DetectorId
     :cognitect.aws.guardduty.GetTrustedEntitySetRequest/TrustedEntitySetId]))

(s/def
  :cognitect.aws.guardduty/UnarchiveFindingsRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.UnarchiveFindingsRequest/DetectorId
     :cognitect.aws.guardduty.UnarchiveFindingsRequest/FindingIds]))

(s/def :cognitect.aws.guardduty/UsageFeatureList (s/coll-of :cognitect.aws.guardduty/UsageFeature))

(s/def
  :cognitect.aws.guardduty/DefaultServerSideEncryption
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.DefaultServerSideEncryption/KmsMasterKeyArn
     :cognitect.aws.guardduty.DefaultServerSideEncryption/EncryptionType]))

(s/def
  :cognitect.aws.guardduty/ProcessPath
  (s/spec
    (s/and string? #(<= 0 (count %) 4096))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 0 4096) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.guardduty/EbsVolumesResult
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.EbsVolumesResult/Status
     :cognitect.aws.guardduty.EbsVolumesResult/Reason]))

(s/def
  :cognitect.aws.guardduty/UsageStatisticType
  (s/spec
    string?
    :gen
    #(s/gen
      #{"TOP_ACCOUNTS_BY_FEATURE"
        "TOP_RESOURCES"
        "SUM_BY_ACCOUNT"
        "SUM_BY_DATA_SOURCE"
        "SUM_BY_RESOURCE"
        "SUM_BY_FEATURES"})))

(s/def
  :cognitect.aws.guardduty/OrganizationS3LogsConfigurationResult
  (s/keys :req-un [:cognitect.aws.guardduty.OrganizationS3LogsConfigurationResult/AutoEnable]))

(s/def
  :cognitect.aws.guardduty/Action
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.Action/PortProbeAction
     :cognitect.aws.guardduty.Action/KubernetesApiCallAction
     :cognitect.aws.guardduty.Action/KubernetesRoleBindingDetails
     :cognitect.aws.guardduty.Action/DnsRequestAction
     :cognitect.aws.guardduty.Action/NetworkConnectionAction
     :cognitect.aws.guardduty.Action/RdsLoginAttemptAction
     :cognitect.aws.guardduty.Action/KubernetesPermissionCheckedDetails
     :cognitect.aws.guardduty.Action/KubernetesRoleDetails
     :cognitect.aws.guardduty.Action/ActionType
     :cognitect.aws.guardduty.Action/AwsApiCallAction]))

(s/def
  :cognitect.aws.guardduty/UsageDataSourceResultList
  (s/coll-of :cognitect.aws.guardduty/UsageDataSourceResult))

(s/def
  :cognitect.aws.guardduty/ClusterStatus
  (s/spec string? :gen #(s/gen #{"UPDATING" "PENDING" "DELETING" "CREATING" "ACTIVE" "FAILED"})))

(s/def :cognitect.aws.guardduty/Long (s/spec int? :gen #(gen/choose Long/MIN_VALUE Long/MAX_VALUE)))

(s/def
  :cognitect.aws.guardduty/ProcessName
  (s/spec
    (s/and string? #(<= 0 (count %) 4096))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 0 4096) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.guardduty/ThreatDetectedByName
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.ThreatDetectedByName/ThreatNames
     :cognitect.aws.guardduty.ThreatDetectedByName/UniqueThreatNameCount
     :cognitect.aws.guardduty.ThreatDetectedByName/ItemCount
     :cognitect.aws.guardduty.ThreatDetectedByName/Shortened]))

(s/def
  :cognitect.aws.guardduty/OrganizationFeatureStatisticsResults
  (s/coll-of :cognitect.aws.guardduty/OrganizationFeatureStatistics))

(s/def
  :cognitect.aws.guardduty/GetCoverageStatisticsResponse
  (s/keys :opt-un [:cognitect.aws.guardduty.GetCoverageStatisticsResponse/CoverageStatistics]))

(s/def
  :cognitect.aws.guardduty/BadRequestException
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.BadRequestException/Type
     :cognitect.aws.guardduty.BadRequestException/Message]))

(s/def
  :cognitect.aws.guardduty/OrganizationStatistics
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.OrganizationStatistics/EnabledAccountsCount
     :cognitect.aws.guardduty.OrganizationStatistics/TotalAccountsCount
     :cognitect.aws.guardduty.OrganizationStatistics/CountByFeature
     :cognitect.aws.guardduty.OrganizationStatistics/ActiveAccountsCount
     :cognitect.aws.guardduty.OrganizationStatistics/MemberAccountsCount]))

(s/def
  :cognitect.aws.guardduty/PermissionConfiguration
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.PermissionConfiguration/BucketLevelPermissions
     :cognitect.aws.guardduty.PermissionConfiguration/AccountLevelPermissions]))

(s/def
  :cognitect.aws.guardduty/ScanCriterion
  (s/map-of :cognitect.aws.guardduty/ScanCriterionKey :cognitect.aws.guardduty/ScanCondition))

(s/def
  :cognitect.aws.guardduty/NetworkConnectionAction
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.NetworkConnectionAction/ConnectionDirection
     :cognitect.aws.guardduty.NetworkConnectionAction/Blocked
     :cognitect.aws.guardduty.NetworkConnectionAction/LocalPortDetails
     :cognitect.aws.guardduty.NetworkConnectionAction/RemoteIpDetails
     :cognitect.aws.guardduty.NetworkConnectionAction/Protocol
     :cognitect.aws.guardduty.NetworkConnectionAction/LocalIpDetails
     :cognitect.aws.guardduty.NetworkConnectionAction/RemotePortDetails
     :cognitect.aws.guardduty.NetworkConnectionAction/LocalNetworkInterface]))

(s/def
  :cognitect.aws.guardduty/ExpectedBucketOwner
  (s/spec #(re-matches (re-pattern "^[0-9]+$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.guardduty/NetworkGeoLocation
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.NetworkGeoLocation/City
     :cognitect.aws.guardduty.NetworkGeoLocation/Country
     :cognitect.aws.guardduty.NetworkGeoLocation/Latitude
     :cognitect.aws.guardduty.NetworkGeoLocation/Longitude]))

(s/def
  :cognitect.aws.guardduty/RemoteIpDetails
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.RemoteIpDetails/IpAddressV6
     :cognitect.aws.guardduty.RemoteIpDetails/Country
     :cognitect.aws.guardduty.RemoteIpDetails/Organization
     :cognitect.aws.guardduty.RemoteIpDetails/IpAddressV4
     :cognitect.aws.guardduty.RemoteIpDetails/GeoLocation
     :cognitect.aws.guardduty.RemoteIpDetails/City]))

(s/def
  :cognitect.aws.guardduty/ScanCriterionKey
  (s/spec string? :gen #(s/gen #{"EC2_INSTANCE_TAG"})))

(s/def
  :cognitect.aws.guardduty/Issues
  (s/coll-of :cognitect.aws.guardduty/String :min-count 0 :max-count 50))

(s/def
  :cognitect.aws.guardduty/DetectorFeatureConfiguration
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.DetectorFeatureConfiguration/Status
     :cognitect.aws.guardduty.DetectorFeatureConfiguration/AdditionalConfiguration
     :cognitect.aws.guardduty.DetectorFeatureConfiguration/Name]))

(s/def
  :cognitect.aws.guardduty/Indicators
  (s/coll-of :cognitect.aws.guardduty/Indicator :max-count 400))

(s/def
  :cognitect.aws.guardduty/AutonomousSystem
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.AutonomousSystem/Name
     :cognitect.aws.guardduty.AutonomousSystem/Number]))

(s/def
  :cognitect.aws.guardduty/OrganizationEbsVolumesResult
  (s/keys :opt-un [:cognitect.aws.guardduty.OrganizationEbsVolumesResult/AutoEnable]))

(s/def
  :cognitect.aws.guardduty/GeoLocation
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.GeoLocation/Lat :cognitect.aws.guardduty.GeoLocation/Lon]))

(s/def
  :cognitect.aws.guardduty/NetworkEndpoints
  (s/coll-of :cognitect.aws.guardduty/NetworkEndpoint :max-count 400))

(s/def
  :cognitect.aws.guardduty/EbsSnapshotPreservation
  (s/spec string? :gen #(s/gen #{"RETENTION_WITH_FINDING" "NO_RETENTION"})))

(s/def
  :cognitect.aws.guardduty/MemberAdditionalConfigurationResults
  (s/coll-of :cognitect.aws.guardduty/MemberAdditionalConfigurationResult))

(s/def
  :cognitect.aws.guardduty/ScanDetections
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.ScanDetections/HighestSeverityThreatDetails
     :cognitect.aws.guardduty.ScanDetections/ThreatsDetectedItemCount
     :cognitect.aws.guardduty.ScanDetections/ThreatDetectedByName
     :cognitect.aws.guardduty.ScanDetections/ScannedItemCount]))

(s/def
  :cognitect.aws.guardduty/Finding
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.Finding/AccountId
     :cognitect.aws.guardduty.Finding/Arn
     :cognitect.aws.guardduty.Finding/CreatedAt
     :cognitect.aws.guardduty.Finding/Id
     :cognitect.aws.guardduty.Finding/Region
     :cognitect.aws.guardduty.Finding/Resource
     :cognitect.aws.guardduty.Finding/SchemaVersion
     :cognitect.aws.guardduty.Finding/Severity
     :cognitect.aws.guardduty.Finding/Type
     :cognitect.aws.guardduty.Finding/UpdatedAt]
    :opt-un
    [:cognitect.aws.guardduty.Finding/Title
     :cognitect.aws.guardduty.Finding/Description
     :cognitect.aws.guardduty.Finding/Partition
     :cognitect.aws.guardduty.Finding/Service
     :cognitect.aws.guardduty.Finding/AssociatedAttackSequenceArn
     :cognitect.aws.guardduty.Finding/Confidence]))

(s/def
  :cognitect.aws.guardduty/AccessKeyDetails
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.AccessKeyDetails/PrincipalId
     :cognitect.aws.guardduty.AccessKeyDetails/UserType
     :cognitect.aws.guardduty.AccessKeyDetails/UserName
     :cognitect.aws.guardduty.AccessKeyDetails/AccessKeyId]))

(s/def
  :cognitect.aws.guardduty/OrgFeatureAdditionalConfiguration
  (s/spec
    string?
    :gen
    #(s/gen #{"EKS_ADDON_MANAGEMENT" "EC2_AGENT_MANAGEMENT" "ECS_FARGATE_AGENT_MANAGEMENT"})))

(s/def
  :cognitect.aws.guardduty/IamInstanceProfile
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.IamInstanceProfile/Arn
     :cognitect.aws.guardduty.IamInstanceProfile/Id]))

(s/def
  :cognitect.aws.guardduty/GetAdministratorAccountResponse
  (s/keys :req-un [:cognitect.aws.guardduty.GetAdministratorAccountResponse/Administrator]))

(s/def
  :cognitect.aws.guardduty/ListOrganizationAdminAccountsRequest
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.ListOrganizationAdminAccountsRequest/NextToken
     :cognitect.aws.guardduty.ListOrganizationAdminAccountsRequest/MaxResults]))

(s/def
  :cognitect.aws.guardduty/ResourceData
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.ResourceData/EksCluster
     :cognitect.aws.guardduty.ResourceData/Ec2Instance
     :cognitect.aws.guardduty.ResourceData/Ec2NetworkInterface
     :cognitect.aws.guardduty.ResourceData/KubernetesWorkload
     :cognitect.aws.guardduty.ResourceData/Container
     :cognitect.aws.guardduty.ResourceData/S3Bucket
     :cognitect.aws.guardduty.ResourceData/AccessKey
     :cognitect.aws.guardduty.ResourceData/S3Object]))

(s/def :cognitect.aws.guardduty/LoginAttributes (s/coll-of :cognitect.aws.guardduty/LoginAttribute))

(s/def :cognitect.aws.guardduty/Volumes (s/coll-of :cognitect.aws.guardduty/Volume))

(s/def
  :cognitect.aws.guardduty/DataSourceStatus
  (s/spec string? :gen #(s/gen #{"DISABLED" "ENABLED"})))

(s/def
  :cognitect.aws.guardduty/CoverageResource
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.CoverageResource/ResourceDetails
     :cognitect.aws.guardduty.CoverageResource/ResourceId
     :cognitect.aws.guardduty.CoverageResource/Issue
     :cognitect.aws.guardduty.CoverageResource/AccountId
     :cognitect.aws.guardduty.CoverageResource/CoverageStatus
     :cognitect.aws.guardduty.CoverageResource/DetectorId
     :cognitect.aws.guardduty.CoverageResource/UpdatedAt]))

(s/def
  :cognitect.aws.guardduty/MalwareProtectionPlanSummary
  (s/keys :opt-un [:cognitect.aws.guardduty.MalwareProtectionPlanSummary/MalwareProtectionPlanId]))

(s/def
  :cognitect.aws.guardduty/DeleteMalwareProtectionPlanRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.DeleteMalwareProtectionPlanRequest/MalwareProtectionPlanId]))

(s/def
  :cognitect.aws.guardduty/Name
  (s/spec
    (s/and string? #(<= 1 (count %) 300))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 1 300) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.guardduty/MaxResults
  (s/spec (s/and int? #(<= 1 % 50)) :gen #(gen/choose 1 50)))

(s/def :cognitect.aws.guardduty/UpdateTrustedEntitySetResponse (s/keys))

(s/def
  :cognitect.aws.guardduty/UntagResourceRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.UntagResourceRequest/ResourceArn
     :cognitect.aws.guardduty.UntagResourceRequest/TagKeys]))

(s/def
  :cognitect.aws.guardduty/TagKey
  (s/spec #(re-matches (re-pattern "^(?!aws:)[a-zA-Z+-=._:/]+$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.guardduty/GetUsageStatisticsRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.GetUsageStatisticsRequest/DetectorId
     :cognitect.aws.guardduty.GetUsageStatisticsRequest/UsageStatisticType
     :cognitect.aws.guardduty.GetUsageStatisticsRequest/UsageCriteria]
    :opt-un
    [:cognitect.aws.guardduty.GetUsageStatisticsRequest/Unit
     :cognitect.aws.guardduty.GetUsageStatisticsRequest/NextToken
     :cognitect.aws.guardduty.GetUsageStatisticsRequest/MaxResults]))

(s/def
  :cognitect.aws.guardduty/SortCriteria
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.SortCriteria/AttributeName
     :cognitect.aws.guardduty.SortCriteria/OrderBy]))

(s/def
  :cognitect.aws.guardduty/FilterCriterion
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.FilterCriterion/FilterCondition
     :cognitect.aws.guardduty.FilterCriterion/CriterionKey]))

(s/def
  :cognitect.aws.guardduty/Resource
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.Resource/EbsVolumeDetails
     :cognitect.aws.guardduty.Resource/KubernetesDetails
     :cognitect.aws.guardduty.Resource/EksClusterDetails
     :cognitect.aws.guardduty.Resource/RdsLimitlessDbDetails
     :cognitect.aws.guardduty.Resource/InstanceDetails
     :cognitect.aws.guardduty.Resource/ContainerDetails
     :cognitect.aws.guardduty.Resource/EcsClusterDetails
     :cognitect.aws.guardduty.Resource/AccessKeyDetails
     :cognitect.aws.guardduty.Resource/ResourceType
     :cognitect.aws.guardduty.Resource/RdsDbUserDetails
     :cognitect.aws.guardduty.Resource/LambdaDetails
     :cognitect.aws.guardduty.Resource/S3BucketDetails
     :cognitect.aws.guardduty.Resource/RdsDbInstanceDetails]))

(s/def
  :cognitect.aws.guardduty/OrgFeatureStatus
  (s/spec string? :gen #(s/gen #{"NEW" "NONE" "ALL"})))

(s/def :cognitect.aws.guardduty/UpdateThreatEntitySetResponse (s/keys))

(s/def
  :cognitect.aws.guardduty/OrganizationFeaturesConfigurations
  (s/coll-of :cognitect.aws.guardduty/OrganizationFeatureConfiguration))

(s/def
  :cognitect.aws.guardduty/UsageTopAccountsResult
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.UsageTopAccountsResult/Feature
     :cognitect.aws.guardduty.UsageTopAccountsResult/Accounts]))

(s/def
  :cognitect.aws.guardduty/UpdateThreatIntelSetRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.UpdateThreatIntelSetRequest/DetectorId
     :cognitect.aws.guardduty.UpdateThreatIntelSetRequest/ThreatIntelSetId]
    :opt-un
    [:cognitect.aws.guardduty.UpdateThreatIntelSetRequest/Location
     :cognitect.aws.guardduty.UpdateThreatIntelSetRequest/ExpectedBucketOwner
     :cognitect.aws.guardduty.UpdateThreatIntelSetRequest/Name
     :cognitect.aws.guardduty.UpdateThreatIntelSetRequest/Activate]))

(s/def
  :cognitect.aws.guardduty/GroupedBySeverity
  (s/coll-of :cognitect.aws.guardduty/SeverityStatistics))

(s/def
  :cognitect.aws.guardduty/KubernetesDetails
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.KubernetesDetails/KubernetesWorkloadDetails
     :cognitect.aws.guardduty.KubernetesDetails/KubernetesUserDetails]))

(s/def
  :cognitect.aws.guardduty/LineageObject
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.LineageObject/Euid
     :cognitect.aws.guardduty.LineageObject/Pid
     :cognitect.aws.guardduty.LineageObject/StartTime
     :cognitect.aws.guardduty.LineageObject/UserId
     :cognitect.aws.guardduty.LineageObject/Uuid
     :cognitect.aws.guardduty.LineageObject/NamespacePid
     :cognitect.aws.guardduty.LineageObject/ExecutablePath
     :cognitect.aws.guardduty.LineageObject/Name
     :cognitect.aws.guardduty.LineageObject/ParentUuid]))

(s/def
  :cognitect.aws.guardduty/ProcessDetails
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.ProcessDetails/Lineage
     :cognitect.aws.guardduty.ProcessDetails/Euid
     :cognitect.aws.guardduty.ProcessDetails/ExecutableSha256
     :cognitect.aws.guardduty.ProcessDetails/Pid
     :cognitect.aws.guardduty.ProcessDetails/Pwd
     :cognitect.aws.guardduty.ProcessDetails/StartTime
     :cognitect.aws.guardduty.ProcessDetails/UserId
     :cognitect.aws.guardduty.ProcessDetails/Uuid
     :cognitect.aws.guardduty.ProcessDetails/User
     :cognitect.aws.guardduty.ProcessDetails/NamespacePid
     :cognitect.aws.guardduty.ProcessDetails/ExecutablePath
     :cognitect.aws.guardduty.ProcessDetails/Name
     :cognitect.aws.guardduty.ProcessDetails/ParentUuid]))

(s/def :cognitect.aws.guardduty/ThreatNames (s/coll-of :cognitect.aws.guardduty/String))

(s/def
  :cognitect.aws.guardduty/IpSetStatus
  (s/spec
    string?
    :gen
    #(s/gen #{"DEACTIVATING" "ACTIVATING" "ERROR" "INACTIVE" "ACTIVE" "DELETE_PENDING" "DELETED"})))

(s/def
  :cognitect.aws.guardduty/AccessKey
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.AccessKey/PrincipalId
     :cognitect.aws.guardduty.AccessKey/UserType
     :cognitect.aws.guardduty.AccessKey/UserName]))

(s/def
  :cognitect.aws.guardduty/DisassociateFromAdministratorAccountRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.DisassociateFromAdministratorAccountRequest/DetectorId]))

(s/def
  :cognitect.aws.guardduty/DateStatistics
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.DateStatistics/Severity
     :cognitect.aws.guardduty.DateStatistics/LastGeneratedAt
     :cognitect.aws.guardduty.DateStatistics/Date
     :cognitect.aws.guardduty.DateStatistics/TotalFindings]))

(s/def
  :cognitect.aws.guardduty/UsageResourceResultList
  (s/coll-of :cognitect.aws.guardduty/UsageResourceResult))

(s/def
  :cognitect.aws.guardduty/ClientToken
  (s/spec
    (s/and string? #(<= 0 (count %) 64))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 0 64) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.guardduty/UpdateMemberDetectorsRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.UpdateMemberDetectorsRequest/DetectorId
     :cognitect.aws.guardduty.UpdateMemberDetectorsRequest/AccountIds]
    :opt-un
    [:cognitect.aws.guardduty.UpdateMemberDetectorsRequest/DataSources
     :cognitect.aws.guardduty.UpdateMemberDetectorsRequest/Features]))

(s/def :cognitect.aws.guardduty/ResourceList (s/coll-of :cognitect.aws.guardduty/String))

(s/def
  :cognitect.aws.guardduty/CoverageFilterCriterionList
  (s/coll-of :cognitect.aws.guardduty/CoverageFilterCriterion :min-count 0 :max-count 50))

(s/def
  :cognitect.aws.guardduty/CoverageSortKey
  (s/spec
    string?
    :gen
    #(s/gen
      #{"ECS_CLUSTER_NAME"
        "INSTANCE_ID"
        "COVERAGE_STATUS"
        "EKS_CLUSTER_NAME"
        "CLUSTER_NAME"
        "UPDATED_AT"
        "ACCOUNT_ID"
        "ISSUE"
        "ADDON_VERSION"})))

(s/def
  :cognitect.aws.guardduty/KubernetesUserDetails
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.KubernetesUserDetails/Username
     :cognitect.aws.guardduty.KubernetesUserDetails/ImpersonatedUser
     :cognitect.aws.guardduty.KubernetesUserDetails/Uid
     :cognitect.aws.guardduty.KubernetesUserDetails/Groups
     :cognitect.aws.guardduty.KubernetesUserDetails/SessionName]))

(s/def
  :cognitect.aws.guardduty/DomainDetails
  (s/keys :opt-un [:cognitect.aws.guardduty.DomainDetails/Domain]))

(s/def :cognitect.aws.guardduty/Timestamp inst?)

(s/def
  :cognitect.aws.guardduty/FilterName
  (s/spec
    (s/and string? #(<= 3 (count %) 64))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 3 64) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.guardduty/PrivateIpAddressDetails
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.PrivateIpAddressDetails/PrivateDnsName
     :cognitect.aws.guardduty.PrivateIpAddressDetails/PrivateIpAddress]))

(s/def
  :cognitect.aws.guardduty/UpdateIPSetRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.UpdateIPSetRequest/DetectorId
     :cognitect.aws.guardduty.UpdateIPSetRequest/IpSetId]
    :opt-un
    [:cognitect.aws.guardduty.UpdateIPSetRequest/Location
     :cognitect.aws.guardduty.UpdateIPSetRequest/ExpectedBucketOwner
     :cognitect.aws.guardduty.UpdateIPSetRequest/Name
     :cognitect.aws.guardduty.UpdateIPSetRequest/Activate]))

(s/def :cognitect.aws.guardduty/SessionNameList (s/coll-of :cognitect.aws.guardduty/String))

(s/def
  :cognitect.aws.guardduty/UpdateFindingsFeedbackRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.UpdateFindingsFeedbackRequest/DetectorId
     :cognitect.aws.guardduty.UpdateFindingsFeedbackRequest/FindingIds
     :cognitect.aws.guardduty.UpdateFindingsFeedbackRequest/Feedback]
    :opt-un
    [:cognitect.aws.guardduty.UpdateFindingsFeedbackRequest/Comments]))

(s/def
  :cognitect.aws.guardduty/AdminAccount
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.AdminAccount/AdminStatus
     :cognitect.aws.guardduty.AdminAccount/AdminAccountId]))

(s/def
  :cognitect.aws.guardduty/DeleteFilterRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.DeleteFilterRequest/DetectorId
     :cognitect.aws.guardduty.DeleteFilterRequest/FilterName]))

(s/def
  :cognitect.aws.guardduty/KubernetesRoleDetails
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.KubernetesRoleDetails/Uid
     :cognitect.aws.guardduty.KubernetesRoleDetails/Name
     :cognitect.aws.guardduty.KubernetesRoleDetails/Kind]))

(s/def
  :cognitect.aws.guardduty/FindingTypeStatistics
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.FindingTypeStatistics/LastGeneratedAt
     :cognitect.aws.guardduty.FindingTypeStatistics/TotalFindings
     :cognitect.aws.guardduty.FindingTypeStatistics/FindingType]))

(s/def
  :cognitect.aws.guardduty/AcceptAdministratorInvitationRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.AcceptAdministratorInvitationRequest/DetectorId
     :cognitect.aws.guardduty.AcceptAdministratorInvitationRequest/AdministratorId
     :cognitect.aws.guardduty.AcceptAdministratorInvitationRequest/InvitationId]))

(s/def
  :cognitect.aws.guardduty/GetMalwareScanSettingsRequest
  (s/keys :req-un [:cognitect.aws.guardduty.GetMalwareScanSettingsRequest/DetectorId]))

(s/def :cognitect.aws.guardduty/FilePaths (s/coll-of :cognitect.aws.guardduty/ScanFilePath))

(s/def :cognitect.aws.guardduty/DisableOrganizationAdminAccountResponse (s/keys))

(s/def
  :cognitect.aws.guardduty/VolumeMount
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.VolumeMount/MountPath :cognitect.aws.guardduty.VolumeMount/Name]))

(s/def
  :cognitect.aws.guardduty/LongValue
  (s/spec int? :gen #(gen/choose Long/MIN_VALUE Long/MAX_VALUE)))

(s/def
  :cognitect.aws.guardduty/SignalType
  (s/spec
    string?
    :gen
    #(s/gen
      #{"RUNTIME_MONITORING"
        "S3_DATA_EVENTS"
        "FINDING"
        "EKS_AUDIT_LOGS"
        "FLOW_LOGS"
        "CLOUD_TRAIL"
        "DNS_LOGS"})))

(s/def
  :cognitect.aws.guardduty/OrganizationKubernetesConfiguration
  (s/keys :req-un [:cognitect.aws.guardduty.OrganizationKubernetesConfiguration/AuditLogs]))

(s/def
  :cognitect.aws.guardduty/Country
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.Country/CountryName :cognitect.aws.guardduty.Country/CountryCode]))

(s/def
  :cognitect.aws.guardduty/GetIPSetResponse
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.GetIPSetResponse/Name
     :cognitect.aws.guardduty.GetIPSetResponse/Format
     :cognitect.aws.guardduty.GetIPSetResponse/Location
     :cognitect.aws.guardduty.GetIPSetResponse/Status]
    :opt-un
    [:cognitect.aws.guardduty.GetIPSetResponse/Tags
     :cognitect.aws.guardduty.GetIPSetResponse/ExpectedBucketOwner]))

(s/def
  :cognitect.aws.guardduty/ArchiveFindingsRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.ArchiveFindingsRequest/DetectorId
     :cognitect.aws.guardduty.ArchiveFindingsRequest/FindingIds]))

(s/def
  :cognitect.aws.guardduty/ResourceArn
  (s/spec
    #(re-matches
      (re-pattern
        "^arn:[A-Za-z-]+:[A-Za-z0-9]+:[A-Za-z0-9-]+:\\d+:(([A-Za-z0-9-]+)[:\\/])?[A-Za-z0-9-]*$")
      %)
    :gen
    #(gen/string)))

(s/def :cognitect.aws.guardduty/UpdateOrganizationConfigurationResponse (s/keys))

(s/def
  :cognitect.aws.guardduty/ListPublishingDestinationsRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.ListPublishingDestinationsRequest/DetectorId]
    :opt-un
    [:cognitect.aws.guardduty.ListPublishingDestinationsRequest/NextToken
     :cognitect.aws.guardduty.ListPublishingDestinationsRequest/MaxResults]))

(s/def :cognitect.aws.guardduty/DeletePublishingDestinationResponse (s/keys))

(s/def
  :cognitect.aws.guardduty/Condition
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.Condition/GreaterThanOrEqual
     :cognitect.aws.guardduty.Condition/Lte
     :cognitect.aws.guardduty.Condition/Lt
     :cognitect.aws.guardduty.Condition/NotEquals
     :cognitect.aws.guardduty.Condition/LessThan
     :cognitect.aws.guardduty.Condition/Neq
     :cognitect.aws.guardduty.Condition/Gt
     :cognitect.aws.guardduty.Condition/GreaterThan
     :cognitect.aws.guardduty.Condition/Gte
     :cognitect.aws.guardduty.Condition/LessThanOrEqual
     :cognitect.aws.guardduty.Condition/Eq
     :cognitect.aws.guardduty.Condition/Equals]))

(s/def
  :cognitect.aws.guardduty/CreateThreatEntitySetResponse
  (s/keys :req-un [:cognitect.aws.guardduty.CreateThreatEntitySetResponse/ThreatEntitySetId]))

(s/def
  :cognitect.aws.guardduty/AdminStatus
  (s/spec string? :gen #(s/gen #{"ENABLED" "DISABLE_IN_PROGRESS"})))

(s/def
  :cognitect.aws.guardduty/LambdaDetails
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.LambdaDetails/FunctionArn
     :cognitect.aws.guardduty.LambdaDetails/FunctionName
     :cognitect.aws.guardduty.LambdaDetails/Tags
     :cognitect.aws.guardduty.LambdaDetails/FunctionVersion
     :cognitect.aws.guardduty.LambdaDetails/Role
     :cognitect.aws.guardduty.LambdaDetails/Description
     :cognitect.aws.guardduty.LambdaDetails/VpcConfig
     :cognitect.aws.guardduty.LambdaDetails/RevisionId
     :cognitect.aws.guardduty.LambdaDetails/LastModifiedAt]))

(s/def
  :cognitect.aws.guardduty/MalwareProtectionPlanStatusReason
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.MalwareProtectionPlanStatusReason/Message
     :cognitect.aws.guardduty.MalwareProtectionPlanStatusReason/Code]))

(s/def
  :cognitect.aws.guardduty/CoverageEcsClusterDetails
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.CoverageEcsClusterDetails/ContainerInstanceDetails
     :cognitect.aws.guardduty.CoverageEcsClusterDetails/FargateDetails
     :cognitect.aws.guardduty.CoverageEcsClusterDetails/ClusterName]))

(s/def
  :cognitect.aws.guardduty/PortProbeDetail
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.PortProbeDetail/LocalPortDetails
     :cognitect.aws.guardduty.PortProbeDetail/RemoteIpDetails
     :cognitect.aws.guardduty.PortProbeDetail/LocalIpDetails]))

(s/def
  :cognitect.aws.guardduty/AccountFreeTrialInfo
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.AccountFreeTrialInfo/DataSources
     :cognitect.aws.guardduty.AccountFreeTrialInfo/AccountId
     :cognitect.aws.guardduty.AccountFreeTrialInfo/Features]))

(s/def
  :cognitect.aws.guardduty/KubernetesWorkload
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.KubernetesWorkload/Namespace
     :cognitect.aws.guardduty.KubernetesWorkload/KubernetesResourcesTypes
     :cognitect.aws.guardduty.KubernetesWorkload/ContainerUids]))

(s/def
  :cognitect.aws.guardduty/DataSource
  (s/spec
    string?
    :gen
    #(s/gen
      #{"S3_LOGS"
        "KUBERNETES_AUDIT_LOGS"
        "FLOW_LOGS"
        "EC2_MALWARE_SCAN"
        "CLOUD_TRAIL"
        "DNS_LOGS"})))

(s/def
  :cognitect.aws.guardduty/ListIPSetsRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.ListIPSetsRequest/DetectorId]
    :opt-un
    [:cognitect.aws.guardduty.ListIPSetsRequest/NextToken
     :cognitect.aws.guardduty.ListIPSetsRequest/MaxResults]))

(s/def
  :cognitect.aws.guardduty/MalwareProtectionPlanTaggingAction
  (s/keys :opt-un [:cognitect.aws.guardduty.MalwareProtectionPlanTaggingAction/Status]))

(s/def
  :cognitect.aws.guardduty/SeverityStatistics
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.SeverityStatistics/Severity
     :cognitect.aws.guardduty.SeverityStatistics/LastGeneratedAt
     :cognitect.aws.guardduty.SeverityStatistics/TotalFindings]))

(s/def
  :cognitect.aws.guardduty/CoverageFilterCriterionKey
  (s/spec
    string?
    :gen
    #(s/gen
      #{"ECS_CLUSTER_NAME"
        "AGENT_VERSION"
        "INSTANCE_ID"
        "COVERAGE_STATUS"
        "EKS_CLUSTER_NAME"
        "CLUSTER_ARN"
        "CLUSTER_NAME"
        "RESOURCE_TYPE"
        "ACCOUNT_ID"
        "MANAGEMENT_TYPE"
        "ADDON_VERSION"})))

(s/def
  :cognitect.aws.guardduty/DeleteDetectorRequest
  (s/keys :req-un [:cognitect.aws.guardduty.DeleteDetectorRequest/DetectorId]))

(s/def
  :cognitect.aws.guardduty/UpdateTrustedEntitySetRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.UpdateTrustedEntitySetRequest/DetectorId
     :cognitect.aws.guardduty.UpdateTrustedEntitySetRequest/TrustedEntitySetId]
    :opt-un
    [:cognitect.aws.guardduty.UpdateTrustedEntitySetRequest/Location
     :cognitect.aws.guardduty.UpdateTrustedEntitySetRequest/ExpectedBucketOwner
     :cognitect.aws.guardduty.UpdateTrustedEntitySetRequest/Name
     :cognitect.aws.guardduty.UpdateTrustedEntitySetRequest/Activate]))

(s/def
  :cognitect.aws.guardduty/EbsVolumeDetails
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.EbsVolumeDetails/ScannedVolumeDetails
     :cognitect.aws.guardduty.EbsVolumeDetails/SkippedVolumeDetails]))

(s/def
  :cognitect.aws.guardduty/CoverageResources
  (s/coll-of :cognitect.aws.guardduty/CoverageResource))

(s/def :cognitect.aws.guardduty/GetInvitationsCountRequest (s/keys))

(s/def :cognitect.aws.guardduty/VolumeDetails (s/coll-of :cognitect.aws.guardduty/VolumeDetail))

(s/def
  :cognitect.aws.guardduty/GetMemberDetectorsResponse
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.GetMemberDetectorsResponse/MemberDataSourceConfigurations
     :cognitect.aws.guardduty.GetMemberDetectorsResponse/UnprocessedAccounts]))

(s/def
  :cognitect.aws.guardduty/CreateThreatIntelSetRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.CreateThreatIntelSetRequest/DetectorId
     :cognitect.aws.guardduty.CreateThreatIntelSetRequest/Name
     :cognitect.aws.guardduty.CreateThreatIntelSetRequest/Format
     :cognitect.aws.guardduty.CreateThreatIntelSetRequest/Location
     :cognitect.aws.guardduty.CreateThreatIntelSetRequest/Activate]
    :opt-un
    [:cognitect.aws.guardduty.CreateThreatIntelSetRequest/Tags
     :cognitect.aws.guardduty.CreateThreatIntelSetRequest/ExpectedBucketOwner
     :cognitect.aws.guardduty.CreateThreatIntelSetRequest/ClientToken]))

(s/def
  :cognitect.aws.guardduty/Indicator
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.Indicator/Key]
    :opt-un
    [:cognitect.aws.guardduty.Indicator/Title :cognitect.aws.guardduty.Indicator/Values]))

(s/def :cognitect.aws.guardduty/UpdateFindingsFeedbackResponse (s/keys))

(s/def
  :cognitect.aws.guardduty/DeleteMembersResponse
  (s/keys :req-un [:cognitect.aws.guardduty.DeleteMembersResponse/UnprocessedAccounts]))

(s/def
  :cognitect.aws.guardduty/Double
  (s/spec double? :gen #(gen/double* {:infinite? false, :NaN? false})))

(s/def
  :cognitect.aws.guardduty/FindingStatistics
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.FindingStatistics/GroupedByFindingType
     :cognitect.aws.guardduty.FindingStatistics/CountBySeverity
     :cognitect.aws.guardduty.FindingStatistics/GroupedBySeverity
     :cognitect.aws.guardduty.FindingStatistics/GroupedByDate
     :cognitect.aws.guardduty.FindingStatistics/GroupedByResource
     :cognitect.aws.guardduty.FindingStatistics/GroupedByAccount]))

(s/def
  :cognitect.aws.guardduty/Location
  (s/spec
    (s/and string? #(<= 1 (count %) 300))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 1 300) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.guardduty/PortProbeAction
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.PortProbeAction/Blocked
     :cognitect.aws.guardduty.PortProbeAction/PortProbeDetails]))

(s/def
  :cognitect.aws.guardduty/Organization
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.Organization/Isp
     :cognitect.aws.guardduty.Organization/AsnOrg
     :cognitect.aws.guardduty.Organization/Org
     :cognitect.aws.guardduty.Organization/Asn]))

(s/def
  :cognitect.aws.guardduty/MalwareProtectionPlansSummary
  (s/coll-of :cognitect.aws.guardduty/MalwareProtectionPlanSummary))

(s/def
  :cognitect.aws.guardduty/CreateThreatEntitySetRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.CreateThreatEntitySetRequest/DetectorId
     :cognitect.aws.guardduty.CreateThreatEntitySetRequest/Name
     :cognitect.aws.guardduty.CreateThreatEntitySetRequest/Format
     :cognitect.aws.guardduty.CreateThreatEntitySetRequest/Location
     :cognitect.aws.guardduty.CreateThreatEntitySetRequest/Activate]
    :opt-un
    [:cognitect.aws.guardduty.CreateThreatEntitySetRequest/Tags
     :cognitect.aws.guardduty.CreateThreatEntitySetRequest/ExpectedBucketOwner
     :cognitect.aws.guardduty.CreateThreatEntitySetRequest/ClientToken]))

(s/def
  :cognitect.aws.guardduty/Session
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.Session/MfaStatus
     :cognitect.aws.guardduty.Session/Uid
     :cognitect.aws.guardduty.Session/CreatedTime
     :cognitect.aws.guardduty.Session/Issuer]))

(s/def :cognitect.aws.guardduty/Equals (s/coll-of :cognitect.aws.guardduty/String))

(s/def
  :cognitect.aws.guardduty/DetectorFeature
  (s/spec
    string?
    :gen
    #(s/gen
      #{"EBS_MALWARE_PROTECTION"
        "EKS_RUNTIME_MONITORING"
        "LAMBDA_NETWORK_LOGS"
        "RUNTIME_MONITORING"
        "RDS_LOGIN_EVENTS"
        "S3_DATA_EVENTS"
        "EKS_AUDIT_LOGS"})))

(s/def
  :cognitect.aws.guardduty/EndpointIds
  (s/coll-of :cognitect.aws.guardduty/String :max-count 400))

(s/def :cognitect.aws.guardduty/FilterAction (s/spec string? :gen #(s/gen #{"NOOP" "ARCHIVE"})))

(s/def
  :cognitect.aws.guardduty/AdminAccounts
  (s/coll-of :cognitect.aws.guardduty/AdminAccount :min-count 0 :max-count 1))

(s/def
  :cognitect.aws.guardduty/NetworkInterfaces
  (s/coll-of :cognitect.aws.guardduty/NetworkInterface))

(s/def :cognitect.aws.guardduty/DeleteTrustedEntitySetResponse (s/keys))

(s/def
  :cognitect.aws.guardduty/DataSourcesFreeTrial
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.DataSourcesFreeTrial/DnsLogs
     :cognitect.aws.guardduty.DataSourcesFreeTrial/S3Logs
     :cognitect.aws.guardduty.DataSourcesFreeTrial/CloudTrail
     :cognitect.aws.guardduty.DataSourcesFreeTrial/MalwareProtection
     :cognitect.aws.guardduty.DataSourcesFreeTrial/FlowLogs
     :cognitect.aws.guardduty.DataSourcesFreeTrial/Kubernetes]))

(s/def :cognitect.aws.guardduty/DeleteFilterResponse (s/keys))

(s/def
  :cognitect.aws.guardduty/MemberAdditionalConfigurations
  (s/coll-of :cognitect.aws.guardduty/MemberAdditionalConfiguration))

(s/def
  :cognitect.aws.guardduty/FreeTrialFeatureResult
  (s/spec
    string?
    :gen
    #(s/gen
      #{"EBS_MALWARE_PROTECTION"
        "EKS_RUNTIME_MONITORING"
        "LAMBDA_NETWORK_LOGS"
        "RDS_LOGIN_EVENTS"
        "S3_DATA_EVENTS"
        "EKS_AUDIT_LOGS"
        "EC2_RUNTIME_MONITORING"
        "FLOW_LOGS"
        "FARGATE_RUNTIME_MONITORING"
        "CLOUD_TRAIL"
        "DNS_LOGS"})))

(s/def
  :cognitect.aws.guardduty/ListTagsForResourceResponse
  (s/keys :opt-un [:cognitect.aws.guardduty.ListTagsForResourceResponse/Tags]))

(s/def
  :cognitect.aws.guardduty/DetectorStatus
  (s/spec string? :gen #(s/gen #{"DISABLED" "ENABLED"})))

(s/def
  :cognitect.aws.guardduty/MemberAdditionalConfiguration
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.MemberAdditionalConfiguration/Status
     :cognitect.aws.guardduty.MemberAdditionalConfiguration/Name]))

(s/def
  :cognitect.aws.guardduty/MemberAdditionalConfigurationResult
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.MemberAdditionalConfigurationResult/Status
     :cognitect.aws.guardduty.MemberAdditionalConfigurationResult/UpdatedAt
     :cognitect.aws.guardduty.MemberAdditionalConfigurationResult/Name]))

(s/def
  :cognitect.aws.guardduty/CoverageStatus
  (s/spec string? :gen #(s/gen #{"HEALTHY" "UNHEALTHY"})))

(s/def
  :cognitect.aws.guardduty/GuardDutyArn
  (s/spec
    #(re-matches
      (re-pattern
        "^arn:[A-Za-z_.-]{1,20}:guardduty:[A-Za-z0-9_/.-]{0,63}:\\d+:detector/[A-Za-z0-9_/.-]{32,264}$")
      %)
    :gen
    #(gen/string)))

(s/def
  :cognitect.aws.guardduty/FilterCondition
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.FilterCondition/LessThan
     :cognitect.aws.guardduty.FilterCondition/GreaterThan
     :cognitect.aws.guardduty.FilterCondition/EqualsValue]))

(s/def
  :cognitect.aws.guardduty/DnsRequestAction
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.DnsRequestAction/Blocked
     :cognitect.aws.guardduty.DnsRequestAction/Domain
     :cognitect.aws.guardduty.DnsRequestAction/Protocol
     :cognitect.aws.guardduty.DnsRequestAction/VpcOwnerAccountId
     :cognitect.aws.guardduty.DnsRequestAction/DomainWithSuffix]))

(s/def
  :cognitect.aws.guardduty/DescribePublishingDestinationResponse
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.DescribePublishingDestinationResponse/DestinationId
     :cognitect.aws.guardduty.DescribePublishingDestinationResponse/DestinationType
     :cognitect.aws.guardduty.DescribePublishingDestinationResponse/Status
     :cognitect.aws.guardduty.DescribePublishingDestinationResponse/PublishingFailureStartTimestamp
     :cognitect.aws.guardduty.DescribePublishingDestinationResponse/DestinationProperties]))

(s/def
  :cognitect.aws.guardduty/Master
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.Master/InvitationId
     :cognitect.aws.guardduty.Master/RelationshipStatus
     :cognitect.aws.guardduty.Master/AccountId
     :cognitect.aws.guardduty.Master/InvitedAt]))

(s/def
  :cognitect.aws.guardduty/GetIPSetRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.GetIPSetRequest/DetectorId
     :cognitect.aws.guardduty.GetIPSetRequest/IpSetId]))

(s/def
  :cognitect.aws.guardduty/OrganizationDataSourceConfigurationsResult
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.OrganizationDataSourceConfigurationsResult/S3Logs]
    :opt-un
    [:cognitect.aws.guardduty.OrganizationDataSourceConfigurationsResult/MalwareProtection
     :cognitect.aws.guardduty.OrganizationDataSourceConfigurationsResult/Kubernetes]))

(s/def
  :cognitect.aws.guardduty/DetectorIds
  (s/coll-of :cognitect.aws.guardduty/DetectorId :min-count 0 :max-count 50))

(s/def
  :cognitect.aws.guardduty/Threat
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.Threat/Source
     :cognitect.aws.guardduty.Threat/ItemPaths
     :cognitect.aws.guardduty.Threat/Name]))

(s/def :cognitect.aws.guardduty/SourceIps (s/coll-of :cognitect.aws.guardduty/String))

(s/def
  :cognitect.aws.guardduty/CreateMalwareProtectionPlanResponse
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.CreateMalwareProtectionPlanResponse/MalwareProtectionPlanId]))

(s/def
  :cognitect.aws.guardduty/Detection
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.Detection/Sequence :cognitect.aws.guardduty.Detection/Anomaly]))

(s/def
  :cognitect.aws.guardduty/ListFindingsRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.ListFindingsRequest/DetectorId]
    :opt-un
    [:cognitect.aws.guardduty.ListFindingsRequest/NextToken
     :cognitect.aws.guardduty.ListFindingsRequest/MaxResults
     :cognitect.aws.guardduty.ListFindingsRequest/FindingCriteria
     :cognitect.aws.guardduty.ListFindingsRequest/SortCriteria]))

(s/def
  :cognitect.aws.guardduty/FeatureAdditionalConfiguration
  (s/spec
    string?
    :gen
    #(s/gen #{"EKS_ADDON_MANAGEMENT" "EC2_AGENT_MANAGEMENT" "ECS_FARGATE_AGENT_MANAGEMENT"})))

(s/def
  :cognitect.aws.guardduty/OrganizationScanEc2InstanceWithFindingsResult
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.OrganizationScanEc2InstanceWithFindingsResult/EbsVolumes]))

(s/def
  :cognitect.aws.guardduty/NetworkDirection
  (s/spec string? :gen #(s/gen #{"OUTBOUND" "INBOUND"})))

(s/def
  :cognitect.aws.guardduty/ImpersonatedUser
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.ImpersonatedUser/Username
     :cognitect.aws.guardduty.ImpersonatedUser/Groups]))

(s/def :cognitect.aws.guardduty/CreateSampleFindingsResponse (s/keys))

(s/def
  :cognitect.aws.guardduty/ListIPSetsResponse
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.ListIPSetsResponse/IpSetIds]
    :opt-un
    [:cognitect.aws.guardduty.ListIPSetsResponse/NextToken]))

(s/def
  :cognitect.aws.guardduty/CloudTrailConfigurationResult
  (s/keys :req-un [:cognitect.aws.guardduty.CloudTrailConfigurationResult/Status]))

(s/def
  :cognitect.aws.guardduty/ListThreatIntelSetsResponse
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.ListThreatIntelSetsResponse/ThreatIntelSetIds]
    :opt-un
    [:cognitect.aws.guardduty.ListThreatIntelSetsResponse/NextToken]))

(s/def
  :cognitect.aws.guardduty/AcceptInvitationRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.AcceptInvitationRequest/DetectorId
     :cognitect.aws.guardduty.AcceptInvitationRequest/MasterId
     :cognitect.aws.guardduty.AcceptInvitationRequest/InvitationId]))

(s/def
  :cognitect.aws.guardduty/ListMalwareProtectionPlansRequest
  (s/keys :opt-un [:cognitect.aws.guardduty.ListMalwareProtectionPlansRequest/NextToken]))

(s/def
  :cognitect.aws.guardduty/ContainerInstanceDetails
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.ContainerInstanceDetails/CoveredContainerInstances
     :cognitect.aws.guardduty.ContainerInstanceDetails/CompatibleContainerInstances]))

(s/def
  :cognitect.aws.guardduty/StopMonitoringMembersResponse
  (s/keys :req-un [:cognitect.aws.guardduty.StopMonitoringMembersResponse/UnprocessedAccounts]))

(s/def
  :cognitect.aws.guardduty/CountByResourceType
  (s/map-of :cognitect.aws.guardduty/ResourceType :cognitect.aws.guardduty/Long))

(s/def
  :cognitect.aws.guardduty/EksClusterDetails
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.EksClusterDetails/Tags
     :cognitect.aws.guardduty.EksClusterDetails/Status
     :cognitect.aws.guardduty.EksClusterDetails/VpcId
     :cognitect.aws.guardduty.EksClusterDetails/Arn
     :cognitect.aws.guardduty.EksClusterDetails/CreatedAt
     :cognitect.aws.guardduty.EksClusterDetails/Name]))

(s/def
  :cognitect.aws.guardduty/CoverageFilterCondition
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.CoverageFilterCondition/NotEquals
     :cognitect.aws.guardduty.CoverageFilterCondition/Equals]))

(s/def
  :cognitect.aws.guardduty/DetectorId
  (s/spec
    (s/and string? #(<= 1 (count %) 300))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 1 300) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.guardduty/DeleteInvitationsResponse
  (s/keys :req-un [:cognitect.aws.guardduty.DeleteInvitationsResponse/UnprocessedAccounts]))

(s/def :cognitect.aws.guardduty/Threats (s/coll-of :cognitect.aws.guardduty/Threat))

(s/def
  :cognitect.aws.guardduty/GetMemberDetectorsRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.GetMemberDetectorsRequest/DetectorId
     :cognitect.aws.guardduty.GetMemberDetectorsRequest/AccountIds]))

(s/def
  :cognitect.aws.guardduty/MalwareProtectionPlanObjectPrefixesList
  (s/coll-of :cognitect.aws.guardduty/String :min-count 0 :max-count 5))

(s/def :cognitect.aws.guardduty/String string?)

(s/def
  :cognitect.aws.guardduty/NetworkEndpoint
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.NetworkEndpoint/Id]
    :opt-un
    [:cognitect.aws.guardduty.NetworkEndpoint/AutonomousSystem
     :cognitect.aws.guardduty.NetworkEndpoint/Location
     :cognitect.aws.guardduty.NetworkEndpoint/Domain
     :cognitect.aws.guardduty.NetworkEndpoint/Port
     :cognitect.aws.guardduty.NetworkEndpoint/Ip
     :cognitect.aws.guardduty.NetworkEndpoint/Connection]))

(s/def
  :cognitect.aws.guardduty/AnomalyUnusual
  (s/keys :opt-un [:cognitect.aws.guardduty.AnomalyUnusual/Behavior]))

(s/def
  :cognitect.aws.guardduty/CreateMalwareProtectionPlanRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.CreateMalwareProtectionPlanRequest/Role
     :cognitect.aws.guardduty.CreateMalwareProtectionPlanRequest/ProtectedResource]
    :opt-un
    [:cognitect.aws.guardduty.CreateMalwareProtectionPlanRequest/Actions
     :cognitect.aws.guardduty.CreateMalwareProtectionPlanRequest/Tags
     :cognitect.aws.guardduty.CreateMalwareProtectionPlanRequest/ClientToken]))

(s/def
  :cognitect.aws.guardduty/UpdateMalwareScanSettingsRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.UpdateMalwareScanSettingsRequest/DetectorId]
    :opt-un
    [:cognitect.aws.guardduty.UpdateMalwareScanSettingsRequest/EbsSnapshotPreservation
     :cognitect.aws.guardduty.UpdateMalwareScanSettingsRequest/ScanResourceCriteria]))

(s/def
  :cognitect.aws.guardduty/KubernetesConfigurationResult
  (s/keys :req-un [:cognitect.aws.guardduty.KubernetesConfigurationResult/AuditLogs]))

(s/def :cognitect.aws.guardduty/Ipv6Addresses (s/coll-of :cognitect.aws.guardduty/String))

(s/def
  :cognitect.aws.guardduty/DisableOrganizationAdminAccountRequest
  (s/keys :req-un [:cognitect.aws.guardduty.DisableOrganizationAdminAccountRequest/AdminAccountId]))

(s/def
  :cognitect.aws.guardduty/UsageTopAccountsByFeatureList
  (s/coll-of :cognitect.aws.guardduty/UsageTopAccountResult))

(s/def
  :cognitect.aws.guardduty/Findings
  (s/coll-of :cognitect.aws.guardduty/Finding :min-count 0 :max-count 50))

(s/def
  :cognitect.aws.guardduty/ListThreatEntitySetsRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.ListThreatEntitySetsRequest/DetectorId]
    :opt-un
    [:cognitect.aws.guardduty.ListThreatEntitySetsRequest/NextToken
     :cognitect.aws.guardduty.ListThreatEntitySetsRequest/MaxResults]))

(s/def
  :cognitect.aws.guardduty/AccountIds
  (s/coll-of :cognitect.aws.guardduty/AccountId :min-count 1 :max-count 50))

(s/def
  :cognitect.aws.guardduty/DetectorAdditionalConfigurationResults
  (s/coll-of :cognitect.aws.guardduty/DetectorAdditionalConfigurationResult))

(s/def
  :cognitect.aws.guardduty/ResourceNotFoundException
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.ResourceNotFoundException/Type
     :cognitect.aws.guardduty.ResourceNotFoundException/Message]))

(s/def
  :cognitect.aws.guardduty/CoverageStatistics
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.CoverageStatistics/CountByCoverageStatus
     :cognitect.aws.guardduty.CoverageStatistics/CountByResourceType]))

(s/def
  :cognitect.aws.guardduty/ConflictException
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.ConflictException/Type
     :cognitect.aws.guardduty.ConflictException/Message]))

(s/def
  :cognitect.aws.guardduty/DataSourceConfigurations
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.DataSourceConfigurations/S3Logs
     :cognitect.aws.guardduty.DataSourceConfigurations/MalwareProtection
     :cognitect.aws.guardduty.DataSourceConfigurations/Kubernetes]))

(s/def
  :cognitect.aws.guardduty/TagKeyList
  (s/coll-of :cognitect.aws.guardduty/TagKey :min-count 1 :max-count 200))

(s/def
  :cognitect.aws.guardduty/CoverageEc2InstanceDetails
  (s/keys
    :opt-un
    [:cognitect.aws.guardduty.CoverageEc2InstanceDetails/InstanceId
     :cognitect.aws.guardduty.CoverageEc2InstanceDetails/ClusterArn
     :cognitect.aws.guardduty.CoverageEc2InstanceDetails/InstanceType
     :cognitect.aws.guardduty.CoverageEc2InstanceDetails/ManagementType
     :cognitect.aws.guardduty.CoverageEc2InstanceDetails/AgentDetails]))

(s/def
  :cognitect.aws.guardduty/GetAdministratorAccountRequest
  (s/keys :req-un [:cognitect.aws.guardduty.GetAdministratorAccountRequest/DetectorId]))

(s/def :cognitect.aws.guardduty/SensitiveString string?)

(s/def
  :cognitect.aws.guardduty/ListThreatIntelSetsRequest
  (s/keys
    :req-un
    [:cognitect.aws.guardduty.ListThreatIntelSetsRequest/DetectorId]
    :opt-un
    [:cognitect.aws.guardduty.ListThreatIntelSetsRequest/NextToken
     :cognitect.aws.guardduty.ListThreatIntelSetsRequest/MaxResults]))

(s/def
  :cognitect.aws.guardduty/IndicatorValueString
  (s/spec
    (s/and string? #(<= 1 (count %) 256))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 1 256) #(gen/vector (gen/char-alpha) %))))))

(s/def :cognitect.aws.guardduty/UpdateIPSetResponse (s/keys))

(s/def :cognitect.aws.guardduty/ContainerUids (s/coll-of :cognitect.aws.guardduty/ContainerUid))

(s/def :cognitect.aws.guardduty/DisassociateFromAdministratorAccountResponse (s/keys))

(s/def
  :cognitect.aws.guardduty.DataSourceFreeTrial/FreeTrialDaysRemaining
  :cognitect.aws.guardduty/Integer)

(s/def
  :cognitect.aws.guardduty.OrganizationFeatureStatistics/Name
  :cognitect.aws.guardduty/OrgFeature)

(s/def
  :cognitect.aws.guardduty.OrganizationFeatureStatistics/EnabledAccountsCount
  :cognitect.aws.guardduty/Integer)

(s/def
  :cognitect.aws.guardduty.OrganizationFeatureStatistics/AdditionalConfiguration
  :cognitect.aws.guardduty/OrganizationFeatureStatisticsAdditionalConfigurations)

(s/def :cognitect.aws.guardduty.CreateIPSetResponse/IpSetId :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.GetDetectorResponse/CreatedAt :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.GetDetectorResponse/FindingPublishingFrequency
  :cognitect.aws.guardduty/FindingPublishingFrequency)

(s/def :cognitect.aws.guardduty.GetDetectorResponse/ServiceRole :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.GetDetectorResponse/Status :cognitect.aws.guardduty/DetectorStatus)

(s/def :cognitect.aws.guardduty.GetDetectorResponse/UpdatedAt :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.GetDetectorResponse/DataSources
  :cognitect.aws.guardduty/DataSourceConfigurationsResult)

(s/def :cognitect.aws.guardduty.GetDetectorResponse/Tags :cognitect.aws.guardduty/TagMap)

(s/def
  :cognitect.aws.guardduty.GetDetectorResponse/Features
  :cognitect.aws.guardduty/DetectorFeatureConfigurationsResults)

(s/def :cognitect.aws.guardduty.SecurityContext/Privileged :cognitect.aws.guardduty/Boolean)

(s/def
  :cognitect.aws.guardduty.SecurityContext/AllowPrivilegeEscalation
  :cognitect.aws.guardduty/Boolean)

(s/def :cognitect.aws.guardduty.Service/Evidence :cognitect.aws.guardduty/Evidence)

(s/def
  :cognitect.aws.guardduty.Service/MalwareScanDetails
  :cognitect.aws.guardduty/MalwareScanDetails)

(s/def :cognitect.aws.guardduty.Service/RuntimeDetails :cognitect.aws.guardduty/RuntimeDetails)

(s/def :cognitect.aws.guardduty.Service/EventLastSeen :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.Service/EbsVolumeScanDetails
  :cognitect.aws.guardduty/EbsVolumeScanDetails)

(s/def :cognitect.aws.guardduty.Service/Archived :cognitect.aws.guardduty/Boolean)

(s/def :cognitect.aws.guardduty.Service/EventFirstSeen :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Service/FeatureName :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Service/Action :cognitect.aws.guardduty/Action)

(s/def :cognitect.aws.guardduty.Service/UserFeedback :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.Service/AdditionalInfo
  :cognitect.aws.guardduty/ServiceAdditionalInfo)

(s/def :cognitect.aws.guardduty.Service/Count :cognitect.aws.guardduty/Integer)

(s/def :cognitect.aws.guardduty.Service/Detection :cognitect.aws.guardduty/Detection)

(s/def :cognitect.aws.guardduty.Service/DetectorId :cognitect.aws.guardduty/DetectorId)

(s/def :cognitect.aws.guardduty.Service/ResourceRole :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Service/ServiceName :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.TagResourceRequest/ResourceArn
  :cognitect.aws.guardduty/GuardDutyArn)

(s/def :cognitect.aws.guardduty.TagResourceRequest/Tags :cognitect.aws.guardduty/TagMap)

(s/def
  :cognitect.aws.guardduty.ListCoverageResponse/Resources
  :cognitect.aws.guardduty/CoverageResources)

(s/def :cognitect.aws.guardduty.ListCoverageResponse/NextToken :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.UpdatePublishingDestinationRequest/DetectorId
  :cognitect.aws.guardduty/DetectorId)

(s/def
  :cognitect.aws.guardduty.UpdatePublishingDestinationRequest/DestinationId
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.UpdatePublishingDestinationRequest/DestinationProperties
  :cognitect.aws.guardduty/DestinationProperties)

(s/def :cognitect.aws.guardduty.DescribeMalwareScansResponse/Scans :cognitect.aws.guardduty/Scans)

(s/def
  :cognitect.aws.guardduty.DescribeMalwareScansResponse/NextToken
  :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.ActorProcess/Name :cognitect.aws.guardduty/ProcessName)

(s/def :cognitect.aws.guardduty.ActorProcess/Path :cognitect.aws.guardduty/ProcessPath)

(s/def :cognitect.aws.guardduty.ActorProcess/Sha256 :cognitect.aws.guardduty/ProcessSha256)

(s/def :cognitect.aws.guardduty.VolumeDetail/VolumeArn :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.VolumeDetail/VolumeType :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.VolumeDetail/DeviceName :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.VolumeDetail/VolumeSizeInGB :cognitect.aws.guardduty/Integer)

(s/def :cognitect.aws.guardduty.VolumeDetail/EncryptionType :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.VolumeDetail/SnapshotArn :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.VolumeDetail/KmsKeyArn :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.StartMalwareScanResponse/ScanId
  :cognitect.aws.guardduty/NonEmptyString)

(s/def
  :cognitect.aws.guardduty.HighestSeverityThreatDetails/Severity
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.HighestSeverityThreatDetails/ThreatName
  :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.HighestSeverityThreatDetails/Count :cognitect.aws.guardduty/Integer)

(s/def :cognitect.aws.guardduty.UsageCriteria/AccountIds :cognitect.aws.guardduty/AccountIds)

(s/def :cognitect.aws.guardduty.UsageCriteria/DataSources :cognitect.aws.guardduty/DataSourceList)

(s/def :cognitect.aws.guardduty.UsageCriteria/Resources :cognitect.aws.guardduty/ResourceList)

(s/def :cognitect.aws.guardduty.UsageCriteria/Features :cognitect.aws.guardduty/UsageFeatureList)

(s/def
  :cognitect.aws.guardduty.DestinationProperties/DestinationArn
  :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.DestinationProperties/KmsKeyArn :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.CreateTrustedEntitySetResponse/TrustedEntitySetId
  :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Ec2Instance/AvailabilityZone :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Ec2Instance/InstanceType :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Ec2Instance/OutpostArn :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.Ec2Instance/Ec2NetworkInterfaceUids
  :cognitect.aws.guardduty/Ec2NetworkInterfaceUids)

(s/def :cognitect.aws.guardduty.Ec2Instance/ProductCodes :cognitect.aws.guardduty/ProductCodes)

(s/def :cognitect.aws.guardduty.Ec2Instance/Platform :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Ec2Instance/ImageDescription :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.Ec2Instance/IamInstanceProfile
  :cognitect.aws.guardduty/IamInstanceProfile)

(s/def :cognitect.aws.guardduty.Ec2Instance/InstanceState :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.AccountDetail/AccountId :cognitect.aws.guardduty/AccountId)

(s/def :cognitect.aws.guardduty.AccountDetail/Email :cognitect.aws.guardduty/Email)

(s/def
  :cognitect.aws.guardduty.GetRemainingFreeTrialDaysResponse/Accounts
  :cognitect.aws.guardduty/AccountFreeTrialInfos)

(s/def
  :cognitect.aws.guardduty.GetRemainingFreeTrialDaysResponse/UnprocessedAccounts
  :cognitect.aws.guardduty/UnprocessedAccounts)

(s/def
  :cognitect.aws.guardduty.CreateThreatIntelSetResponse/ThreatIntelSetId
  :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.LocalPortDetails/Port :cognitect.aws.guardduty/Integer)

(s/def :cognitect.aws.guardduty.LocalPortDetails/PortName :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.FilterCriteria/FilterCriterion
  :cognitect.aws.guardduty/FilterCriterionList)

(s/def
  :cognitect.aws.guardduty.DisassociateMembersResponse/UnprocessedAccounts
  :cognitect.aws.guardduty/UnprocessedAccounts)

(s/def
  :cognitect.aws.guardduty.OrganizationFeatureConfigurationResult/Name
  :cognitect.aws.guardduty/OrgFeature)

(s/def
  :cognitect.aws.guardduty.OrganizationFeatureConfigurationResult/AutoEnable
  :cognitect.aws.guardduty/OrgFeatureStatus)

(s/def
  :cognitect.aws.guardduty.OrganizationFeatureConfigurationResult/AdditionalConfiguration
  :cognitect.aws.guardduty/OrganizationAdditionalConfigurationResults)

(s/def
  :cognitect.aws.guardduty.DeclineInvitationsResponse/UnprocessedAccounts
  :cognitect.aws.guardduty/UnprocessedAccounts)

(s/def :cognitect.aws.guardduty.FargateDetails/Issues :cognitect.aws.guardduty/Issues)

(s/def
  :cognitect.aws.guardduty.FargateDetails/ManagementType
  :cognitect.aws.guardduty/ManagementType)

(s/def
  :cognitect.aws.guardduty.GetMalwareProtectionPlanResponse/Arn
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.GetMalwareProtectionPlanResponse/Role
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.GetMalwareProtectionPlanResponse/ProtectedResource
  :cognitect.aws.guardduty/CreateProtectedResource)

(s/def
  :cognitect.aws.guardduty.GetMalwareProtectionPlanResponse/Actions
  :cognitect.aws.guardduty/MalwareProtectionPlanActions)

(s/def
  :cognitect.aws.guardduty.GetMalwareProtectionPlanResponse/CreatedAt
  :cognitect.aws.guardduty/Timestamp)

(s/def
  :cognitect.aws.guardduty.GetMalwareProtectionPlanResponse/Status
  :cognitect.aws.guardduty/MalwareProtectionPlanStatus)

(s/def
  :cognitect.aws.guardduty.GetMalwareProtectionPlanResponse/StatusReasons
  :cognitect.aws.guardduty/MalwareProtectionPlanStatusReasonsList)

(s/def
  :cognitect.aws.guardduty.GetMalwareProtectionPlanResponse/Tags
  :cognitect.aws.guardduty/TagMap)

(s/def
  :cognitect.aws.guardduty.GetMasterAccountRequest/DetectorId
  :cognitect.aws.guardduty/DetectorId)

(s/def :cognitect.aws.guardduty.Tag/Key :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Tag/Value :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.ResourceStatistics/AccountId :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.ResourceStatistics/LastGeneratedAt
  :cognitect.aws.guardduty/Timestamp)

(s/def :cognitect.aws.guardduty.ResourceStatistics/ResourceId :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.ResourceStatistics/ResourceType :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.ResourceStatistics/TotalFindings :cognitect.aws.guardduty/Integer)

(s/def :cognitect.aws.guardduty.AwsApiCallAction/ErrorCode :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.AwsApiCallAction/Api :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.AwsApiCallAction/CallerType :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.AwsApiCallAction/AffectedResources
  :cognitect.aws.guardduty/AffectedResources)

(s/def
  :cognitect.aws.guardduty.AwsApiCallAction/RemoteAccountDetails
  :cognitect.aws.guardduty/RemoteAccountDetails)

(s/def
  :cognitect.aws.guardduty.AwsApiCallAction/RemoteIpDetails
  :cognitect.aws.guardduty/RemoteIpDetails)

(s/def
  :cognitect.aws.guardduty.AwsApiCallAction/DomainDetails
  :cognitect.aws.guardduty/DomainDetails)

(s/def :cognitect.aws.guardduty.AwsApiCallAction/UserAgent :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.AwsApiCallAction/ServiceName :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.GetThreatEntitySetResponse/CreatedAt
  :cognitect.aws.guardduty/Timestamp)

(s/def
  :cognitect.aws.guardduty.GetThreatEntitySetResponse/UpdatedAt
  :cognitect.aws.guardduty/Timestamp)

(s/def :cognitect.aws.guardduty.GetThreatEntitySetResponse/Tags :cognitect.aws.guardduty/TagMap)

(s/def
  :cognitect.aws.guardduty.GetThreatEntitySetResponse/ErrorDetails
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.GetThreatEntitySetResponse/Format
  :cognitect.aws.guardduty/ThreatEntitySetFormat)

(s/def
  :cognitect.aws.guardduty.GetThreatEntitySetResponse/ExpectedBucketOwner
  :cognitect.aws.guardduty/ExpectedBucketOwner)

(s/def :cognitect.aws.guardduty.GetThreatEntitySetResponse/Name :cognitect.aws.guardduty/Name)

(s/def
  :cognitect.aws.guardduty.GetThreatEntitySetResponse/Location
  :cognitect.aws.guardduty/Location)

(s/def
  :cognitect.aws.guardduty.GetThreatEntitySetResponse/Status
  :cognitect.aws.guardduty/ThreatEntitySetStatus)

(s/def :cognitect.aws.guardduty.UsageTopAccountResult/AccountId :cognitect.aws.guardduty/AccountId)

(s/def :cognitect.aws.guardduty.UsageTopAccountResult/Total :cognitect.aws.guardduty/Total)

(s/def :cognitect.aws.guardduty.ScanResultDetails/ScanResult :cognitect.aws.guardduty/ScanResult)

(s/def
  :cognitect.aws.guardduty.ScanEc2InstanceWithFindings/EbsVolumes
  :cognitect.aws.guardduty/Boolean)

(s/def
  :cognitect.aws.guardduty.Evidence/ThreatIntelligenceDetails
  :cognitect.aws.guardduty/ThreatIntelligenceDetails)

(s/def :cognitect.aws.guardduty.UsageAccountResult/AccountId :cognitect.aws.guardduty/AccountId)

(s/def :cognitect.aws.guardduty.UsageAccountResult/Total :cognitect.aws.guardduty/Total)

(s/def
  :cognitect.aws.guardduty.DeleteInvitationsRequest/AccountIds
  :cognitect.aws.guardduty/AccountIds)

(s/def :cognitect.aws.guardduty.ListMembersResponse/Members :cognitect.aws.guardduty/Members)

(s/def :cognitect.aws.guardduty.ListMembersResponse/NextToken :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.ServiceAdditionalInfo/Value :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.ServiceAdditionalInfo/Type :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.GetFindingsResponse/Findings :cognitect.aws.guardduty/Findings)

(s/def
  :cognitect.aws.guardduty.DeleteTrustedEntitySetRequest/DetectorId
  :cognitect.aws.guardduty/DetectorId)

(s/def
  :cognitect.aws.guardduty.DeleteTrustedEntitySetRequest/TrustedEntitySetId
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.DescribeMalwareScansRequest/DetectorId
  :cognitect.aws.guardduty/DetectorId)

(s/def
  :cognitect.aws.guardduty.DescribeMalwareScansRequest/NextToken
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.DescribeMalwareScansRequest/MaxResults
  :cognitect.aws.guardduty/IntegerValueWithMax)

(s/def
  :cognitect.aws.guardduty.DescribeMalwareScansRequest/FilterCriteria
  :cognitect.aws.guardduty/FilterCriteria)

(s/def
  :cognitect.aws.guardduty.DescribeMalwareScansRequest/SortCriteria
  :cognitect.aws.guardduty/SortCriteria)

(s/def
  :cognitect.aws.guardduty.OrganizationMalwareProtectionConfigurationResult/ScanEc2InstanceWithFindings
  :cognitect.aws.guardduty/OrganizationScanEc2InstanceWithFindingsResult)

(s/def
  :cognitect.aws.guardduty.GetTrustedEntitySetResponse/CreatedAt
  :cognitect.aws.guardduty/Timestamp)

(s/def
  :cognitect.aws.guardduty.GetTrustedEntitySetResponse/UpdatedAt
  :cognitect.aws.guardduty/Timestamp)

(s/def :cognitect.aws.guardduty.GetTrustedEntitySetResponse/Tags :cognitect.aws.guardduty/TagMap)

(s/def
  :cognitect.aws.guardduty.GetTrustedEntitySetResponse/ErrorDetails
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.GetTrustedEntitySetResponse/Format
  :cognitect.aws.guardduty/TrustedEntitySetFormat)

(s/def
  :cognitect.aws.guardduty.GetTrustedEntitySetResponse/ExpectedBucketOwner
  :cognitect.aws.guardduty/ExpectedBucketOwner)

(s/def :cognitect.aws.guardduty.GetTrustedEntitySetResponse/Name :cognitect.aws.guardduty/Name)

(s/def
  :cognitect.aws.guardduty.GetTrustedEntitySetResponse/Location
  :cognitect.aws.guardduty/Location)

(s/def
  :cognitect.aws.guardduty.GetTrustedEntitySetResponse/Status
  :cognitect.aws.guardduty/TrustedEntitySetStatus)

(s/def
  :cognitect.aws.guardduty.InviteMembersResponse/UnprocessedAccounts
  :cognitect.aws.guardduty/UnprocessedAccounts)

(s/def :cognitect.aws.guardduty.KubernetesRoleBindingDetails/Kind :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.KubernetesRoleBindingDetails/Name :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.KubernetesRoleBindingDetails/Uid :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.KubernetesRoleBindingDetails/RoleRefName
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.KubernetesRoleBindingDetails/RoleRefKind
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.KubernetesAuditLogsConfigurationResult/Status
  :cognitect.aws.guardduty/DataSourceStatus)

(s/def
  :cognitect.aws.guardduty.FlowLogsConfigurationResult/Status
  :cognitect.aws.guardduty/DataSourceStatus)

(s/def :cognitect.aws.guardduty.Scan/ScanStartTime :cognitect.aws.guardduty/Timestamp)

(s/def :cognitect.aws.guardduty.Scan/ScanResultDetails :cognitect.aws.guardduty/ScanResultDetails)

(s/def :cognitect.aws.guardduty.Scan/ScanStatus :cognitect.aws.guardduty/ScanStatus)

(s/def :cognitect.aws.guardduty.Scan/AccountId :cognitect.aws.guardduty/AccountId)

(s/def :cognitect.aws.guardduty.Scan/AttachedVolumes :cognitect.aws.guardduty/VolumeDetails)

(s/def :cognitect.aws.guardduty.Scan/TriggerDetails :cognitect.aws.guardduty/TriggerDetails)

(s/def :cognitect.aws.guardduty.Scan/ScanType :cognitect.aws.guardduty/ScanType)

(s/def :cognitect.aws.guardduty.Scan/ResourceDetails :cognitect.aws.guardduty/ResourceDetails)

(s/def :cognitect.aws.guardduty.Scan/TotalBytes :cognitect.aws.guardduty/PositiveLong)

(s/def :cognitect.aws.guardduty.Scan/ScanId :cognitect.aws.guardduty/NonEmptyString)

(s/def :cognitect.aws.guardduty.Scan/ScanEndTime :cognitect.aws.guardduty/Timestamp)

(s/def :cognitect.aws.guardduty.Scan/FileCount :cognitect.aws.guardduty/PositiveLong)

(s/def :cognitect.aws.guardduty.Scan/AdminDetectorId :cognitect.aws.guardduty/DetectorId)

(s/def :cognitect.aws.guardduty.Scan/FailureReason :cognitect.aws.guardduty/NonEmptyString)

(s/def :cognitect.aws.guardduty.Scan/DetectorId :cognitect.aws.guardduty/DetectorId)

(s/def
  :cognitect.aws.guardduty.BucketPolicy/AllowsPublicReadAccess
  :cognitect.aws.guardduty/Boolean)

(s/def
  :cognitect.aws.guardduty.BucketPolicy/AllowsPublicWriteAccess
  :cognitect.aws.guardduty/Boolean)

(s/def :cognitect.aws.guardduty.BlockPublicAccess/IgnorePublicAcls :cognitect.aws.guardduty/Boolean)

(s/def
  :cognitect.aws.guardduty.BlockPublicAccess/RestrictPublicBuckets
  :cognitect.aws.guardduty/Boolean)

(s/def :cognitect.aws.guardduty.BlockPublicAccess/BlockPublicAcls :cognitect.aws.guardduty/Boolean)

(s/def
  :cognitect.aws.guardduty.BlockPublicAccess/BlockPublicPolicy
  :cognitect.aws.guardduty/Boolean)

(s/def :cognitect.aws.guardduty.VpcConfig/SubnetIds :cognitect.aws.guardduty/SubnetIds)

(s/def :cognitect.aws.guardduty.VpcConfig/VpcId :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.VpcConfig/SecurityGroups :cognitect.aws.guardduty/SecurityGroups)

(s/def
  :cognitect.aws.guardduty.MalwareProtectionConfigurationResult/ScanEc2InstanceWithFindings
  :cognitect.aws.guardduty/ScanEc2InstanceWithFindingsResult)

(s/def
  :cognitect.aws.guardduty.MalwareProtectionConfigurationResult/ServiceRole
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.MemberDataSourceConfiguration/AccountId
  :cognitect.aws.guardduty/AccountId)

(s/def
  :cognitect.aws.guardduty.MemberDataSourceConfiguration/DataSources
  :cognitect.aws.guardduty/DataSourceConfigurationsResult)

(s/def
  :cognitect.aws.guardduty.MemberDataSourceConfiguration/Features
  :cognitect.aws.guardduty/MemberFeaturesConfigurationsResults)

(s/def
  :cognitect.aws.guardduty.MalwareProtectionDataSourceFreeTrial/ScanEc2InstanceWithFindings
  :cognitect.aws.guardduty/DataSourceFreeTrial)

(s/def
  :cognitect.aws.guardduty.GetFindingsStatisticsRequest/DetectorId
  :cognitect.aws.guardduty/DetectorId)

(s/def
  :cognitect.aws.guardduty.GetFindingsStatisticsRequest/FindingStatisticTypes
  :cognitect.aws.guardduty/FindingStatisticTypes)

(s/def
  :cognitect.aws.guardduty.GetFindingsStatisticsRequest/FindingCriteria
  :cognitect.aws.guardduty/FindingCriteria)

(s/def
  :cognitect.aws.guardduty.GetFindingsStatisticsRequest/GroupBy
  :cognitect.aws.guardduty/GroupByType)

(s/def
  :cognitect.aws.guardduty.GetFindingsStatisticsRequest/OrderBy
  :cognitect.aws.guardduty/OrderBy)

(s/def
  :cognitect.aws.guardduty.GetFindingsStatisticsRequest/MaxResults
  :cognitect.aws.guardduty/MaxResults100)

(s/def
  :cognitect.aws.guardduty.EnableOrganizationAdminAccountRequest/AdminAccountId
  :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.MalwareScanDetails/Threats :cognitect.aws.guardduty/Threats)

(s/def
  :cognitect.aws.guardduty.UpdateDetectorRequest/DetectorId
  :cognitect.aws.guardduty/DetectorId)

(s/def :cognitect.aws.guardduty.UpdateDetectorRequest/Enable :cognitect.aws.guardduty/Boolean)

(s/def
  :cognitect.aws.guardduty.UpdateDetectorRequest/FindingPublishingFrequency
  :cognitect.aws.guardduty/FindingPublishingFrequency)

(s/def
  :cognitect.aws.guardduty.UpdateDetectorRequest/DataSources
  :cognitect.aws.guardduty/DataSourceConfigurations)

(s/def
  :cognitect.aws.guardduty.UpdateDetectorRequest/Features
  :cognitect.aws.guardduty/DetectorFeatureConfigurations)

(s/def
  :cognitect.aws.guardduty.CoverageFilterCriteria/FilterCriterion
  :cognitect.aws.guardduty/CoverageFilterCriterionList)

(s/def :cognitect.aws.guardduty.Invitation/AccountId :cognitect.aws.guardduty/AccountId)

(s/def :cognitect.aws.guardduty.Invitation/InvitationId :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Invitation/RelationshipStatus :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Invitation/InvitedAt :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.OrganizationFeatureStatisticsAdditionalConfiguration/Name
  :cognitect.aws.guardduty/OrgFeatureAdditionalConfiguration)

(s/def
  :cognitect.aws.guardduty.OrganizationFeatureStatisticsAdditionalConfiguration/EnabledAccountsCount
  :cognitect.aws.guardduty/Integer)

(s/def :cognitect.aws.guardduty.Container/ContainerRuntime :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Container/Id :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Container/Name :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Container/Image :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Container/ImagePrefix :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Container/VolumeMounts :cognitect.aws.guardduty/VolumeMounts)

(s/def :cognitect.aws.guardduty.Container/SecurityContext :cognitect.aws.guardduty/SecurityContext)

(s/def
  :cognitect.aws.guardduty.UpdateMalwareProtectionPlanRequest/MalwareProtectionPlanId
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.UpdateMalwareProtectionPlanRequest/Role
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.UpdateMalwareProtectionPlanRequest/Actions
  :cognitect.aws.guardduty/MalwareProtectionPlanActions)

(s/def
  :cognitect.aws.guardduty.UpdateMalwareProtectionPlanRequest/ProtectedResource
  :cognitect.aws.guardduty/UpdateProtectedResource)

(s/def :cognitect.aws.guardduty.AccessDeniedException/Message :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.AccessDeniedException/Type :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.DescribePublishingDestinationRequest/DetectorId
  :cognitect.aws.guardduty/DetectorId)

(s/def
  :cognitect.aws.guardduty.DescribePublishingDestinationRequest/DestinationId
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.OrganizationAdditionalConfiguration/Name
  :cognitect.aws.guardduty/OrgFeatureAdditionalConfiguration)

(s/def
  :cognitect.aws.guardduty.OrganizationAdditionalConfiguration/AutoEnable
  :cognitect.aws.guardduty/OrgFeatureStatus)

(s/def :cognitect.aws.guardduty.UnprocessedAccount/AccountId :cognitect.aws.guardduty/AccountId)

(s/def :cognitect.aws.guardduty.UnprocessedAccount/Result :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Member/AccountId :cognitect.aws.guardduty/AccountId)

(s/def :cognitect.aws.guardduty.Member/DetectorId :cognitect.aws.guardduty/DetectorId)

(s/def :cognitect.aws.guardduty.Member/MasterId :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Member/Email :cognitect.aws.guardduty/Email)

(s/def :cognitect.aws.guardduty.Member/RelationshipStatus :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Member/InvitedAt :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Member/UpdatedAt :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Member/AdministratorId :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Observations/Text :cognitect.aws.guardduty/ObservationTexts)

(s/def :cognitect.aws.guardduty.RuntimeDetails/Process :cognitect.aws.guardduty/ProcessDetails)

(s/def :cognitect.aws.guardduty.RuntimeDetails/Context :cognitect.aws.guardduty/RuntimeContext)

(s/def :cognitect.aws.guardduty.GetMasterAccountResponse/Master :cognitect.aws.guardduty/Master)

(s/def
  :cognitect.aws.guardduty.GetCoverageStatisticsRequest/DetectorId
  :cognitect.aws.guardduty/DetectorId)

(s/def
  :cognitect.aws.guardduty.GetCoverageStatisticsRequest/FilterCriteria
  :cognitect.aws.guardduty/CoverageFilterCriteria)

(s/def
  :cognitect.aws.guardduty.GetCoverageStatisticsRequest/StatisticsType
  :cognitect.aws.guardduty/CoverageStatisticsTypeList)

(s/def :cognitect.aws.guardduty.ListMembersRequest/DetectorId :cognitect.aws.guardduty/DetectorId)

(s/def :cognitect.aws.guardduty.ListMembersRequest/MaxResults :cognitect.aws.guardduty/MaxResults)

(s/def :cognitect.aws.guardduty.ListMembersRequest/NextToken :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.ListMembersRequest/OnlyAssociated :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.ProductCode/Code :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.ProductCode/ProductType :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.DescribeOrganizationConfigurationResponse/AutoEnable
  :cognitect.aws.guardduty/Boolean)

(s/def
  :cognitect.aws.guardduty.DescribeOrganizationConfigurationResponse/MemberAccountLimitReached
  :cognitect.aws.guardduty/Boolean)

(s/def
  :cognitect.aws.guardduty.DescribeOrganizationConfigurationResponse/DataSources
  :cognitect.aws.guardduty/OrganizationDataSourceConfigurationsResult)

(s/def
  :cognitect.aws.guardduty.DescribeOrganizationConfigurationResponse/Features
  :cognitect.aws.guardduty/OrganizationFeaturesConfigurationsResults)

(s/def
  :cognitect.aws.guardduty.DescribeOrganizationConfigurationResponse/NextToken
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.DescribeOrganizationConfigurationResponse/AutoEnableOrganizationMembers
  :cognitect.aws.guardduty/AutoEnableMembers)

(s/def
  :cognitect.aws.guardduty.ListFiltersResponse/FilterNames
  :cognitect.aws.guardduty/FilterNames)

(s/def :cognitect.aws.guardduty.ListFiltersResponse/NextToken :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.UpdateThreatEntitySetRequest/DetectorId
  :cognitect.aws.guardduty/DetectorId)

(s/def
  :cognitect.aws.guardduty.UpdateThreatEntitySetRequest/ThreatEntitySetId
  :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.UpdateThreatEntitySetRequest/Name :cognitect.aws.guardduty/Name)

(s/def
  :cognitect.aws.guardduty.UpdateThreatEntitySetRequest/Location
  :cognitect.aws.guardduty/Location)

(s/def
  :cognitect.aws.guardduty.UpdateThreatEntitySetRequest/ExpectedBucketOwner
  :cognitect.aws.guardduty/ExpectedBucketOwner)

(s/def
  :cognitect.aws.guardduty.UpdateThreatEntitySetRequest/Activate
  :cognitect.aws.guardduty/Boolean)

(s/def
  :cognitect.aws.guardduty.DescribeOrganizationConfigurationRequest/DetectorId
  :cognitect.aws.guardduty/DetectorId)

(s/def
  :cognitect.aws.guardduty.DescribeOrganizationConfigurationRequest/MaxResults
  :cognitect.aws.guardduty/MaxResults)

(s/def
  :cognitect.aws.guardduty.DescribeOrganizationConfigurationRequest/NextToken
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.UpdateOrganizationConfigurationRequest/DetectorId
  :cognitect.aws.guardduty/DetectorId)

(s/def
  :cognitect.aws.guardduty.UpdateOrganizationConfigurationRequest/AutoEnable
  :cognitect.aws.guardduty/Boolean)

(s/def
  :cognitect.aws.guardduty.UpdateOrganizationConfigurationRequest/DataSources
  :cognitect.aws.guardduty/OrganizationDataSourceConfigurations)

(s/def
  :cognitect.aws.guardduty.UpdateOrganizationConfigurationRequest/Features
  :cognitect.aws.guardduty/OrganizationFeaturesConfigurations)

(s/def
  :cognitect.aws.guardduty.UpdateOrganizationConfigurationRequest/AutoEnableOrganizationMembers
  :cognitect.aws.guardduty/AutoEnableMembers)

(s/def
  :cognitect.aws.guardduty.DetectorAdditionalConfiguration/Name
  :cognitect.aws.guardduty/FeatureAdditionalConfiguration)

(s/def
  :cognitect.aws.guardduty.DetectorAdditionalConfiguration/Status
  :cognitect.aws.guardduty/FeatureStatus)

(s/def
  :cognitect.aws.guardduty.GetOrganizationStatisticsResponse/OrganizationDetails
  :cognitect.aws.guardduty/OrganizationDetails)

(s/def :cognitect.aws.guardduty.AnomalyObject/ProfileType :cognitect.aws.guardduty/ProfileType)

(s/def
  :cognitect.aws.guardduty.AnomalyObject/ProfileSubtype
  :cognitect.aws.guardduty/ProfileSubtype)

(s/def :cognitect.aws.guardduty.AnomalyObject/Observations :cognitect.aws.guardduty/Observations)

(s/def
  :cognitect.aws.guardduty.ListDetectorsResponse/DetectorIds
  :cognitect.aws.guardduty/DetectorIds)

(s/def :cognitect.aws.guardduty.ListDetectorsResponse/NextToken :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.ListCoverageRequest/DetectorId :cognitect.aws.guardduty/DetectorId)

(s/def :cognitect.aws.guardduty.ListCoverageRequest/NextToken :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.ListCoverageRequest/MaxResults :cognitect.aws.guardduty/MaxResults)

(s/def
  :cognitect.aws.guardduty.ListCoverageRequest/FilterCriteria
  :cognitect.aws.guardduty/CoverageFilterCriteria)

(s/def
  :cognitect.aws.guardduty.ListCoverageRequest/SortCriteria
  :cognitect.aws.guardduty/CoverageSortCriteria)

(s/def :cognitect.aws.guardduty.SecurityGroup/GroupId :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.SecurityGroup/GroupName :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.DetectorFeatureConfigurationResult/Name
  :cognitect.aws.guardduty/DetectorFeatureResult)

(s/def
  :cognitect.aws.guardduty.DetectorFeatureConfigurationResult/Status
  :cognitect.aws.guardduty/FeatureStatus)

(s/def
  :cognitect.aws.guardduty.DetectorFeatureConfigurationResult/UpdatedAt
  :cognitect.aws.guardduty/Timestamp)

(s/def
  :cognitect.aws.guardduty.DetectorFeatureConfigurationResult/AdditionalConfiguration
  :cognitect.aws.guardduty/DetectorAdditionalConfigurationResults)

(s/def :cognitect.aws.guardduty.Owner/Id :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.NetworkConnection/Direction
  :cognitect.aws.guardduty/NetworkDirection)

(s/def :cognitect.aws.guardduty.Destination/DestinationId :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.Destination/DestinationType
  :cognitect.aws.guardduty/DestinationType)

(s/def :cognitect.aws.guardduty.Destination/Status :cognitect.aws.guardduty/PublishingStatus)

(s/def
  :cognitect.aws.guardduty.DeclineInvitationsRequest/AccountIds
  :cognitect.aws.guardduty/AccountIds)

(s/def :cognitect.aws.guardduty.Sequence/Uid :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Sequence/Description :cognitect.aws.guardduty/SequenceDescription)

(s/def :cognitect.aws.guardduty.Sequence/Actors :cognitect.aws.guardduty/Actors)

(s/def :cognitect.aws.guardduty.Sequence/Resources :cognitect.aws.guardduty/Resources)

(s/def :cognitect.aws.guardduty.Sequence/Endpoints :cognitect.aws.guardduty/NetworkEndpoints)

(s/def :cognitect.aws.guardduty.Sequence/Signals :cognitect.aws.guardduty/Signals)

(s/def :cognitect.aws.guardduty.Sequence/SequenceIndicators :cognitect.aws.guardduty/Indicators)

(s/def
  :cognitect.aws.guardduty.Sequence/AdditionalSequenceTypes
  :cognitect.aws.guardduty/AdditionalSequenceTypes)

(s/def :cognitect.aws.guardduty.LoginAttribute/User :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.LoginAttribute/Application :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.LoginAttribute/FailedLoginAttempts :cognitect.aws.guardduty/Integer)

(s/def
  :cognitect.aws.guardduty.LoginAttribute/SuccessfulLoginAttempts
  :cognitect.aws.guardduty/Integer)

(s/def :cognitect.aws.guardduty.HostPath/Path :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.ResourceV2/Service :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.ResourceV2/ResourceType
  :cognitect.aws.guardduty/FindingResourceType)

(s/def :cognitect.aws.guardduty.ResourceV2/CloudPartition :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.ResourceV2/AccountId :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.ResourceV2/Data :cognitect.aws.guardduty/ResourceData)

(s/def :cognitect.aws.guardduty.ResourceV2/Tags :cognitect.aws.guardduty/Tags)

(s/def :cognitect.aws.guardduty.ResourceV2/Region :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.ResourceV2/Uid :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.ResourceV2/Name :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.S3LogsConfigurationResult/Status
  :cognitect.aws.guardduty/DataSourceStatus)

(s/def :cognitect.aws.guardduty.EcsTaskDetails/Group :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.EcsTaskDetails/DefinitionArn :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.EcsTaskDetails/Tags :cognitect.aws.guardduty/Tags)

(s/def :cognitect.aws.guardduty.EcsTaskDetails/Containers :cognitect.aws.guardduty/Containers)

(s/def :cognitect.aws.guardduty.EcsTaskDetails/StartedBy :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.EcsTaskDetails/TaskCreatedAt :cognitect.aws.guardduty/Timestamp)

(s/def :cognitect.aws.guardduty.EcsTaskDetails/StartedAt :cognitect.aws.guardduty/Timestamp)

(s/def :cognitect.aws.guardduty.EcsTaskDetails/Arn :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.EcsTaskDetails/Volumes :cognitect.aws.guardduty/Volumes)

(s/def :cognitect.aws.guardduty.EcsTaskDetails/LaunchType :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.EcsTaskDetails/Version :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.OrganizationKubernetesConfigurationResult/AuditLogs
  :cognitect.aws.guardduty/OrganizationKubernetesAuditLogsConfigurationResult)

(s/def
  :cognitect.aws.guardduty.GetRemainingFreeTrialDaysRequest/DetectorId
  :cognitect.aws.guardduty/DetectorId)

(s/def
  :cognitect.aws.guardduty.GetRemainingFreeTrialDaysRequest/AccountIds
  :cognitect.aws.guardduty/AccountIds)

(s/def :cognitect.aws.guardduty.GetMembersRequest/DetectorId :cognitect.aws.guardduty/DetectorId)

(s/def :cognitect.aws.guardduty.GetMembersRequest/AccountIds :cognitect.aws.guardduty/AccountIds)

(s/def
  :cognitect.aws.guardduty.GetMalwareScanSettingsResponse/ScanResourceCriteria
  :cognitect.aws.guardduty/ScanResourceCriteria)

(s/def
  :cognitect.aws.guardduty.GetMalwareScanSettingsResponse/EbsSnapshotPreservation
  :cognitect.aws.guardduty/EbsSnapshotPreservation)

(s/def :cognitect.aws.guardduty.EbsVolumeScanDetails/ScanId :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.EbsVolumeScanDetails/ScanStartedAt
  :cognitect.aws.guardduty/Timestamp)

(s/def
  :cognitect.aws.guardduty.EbsVolumeScanDetails/ScanCompletedAt
  :cognitect.aws.guardduty/Timestamp)

(s/def
  :cognitect.aws.guardduty.EbsVolumeScanDetails/TriggerFindingId
  :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.EbsVolumeScanDetails/Sources :cognitect.aws.guardduty/Sources)

(s/def
  :cognitect.aws.guardduty.EbsVolumeScanDetails/ScanDetections
  :cognitect.aws.guardduty/ScanDetections)

(s/def :cognitect.aws.guardduty.EbsVolumeScanDetails/ScanType :cognitect.aws.guardduty/ScanType)

(s/def :cognitect.aws.guardduty.DeleteIPSetRequest/DetectorId :cognitect.aws.guardduty/DetectorId)

(s/def :cognitect.aws.guardduty.DeleteIPSetRequest/IpSetId :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.NetworkInterface/PrivateIpAddress
  :cognitect.aws.guardduty/SensitiveString)

(s/def
  :cognitect.aws.guardduty.NetworkInterface/PrivateIpAddresses
  :cognitect.aws.guardduty/PrivateIpAddresses)

(s/def :cognitect.aws.guardduty.NetworkInterface/PublicIp :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.NetworkInterface/SubnetId :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.NetworkInterface/SecurityGroups
  :cognitect.aws.guardduty/SecurityGroups)

(s/def :cognitect.aws.guardduty.NetworkInterface/PrivateDnsName :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.NetworkInterface/PublicDnsName :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.NetworkInterface/VpcId :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.NetworkInterface/Ipv6Addresses
  :cognitect.aws.guardduty/Ipv6Addresses)

(s/def :cognitect.aws.guardduty.NetworkInterface/NetworkInterfaceId :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.UsageStatistics/SumByAccount
  :cognitect.aws.guardduty/UsageAccountResultList)

(s/def
  :cognitect.aws.guardduty.UsageStatistics/TopAccountsByFeature
  :cognitect.aws.guardduty/UsageTopAccountsResultList)

(s/def
  :cognitect.aws.guardduty.UsageStatistics/SumByDataSource
  :cognitect.aws.guardduty/UsageDataSourceResultList)

(s/def
  :cognitect.aws.guardduty.UsageStatistics/SumByResource
  :cognitect.aws.guardduty/UsageResourceResultList)

(s/def
  :cognitect.aws.guardduty.UsageStatistics/TopResources
  :cognitect.aws.guardduty/UsageResourceResultList)

(s/def
  :cognitect.aws.guardduty.UsageStatistics/SumByFeature
  :cognitect.aws.guardduty/UsageFeatureResultList)

(s/def
  :cognitect.aws.guardduty.DetectorAdditionalConfigurationResult/Name
  :cognitect.aws.guardduty/FeatureAdditionalConfiguration)

(s/def
  :cognitect.aws.guardduty.DetectorAdditionalConfigurationResult/Status
  :cognitect.aws.guardduty/FeatureStatus)

(s/def
  :cognitect.aws.guardduty.DetectorAdditionalConfigurationResult/UpdatedAt
  :cognitect.aws.guardduty/Timestamp)

(s/def :cognitect.aws.guardduty.UsageResourceResult/Resource :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.UsageResourceResult/Total :cognitect.aws.guardduty/Total)

(s/def :cognitect.aws.guardduty.UpdateFilterResponse/Name :cognitect.aws.guardduty/FilterName)

(s/def :cognitect.aws.guardduty.Anomaly/Profiles :cognitect.aws.guardduty/AnomalyProfiles)

(s/def :cognitect.aws.guardduty.Anomaly/Unusual :cognitect.aws.guardduty/AnomalyUnusual)

(s/def :cognitect.aws.guardduty.City/CityName :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.DeleteThreatIntelSetRequest/DetectorId
  :cognitect.aws.guardduty/DetectorId)

(s/def
  :cognitect.aws.guardduty.DeleteThreatIntelSetRequest/ThreatIntelSetId
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.StartMonitoringMembersRequest/DetectorId
  :cognitect.aws.guardduty/DetectorId)

(s/def
  :cognitect.aws.guardduty.StartMonitoringMembersRequest/AccountIds
  :cognitect.aws.guardduty/AccountIds)

(s/def
  :cognitect.aws.guardduty.StopMonitoringMembersRequest/DetectorId
  :cognitect.aws.guardduty/DetectorId)

(s/def
  :cognitect.aws.guardduty.StopMonitoringMembersRequest/AccountIds
  :cognitect.aws.guardduty/AccountIds)

(s/def
  :cognitect.aws.guardduty.DisassociateMembersRequest/DetectorId
  :cognitect.aws.guardduty/DetectorId)

(s/def
  :cognitect.aws.guardduty.DisassociateMembersRequest/AccountIds
  :cognitect.aws.guardduty/AccountIds)

(s/def
  :cognitect.aws.guardduty.GetMalwareProtectionPlanRequest/MalwareProtectionPlanId
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.CreatePublishingDestinationRequest/DetectorId
  :cognitect.aws.guardduty/DetectorId)

(s/def
  :cognitect.aws.guardduty.CreatePublishingDestinationRequest/DestinationType
  :cognitect.aws.guardduty/DestinationType)

(s/def
  :cognitect.aws.guardduty.CreatePublishingDestinationRequest/DestinationProperties
  :cognitect.aws.guardduty/DestinationProperties)

(s/def
  :cognitect.aws.guardduty.CreatePublishingDestinationRequest/ClientToken
  :cognitect.aws.guardduty/ClientToken)

(s/def :cognitect.aws.guardduty.LocalIpDetails/IpAddressV4 :cognitect.aws.guardduty/SensitiveString)

(s/def :cognitect.aws.guardduty.LocalIpDetails/IpAddressV6 :cognitect.aws.guardduty/SensitiveString)

(s/def
  :cognitect.aws.guardduty.AccessControlList/AllowsPublicReadAccess
  :cognitect.aws.guardduty/Boolean)

(s/def
  :cognitect.aws.guardduty.AccessControlList/AllowsPublicWriteAccess
  :cognitect.aws.guardduty/Boolean)

(s/def
  :cognitect.aws.guardduty.TriggerDetails/GuardDutyFindingId
  :cognitect.aws.guardduty/NonEmptyString)

(s/def :cognitect.aws.guardduty.TriggerDetails/Description :cognitect.aws.guardduty/NonEmptyString)

(s/def :cognitect.aws.guardduty.ScanFilePath/FilePath :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.ScanFilePath/VolumeArn :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.ScanFilePath/Hash :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.ScanFilePath/FileName :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.CreateMembersRequest/DetectorId :cognitect.aws.guardduty/DetectorId)

(s/def
  :cognitect.aws.guardduty.CreateMembersRequest/AccountDetails
  :cognitect.aws.guardduty/AccountDetails)

(s/def
  :cognitect.aws.guardduty.Ec2NetworkInterface/Ipv6Addresses
  :cognitect.aws.guardduty/Ipv6Addresses)

(s/def
  :cognitect.aws.guardduty.Ec2NetworkInterface/PrivateIpAddresses
  :cognitect.aws.guardduty/PrivateIpAddresses)

(s/def :cognitect.aws.guardduty.Ec2NetworkInterface/PublicIp :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.Ec2NetworkInterface/SecurityGroups
  :cognitect.aws.guardduty/SecurityGroups)

(s/def :cognitect.aws.guardduty.Ec2NetworkInterface/SubNetId :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Ec2NetworkInterface/VpcId :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.GetThreatEntitySetRequest/DetectorId
  :cognitect.aws.guardduty/DetectorId)

(s/def
  :cognitect.aws.guardduty.GetThreatEntitySetRequest/ThreatEntitySetId
  :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.ScanConditionPair/Key :cognitect.aws.guardduty/TagKey)

(s/def :cognitect.aws.guardduty.ScanConditionPair/Value :cognitect.aws.guardduty/TagValue)

(s/def
  :cognitect.aws.guardduty.GetInvitationsCountResponse/InvitationsCount
  :cognitect.aws.guardduty/Integer)

(s/def
  :cognitect.aws.guardduty.PublicAccessConfiguration/PublicAclAccess
  :cognitect.aws.guardduty/PublicAccessStatus)

(s/def
  :cognitect.aws.guardduty.PublicAccessConfiguration/PublicPolicyAccess
  :cognitect.aws.guardduty/PublicAccessStatus)

(s/def
  :cognitect.aws.guardduty.PublicAccessConfiguration/PublicAclIgnoreBehavior
  :cognitect.aws.guardduty/PublicAclIgnoreBehavior)

(s/def
  :cognitect.aws.guardduty.PublicAccessConfiguration/PublicBucketRestrictBehavior
  :cognitect.aws.guardduty/PublicBucketRestrictBehavior)

(s/def :cognitect.aws.guardduty.CreateS3BucketResource/BucketName :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.CreateS3BucketResource/ObjectPrefixes
  :cognitect.aws.guardduty/MalwareProtectionPlanObjectPrefixesList)

(s/def :cognitect.aws.guardduty.AgentDetails/Version :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.UsageDataSourceResult/DataSource
  :cognitect.aws.guardduty/DataSource)

(s/def :cognitect.aws.guardduty.UsageDataSourceResult/Total :cognitect.aws.guardduty/Total)

(s/def
  :cognitect.aws.guardduty.GetFindingsStatisticsResponse/FindingStatistics
  :cognitect.aws.guardduty/FindingStatistics)

(s/def
  :cognitect.aws.guardduty.GetFindingsStatisticsResponse/NextToken
  :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.CreateFilterRequest/DetectorId :cognitect.aws.guardduty/DetectorId)

(s/def :cognitect.aws.guardduty.CreateFilterRequest/Name :cognitect.aws.guardduty/FilterName)

(s/def
  :cognitect.aws.guardduty.CreateFilterRequest/Description
  :cognitect.aws.guardduty/FilterDescription)

(s/def :cognitect.aws.guardduty.CreateFilterRequest/Action :cognitect.aws.guardduty/FilterAction)

(s/def :cognitect.aws.guardduty.CreateFilterRequest/Rank :cognitect.aws.guardduty/FilterRank)

(s/def
  :cognitect.aws.guardduty.CreateFilterRequest/FindingCriteria
  :cognitect.aws.guardduty/FindingCriteria)

(s/def
  :cognitect.aws.guardduty.CreateFilterRequest/ClientToken
  :cognitect.aws.guardduty/ClientToken)

(s/def :cognitect.aws.guardduty.CreateFilterRequest/Tags :cognitect.aws.guardduty/TagMap)

(s/def
  :cognitect.aws.guardduty.OrganizationMalwareProtectionConfiguration/ScanEc2InstanceWithFindings
  :cognitect.aws.guardduty/OrganizationScanEc2InstanceWithFindings)

(s/def
  :cognitect.aws.guardduty.CreateSampleFindingsRequest/DetectorId
  :cognitect.aws.guardduty/DetectorId)

(s/def
  :cognitect.aws.guardduty.CreateSampleFindingsRequest/FindingTypes
  :cognitect.aws.guardduty/FindingTypes)

(s/def
  :cognitect.aws.guardduty.ListInvitationsResponse/Invitations
  :cognitect.aws.guardduty/Invitations)

(s/def :cognitect.aws.guardduty.ListInvitationsResponse/NextToken :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.GetThreatIntelSetRequest/DetectorId
  :cognitect.aws.guardduty/DetectorId)

(s/def
  :cognitect.aws.guardduty.GetThreatIntelSetRequest/ThreatIntelSetId
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.ListOrganizationAdminAccountsResponse/AdminAccounts
  :cognitect.aws.guardduty/AdminAccounts)

(s/def
  :cognitect.aws.guardduty.ListOrganizationAdminAccountsResponse/NextToken
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.RdsLoginAttemptAction/RemoteIpDetails
  :cognitect.aws.guardduty/RemoteIpDetails)

(s/def
  :cognitect.aws.guardduty.RdsLoginAttemptAction/LoginAttributes
  :cognitect.aws.guardduty/LoginAttributes)

(s/def
  :cognitect.aws.guardduty.CoverageEksClusterDetails/ClusterName
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.CoverageEksClusterDetails/CoveredNodes
  :cognitect.aws.guardduty/Long)

(s/def
  :cognitect.aws.guardduty.CoverageEksClusterDetails/CompatibleNodes
  :cognitect.aws.guardduty/Long)

(s/def
  :cognitect.aws.guardduty.CoverageEksClusterDetails/AddonDetails
  :cognitect.aws.guardduty/AddonDetails)

(s/def
  :cognitect.aws.guardduty.CoverageEksClusterDetails/ManagementType
  :cognitect.aws.guardduty/ManagementType)

(s/def :cognitect.aws.guardduty.GetFilterRequest/DetectorId :cognitect.aws.guardduty/DetectorId)

(s/def :cognitect.aws.guardduty.GetFilterRequest/FilterName :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.ScanCondition/MapEquals :cognitect.aws.guardduty/MapEquals)

(s/def :cognitect.aws.guardduty.S3Bucket/CreatedAt :cognitect.aws.guardduty/Timestamp)

(s/def
  :cognitect.aws.guardduty.S3Bucket/AccountPublicAccess
  :cognitect.aws.guardduty/PublicAccessConfiguration)

(s/def :cognitect.aws.guardduty.S3Bucket/S3ObjectUids :cognitect.aws.guardduty/S3ObjectUids)

(s/def :cognitect.aws.guardduty.S3Bucket/EffectivePermission :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.S3Bucket/BucketPublicAccess
  :cognitect.aws.guardduty/PublicAccessConfiguration)

(s/def :cognitect.aws.guardduty.S3Bucket/EncryptionType :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.S3Bucket/OwnerId :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.S3Bucket/PublicReadAccess
  :cognitect.aws.guardduty/PublicAccessStatus)

(s/def :cognitect.aws.guardduty.S3Bucket/EncryptionKeyArn :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.S3Bucket/PublicWriteAccess
  :cognitect.aws.guardduty/PublicAccessStatus)

(s/def
  :cognitect.aws.guardduty.OrganizationS3LogsConfiguration/AutoEnable
  :cognitect.aws.guardduty/Boolean)

(s/def
  :cognitect.aws.guardduty.CreateProtectedResource/S3Bucket
  :cognitect.aws.guardduty/CreateS3BucketResource)

(s/def :cognitect.aws.guardduty.OrganizationEbsVolumes/AutoEnable :cognitect.aws.guardduty/Boolean)

(s/def
  :cognitect.aws.guardduty.RdsDbInstanceDetails/DbInstanceIdentifier
  :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.RdsDbInstanceDetails/Engine :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.RdsDbInstanceDetails/EngineVersion :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.RdsDbInstanceDetails/DbClusterIdentifier
  :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.RdsDbInstanceDetails/DbInstanceArn :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.RdsDbInstanceDetails/Tags :cognitect.aws.guardduty/Tags)

(s/def
  :cognitect.aws.guardduty.CreateTrustedEntitySetRequest/DetectorId
  :cognitect.aws.guardduty/DetectorId)

(s/def :cognitect.aws.guardduty.CreateTrustedEntitySetRequest/Name :cognitect.aws.guardduty/Name)

(s/def
  :cognitect.aws.guardduty.CreateTrustedEntitySetRequest/Format
  :cognitect.aws.guardduty/TrustedEntitySetFormat)

(s/def
  :cognitect.aws.guardduty.CreateTrustedEntitySetRequest/Location
  :cognitect.aws.guardduty/Location)

(s/def
  :cognitect.aws.guardduty.CreateTrustedEntitySetRequest/ExpectedBucketOwner
  :cognitect.aws.guardduty/ExpectedBucketOwner)

(s/def
  :cognitect.aws.guardduty.CreateTrustedEntitySetRequest/Activate
  :cognitect.aws.guardduty/Boolean)

(s/def
  :cognitect.aws.guardduty.CreateTrustedEntitySetRequest/ClientToken
  :cognitect.aws.guardduty/ClientToken)

(s/def :cognitect.aws.guardduty.CreateTrustedEntitySetRequest/Tags :cognitect.aws.guardduty/TagMap)

(s/def
  :cognitect.aws.guardduty.MalwareProtectionPlanActions/Tagging
  :cognitect.aws.guardduty/MalwareProtectionPlanTaggingAction)

(s/def :cognitect.aws.guardduty.GetThreatIntelSetResponse/Name :cognitect.aws.guardduty/Name)

(s/def
  :cognitect.aws.guardduty.GetThreatIntelSetResponse/Format
  :cognitect.aws.guardduty/ThreatIntelSetFormat)

(s/def
  :cognitect.aws.guardduty.GetThreatIntelSetResponse/Location
  :cognitect.aws.guardduty/Location)

(s/def
  :cognitect.aws.guardduty.GetThreatIntelSetResponse/Status
  :cognitect.aws.guardduty/ThreatIntelSetStatus)

(s/def :cognitect.aws.guardduty.GetThreatIntelSetResponse/Tags :cognitect.aws.guardduty/TagMap)

(s/def
  :cognitect.aws.guardduty.GetThreatIntelSetResponse/ExpectedBucketOwner
  :cognitect.aws.guardduty/AccountId)

(s/def :cognitect.aws.guardduty.CreateFilterResponse/Name :cognitect.aws.guardduty/FilterName)

(s/def
  :cognitect.aws.guardduty.MemberFeaturesConfigurationResult/Name
  :cognitect.aws.guardduty/OrgFeature)

(s/def
  :cognitect.aws.guardduty.MemberFeaturesConfigurationResult/Status
  :cognitect.aws.guardduty/FeatureStatus)

(s/def
  :cognitect.aws.guardduty.MemberFeaturesConfigurationResult/UpdatedAt
  :cognitect.aws.guardduty/Timestamp)

(s/def
  :cognitect.aws.guardduty.MemberFeaturesConfigurationResult/AdditionalConfiguration
  :cognitect.aws.guardduty/MemberAdditionalConfigurationResults)

(s/def
  :cognitect.aws.guardduty.OrganizationScanEc2InstanceWithFindings/EbsVolumes
  :cognitect.aws.guardduty/OrganizationEbsVolumes)

(s/def :cognitect.aws.guardduty.ResourceDetails/InstanceArn :cognitect.aws.guardduty/InstanceArn)

(s/def
  :cognitect.aws.guardduty.CoverageFilterCriterion/CriterionKey
  :cognitect.aws.guardduty/CoverageFilterCriterionKey)

(s/def
  :cognitect.aws.guardduty.CoverageFilterCriterion/FilterCondition
  :cognitect.aws.guardduty/CoverageFilterCondition)

(s/def
  :cognitect.aws.guardduty.KubernetesConfiguration/AuditLogs
  :cognitect.aws.guardduty/KubernetesAuditLogsConfiguration)

(s/def :cognitect.aws.guardduty.Administrator/AccountId :cognitect.aws.guardduty/AccountId)

(s/def :cognitect.aws.guardduty.Administrator/InvitationId :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Administrator/RelationshipStatus :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Administrator/InvitedAt :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.ListTrustedEntitySetsRequest/DetectorId
  :cognitect.aws.guardduty/DetectorId)

(s/def
  :cognitect.aws.guardduty.ListTrustedEntitySetsRequest/MaxResults
  :cognitect.aws.guardduty/MaxResults)

(s/def
  :cognitect.aws.guardduty.ListTrustedEntitySetsRequest/NextToken
  :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.EksCluster/Arn :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.EksCluster/CreatedAt :cognitect.aws.guardduty/Timestamp)

(s/def :cognitect.aws.guardduty.EksCluster/Status :cognitect.aws.guardduty/ClusterStatus)

(s/def :cognitect.aws.guardduty.EksCluster/VpcId :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.EksCluster/Ec2InstanceUids :cognitect.aws.guardduty/Ec2InstanceUids)

(s/def
  :cognitect.aws.guardduty.DNSLogsConfigurationResult/Status
  :cognitect.aws.guardduty/DataSourceStatus)

(s/def
  :cognitect.aws.guardduty.MalwareProtectionConfiguration/ScanEc2InstanceWithFindings
  :cognitect.aws.guardduty/ScanEc2InstanceWithFindings)

(s/def :cognitect.aws.guardduty.DeleteMembersRequest/DetectorId :cognitect.aws.guardduty/DetectorId)

(s/def :cognitect.aws.guardduty.DeleteMembersRequest/AccountIds :cognitect.aws.guardduty/AccountIds)

(s/def
  :cognitect.aws.guardduty.ListPublishingDestinationsResponse/Destinations
  :cognitect.aws.guardduty/Destinations)

(s/def
  :cognitect.aws.guardduty.ListPublishingDestinationsResponse/NextToken
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.DeletePublishingDestinationRequest/DetectorId
  :cognitect.aws.guardduty/DetectorId)

(s/def
  :cognitect.aws.guardduty.DeletePublishingDestinationRequest/DestinationId
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.DataSourceConfigurationsResult/CloudTrail
  :cognitect.aws.guardduty/CloudTrailConfigurationResult)

(s/def
  :cognitect.aws.guardduty.DataSourceConfigurationsResult/DNSLogs
  :cognitect.aws.guardduty/DNSLogsConfigurationResult)

(s/def
  :cognitect.aws.guardduty.DataSourceConfigurationsResult/FlowLogs
  :cognitect.aws.guardduty/FlowLogsConfigurationResult)

(s/def
  :cognitect.aws.guardduty.DataSourceConfigurationsResult/S3Logs
  :cognitect.aws.guardduty/S3LogsConfigurationResult)

(s/def
  :cognitect.aws.guardduty.DataSourceConfigurationsResult/Kubernetes
  :cognitect.aws.guardduty/KubernetesConfigurationResult)

(s/def
  :cognitect.aws.guardduty.DataSourceConfigurationsResult/MalwareProtection
  :cognitect.aws.guardduty/MalwareProtectionConfigurationResult)

(s/def
  :cognitect.aws.guardduty.CoverageSortCriteria/AttributeName
  :cognitect.aws.guardduty/CoverageSortKey)

(s/def :cognitect.aws.guardduty.CoverageSortCriteria/OrderBy :cognitect.aws.guardduty/OrderBy)

(s/def
  :cognitect.aws.guardduty.OrganizationAdditionalConfigurationResult/Name
  :cognitect.aws.guardduty/OrgFeatureAdditionalConfiguration)

(s/def
  :cognitect.aws.guardduty.OrganizationAdditionalConfigurationResult/AutoEnable
  :cognitect.aws.guardduty/OrgFeatureStatus)

(s/def
  :cognitect.aws.guardduty.AccountLevelPermissions/BlockPublicAccess
  :cognitect.aws.guardduty/BlockPublicAccess)

(s/def
  :cognitect.aws.guardduty.RdsLimitlessDbDetails/DbShardGroupIdentifier
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.RdsLimitlessDbDetails/DbShardGroupResourceId
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.RdsLimitlessDbDetails/DbShardGroupArn
  :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.RdsLimitlessDbDetails/Engine :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.RdsLimitlessDbDetails/EngineVersion :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.RdsLimitlessDbDetails/DbClusterIdentifier
  :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.RdsLimitlessDbDetails/Tags :cognitect.aws.guardduty/Tags)

(s/def
  :cognitect.aws.guardduty.KubernetesDataSourceFreeTrial/AuditLogs
  :cognitect.aws.guardduty/DataSourceFreeTrial)

(s/def :cognitect.aws.guardduty.ItemPath/NestedItemPath :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.ItemPath/Hash :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.ListFindingsResponse/FindingIds :cognitect.aws.guardduty/FindingIds)

(s/def :cognitect.aws.guardduty.ListFindingsResponse/NextToken :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.RemoteAccountDetails/AccountId :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.RemoteAccountDetails/Affiliated :cognitect.aws.guardduty/Boolean)

(s/def
  :cognitect.aws.guardduty.ListTrustedEntitySetsResponse/TrustedEntitySetIds
  :cognitect.aws.guardduty/TrustedEntitySetIds)

(s/def
  :cognitect.aws.guardduty.ListTrustedEntitySetsResponse/NextToken
  :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.GetDetectorRequest/DetectorId :cognitect.aws.guardduty/DetectorId)

(s/def :cognitect.aws.guardduty.CreateIPSetRequest/DetectorId :cognitect.aws.guardduty/DetectorId)

(s/def :cognitect.aws.guardduty.CreateIPSetRequest/Name :cognitect.aws.guardduty/Name)

(s/def :cognitect.aws.guardduty.CreateIPSetRequest/Format :cognitect.aws.guardduty/IpSetFormat)

(s/def :cognitect.aws.guardduty.CreateIPSetRequest/Location :cognitect.aws.guardduty/Location)

(s/def :cognitect.aws.guardduty.CreateIPSetRequest/Activate :cognitect.aws.guardduty/Boolean)

(s/def :cognitect.aws.guardduty.CreateIPSetRequest/ClientToken :cognitect.aws.guardduty/ClientToken)

(s/def :cognitect.aws.guardduty.CreateIPSetRequest/Tags :cognitect.aws.guardduty/TagMap)

(s/def
  :cognitect.aws.guardduty.CreateIPSetRequest/ExpectedBucketOwner
  :cognitect.aws.guardduty/AccountId)

(s/def
  :cognitect.aws.guardduty.CreateMembersResponse/UnprocessedAccounts
  :cognitect.aws.guardduty/UnprocessedAccounts)

(s/def
  :cognitect.aws.guardduty.KubernetesWorkloadDetails/ServiceAccountName
  :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.KubernetesWorkloadDetails/HostIPC :cognitect.aws.guardduty/Boolean)

(s/def
  :cognitect.aws.guardduty.KubernetesWorkloadDetails/Containers
  :cognitect.aws.guardduty/Containers)

(s/def
  :cognitect.aws.guardduty.KubernetesWorkloadDetails/HostNetwork
  :cognitect.aws.guardduty/Boolean)

(s/def :cognitect.aws.guardduty.KubernetesWorkloadDetails/Uid :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.KubernetesWorkloadDetails/Namespace :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.KubernetesWorkloadDetails/Volumes :cognitect.aws.guardduty/Volumes)

(s/def :cognitect.aws.guardduty.KubernetesWorkloadDetails/Name :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.KubernetesWorkloadDetails/Type :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.KubernetesWorkloadDetails/HostPID :cognitect.aws.guardduty/Boolean)

(s/def :cognitect.aws.guardduty.InstanceDetails/AvailabilityZone :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.InstanceDetails/LaunchTime :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.InstanceDetails/ImageId :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.InstanceDetails/Tags :cognitect.aws.guardduty/Tags)

(s/def :cognitect.aws.guardduty.InstanceDetails/InstanceId :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.InstanceDetails/InstanceType :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.InstanceDetails/OutpostArn :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.InstanceDetails/ProductCodes :cognitect.aws.guardduty/ProductCodes)

(s/def :cognitect.aws.guardduty.InstanceDetails/Platform :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.InstanceDetails/ImageDescription :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.InstanceDetails/IamInstanceProfile
  :cognitect.aws.guardduty/IamInstanceProfile)

(s/def :cognitect.aws.guardduty.InstanceDetails/InstanceState :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.InstanceDetails/NetworkInterfaces
  :cognitect.aws.guardduty/NetworkInterfaces)

(s/def :cognitect.aws.guardduty.ScanResourceCriteria/Include :cognitect.aws.guardduty/ScanCriterion)

(s/def :cognitect.aws.guardduty.ScanResourceCriteria/Exclude :cognitect.aws.guardduty/ScanCriterion)

(s/def :cognitect.aws.guardduty.Volume/Name :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Volume/HostPath :cognitect.aws.guardduty/HostPath)

(s/def :cognitect.aws.guardduty.ListDetectorsRequest/MaxResults :cognitect.aws.guardduty/MaxResults)

(s/def :cognitect.aws.guardduty.ListDetectorsRequest/NextToken :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.StartMonitoringMembersResponse/UnprocessedAccounts
  :cognitect.aws.guardduty/UnprocessedAccounts)

(s/def :cognitect.aws.guardduty.ThreatsDetectedItemCount/Files :cognitect.aws.guardduty/Integer)

(s/def
  :cognitect.aws.guardduty.KubernetesPermissionCheckedDetails/Verb
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.KubernetesPermissionCheckedDetails/Resource
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.KubernetesPermissionCheckedDetails/Namespace
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.KubernetesPermissionCheckedDetails/Allowed
  :cognitect.aws.guardduty/Boolean)

(s/def
  :cognitect.aws.guardduty.StartMalwareScanRequest/ResourceArn
  :cognitect.aws.guardduty/ResourceArn)

(s/def :cognitect.aws.guardduty.GetFilterResponse/Name :cognitect.aws.guardduty/FilterName)

(s/def
  :cognitect.aws.guardduty.GetFilterResponse/Description
  :cognitect.aws.guardduty/FilterDescription)

(s/def :cognitect.aws.guardduty.GetFilterResponse/Action :cognitect.aws.guardduty/FilterAction)

(s/def :cognitect.aws.guardduty.GetFilterResponse/Rank :cognitect.aws.guardduty/FilterRank)

(s/def
  :cognitect.aws.guardduty.GetFilterResponse/FindingCriteria
  :cognitect.aws.guardduty/FindingCriteria)

(s/def :cognitect.aws.guardduty.GetFilterResponse/Tags :cognitect.aws.guardduty/TagMap)

(s/def
  :cognitect.aws.guardduty.UpdateProtectedResource/S3Bucket
  :cognitect.aws.guardduty/UpdateS3BucketResource)

(s/def
  :cognitect.aws.guardduty.DeleteThreatEntitySetRequest/DetectorId
  :cognitect.aws.guardduty/DetectorId)

(s/def
  :cognitect.aws.guardduty.DeleteThreatEntitySetRequest/ThreatEntitySetId
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.ListTagsForResourceRequest/ResourceArn
  :cognitect.aws.guardduty/GuardDutyArn)

(s/def :cognitect.aws.guardduty.ListFiltersRequest/DetectorId :cognitect.aws.guardduty/DetectorId)

(s/def :cognitect.aws.guardduty.ListFiltersRequest/MaxResults :cognitect.aws.guardduty/MaxResults)

(s/def :cognitect.aws.guardduty.ListFiltersRequest/NextToken :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.ThreatIntelligenceDetail/ThreatListName
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.ThreatIntelligenceDetail/ThreatNames
  :cognitect.aws.guardduty/ThreatNames)

(s/def
  :cognitect.aws.guardduty.ThreatIntelligenceDetail/ThreatFileSha256
  :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.S3ObjectDetail/ObjectArn :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.S3ObjectDetail/Key :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.S3ObjectDetail/ETag :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.S3ObjectDetail/Hash :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.S3ObjectDetail/VersionId :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.ScanThreatName/Name :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.ScanThreatName/Severity :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.ScanThreatName/ItemCount :cognitect.aws.guardduty/Integer)

(s/def :cognitect.aws.guardduty.ScanThreatName/FilePaths :cognitect.aws.guardduty/FilePaths)

(s/def
  :cognitect.aws.guardduty.OrganizationFeatureConfiguration/Name
  :cognitect.aws.guardduty/OrgFeature)

(s/def
  :cognitect.aws.guardduty.OrganizationFeatureConfiguration/AutoEnable
  :cognitect.aws.guardduty/OrgFeatureStatus)

(s/def
  :cognitect.aws.guardduty.OrganizationFeatureConfiguration/AdditionalConfiguration
  :cognitect.aws.guardduty/OrganizationAdditionalConfigurations)

(s/def :cognitect.aws.guardduty.GetFindingsRequest/DetectorId :cognitect.aws.guardduty/DetectorId)

(s/def :cognitect.aws.guardduty.GetFindingsRequest/FindingIds :cognitect.aws.guardduty/FindingIds)

(s/def
  :cognitect.aws.guardduty.GetFindingsRequest/SortCriteria
  :cognitect.aws.guardduty/SortCriteria)

(s/def :cognitect.aws.guardduty.GetMembersResponse/Members :cognitect.aws.guardduty/Members)

(s/def
  :cognitect.aws.guardduty.GetMembersResponse/UnprocessedAccounts
  :cognitect.aws.guardduty/UnprocessedAccounts)

(s/def
  :cognitect.aws.guardduty.BucketLevelPermissions/AccessControlList
  :cognitect.aws.guardduty/AccessControlList)

(s/def
  :cognitect.aws.guardduty.BucketLevelPermissions/BucketPolicy
  :cognitect.aws.guardduty/BucketPolicy)

(s/def
  :cognitect.aws.guardduty.BucketLevelPermissions/BlockPublicAccess
  :cognitect.aws.guardduty/BlockPublicAccess)

(s/def
  :cognitect.aws.guardduty.UpdateMemberDetectorsResponse/UnprocessedAccounts
  :cognitect.aws.guardduty/UnprocessedAccounts)

(s/def :cognitect.aws.guardduty.CreateDetectorRequest/Enable :cognitect.aws.guardduty/Boolean)

(s/def
  :cognitect.aws.guardduty.CreateDetectorRequest/ClientToken
  :cognitect.aws.guardduty/ClientToken)

(s/def
  :cognitect.aws.guardduty.CreateDetectorRequest/FindingPublishingFrequency
  :cognitect.aws.guardduty/FindingPublishingFrequency)

(s/def
  :cognitect.aws.guardduty.CreateDetectorRequest/DataSources
  :cognitect.aws.guardduty/DataSourceConfigurations)

(s/def :cognitect.aws.guardduty.CreateDetectorRequest/Tags :cognitect.aws.guardduty/TagMap)

(s/def
  :cognitect.aws.guardduty.CreateDetectorRequest/Features
  :cognitect.aws.guardduty/DetectorFeatureConfigurations)

(s/def :cognitect.aws.guardduty.S3LogsConfiguration/Enable :cognitect.aws.guardduty/Boolean)

(s/def
  :cognitect.aws.guardduty.ListMalwareProtectionPlansResponse/MalwareProtectionPlans
  :cognitect.aws.guardduty/MalwareProtectionPlansSummary)

(s/def
  :cognitect.aws.guardduty.ListMalwareProtectionPlansResponse/NextToken
  :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.UsageFeatureResult/Feature :cognitect.aws.guardduty/UsageFeature)

(s/def :cognitect.aws.guardduty.UsageFeatureResult/Total :cognitect.aws.guardduty/Total)

(s/def :cognitect.aws.guardduty.AddonDetails/AddonVersion :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.AddonDetails/AddonStatus :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.ListInvitationsRequest/MaxResults
  :cognitect.aws.guardduty/MaxResults)

(s/def :cognitect.aws.guardduty.ListInvitationsRequest/NextToken :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.UpdateFilterRequest/DetectorId :cognitect.aws.guardduty/DetectorId)

(s/def :cognitect.aws.guardduty.UpdateFilterRequest/FilterName :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.UpdateFilterRequest/Description
  :cognitect.aws.guardduty/FilterDescription)

(s/def :cognitect.aws.guardduty.UpdateFilterRequest/Action :cognitect.aws.guardduty/FilterAction)

(s/def :cognitect.aws.guardduty.UpdateFilterRequest/Rank :cognitect.aws.guardduty/FilterRank)

(s/def
  :cognitect.aws.guardduty.UpdateFilterRequest/FindingCriteria
  :cognitect.aws.guardduty/FindingCriteria)

(s/def :cognitect.aws.guardduty.User/Name :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.User/Uid :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.User/Type :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.User/CredentialUid :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.User/Account :cognitect.aws.guardduty/Account)

(s/def :cognitect.aws.guardduty.EcsClusterDetails/Name :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.EcsClusterDetails/Arn :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.EcsClusterDetails/Status :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.EcsClusterDetails/ActiveServicesCount
  :cognitect.aws.guardduty/Integer)

(s/def
  :cognitect.aws.guardduty.EcsClusterDetails/RegisteredContainerInstancesCount
  :cognitect.aws.guardduty/Integer)

(s/def
  :cognitect.aws.guardduty.EcsClusterDetails/RunningTasksCount
  :cognitect.aws.guardduty/Integer)

(s/def :cognitect.aws.guardduty.EcsClusterDetails/Tags :cognitect.aws.guardduty/Tags)

(s/def
  :cognitect.aws.guardduty.EcsClusterDetails/TaskDetails
  :cognitect.aws.guardduty/EcsTaskDetails)

(s/def
  :cognitect.aws.guardduty.OrganizationDataSourceConfigurations/S3Logs
  :cognitect.aws.guardduty/OrganizationS3LogsConfiguration)

(s/def
  :cognitect.aws.guardduty.OrganizationDataSourceConfigurations/Kubernetes
  :cognitect.aws.guardduty/OrganizationKubernetesConfiguration)

(s/def
  :cognitect.aws.guardduty.OrganizationDataSourceConfigurations/MalwareProtection
  :cognitect.aws.guardduty/OrganizationMalwareProtectionConfiguration)

(s/def :cognitect.aws.guardduty.FindingCriteria/Criterion :cognitect.aws.guardduty/Criterion)

(s/def
  :cognitect.aws.guardduty.InternalServerErrorException/Message
  :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.InternalServerErrorException/Type :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.ListThreatEntitySetsResponse/ThreatEntitySetIds
  :cognitect.aws.guardduty/ThreatEntitySetIds)

(s/def
  :cognitect.aws.guardduty.ListThreatEntitySetsResponse/NextToken
  :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.RdsDbUserDetails/User :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.RdsDbUserDetails/Application :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.RdsDbUserDetails/Database :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.RdsDbUserDetails/Ssl :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.RdsDbUserDetails/AuthMethod :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.DisassociateFromMasterAccountRequest/DetectorId
  :cognitect.aws.guardduty/DetectorId)

(s/def :cognitect.aws.guardduty.InviteMembersRequest/DetectorId :cognitect.aws.guardduty/DetectorId)

(s/def :cognitect.aws.guardduty.InviteMembersRequest/AccountIds :cognitect.aws.guardduty/AccountIds)

(s/def
  :cognitect.aws.guardduty.InviteMembersRequest/DisableEmailNotification
  :cognitect.aws.guardduty/Boolean)

(s/def :cognitect.aws.guardduty.InviteMembersRequest/Message :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.CreateDetectorResponse/DetectorId
  :cognitect.aws.guardduty/DetectorId)

(s/def
  :cognitect.aws.guardduty.CreateDetectorResponse/UnprocessedDataSources
  :cognitect.aws.guardduty/UnprocessedDataSourcesResult)

(s/def
  :cognitect.aws.guardduty.UpdateS3BucketResource/ObjectPrefixes
  :cognitect.aws.guardduty/MalwareProtectionPlanObjectPrefixesList)

(s/def :cognitect.aws.guardduty.OrganizationDetails/UpdatedAt :cognitect.aws.guardduty/Timestamp)

(s/def
  :cognitect.aws.guardduty.OrganizationDetails/OrganizationStatistics
  :cognitect.aws.guardduty/OrganizationStatistics)

(s/def :cognitect.aws.guardduty.RuntimeContext/RuncBinaryPath :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.RuntimeContext/ModuleName :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.RuntimeContext/CommandLineExample :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.RuntimeContext/ReleaseAgentPath :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.RuntimeContext/MemoryRegions
  :cognitect.aws.guardduty/MemoryRegionsList)

(s/def :cognitect.aws.guardduty.RuntimeContext/ScriptPath :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.RuntimeContext/ShellHistoryFilePath :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.RuntimeContext/ModifyingProcess
  :cognitect.aws.guardduty/ProcessDetails)

(s/def :cognitect.aws.guardduty.RuntimeContext/AddressFamily :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.RuntimeContext/ModuleFilePath :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.RuntimeContext/ToolName :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.RuntimeContext/MountTarget :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.RuntimeContext/ModuleSha256 :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.RuntimeContext/MountSource :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.RuntimeContext/LdPreloadValue :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.RuntimeContext/Flags :cognitect.aws.guardduty/FlagsList)

(s/def :cognitect.aws.guardduty.RuntimeContext/FileSystemType :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.RuntimeContext/ModifiedAt :cognitect.aws.guardduty/Timestamp)

(s/def :cognitect.aws.guardduty.RuntimeContext/LibraryPath :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.RuntimeContext/SocketPath :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.RuntimeContext/TargetProcess
  :cognitect.aws.guardduty/ProcessDetails)

(s/def :cognitect.aws.guardduty.RuntimeContext/ThreatFilePath :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.RuntimeContext/IanaProtocolNumber :cognitect.aws.guardduty/Integer)

(s/def :cognitect.aws.guardduty.RuntimeContext/ToolCategory :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.RuntimeContext/ServiceName :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.CoverageResourceDetails/EksClusterDetails
  :cognitect.aws.guardduty/CoverageEksClusterDetails)

(s/def
  :cognitect.aws.guardduty.CoverageResourceDetails/ResourceType
  :cognitect.aws.guardduty/ResourceType)

(s/def
  :cognitect.aws.guardduty.CoverageResourceDetails/EcsClusterDetails
  :cognitect.aws.guardduty/CoverageEcsClusterDetails)

(s/def
  :cognitect.aws.guardduty.CoverageResourceDetails/Ec2InstanceDetails
  :cognitect.aws.guardduty/CoverageEc2InstanceDetails)

(s/def :cognitect.aws.guardduty.ScannedItemCount/TotalGb :cognitect.aws.guardduty/Integer)

(s/def :cognitect.aws.guardduty.ScannedItemCount/Files :cognitect.aws.guardduty/Integer)

(s/def :cognitect.aws.guardduty.ScannedItemCount/Volumes :cognitect.aws.guardduty/Integer)

(s/def :cognitect.aws.guardduty.Actor/Id :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Actor/User :cognitect.aws.guardduty/User)

(s/def :cognitect.aws.guardduty.Actor/Session :cognitect.aws.guardduty/Session)

(s/def :cognitect.aws.guardduty.Actor/Process :cognitect.aws.guardduty/ActorProcess)

(s/def
  :cognitect.aws.guardduty.ScanEc2InstanceWithFindingsResult/EbsVolumes
  :cognitect.aws.guardduty/EbsVolumesResult)

(s/def
  :cognitect.aws.guardduty.KubernetesAuditLogsConfiguration/Enable
  :cognitect.aws.guardduty/Boolean)

(s/def :cognitect.aws.guardduty.RemotePortDetails/Port :cognitect.aws.guardduty/Integer)

(s/def :cognitect.aws.guardduty.RemotePortDetails/PortName :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Total/Amount :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Total/Unit :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.ContainerFindingResource/Image :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.ContainerFindingResource/ImageUid
  :cognitect.aws.guardduty/ContainerImageUid)

(s/def :cognitect.aws.guardduty.KubernetesApiCallAction/RequestUri :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.KubernetesApiCallAction/Parameters :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.KubernetesApiCallAction/StatusCode :cognitect.aws.guardduty/Integer)

(s/def :cognitect.aws.guardduty.KubernetesApiCallAction/Subresource :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.KubernetesApiCallAction/Namespace :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.KubernetesApiCallAction/RemoteIpDetails
  :cognitect.aws.guardduty/RemoteIpDetails)

(s/def :cognitect.aws.guardduty.KubernetesApiCallAction/Verb :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.KubernetesApiCallAction/Resource :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.KubernetesApiCallAction/ResourceName
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.KubernetesApiCallAction/SourceIps
  :cognitect.aws.guardduty/SourceIps)

(s/def :cognitect.aws.guardduty.KubernetesApiCallAction/UserAgent :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.FreeTrialFeatureConfigurationResult/Name
  :cognitect.aws.guardduty/FreeTrialFeatureResult)

(s/def
  :cognitect.aws.guardduty.FreeTrialFeatureConfigurationResult/FreeTrialDaysRemaining
  :cognitect.aws.guardduty/Integer)

(s/def :cognitect.aws.guardduty.S3Object/ETag :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.S3Object/Key :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.S3Object/VersionId :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Signal/Severity :cognitect.aws.guardduty/Double)

(s/def :cognitect.aws.guardduty.Signal/ActorIds :cognitect.aws.guardduty/ActorIds)

(s/def :cognitect.aws.guardduty.Signal/CreatedAt :cognitect.aws.guardduty/Timestamp)

(s/def :cognitect.aws.guardduty.Signal/LastSeenAt :cognitect.aws.guardduty/Timestamp)

(s/def :cognitect.aws.guardduty.Signal/FirstSeenAt :cognitect.aws.guardduty/Timestamp)

(s/def :cognitect.aws.guardduty.Signal/UpdatedAt :cognitect.aws.guardduty/Timestamp)

(s/def :cognitect.aws.guardduty.Signal/ResourceUids :cognitect.aws.guardduty/ResourceUids)

(s/def :cognitect.aws.guardduty.Signal/SignalIndicators :cognitect.aws.guardduty/Indicators)

(s/def :cognitect.aws.guardduty.Signal/Uid :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Signal/Name :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Signal/Description :cognitect.aws.guardduty/SignalDescription)

(s/def :cognitect.aws.guardduty.Signal/Type :cognitect.aws.guardduty/SignalType)

(s/def :cognitect.aws.guardduty.Signal/Count :cognitect.aws.guardduty/Integer)

(s/def :cognitect.aws.guardduty.Signal/EndpointIds :cognitect.aws.guardduty/EndpointIds)

(s/def
  :cognitect.aws.guardduty.MemberFeaturesConfiguration/Name
  :cognitect.aws.guardduty/OrgFeature)

(s/def
  :cognitect.aws.guardduty.MemberFeaturesConfiguration/Status
  :cognitect.aws.guardduty/FeatureStatus)

(s/def
  :cognitect.aws.guardduty.MemberFeaturesConfiguration/AdditionalConfiguration
  :cognitect.aws.guardduty/MemberAdditionalConfigurations)

(s/def
  :cognitect.aws.guardduty.PublicAccess/PermissionConfiguration
  :cognitect.aws.guardduty/PermissionConfiguration)

(s/def :cognitect.aws.guardduty.PublicAccess/EffectivePermission :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Account/Uid :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Account/Name :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.GetUsageStatisticsResponse/UsageStatistics
  :cognitect.aws.guardduty/UsageStatistics)

(s/def
  :cognitect.aws.guardduty.GetUsageStatisticsResponse/NextToken
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.UnprocessedDataSourcesResult/MalwareProtection
  :cognitect.aws.guardduty/MalwareProtectionConfigurationResult)

(s/def
  :cognitect.aws.guardduty.OrganizationKubernetesAuditLogsConfiguration/AutoEnable
  :cognitect.aws.guardduty/Boolean)

(s/def
  :cognitect.aws.guardduty.OrganizationKubernetesAuditLogsConfigurationResult/AutoEnable
  :cognitect.aws.guardduty/Boolean)

(s/def :cognitect.aws.guardduty.AccountStatistics/AccountId :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.AccountStatistics/LastGeneratedAt
  :cognitect.aws.guardduty/Timestamp)

(s/def :cognitect.aws.guardduty.AccountStatistics/TotalFindings :cognitect.aws.guardduty/Integer)

(s/def
  :cognitect.aws.guardduty.CreatePublishingDestinationResponse/DestinationId
  :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.S3BucketDetail/CreatedAt :cognitect.aws.guardduty/Timestamp)

(s/def
  :cognitect.aws.guardduty.S3BucketDetail/S3ObjectDetails
  :cognitect.aws.guardduty/S3ObjectDetails)

(s/def :cognitect.aws.guardduty.S3BucketDetail/Owner :cognitect.aws.guardduty/Owner)

(s/def :cognitect.aws.guardduty.S3BucketDetail/Tags :cognitect.aws.guardduty/Tags)

(s/def :cognitect.aws.guardduty.S3BucketDetail/PublicAccess :cognitect.aws.guardduty/PublicAccess)

(s/def
  :cognitect.aws.guardduty.S3BucketDetail/DefaultServerSideEncryption
  :cognitect.aws.guardduty/DefaultServerSideEncryption)

(s/def :cognitect.aws.guardduty.S3BucketDetail/Arn :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.S3BucketDetail/Name :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.S3BucketDetail/Type :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.GetTrustedEntitySetRequest/DetectorId
  :cognitect.aws.guardduty/DetectorId)

(s/def
  :cognitect.aws.guardduty.GetTrustedEntitySetRequest/TrustedEntitySetId
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.UnarchiveFindingsRequest/DetectorId
  :cognitect.aws.guardduty/DetectorId)

(s/def
  :cognitect.aws.guardduty.UnarchiveFindingsRequest/FindingIds
  :cognitect.aws.guardduty/FindingIds)

(s/def
  :cognitect.aws.guardduty.DefaultServerSideEncryption/EncryptionType
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.DefaultServerSideEncryption/KmsMasterKeyArn
  :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.EbsVolumesResult/Status :cognitect.aws.guardduty/DataSourceStatus)

(s/def :cognitect.aws.guardduty.EbsVolumesResult/Reason :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.OrganizationS3LogsConfigurationResult/AutoEnable
  :cognitect.aws.guardduty/Boolean)

(s/def :cognitect.aws.guardduty.Action/AwsApiCallAction :cognitect.aws.guardduty/AwsApiCallAction)

(s/def
  :cognitect.aws.guardduty.Action/KubernetesRoleBindingDetails
  :cognitect.aws.guardduty/KubernetesRoleBindingDetails)

(s/def :cognitect.aws.guardduty.Action/ActionType :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.Action/RdsLoginAttemptAction
  :cognitect.aws.guardduty/RdsLoginAttemptAction)

(s/def
  :cognitect.aws.guardduty.Action/KubernetesPermissionCheckedDetails
  :cognitect.aws.guardduty/KubernetesPermissionCheckedDetails)

(s/def
  :cognitect.aws.guardduty.Action/KubernetesApiCallAction
  :cognitect.aws.guardduty/KubernetesApiCallAction)

(s/def
  :cognitect.aws.guardduty.Action/NetworkConnectionAction
  :cognitect.aws.guardduty/NetworkConnectionAction)

(s/def
  :cognitect.aws.guardduty.Action/KubernetesRoleDetails
  :cognitect.aws.guardduty/KubernetesRoleDetails)

(s/def :cognitect.aws.guardduty.Action/PortProbeAction :cognitect.aws.guardduty/PortProbeAction)

(s/def :cognitect.aws.guardduty.Action/DnsRequestAction :cognitect.aws.guardduty/DnsRequestAction)

(s/def :cognitect.aws.guardduty.ThreatDetectedByName/ItemCount :cognitect.aws.guardduty/Integer)

(s/def
  :cognitect.aws.guardduty.ThreatDetectedByName/UniqueThreatNameCount
  :cognitect.aws.guardduty/Integer)

(s/def :cognitect.aws.guardduty.ThreatDetectedByName/Shortened :cognitect.aws.guardduty/Boolean)

(s/def
  :cognitect.aws.guardduty.ThreatDetectedByName/ThreatNames
  :cognitect.aws.guardduty/ScanThreatNames)

(s/def
  :cognitect.aws.guardduty.GetCoverageStatisticsResponse/CoverageStatistics
  :cognitect.aws.guardduty/CoverageStatistics)

(s/def :cognitect.aws.guardduty.BadRequestException/Message :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.BadRequestException/Type :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.OrganizationStatistics/TotalAccountsCount
  :cognitect.aws.guardduty/Integer)

(s/def
  :cognitect.aws.guardduty.OrganizationStatistics/MemberAccountsCount
  :cognitect.aws.guardduty/Integer)

(s/def
  :cognitect.aws.guardduty.OrganizationStatistics/ActiveAccountsCount
  :cognitect.aws.guardduty/Integer)

(s/def
  :cognitect.aws.guardduty.OrganizationStatistics/EnabledAccountsCount
  :cognitect.aws.guardduty/Integer)

(s/def
  :cognitect.aws.guardduty.OrganizationStatistics/CountByFeature
  :cognitect.aws.guardduty/OrganizationFeatureStatisticsResults)

(s/def
  :cognitect.aws.guardduty.PermissionConfiguration/BucketLevelPermissions
  :cognitect.aws.guardduty/BucketLevelPermissions)

(s/def
  :cognitect.aws.guardduty.PermissionConfiguration/AccountLevelPermissions
  :cognitect.aws.guardduty/AccountLevelPermissions)

(s/def :cognitect.aws.guardduty.NetworkConnectionAction/Blocked :cognitect.aws.guardduty/Boolean)

(s/def
  :cognitect.aws.guardduty.NetworkConnectionAction/ConnectionDirection
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.NetworkConnectionAction/LocalPortDetails
  :cognitect.aws.guardduty/LocalPortDetails)

(s/def :cognitect.aws.guardduty.NetworkConnectionAction/Protocol :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.NetworkConnectionAction/LocalIpDetails
  :cognitect.aws.guardduty/LocalIpDetails)

(s/def
  :cognitect.aws.guardduty.NetworkConnectionAction/LocalNetworkInterface
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.NetworkConnectionAction/RemoteIpDetails
  :cognitect.aws.guardduty/RemoteIpDetails)

(s/def
  :cognitect.aws.guardduty.NetworkConnectionAction/RemotePortDetails
  :cognitect.aws.guardduty/RemotePortDetails)

(s/def :cognitect.aws.guardduty.NetworkGeoLocation/City :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.NetworkGeoLocation/Country :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.NetworkGeoLocation/Latitude :cognitect.aws.guardduty/Double)

(s/def :cognitect.aws.guardduty.NetworkGeoLocation/Longitude :cognitect.aws.guardduty/Double)

(s/def :cognitect.aws.guardduty.RemoteIpDetails/City :cognitect.aws.guardduty/City)

(s/def :cognitect.aws.guardduty.RemoteIpDetails/Country :cognitect.aws.guardduty/Country)

(s/def :cognitect.aws.guardduty.RemoteIpDetails/GeoLocation :cognitect.aws.guardduty/GeoLocation)

(s/def
  :cognitect.aws.guardduty.RemoteIpDetails/IpAddressV4
  :cognitect.aws.guardduty/SensitiveString)

(s/def
  :cognitect.aws.guardduty.RemoteIpDetails/IpAddressV6
  :cognitect.aws.guardduty/SensitiveString)

(s/def :cognitect.aws.guardduty.RemoteIpDetails/Organization :cognitect.aws.guardduty/Organization)

(s/def
  :cognitect.aws.guardduty.DetectorFeatureConfiguration/Name
  :cognitect.aws.guardduty/DetectorFeature)

(s/def
  :cognitect.aws.guardduty.DetectorFeatureConfiguration/Status
  :cognitect.aws.guardduty/FeatureStatus)

(s/def
  :cognitect.aws.guardduty.DetectorFeatureConfiguration/AdditionalConfiguration
  :cognitect.aws.guardduty/DetectorAdditionalConfigurations)

(s/def :cognitect.aws.guardduty.AutonomousSystem/Name :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.AutonomousSystem/Number :cognitect.aws.guardduty/Integer)

(s/def
  :cognitect.aws.guardduty.OrganizationEbsVolumesResult/AutoEnable
  :cognitect.aws.guardduty/Boolean)

(s/def :cognitect.aws.guardduty.GeoLocation/Lat :cognitect.aws.guardduty/Double)

(s/def :cognitect.aws.guardduty.GeoLocation/Lon :cognitect.aws.guardduty/Double)

(s/def
  :cognitect.aws.guardduty.ScanDetections/ScannedItemCount
  :cognitect.aws.guardduty/ScannedItemCount)

(s/def
  :cognitect.aws.guardduty.ScanDetections/ThreatsDetectedItemCount
  :cognitect.aws.guardduty/ThreatsDetectedItemCount)

(s/def
  :cognitect.aws.guardduty.ScanDetections/HighestSeverityThreatDetails
  :cognitect.aws.guardduty/HighestSeverityThreatDetails)

(s/def
  :cognitect.aws.guardduty.ScanDetections/ThreatDetectedByName
  :cognitect.aws.guardduty/ThreatDetectedByName)

(s/def :cognitect.aws.guardduty.Finding/SchemaVersion :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Finding/Severity :cognitect.aws.guardduty/Double)

(s/def :cognitect.aws.guardduty.Finding/Service :cognitect.aws.guardduty/Service)

(s/def :cognitect.aws.guardduty.Finding/AssociatedAttackSequenceArn :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Finding/CreatedAt :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Finding/UpdatedAt :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Finding/AccountId :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Finding/Id :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Finding/Title :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Finding/Region :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Finding/Arn :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Finding/Resource :cognitect.aws.guardduty/Resource)

(s/def :cognitect.aws.guardduty.Finding/Description :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Finding/Type :cognitect.aws.guardduty/FindingType)

(s/def :cognitect.aws.guardduty.Finding/Partition :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Finding/Confidence :cognitect.aws.guardduty/Double)

(s/def :cognitect.aws.guardduty.AccessKeyDetails/AccessKeyId :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.AccessKeyDetails/PrincipalId :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.AccessKeyDetails/UserName :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.AccessKeyDetails/UserType :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.IamInstanceProfile/Arn :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.IamInstanceProfile/Id :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.GetAdministratorAccountResponse/Administrator
  :cognitect.aws.guardduty/Administrator)

(s/def
  :cognitect.aws.guardduty.ListOrganizationAdminAccountsRequest/MaxResults
  :cognitect.aws.guardduty/MaxResults)

(s/def
  :cognitect.aws.guardduty.ListOrganizationAdminAccountsRequest/NextToken
  :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.ResourceData/S3Bucket :cognitect.aws.guardduty/S3Bucket)

(s/def :cognitect.aws.guardduty.ResourceData/Ec2Instance :cognitect.aws.guardduty/Ec2Instance)

(s/def :cognitect.aws.guardduty.ResourceData/AccessKey :cognitect.aws.guardduty/AccessKey)

(s/def
  :cognitect.aws.guardduty.ResourceData/Ec2NetworkInterface
  :cognitect.aws.guardduty/Ec2NetworkInterface)

(s/def :cognitect.aws.guardduty.ResourceData/S3Object :cognitect.aws.guardduty/S3Object)

(s/def :cognitect.aws.guardduty.ResourceData/EksCluster :cognitect.aws.guardduty/EksCluster)

(s/def
  :cognitect.aws.guardduty.ResourceData/KubernetesWorkload
  :cognitect.aws.guardduty/KubernetesWorkload)

(s/def
  :cognitect.aws.guardduty.ResourceData/Container
  :cognitect.aws.guardduty/ContainerFindingResource)

(s/def :cognitect.aws.guardduty.CoverageResource/ResourceId :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.CoverageResource/DetectorId :cognitect.aws.guardduty/DetectorId)

(s/def :cognitect.aws.guardduty.CoverageResource/AccountId :cognitect.aws.guardduty/AccountId)

(s/def
  :cognitect.aws.guardduty.CoverageResource/ResourceDetails
  :cognitect.aws.guardduty/CoverageResourceDetails)

(s/def
  :cognitect.aws.guardduty.CoverageResource/CoverageStatus
  :cognitect.aws.guardduty/CoverageStatus)

(s/def :cognitect.aws.guardduty.CoverageResource/Issue :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.CoverageResource/UpdatedAt :cognitect.aws.guardduty/Timestamp)

(s/def
  :cognitect.aws.guardduty.MalwareProtectionPlanSummary/MalwareProtectionPlanId
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.DeleteMalwareProtectionPlanRequest/MalwareProtectionPlanId
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.UntagResourceRequest/ResourceArn
  :cognitect.aws.guardduty/GuardDutyArn)

(s/def :cognitect.aws.guardduty.UntagResourceRequest/TagKeys :cognitect.aws.guardduty/TagKeyList)

(s/def
  :cognitect.aws.guardduty.GetUsageStatisticsRequest/DetectorId
  :cognitect.aws.guardduty/DetectorId)

(s/def
  :cognitect.aws.guardduty.GetUsageStatisticsRequest/UsageStatisticType
  :cognitect.aws.guardduty/UsageStatisticType)

(s/def
  :cognitect.aws.guardduty.GetUsageStatisticsRequest/UsageCriteria
  :cognitect.aws.guardduty/UsageCriteria)

(s/def :cognitect.aws.guardduty.GetUsageStatisticsRequest/Unit :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.GetUsageStatisticsRequest/MaxResults
  :cognitect.aws.guardduty/MaxResults)

(s/def :cognitect.aws.guardduty.GetUsageStatisticsRequest/NextToken :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.SortCriteria/AttributeName :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.SortCriteria/OrderBy :cognitect.aws.guardduty/OrderBy)

(s/def :cognitect.aws.guardduty.FilterCriterion/CriterionKey :cognitect.aws.guardduty/CriterionKey)

(s/def
  :cognitect.aws.guardduty.FilterCriterion/FilterCondition
  :cognitect.aws.guardduty/FilterCondition)

(s/def :cognitect.aws.guardduty.Resource/ResourceType :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Resource/ContainerDetails :cognitect.aws.guardduty/Container)

(s/def
  :cognitect.aws.guardduty.Resource/RdsDbInstanceDetails
  :cognitect.aws.guardduty/RdsDbInstanceDetails)

(s/def
  :cognitect.aws.guardduty.Resource/RdsLimitlessDbDetails
  :cognitect.aws.guardduty/RdsLimitlessDbDetails)

(s/def :cognitect.aws.guardduty.Resource/InstanceDetails :cognitect.aws.guardduty/InstanceDetails)

(s/def
  :cognitect.aws.guardduty.Resource/EcsClusterDetails
  :cognitect.aws.guardduty/EcsClusterDetails)

(s/def :cognitect.aws.guardduty.Resource/S3BucketDetails :cognitect.aws.guardduty/S3BucketDetails)

(s/def :cognitect.aws.guardduty.Resource/RdsDbUserDetails :cognitect.aws.guardduty/RdsDbUserDetails)

(s/def :cognitect.aws.guardduty.Resource/AccessKeyDetails :cognitect.aws.guardduty/AccessKeyDetails)

(s/def
  :cognitect.aws.guardduty.Resource/KubernetesDetails
  :cognitect.aws.guardduty/KubernetesDetails)

(s/def :cognitect.aws.guardduty.Resource/LambdaDetails :cognitect.aws.guardduty/LambdaDetails)

(s/def :cognitect.aws.guardduty.Resource/EbsVolumeDetails :cognitect.aws.guardduty/EbsVolumeDetails)

(s/def
  :cognitect.aws.guardduty.Resource/EksClusterDetails
  :cognitect.aws.guardduty/EksClusterDetails)

(s/def
  :cognitect.aws.guardduty.UsageTopAccountsResult/Feature
  :cognitect.aws.guardduty/UsageFeature)

(s/def
  :cognitect.aws.guardduty.UsageTopAccountsResult/Accounts
  :cognitect.aws.guardduty/UsageTopAccountsByFeatureList)

(s/def
  :cognitect.aws.guardduty.UpdateThreatIntelSetRequest/DetectorId
  :cognitect.aws.guardduty/DetectorId)

(s/def
  :cognitect.aws.guardduty.UpdateThreatIntelSetRequest/ThreatIntelSetId
  :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.UpdateThreatIntelSetRequest/Name :cognitect.aws.guardduty/Name)

(s/def
  :cognitect.aws.guardduty.UpdateThreatIntelSetRequest/Location
  :cognitect.aws.guardduty/Location)

(s/def
  :cognitect.aws.guardduty.UpdateThreatIntelSetRequest/Activate
  :cognitect.aws.guardduty/Boolean)

(s/def
  :cognitect.aws.guardduty.UpdateThreatIntelSetRequest/ExpectedBucketOwner
  :cognitect.aws.guardduty/AccountId)

(s/def
  :cognitect.aws.guardduty.KubernetesDetails/KubernetesUserDetails
  :cognitect.aws.guardduty/KubernetesUserDetails)

(s/def
  :cognitect.aws.guardduty.KubernetesDetails/KubernetesWorkloadDetails
  :cognitect.aws.guardduty/KubernetesWorkloadDetails)

(s/def :cognitect.aws.guardduty.LineageObject/ExecutablePath :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.LineageObject/Uuid :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.LineageObject/Pid :cognitect.aws.guardduty/Integer)

(s/def :cognitect.aws.guardduty.LineageObject/UserId :cognitect.aws.guardduty/Integer)

(s/def :cognitect.aws.guardduty.LineageObject/StartTime :cognitect.aws.guardduty/Timestamp)

(s/def :cognitect.aws.guardduty.LineageObject/ParentUuid :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.LineageObject/Name :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.LineageObject/NamespacePid :cognitect.aws.guardduty/Integer)

(s/def :cognitect.aws.guardduty.LineageObject/Euid :cognitect.aws.guardduty/Integer)

(s/def :cognitect.aws.guardduty.ProcessDetails/Lineage :cognitect.aws.guardduty/Lineage)

(s/def :cognitect.aws.guardduty.ProcessDetails/ExecutablePath :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.ProcessDetails/Uuid :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.ProcessDetails/Pid :cognitect.aws.guardduty/Integer)

(s/def :cognitect.aws.guardduty.ProcessDetails/UserId :cognitect.aws.guardduty/Integer)

(s/def :cognitect.aws.guardduty.ProcessDetails/User :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.ProcessDetails/StartTime :cognitect.aws.guardduty/Timestamp)

(s/def :cognitect.aws.guardduty.ProcessDetails/ParentUuid :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.ProcessDetails/Pwd :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.ProcessDetails/ExecutableSha256 :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.ProcessDetails/Name :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.ProcessDetails/NamespacePid :cognitect.aws.guardduty/Integer)

(s/def :cognitect.aws.guardduty.ProcessDetails/Euid :cognitect.aws.guardduty/Integer)

(s/def :cognitect.aws.guardduty.AccessKey/PrincipalId :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.AccessKey/UserName :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.AccessKey/UserType :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.DisassociateFromAdministratorAccountRequest/DetectorId
  :cognitect.aws.guardduty/DetectorId)

(s/def :cognitect.aws.guardduty.DateStatistics/Date :cognitect.aws.guardduty/Timestamp)

(s/def :cognitect.aws.guardduty.DateStatistics/LastGeneratedAt :cognitect.aws.guardduty/Timestamp)

(s/def :cognitect.aws.guardduty.DateStatistics/Severity :cognitect.aws.guardduty/Double)

(s/def :cognitect.aws.guardduty.DateStatistics/TotalFindings :cognitect.aws.guardduty/Integer)

(s/def
  :cognitect.aws.guardduty.UpdateMemberDetectorsRequest/DetectorId
  :cognitect.aws.guardduty/DetectorId)

(s/def
  :cognitect.aws.guardduty.UpdateMemberDetectorsRequest/AccountIds
  :cognitect.aws.guardduty/AccountIds)

(s/def
  :cognitect.aws.guardduty.UpdateMemberDetectorsRequest/DataSources
  :cognitect.aws.guardduty/DataSourceConfigurations)

(s/def
  :cognitect.aws.guardduty.UpdateMemberDetectorsRequest/Features
  :cognitect.aws.guardduty/MemberFeaturesConfigurations)

(s/def :cognitect.aws.guardduty.KubernetesUserDetails/Username :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.KubernetesUserDetails/Uid :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.KubernetesUserDetails/Groups :cognitect.aws.guardduty/Groups)

(s/def
  :cognitect.aws.guardduty.KubernetesUserDetails/SessionName
  :cognitect.aws.guardduty/SessionNameList)

(s/def
  :cognitect.aws.guardduty.KubernetesUserDetails/ImpersonatedUser
  :cognitect.aws.guardduty/ImpersonatedUser)

(s/def :cognitect.aws.guardduty.DomainDetails/Domain :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.PrivateIpAddressDetails/PrivateDnsName
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.PrivateIpAddressDetails/PrivateIpAddress
  :cognitect.aws.guardduty/SensitiveString)

(s/def :cognitect.aws.guardduty.UpdateIPSetRequest/DetectorId :cognitect.aws.guardduty/DetectorId)

(s/def :cognitect.aws.guardduty.UpdateIPSetRequest/IpSetId :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.UpdateIPSetRequest/Name :cognitect.aws.guardduty/Name)

(s/def :cognitect.aws.guardduty.UpdateIPSetRequest/Location :cognitect.aws.guardduty/Location)

(s/def :cognitect.aws.guardduty.UpdateIPSetRequest/Activate :cognitect.aws.guardduty/Boolean)

(s/def
  :cognitect.aws.guardduty.UpdateIPSetRequest/ExpectedBucketOwner
  :cognitect.aws.guardduty/AccountId)

(s/def
  :cognitect.aws.guardduty.UpdateFindingsFeedbackRequest/DetectorId
  :cognitect.aws.guardduty/DetectorId)

(s/def
  :cognitect.aws.guardduty.UpdateFindingsFeedbackRequest/FindingIds
  :cognitect.aws.guardduty/FindingIds)

(s/def
  :cognitect.aws.guardduty.UpdateFindingsFeedbackRequest/Feedback
  :cognitect.aws.guardduty/Feedback)

(s/def
  :cognitect.aws.guardduty.UpdateFindingsFeedbackRequest/Comments
  :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.AdminAccount/AdminAccountId :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.AdminAccount/AdminStatus :cognitect.aws.guardduty/AdminStatus)

(s/def :cognitect.aws.guardduty.DeleteFilterRequest/DetectorId :cognitect.aws.guardduty/DetectorId)

(s/def :cognitect.aws.guardduty.DeleteFilterRequest/FilterName :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.KubernetesRoleDetails/Kind :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.KubernetesRoleDetails/Name :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.KubernetesRoleDetails/Uid :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.FindingTypeStatistics/FindingType :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.FindingTypeStatistics/LastGeneratedAt
  :cognitect.aws.guardduty/Timestamp)

(s/def
  :cognitect.aws.guardduty.FindingTypeStatistics/TotalFindings
  :cognitect.aws.guardduty/Integer)

(s/def
  :cognitect.aws.guardduty.AcceptAdministratorInvitationRequest/DetectorId
  :cognitect.aws.guardduty/DetectorId)

(s/def
  :cognitect.aws.guardduty.AcceptAdministratorInvitationRequest/AdministratorId
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.AcceptAdministratorInvitationRequest/InvitationId
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.GetMalwareScanSettingsRequest/DetectorId
  :cognitect.aws.guardduty/DetectorId)

(s/def :cognitect.aws.guardduty.VolumeMount/Name :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.VolumeMount/MountPath :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.OrganizationKubernetesConfiguration/AuditLogs
  :cognitect.aws.guardduty/OrganizationKubernetesAuditLogsConfiguration)

(s/def :cognitect.aws.guardduty.Country/CountryCode :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Country/CountryName :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.GetIPSetResponse/Name :cognitect.aws.guardduty/Name)

(s/def :cognitect.aws.guardduty.GetIPSetResponse/Format :cognitect.aws.guardduty/IpSetFormat)

(s/def :cognitect.aws.guardduty.GetIPSetResponse/Location :cognitect.aws.guardduty/Location)

(s/def :cognitect.aws.guardduty.GetIPSetResponse/Status :cognitect.aws.guardduty/IpSetStatus)

(s/def :cognitect.aws.guardduty.GetIPSetResponse/Tags :cognitect.aws.guardduty/TagMap)

(s/def
  :cognitect.aws.guardduty.GetIPSetResponse/ExpectedBucketOwner
  :cognitect.aws.guardduty/AccountId)

(s/def
  :cognitect.aws.guardduty.ArchiveFindingsRequest/DetectorId
  :cognitect.aws.guardduty/DetectorId)

(s/def
  :cognitect.aws.guardduty.ArchiveFindingsRequest/FindingIds
  :cognitect.aws.guardduty/FindingIds)

(s/def
  :cognitect.aws.guardduty.ListPublishingDestinationsRequest/DetectorId
  :cognitect.aws.guardduty/DetectorId)

(s/def
  :cognitect.aws.guardduty.ListPublishingDestinationsRequest/MaxResults
  :cognitect.aws.guardduty/MaxResults)

(s/def
  :cognitect.aws.guardduty.ListPublishingDestinationsRequest/NextToken
  :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Condition/Neq :cognitect.aws.guardduty/Neq)

(s/def :cognitect.aws.guardduty.Condition/Eq :cognitect.aws.guardduty/Eq)

(s/def :cognitect.aws.guardduty.Condition/GreaterThanOrEqual :cognitect.aws.guardduty/Long)

(s/def :cognitect.aws.guardduty.Condition/NotEquals :cognitect.aws.guardduty/NotEquals)

(s/def :cognitect.aws.guardduty.Condition/LessThanOrEqual :cognitect.aws.guardduty/Long)

(s/def :cognitect.aws.guardduty.Condition/Gt :cognitect.aws.guardduty/Integer)

(s/def :cognitect.aws.guardduty.Condition/LessThan :cognitect.aws.guardduty/Long)

(s/def :cognitect.aws.guardduty.Condition/Gte :cognitect.aws.guardduty/Integer)

(s/def :cognitect.aws.guardduty.Condition/GreaterThan :cognitect.aws.guardduty/Long)

(s/def :cognitect.aws.guardduty.Condition/Lte :cognitect.aws.guardduty/Integer)

(s/def :cognitect.aws.guardduty.Condition/Equals :cognitect.aws.guardduty/Equals)

(s/def :cognitect.aws.guardduty.Condition/Lt :cognitect.aws.guardduty/Integer)

(s/def
  :cognitect.aws.guardduty.CreateThreatEntitySetResponse/ThreatEntitySetId
  :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.LambdaDetails/VpcConfig :cognitect.aws.guardduty/VpcConfig)

(s/def :cognitect.aws.guardduty.LambdaDetails/Tags :cognitect.aws.guardduty/Tags)

(s/def :cognitect.aws.guardduty.LambdaDetails/LastModifiedAt :cognitect.aws.guardduty/Timestamp)

(s/def :cognitect.aws.guardduty.LambdaDetails/FunctionName :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.LambdaDetails/FunctionArn :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.LambdaDetails/RevisionId :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.LambdaDetails/Description :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.LambdaDetails/FunctionVersion :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.LambdaDetails/Role :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.MalwareProtectionPlanStatusReason/Code
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.MalwareProtectionPlanStatusReason/Message
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.CoverageEcsClusterDetails/ClusterName
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.CoverageEcsClusterDetails/FargateDetails
  :cognitect.aws.guardduty/FargateDetails)

(s/def
  :cognitect.aws.guardduty.CoverageEcsClusterDetails/ContainerInstanceDetails
  :cognitect.aws.guardduty/ContainerInstanceDetails)

(s/def
  :cognitect.aws.guardduty.PortProbeDetail/LocalPortDetails
  :cognitect.aws.guardduty/LocalPortDetails)

(s/def
  :cognitect.aws.guardduty.PortProbeDetail/LocalIpDetails
  :cognitect.aws.guardduty/LocalIpDetails)

(s/def
  :cognitect.aws.guardduty.PortProbeDetail/RemoteIpDetails
  :cognitect.aws.guardduty/RemoteIpDetails)

(s/def :cognitect.aws.guardduty.AccountFreeTrialInfo/AccountId :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.AccountFreeTrialInfo/DataSources
  :cognitect.aws.guardduty/DataSourcesFreeTrial)

(s/def
  :cognitect.aws.guardduty.AccountFreeTrialInfo/Features
  :cognitect.aws.guardduty/FreeTrialFeatureConfigurationsResults)

(s/def
  :cognitect.aws.guardduty.KubernetesWorkload/ContainerUids
  :cognitect.aws.guardduty/ContainerUids)

(s/def :cognitect.aws.guardduty.KubernetesWorkload/Namespace :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.KubernetesWorkload/KubernetesResourcesTypes
  :cognitect.aws.guardduty/KubernetesResourcesTypes)

(s/def :cognitect.aws.guardduty.ListIPSetsRequest/DetectorId :cognitect.aws.guardduty/DetectorId)

(s/def :cognitect.aws.guardduty.ListIPSetsRequest/MaxResults :cognitect.aws.guardduty/MaxResults)

(s/def :cognitect.aws.guardduty.ListIPSetsRequest/NextToken :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.MalwareProtectionPlanTaggingAction/Status
  :cognitect.aws.guardduty/MalwareProtectionPlanTaggingActionStatus)

(s/def
  :cognitect.aws.guardduty.SeverityStatistics/LastGeneratedAt
  :cognitect.aws.guardduty/Timestamp)

(s/def :cognitect.aws.guardduty.SeverityStatistics/Severity :cognitect.aws.guardduty/Double)

(s/def :cognitect.aws.guardduty.SeverityStatistics/TotalFindings :cognitect.aws.guardduty/Integer)

(s/def
  :cognitect.aws.guardduty.DeleteDetectorRequest/DetectorId
  :cognitect.aws.guardduty/DetectorId)

(s/def
  :cognitect.aws.guardduty.UpdateTrustedEntitySetRequest/DetectorId
  :cognitect.aws.guardduty/DetectorId)

(s/def
  :cognitect.aws.guardduty.UpdateTrustedEntitySetRequest/TrustedEntitySetId
  :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.UpdateTrustedEntitySetRequest/Name :cognitect.aws.guardduty/Name)

(s/def
  :cognitect.aws.guardduty.UpdateTrustedEntitySetRequest/Location
  :cognitect.aws.guardduty/Location)

(s/def
  :cognitect.aws.guardduty.UpdateTrustedEntitySetRequest/ExpectedBucketOwner
  :cognitect.aws.guardduty/ExpectedBucketOwner)

(s/def
  :cognitect.aws.guardduty.UpdateTrustedEntitySetRequest/Activate
  :cognitect.aws.guardduty/Boolean)

(s/def
  :cognitect.aws.guardduty.EbsVolumeDetails/ScannedVolumeDetails
  :cognitect.aws.guardduty/VolumeDetails)

(s/def
  :cognitect.aws.guardduty.EbsVolumeDetails/SkippedVolumeDetails
  :cognitect.aws.guardduty/VolumeDetails)

(s/def
  :cognitect.aws.guardduty.GetMemberDetectorsResponse/MemberDataSourceConfigurations
  :cognitect.aws.guardduty/MemberDataSourceConfigurations)

(s/def
  :cognitect.aws.guardduty.GetMemberDetectorsResponse/UnprocessedAccounts
  :cognitect.aws.guardduty/UnprocessedAccounts)

(s/def
  :cognitect.aws.guardduty.CreateThreatIntelSetRequest/DetectorId
  :cognitect.aws.guardduty/DetectorId)

(s/def :cognitect.aws.guardduty.CreateThreatIntelSetRequest/Name :cognitect.aws.guardduty/Name)

(s/def
  :cognitect.aws.guardduty.CreateThreatIntelSetRequest/Format
  :cognitect.aws.guardduty/ThreatIntelSetFormat)

(s/def
  :cognitect.aws.guardduty.CreateThreatIntelSetRequest/Location
  :cognitect.aws.guardduty/Location)

(s/def
  :cognitect.aws.guardduty.CreateThreatIntelSetRequest/Activate
  :cognitect.aws.guardduty/Boolean)

(s/def
  :cognitect.aws.guardduty.CreateThreatIntelSetRequest/ClientToken
  :cognitect.aws.guardduty/ClientToken)

(s/def :cognitect.aws.guardduty.CreateThreatIntelSetRequest/Tags :cognitect.aws.guardduty/TagMap)

(s/def
  :cognitect.aws.guardduty.CreateThreatIntelSetRequest/ExpectedBucketOwner
  :cognitect.aws.guardduty/AccountId)

(s/def :cognitect.aws.guardduty.Indicator/Key :cognitect.aws.guardduty/IndicatorType)

(s/def :cognitect.aws.guardduty.Indicator/Values :cognitect.aws.guardduty/IndicatorValues)

(s/def :cognitect.aws.guardduty.Indicator/Title :cognitect.aws.guardduty/IndicatorTitle)

(s/def
  :cognitect.aws.guardduty.DeleteMembersResponse/UnprocessedAccounts
  :cognitect.aws.guardduty/UnprocessedAccounts)

(s/def
  :cognitect.aws.guardduty.FindingStatistics/CountBySeverity
  :cognitect.aws.guardduty/CountBySeverity)

(s/def
  :cognitect.aws.guardduty.FindingStatistics/GroupedByAccount
  :cognitect.aws.guardduty/GroupedByAccount)

(s/def
  :cognitect.aws.guardduty.FindingStatistics/GroupedByDate
  :cognitect.aws.guardduty/GroupedByDate)

(s/def
  :cognitect.aws.guardduty.FindingStatistics/GroupedByFindingType
  :cognitect.aws.guardduty/GroupedByFindingType)

(s/def
  :cognitect.aws.guardduty.FindingStatistics/GroupedByResource
  :cognitect.aws.guardduty/GroupedByResource)

(s/def
  :cognitect.aws.guardduty.FindingStatistics/GroupedBySeverity
  :cognitect.aws.guardduty/GroupedBySeverity)

(s/def :cognitect.aws.guardduty.PortProbeAction/Blocked :cognitect.aws.guardduty/Boolean)

(s/def
  :cognitect.aws.guardduty.PortProbeAction/PortProbeDetails
  :cognitect.aws.guardduty/PortProbeDetails)

(s/def :cognitect.aws.guardduty.Organization/Asn :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Organization/AsnOrg :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Organization/Isp :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Organization/Org :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.CreateThreatEntitySetRequest/DetectorId
  :cognitect.aws.guardduty/DetectorId)

(s/def :cognitect.aws.guardduty.CreateThreatEntitySetRequest/Name :cognitect.aws.guardduty/Name)

(s/def
  :cognitect.aws.guardduty.CreateThreatEntitySetRequest/Format
  :cognitect.aws.guardduty/ThreatEntitySetFormat)

(s/def
  :cognitect.aws.guardduty.CreateThreatEntitySetRequest/Location
  :cognitect.aws.guardduty/Location)

(s/def
  :cognitect.aws.guardduty.CreateThreatEntitySetRequest/ExpectedBucketOwner
  :cognitect.aws.guardduty/ExpectedBucketOwner)

(s/def
  :cognitect.aws.guardduty.CreateThreatEntitySetRequest/Activate
  :cognitect.aws.guardduty/Boolean)

(s/def
  :cognitect.aws.guardduty.CreateThreatEntitySetRequest/ClientToken
  :cognitect.aws.guardduty/ClientToken)

(s/def :cognitect.aws.guardduty.CreateThreatEntitySetRequest/Tags :cognitect.aws.guardduty/TagMap)

(s/def :cognitect.aws.guardduty.Session/Uid :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Session/MfaStatus :cognitect.aws.guardduty/MfaStatus)

(s/def :cognitect.aws.guardduty.Session/CreatedTime :cognitect.aws.guardduty/Timestamp)

(s/def :cognitect.aws.guardduty.Session/Issuer :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.DataSourcesFreeTrial/CloudTrail
  :cognitect.aws.guardduty/DataSourceFreeTrial)

(s/def
  :cognitect.aws.guardduty.DataSourcesFreeTrial/DnsLogs
  :cognitect.aws.guardduty/DataSourceFreeTrial)

(s/def
  :cognitect.aws.guardduty.DataSourcesFreeTrial/FlowLogs
  :cognitect.aws.guardduty/DataSourceFreeTrial)

(s/def
  :cognitect.aws.guardduty.DataSourcesFreeTrial/S3Logs
  :cognitect.aws.guardduty/DataSourceFreeTrial)

(s/def
  :cognitect.aws.guardduty.DataSourcesFreeTrial/Kubernetes
  :cognitect.aws.guardduty/KubernetesDataSourceFreeTrial)

(s/def
  :cognitect.aws.guardduty.DataSourcesFreeTrial/MalwareProtection
  :cognitect.aws.guardduty/MalwareProtectionDataSourceFreeTrial)

(s/def :cognitect.aws.guardduty.ListTagsForResourceResponse/Tags :cognitect.aws.guardduty/TagMap)

(s/def
  :cognitect.aws.guardduty.MemberAdditionalConfiguration/Name
  :cognitect.aws.guardduty/OrgFeatureAdditionalConfiguration)

(s/def
  :cognitect.aws.guardduty.MemberAdditionalConfiguration/Status
  :cognitect.aws.guardduty/FeatureStatus)

(s/def
  :cognitect.aws.guardduty.MemberAdditionalConfigurationResult/Name
  :cognitect.aws.guardduty/OrgFeatureAdditionalConfiguration)

(s/def
  :cognitect.aws.guardduty.MemberAdditionalConfigurationResult/Status
  :cognitect.aws.guardduty/FeatureStatus)

(s/def
  :cognitect.aws.guardduty.MemberAdditionalConfigurationResult/UpdatedAt
  :cognitect.aws.guardduty/Timestamp)

(s/def :cognitect.aws.guardduty.FilterCondition/EqualsValue :cognitect.aws.guardduty/NonEmptyString)

(s/def :cognitect.aws.guardduty.FilterCondition/GreaterThan :cognitect.aws.guardduty/LongValue)

(s/def :cognitect.aws.guardduty.FilterCondition/LessThan :cognitect.aws.guardduty/LongValue)

(s/def :cognitect.aws.guardduty.DnsRequestAction/Domain :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.DnsRequestAction/Protocol :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.DnsRequestAction/Blocked :cognitect.aws.guardduty/Boolean)

(s/def :cognitect.aws.guardduty.DnsRequestAction/DomainWithSuffix :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.DnsRequestAction/VpcOwnerAccountId
  :cognitect.aws.guardduty/AccountId)

(s/def
  :cognitect.aws.guardduty.DescribePublishingDestinationResponse/DestinationId
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.DescribePublishingDestinationResponse/DestinationType
  :cognitect.aws.guardduty/DestinationType)

(s/def
  :cognitect.aws.guardduty.DescribePublishingDestinationResponse/Status
  :cognitect.aws.guardduty/PublishingStatus)

(s/def
  :cognitect.aws.guardduty.DescribePublishingDestinationResponse/PublishingFailureStartTimestamp
  :cognitect.aws.guardduty/Long)

(s/def
  :cognitect.aws.guardduty.DescribePublishingDestinationResponse/DestinationProperties
  :cognitect.aws.guardduty/DestinationProperties)

(s/def :cognitect.aws.guardduty.Master/AccountId :cognitect.aws.guardduty/AccountId)

(s/def :cognitect.aws.guardduty.Master/InvitationId :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Master/RelationshipStatus :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Master/InvitedAt :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.GetIPSetRequest/DetectorId :cognitect.aws.guardduty/DetectorId)

(s/def :cognitect.aws.guardduty.GetIPSetRequest/IpSetId :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.OrganizationDataSourceConfigurationsResult/S3Logs
  :cognitect.aws.guardduty/OrganizationS3LogsConfigurationResult)

(s/def
  :cognitect.aws.guardduty.OrganizationDataSourceConfigurationsResult/Kubernetes
  :cognitect.aws.guardduty/OrganizationKubernetesConfigurationResult)

(s/def
  :cognitect.aws.guardduty.OrganizationDataSourceConfigurationsResult/MalwareProtection
  :cognitect.aws.guardduty/OrganizationMalwareProtectionConfigurationResult)

(s/def :cognitect.aws.guardduty.Threat/Name :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Threat/Source :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Threat/ItemPaths :cognitect.aws.guardduty/ItemPaths)

(s/def
  :cognitect.aws.guardduty.CreateMalwareProtectionPlanResponse/MalwareProtectionPlanId
  :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.Detection/Anomaly :cognitect.aws.guardduty/Anomaly)

(s/def :cognitect.aws.guardduty.Detection/Sequence :cognitect.aws.guardduty/Sequence)

(s/def :cognitect.aws.guardduty.ListFindingsRequest/DetectorId :cognitect.aws.guardduty/DetectorId)

(s/def
  :cognitect.aws.guardduty.ListFindingsRequest/FindingCriteria
  :cognitect.aws.guardduty/FindingCriteria)

(s/def
  :cognitect.aws.guardduty.ListFindingsRequest/SortCriteria
  :cognitect.aws.guardduty/SortCriteria)

(s/def :cognitect.aws.guardduty.ListFindingsRequest/MaxResults :cognitect.aws.guardduty/MaxResults)

(s/def :cognitect.aws.guardduty.ListFindingsRequest/NextToken :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.OrganizationScanEc2InstanceWithFindingsResult/EbsVolumes
  :cognitect.aws.guardduty/OrganizationEbsVolumesResult)

(s/def :cognitect.aws.guardduty.ImpersonatedUser/Username :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.ImpersonatedUser/Groups :cognitect.aws.guardduty/Groups)

(s/def :cognitect.aws.guardduty.ListIPSetsResponse/IpSetIds :cognitect.aws.guardduty/IpSetIds)

(s/def :cognitect.aws.guardduty.ListIPSetsResponse/NextToken :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.CloudTrailConfigurationResult/Status
  :cognitect.aws.guardduty/DataSourceStatus)

(s/def
  :cognitect.aws.guardduty.ListThreatIntelSetsResponse/ThreatIntelSetIds
  :cognitect.aws.guardduty/ThreatIntelSetIds)

(s/def
  :cognitect.aws.guardduty.ListThreatIntelSetsResponse/NextToken
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.AcceptInvitationRequest/DetectorId
  :cognitect.aws.guardduty/DetectorId)

(s/def :cognitect.aws.guardduty.AcceptInvitationRequest/MasterId :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.AcceptInvitationRequest/InvitationId
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.ListMalwareProtectionPlansRequest/NextToken
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.ContainerInstanceDetails/CoveredContainerInstances
  :cognitect.aws.guardduty/Long)

(s/def
  :cognitect.aws.guardduty.ContainerInstanceDetails/CompatibleContainerInstances
  :cognitect.aws.guardduty/Long)

(s/def
  :cognitect.aws.guardduty.StopMonitoringMembersResponse/UnprocessedAccounts
  :cognitect.aws.guardduty/UnprocessedAccounts)

(s/def :cognitect.aws.guardduty.EksClusterDetails/Name :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.EksClusterDetails/Arn :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.EksClusterDetails/VpcId :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.EksClusterDetails/Status :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.EksClusterDetails/Tags :cognitect.aws.guardduty/Tags)

(s/def :cognitect.aws.guardduty.EksClusterDetails/CreatedAt :cognitect.aws.guardduty/Timestamp)

(s/def :cognitect.aws.guardduty.CoverageFilterCondition/Equals :cognitect.aws.guardduty/Equals)

(s/def
  :cognitect.aws.guardduty.CoverageFilterCondition/NotEquals
  :cognitect.aws.guardduty/NotEquals)

(s/def
  :cognitect.aws.guardduty.DeleteInvitationsResponse/UnprocessedAccounts
  :cognitect.aws.guardduty/UnprocessedAccounts)

(s/def
  :cognitect.aws.guardduty.GetMemberDetectorsRequest/DetectorId
  :cognitect.aws.guardduty/DetectorId)

(s/def
  :cognitect.aws.guardduty.GetMemberDetectorsRequest/AccountIds
  :cognitect.aws.guardduty/AccountIds)

(s/def :cognitect.aws.guardduty.NetworkEndpoint/Id :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.NetworkEndpoint/Ip :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.NetworkEndpoint/Domain :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.NetworkEndpoint/Port :cognitect.aws.guardduty/Integer)

(s/def
  :cognitect.aws.guardduty.NetworkEndpoint/Location
  :cognitect.aws.guardduty/NetworkGeoLocation)

(s/def
  :cognitect.aws.guardduty.NetworkEndpoint/AutonomousSystem
  :cognitect.aws.guardduty/AutonomousSystem)

(s/def
  :cognitect.aws.guardduty.NetworkEndpoint/Connection
  :cognitect.aws.guardduty/NetworkConnection)

(s/def :cognitect.aws.guardduty.AnomalyUnusual/Behavior :cognitect.aws.guardduty/Behavior)

(s/def
  :cognitect.aws.guardduty.CreateMalwareProtectionPlanRequest/ClientToken
  :cognitect.aws.guardduty/ClientToken)

(s/def
  :cognitect.aws.guardduty.CreateMalwareProtectionPlanRequest/Role
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.CreateMalwareProtectionPlanRequest/ProtectedResource
  :cognitect.aws.guardduty/CreateProtectedResource)

(s/def
  :cognitect.aws.guardduty.CreateMalwareProtectionPlanRequest/Actions
  :cognitect.aws.guardduty/MalwareProtectionPlanActions)

(s/def
  :cognitect.aws.guardduty.CreateMalwareProtectionPlanRequest/Tags
  :cognitect.aws.guardduty/TagMap)

(s/def
  :cognitect.aws.guardduty.UpdateMalwareScanSettingsRequest/DetectorId
  :cognitect.aws.guardduty/DetectorId)

(s/def
  :cognitect.aws.guardduty.UpdateMalwareScanSettingsRequest/ScanResourceCriteria
  :cognitect.aws.guardduty/ScanResourceCriteria)

(s/def
  :cognitect.aws.guardduty.UpdateMalwareScanSettingsRequest/EbsSnapshotPreservation
  :cognitect.aws.guardduty/EbsSnapshotPreservation)

(s/def
  :cognitect.aws.guardduty.KubernetesConfigurationResult/AuditLogs
  :cognitect.aws.guardduty/KubernetesAuditLogsConfigurationResult)

(s/def
  :cognitect.aws.guardduty.DisableOrganizationAdminAccountRequest/AdminAccountId
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.ListThreatEntitySetsRequest/DetectorId
  :cognitect.aws.guardduty/DetectorId)

(s/def
  :cognitect.aws.guardduty.ListThreatEntitySetsRequest/MaxResults
  :cognitect.aws.guardduty/MaxResults)

(s/def
  :cognitect.aws.guardduty.ListThreatEntitySetsRequest/NextToken
  :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.ResourceNotFoundException/Message :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.ResourceNotFoundException/Type :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.CoverageStatistics/CountByResourceType
  :cognitect.aws.guardduty/CountByResourceType)

(s/def
  :cognitect.aws.guardduty.CoverageStatistics/CountByCoverageStatus
  :cognitect.aws.guardduty/CountByCoverageStatus)

(s/def :cognitect.aws.guardduty.ConflictException/Message :cognitect.aws.guardduty/String)

(s/def :cognitect.aws.guardduty.ConflictException/Type :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.DataSourceConfigurations/S3Logs
  :cognitect.aws.guardduty/S3LogsConfiguration)

(s/def
  :cognitect.aws.guardduty.DataSourceConfigurations/Kubernetes
  :cognitect.aws.guardduty/KubernetesConfiguration)

(s/def
  :cognitect.aws.guardduty.DataSourceConfigurations/MalwareProtection
  :cognitect.aws.guardduty/MalwareProtectionConfiguration)

(s/def
  :cognitect.aws.guardduty.CoverageEc2InstanceDetails/InstanceId
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.CoverageEc2InstanceDetails/InstanceType
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.CoverageEc2InstanceDetails/ClusterArn
  :cognitect.aws.guardduty/String)

(s/def
  :cognitect.aws.guardduty.CoverageEc2InstanceDetails/AgentDetails
  :cognitect.aws.guardduty/AgentDetails)

(s/def
  :cognitect.aws.guardduty.CoverageEc2InstanceDetails/ManagementType
  :cognitect.aws.guardduty/ManagementType)

(s/def
  :cognitect.aws.guardduty.GetAdministratorAccountRequest/DetectorId
  :cognitect.aws.guardduty/DetectorId)

(s/def
  :cognitect.aws.guardduty.ListThreatIntelSetsRequest/DetectorId
  :cognitect.aws.guardduty/DetectorId)

(s/def
  :cognitect.aws.guardduty.ListThreatIntelSetsRequest/MaxResults
  :cognitect.aws.guardduty/MaxResults)

(s/def
  :cognitect.aws.guardduty.ListThreatIntelSetsRequest/NextToken
  :cognitect.aws.guardduty/String)

