;; Copyright (c) Cognitect, Inc.
;; All rights reserved.

(ns cognitect.aws.transfer.specs
  (:require [clojure.spec.alpha :as s] [clojure.spec.gen.alpha :as gen]))

(s/def :cognitect.aws/client map?)

(s/def :core.async/channel any?)

(s/def
  :cognitect.aws.transfer/Url
  (s/spec
    (s/and string? #(>= 255 (count %)))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 0 255) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.transfer/S3InputFileLocation
  (s/keys
    :opt-un
    [:cognitect.aws.transfer.S3InputFileLocation/Bucket
     :cognitect.aws.transfer.S3InputFileLocation/Key]))

(s/def
  :cognitect.aws.transfer/S3TagValue
  (s/spec #(re-matches (re-pattern "^([\\p{L}\\p{Z}\\p{N}_.:/=+\\-@]*)$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.transfer/SourceFileLocation
  (s/spec #(re-matches (re-pattern "^\\$\\{(\\w+.)+\\w+\\}$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.transfer/IdentityProviderType
  (s/spec
    string?
    :gen
    #(s/gen #{"AWS_LAMBDA" "AWS_DIRECTORY_SERVICE" "API_GATEWAY" "SERVICE_MANAGED"})))

(s/def :cognitect.aws.transfer/SubnetIds (s/coll-of :cognitect.aws.transfer/SubnetId))

(s/def :cognitect.aws.transfer/SendWorkflowStepStateResponse (s/keys))

(s/def :cognitect.aws.transfer/Domain (s/spec string? :gen #(s/gen #{"EFS" "S3"})))

(s/def
  :cognitect.aws.transfer/ExecutionId
  (s/spec
    #(re-matches
      (re-pattern
        "^[0-9a-fA-F]{8}\\-[0-9a-fA-F]{4}\\-[0-9a-fA-F]{4}\\-[0-9a-fA-F]{4}\\-[0-9a-fA-F]{12}$")
      %)
    :gen
    #(gen/string)))

(s/def
  :cognitect.aws.transfer/TagResourceRequest
  (s/keys
    :req-un
    [:cognitect.aws.transfer.TagResourceRequest/Arn
     :cognitect.aws.transfer.TagResourceRequest/Tags]))

(s/def
  :cognitect.aws.transfer/WorkflowStepName
  (s/spec #(re-matches (re-pattern "^[\\w-]*$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.transfer/WorkflowSteps
  (s/coll-of :cognitect.aws.transfer/WorkflowStep :max-count 8))

(s/def
  :cognitect.aws.transfer/ExternalId
  (s/spec #(re-matches (re-pattern "^S-1-[\\d-]+$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.transfer/Function
  (s/spec #(re-matches (re-pattern "^arn:[a-z-]+:lambda:.*$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.transfer/TlsSessionResumptionMode
  (s/spec string? :gen #(s/gen #{"DISABLED" "ENFORCED" "ENABLED"})))

(s/def
  :cognitect.aws.transfer/PosixProfile
  (s/keys
    :req-un
    [:cognitect.aws.transfer.PosixProfile/Uid :cognitect.aws.transfer.PosixProfile/Gid]
    :opt-un
    [:cognitect.aws.transfer.PosixProfile/SecondaryGids]))

(s/def
  :cognitect.aws.transfer/CustomStepDetails
  (s/keys
    :opt-un
    [:cognitect.aws.transfer.CustomStepDetails/TimeoutSeconds
     :cognitect.aws.transfer.CustomStepDetails/SourceFileLocation
     :cognitect.aws.transfer.CustomStepDetails/Target
     :cognitect.aws.transfer.CustomStepDetails/Name]))

(s/def
  :cognitect.aws.transfer/SshPublicKeys
  (s/coll-of :cognitect.aws.transfer/SshPublicKey :max-count 5))

(s/def
  :cognitect.aws.transfer/StartServerRequest
  (s/keys :req-un [:cognitect.aws.transfer.StartServerRequest/ServerId]))

(s/def
  :cognitect.aws.transfer/ListedWorkflow
  (s/keys
    :opt-un
    [:cognitect.aws.transfer.ListedWorkflow/Description
     :cognitect.aws.transfer.ListedWorkflow/Arn
     :cognitect.aws.transfer.ListedWorkflow/WorkflowId]))

(s/def
  :cognitect.aws.transfer/ListExecutionsRequest
  (s/keys
    :req-un
    [:cognitect.aws.transfer.ListExecutionsRequest/WorkflowId]
    :opt-un
    [:cognitect.aws.transfer.ListExecutionsRequest/NextToken
     :cognitect.aws.transfer.ListExecutionsRequest/MaxResults]))

(s/def
  :cognitect.aws.transfer/CreateUserResponse
  (s/keys
    :req-un
    [:cognitect.aws.transfer.CreateUserResponse/ServerId
     :cognitect.aws.transfer.CreateUserResponse/UserName]))

(s/def
  :cognitect.aws.transfer/ExecutionResults
  (s/keys
    :opt-un
    [:cognitect.aws.transfer.ExecutionResults/OnExceptionSteps
     :cognitect.aws.transfer.ExecutionResults/Steps]))

(s/def
  :cognitect.aws.transfer/ImportSshPublicKeyRequest
  (s/keys
    :req-un
    [:cognitect.aws.transfer.ImportSshPublicKeyRequest/ServerId
     :cognitect.aws.transfer.ImportSshPublicKeyRequest/SshPublicKeyBody
     :cognitect.aws.transfer.ImportSshPublicKeyRequest/UserName]))

(s/def
  :cognitect.aws.transfer/Tag
  (s/keys :req-un [:cognitect.aws.transfer.Tag/Key :cognitect.aws.transfer.Tag/Value]))

(s/def
  :cognitect.aws.transfer/SshPublicKeyBody
  (s/spec
    #(re-matches (re-pattern "^ssh-rsa\\s+[A-Za-z0-9+/]+[=]{0,3}(\\s+.+)?\\s*$") %)
    :gen
    #(gen/string)))

(s/def
  :cognitect.aws.transfer/StepResultOutputsJson
  (s/spec
    (s/and string? #(>= 65536 (count %)))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 0 65536) #(gen/vector (gen/char-alpha) %))))))

(s/def :cognitect.aws.transfer/ExecutionErrorMessage string?)

(s/def
  :cognitect.aws.transfer/DescribeAccessRequest
  (s/keys
    :req-un
    [:cognitect.aws.transfer.DescribeAccessRequest/ServerId
     :cognitect.aws.transfer.DescribeAccessRequest/ExternalId]))

(s/def
  :cognitect.aws.transfer/DescribedServer
  (s/keys
    :req-un
    [:cognitect.aws.transfer.DescribedServer/Arn]
    :opt-un
    [:cognitect.aws.transfer.DescribedServer/WorkflowDetails
     :cognitect.aws.transfer.DescribedServer/SecurityPolicyName
     :cognitect.aws.transfer.DescribedServer/Protocols
     :cognitect.aws.transfer.DescribedServer/EndpointType
     :cognitect.aws.transfer.DescribedServer/EndpointDetails
     :cognitect.aws.transfer.DescribedServer/Tags
     :cognitect.aws.transfer.DescribedServer/PostAuthenticationLoginBanner
     :cognitect.aws.transfer.DescribedServer/ProtocolDetails
     :cognitect.aws.transfer.DescribedServer/PreAuthenticationLoginBanner
     :cognitect.aws.transfer.DescribedServer/IdentityProviderDetails
     :cognitect.aws.transfer.DescribedServer/Domain
     :cognitect.aws.transfer.DescribedServer/LoggingRole
     :cognitect.aws.transfer.DescribedServer/IdentityProviderType
     :cognitect.aws.transfer.DescribedServer/State
     :cognitect.aws.transfer.DescribedServer/HostKeyFingerprint
     :cognitect.aws.transfer.DescribedServer/UserCount
     :cognitect.aws.transfer.DescribedServer/ServerId
     :cognitect.aws.transfer.DescribedServer/Certificate]))

(s/def
  :cognitect.aws.transfer/CreateWorkflowRequest
  (s/keys
    :req-un
    [:cognitect.aws.transfer.CreateWorkflowRequest/Steps]
    :opt-un
    [:cognitect.aws.transfer.CreateWorkflowRequest/Tags
     :cognitect.aws.transfer.CreateWorkflowRequest/OnExceptionSteps
     :cognitect.aws.transfer.CreateWorkflowRequest/Description]))

(s/def :cognitect.aws.transfer/Message string?)

(s/def
  :cognitect.aws.transfer/HomeDirectory
  (s/spec #(re-matches (re-pattern "^$|/.*") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.transfer/ExecutionError
  (s/keys
    :req-un
    [:cognitect.aws.transfer.ExecutionError/Type :cognitect.aws.transfer.ExecutionError/Message]))

(s/def
  :cognitect.aws.transfer/WorkflowId
  (s/spec #(re-matches (re-pattern "^w-([a-z0-9]{17})$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.transfer/InputFileLocation
  (s/keys
    :opt-un
    [:cognitect.aws.transfer.InputFileLocation/EfsFileLocation
     :cognitect.aws.transfer.InputFileLocation/S3FileLocation]))

(s/def
  :cognitect.aws.transfer/DescribeSecurityPolicyRequest
  (s/keys :req-un [:cognitect.aws.transfer.DescribeSecurityPolicyRequest/SecurityPolicyName]))

(s/def
  :cognitect.aws.transfer/DescribeExecutionResponse
  (s/keys
    :req-un
    [:cognitect.aws.transfer.DescribeExecutionResponse/WorkflowId
     :cognitect.aws.transfer.DescribeExecutionResponse/Execution]))

(s/def
  :cognitect.aws.transfer/DeleteWorkflowRequest
  (s/keys :req-un [:cognitect.aws.transfer.DeleteWorkflowRequest/WorkflowId]))

(s/def
  :cognitect.aws.transfer/CreateWorkflowResponse
  (s/keys :req-un [:cognitect.aws.transfer.CreateWorkflowResponse/WorkflowId]))

(s/def :cognitect.aws.transfer/ListedServers (s/coll-of :cognitect.aws.transfer/ListedServer))

(s/def
  :cognitect.aws.transfer/DescribedWorkflow
  (s/keys
    :req-un
    [:cognitect.aws.transfer.DescribedWorkflow/Arn]
    :opt-un
    [:cognitect.aws.transfer.DescribedWorkflow/Tags
     :cognitect.aws.transfer.DescribedWorkflow/OnExceptionSteps
     :cognitect.aws.transfer.DescribedWorkflow/Description
     :cognitect.aws.transfer.DescribedWorkflow/Steps
     :cognitect.aws.transfer.DescribedWorkflow/WorkflowId]))

(s/def
  :cognitect.aws.transfer/HostKey
  (s/spec
    (s/and string? #(>= 4096 (count %)))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 0 4096) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.transfer/WorkflowStepType
  (s/spec string? :gen #(s/gen #{"DELETE" "CUSTOM" "TAG" "COPY"})))

(s/def
  :cognitect.aws.transfer/PassiveIp
  (s/spec
    (s/and string? #(>= 15 (count %)))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 0 15) #(gen/vector (gen/char-alpha) %))))))

(s/def :cognitect.aws.transfer/ListedExecutions (s/coll-of :cognitect.aws.transfer/ListedExecution))

(s/def
  :cognitect.aws.transfer/DescribeWorkflowRequest
  (s/keys :req-un [:cognitect.aws.transfer.DescribeWorkflowRequest/WorkflowId]))

(s/def
  :cognitect.aws.transfer/DescribedUser
  (s/keys
    :req-un
    [:cognitect.aws.transfer.DescribedUser/Arn]
    :opt-un
    [:cognitect.aws.transfer.DescribedUser/Tags
     :cognitect.aws.transfer.DescribedUser/HomeDirectory
     :cognitect.aws.transfer.DescribedUser/UserName
     :cognitect.aws.transfer.DescribedUser/Role
     :cognitect.aws.transfer.DescribedUser/SshPublicKeys
     :cognitect.aws.transfer.DescribedUser/PosixProfile
     :cognitect.aws.transfer.DescribedUser/HomeDirectoryType
     :cognitect.aws.transfer.DescribedUser/HomeDirectoryMappings
     :cognitect.aws.transfer.DescribedUser/Policy]))

(s/def
  :cognitect.aws.transfer/CreateAccessResponse
  (s/keys
    :req-un
    [:cognitect.aws.transfer.CreateAccessResponse/ServerId
     :cognitect.aws.transfer.CreateAccessResponse/ExternalId]))

(s/def
  :cognitect.aws.transfer/DirectoryId
  (s/spec #(re-matches (re-pattern "^d-[0-9a-f]{10}$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.transfer/S3TagKey
  (s/spec #(re-matches (re-pattern "^([\\p{L}\\p{Z}\\p{N}_.:/=+\\-@]*)$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.transfer/Tags
  (s/coll-of :cognitect.aws.transfer/Tag :min-count 1 :max-count 50))

(s/def
  :cognitect.aws.transfer/ExecutionErrorType
  (s/spec
    string?
    :gen
    #(s/gen
      #{"TIMEOUT" "ALREADY_EXISTS" "INTERNAL_SERVER_ERROR" "NOT_FOUND" "BAD_REQUEST"
        "PERMISSION_DENIED" "CUSTOM_STEP_FAILED" "THROTTLED"})))

(s/def
  :cognitect.aws.transfer/CreateAccessRequest
  (s/keys
    :req-un
    [:cognitect.aws.transfer.CreateAccessRequest/Role
     :cognitect.aws.transfer.CreateAccessRequest/ServerId
     :cognitect.aws.transfer.CreateAccessRequest/ExternalId]
    :opt-un
    [:cognitect.aws.transfer.CreateAccessRequest/HomeDirectory
     :cognitect.aws.transfer.CreateAccessRequest/PosixProfile
     :cognitect.aws.transfer.CreateAccessRequest/HomeDirectoryType
     :cognitect.aws.transfer.CreateAccessRequest/HomeDirectoryMappings
     :cognitect.aws.transfer.CreateAccessRequest/Policy]))

(s/def
  :cognitect.aws.transfer/WorkflowStep
  (s/keys
    :opt-un
    [:cognitect.aws.transfer.WorkflowStep/CopyStepDetails
     :cognitect.aws.transfer.WorkflowStep/Type
     :cognitect.aws.transfer.WorkflowStep/DeleteStepDetails
     :cognitect.aws.transfer.WorkflowStep/TagStepDetails
     :cognitect.aws.transfer.WorkflowStep/CustomStepDetails]))

(s/def
  :cognitect.aws.transfer/EndpointDetails
  (s/keys
    :opt-un
    [:cognitect.aws.transfer.EndpointDetails/SubnetIds
     :cognitect.aws.transfer.EndpointDetails/SecurityGroupIds
     :cognitect.aws.transfer.EndpointDetails/VpcEndpointId
     :cognitect.aws.transfer.EndpointDetails/AddressAllocationIds
     :cognitect.aws.transfer.EndpointDetails/VpcId]))

(s/def
  :cognitect.aws.transfer/ServiceMetadata
  (s/keys :req-un [:cognitect.aws.transfer.ServiceMetadata/UserDetails]))

(s/def
  :cognitect.aws.transfer/CallbackToken
  (s/spec #(re-matches (re-pattern "\\w+") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.transfer/S3Key
  (s/spec #(re-matches (re-pattern "[\\P{M}\\p{M}]*") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.transfer/DescribeExecutionRequest
  (s/keys
    :req-un
    [:cognitect.aws.transfer.DescribeExecutionRequest/ExecutionId
     :cognitect.aws.transfer.DescribeExecutionRequest/WorkflowId]))

(s/def
  :cognitect.aws.transfer/ExecutionStatus
  (s/spec string? :gen #(s/gen #{"COMPLETED" "EXCEPTION" "IN_PROGRESS" "HANDLING_EXCEPTION"})))

(s/def
  :cognitect.aws.transfer/StopServerRequest
  (s/keys :req-un [:cognitect.aws.transfer.StopServerRequest/ServerId]))

(s/def
  :cognitect.aws.transfer/OnUploadWorkflowDetails
  (s/coll-of :cognitect.aws.transfer/WorkflowDetail :max-count 1))

(s/def
  :cognitect.aws.transfer/ListSecurityPoliciesRequest
  (s/keys
    :opt-un
    [:cognitect.aws.transfer.ListSecurityPoliciesRequest/NextToken
     :cognitect.aws.transfer.ListSecurityPoliciesRequest/MaxResults]))

(s/def
  :cognitect.aws.transfer/UserCount
  (s/spec int? :gen #(gen/choose Long/MIN_VALUE Long/MAX_VALUE)))

(s/def
  :cognitect.aws.transfer/CustomStepTimeoutSeconds
  (s/spec (s/and int? #(<= 1 % 1800)) :gen #(gen/choose 1 1800)))

(s/def
  :cognitect.aws.transfer/IdentityProviderDetails
  (s/keys
    :opt-un
    [:cognitect.aws.transfer.IdentityProviderDetails/Url
     :cognitect.aws.transfer.IdentityProviderDetails/DirectoryId
     :cognitect.aws.transfer.IdentityProviderDetails/Function
     :cognitect.aws.transfer.IdentityProviderDetails/InvocationRole]))

(s/def
  :cognitect.aws.transfer/ListedAccess
  (s/keys
    :opt-un
    [:cognitect.aws.transfer.ListedAccess/ExternalId
     :cognitect.aws.transfer.ListedAccess/HomeDirectory
     :cognitect.aws.transfer.ListedAccess/Role
     :cognitect.aws.transfer.ListedAccess/HomeDirectoryType]))

(s/def
  :cognitect.aws.transfer/SecondaryGids
  (s/coll-of :cognitect.aws.transfer/PosixId :min-count 0 :max-count 16))

(s/def
  :cognitect.aws.transfer/DeleteSshPublicKeyRequest
  (s/keys
    :req-un
    [:cognitect.aws.transfer.DeleteSshPublicKeyRequest/ServerId
     :cognitect.aws.transfer.DeleteSshPublicKeyRequest/SshPublicKeyId
     :cognitect.aws.transfer.DeleteSshPublicKeyRequest/UserName]))

(s/def
  :cognitect.aws.transfer/LogGroupName
  (s/spec #(re-matches (re-pattern "[\\.\\-_/#A-Za-z0-9]*") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.transfer/SecurityPolicyOption
  (s/spec
    (s/and string? #(>= 50 (count %)))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 0 50) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.transfer/DescribeUserRequest
  (s/keys
    :req-un
    [:cognitect.aws.transfer.DescribeUserRequest/ServerId
     :cognitect.aws.transfer.DescribeUserRequest/UserName]))

(s/def
  :cognitect.aws.transfer/CreateServerResponse
  (s/keys :req-un [:cognitect.aws.transfer.CreateServerResponse/ServerId]))

(s/def
  :cognitect.aws.transfer/UserPassword
  (s/spec
    (s/and string? #(>= 1024 (count %)))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 0 1024) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.transfer/PostAuthenticationLoginBanner
  (s/spec #(re-matches (re-pattern "[\\x09-\\x0D\\x20-\\x7E]*") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.transfer/Protocols
  (s/coll-of :cognitect.aws.transfer/Protocol :min-count 1 :max-count 3))

(s/def
  :cognitect.aws.transfer/S3Bucket
  (s/spec #(re-matches (re-pattern "^[a-z0-9][\\.\\-a-z0-9]{1,61}[a-z0-9]$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.transfer/UpdateAccessResponse
  (s/keys
    :req-un
    [:cognitect.aws.transfer.UpdateAccessResponse/ServerId
     :cognitect.aws.transfer.UpdateAccessResponse/ExternalId]))

(s/def
  :cognitect.aws.transfer/ListedExecution
  (s/keys
    :opt-un
    [:cognitect.aws.transfer.ListedExecution/InitialFileLocation
     :cognitect.aws.transfer.ListedExecution/Status
     :cognitect.aws.transfer.ListedExecution/ExecutionId
     :cognitect.aws.transfer.ListedExecution/ServiceMetadata]))

(s/def
  :cognitect.aws.transfer/CustomStepTarget
  (s/spec #(re-matches (re-pattern "arn:[a-z-]+:lambda:.*$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.transfer/ListWorkflowsRequest
  (s/keys
    :opt-un
    [:cognitect.aws.transfer.ListWorkflowsRequest/NextToken
     :cognitect.aws.transfer.ListWorkflowsRequest/MaxResults]))

(s/def
  :cognitect.aws.transfer/NextToken
  (s/spec
    (s/and string? #(<= 1 (count %) 6144))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 1 6144) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.transfer/DescribedAccess
  (s/keys
    :opt-un
    [:cognitect.aws.transfer.DescribedAccess/ExternalId
     :cognitect.aws.transfer.DescribedAccess/HomeDirectory
     :cognitect.aws.transfer.DescribedAccess/Role
     :cognitect.aws.transfer.DescribedAccess/PosixProfile
     :cognitect.aws.transfer.DescribedAccess/HomeDirectoryType
     :cognitect.aws.transfer.DescribedAccess/HomeDirectoryMappings
     :cognitect.aws.transfer.DescribedAccess/Policy]))

(s/def
  :cognitect.aws.transfer/DescribedExecution
  (s/keys
    :opt-un
    [:cognitect.aws.transfer.DescribedExecution/InitialFileLocation
     :cognitect.aws.transfer.DescribedExecution/Status
     :cognitect.aws.transfer.DescribedExecution/PosixProfile
     :cognitect.aws.transfer.DescribedExecution/ExecutionId
     :cognitect.aws.transfer.DescribedExecution/LoggingConfiguration
     :cognitect.aws.transfer.DescribedExecution/ServiceMetadata
     :cognitect.aws.transfer.DescribedExecution/ExecutionRole
     :cognitect.aws.transfer.DescribedExecution/Results]))

(s/def
  :cognitect.aws.transfer/HomeDirectoryMapEntry
  (s/keys
    :req-un
    [:cognitect.aws.transfer.HomeDirectoryMapEntry/Entry
     :cognitect.aws.transfer.HomeDirectoryMapEntry/Target]))

(s/def
  :cognitect.aws.transfer/ServerId
  (s/spec #(re-matches (re-pattern "^s-([0-9a-f]{17})$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.transfer/EfsPath
  (s/spec #(re-matches (re-pattern "^[^\\x00]+$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.transfer/PosixId
  (s/spec (s/and int? #(<= 0 % 4294967295)) :gen #(gen/choose 0 4294967295)))

(s/def
  :cognitect.aws.transfer/TagValue
  (s/spec
    (s/and string? #(>= 256 (count %)))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 0 256) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.transfer/S3Tag
  (s/keys :req-un [:cognitect.aws.transfer.S3Tag/Key :cognitect.aws.transfer.S3Tag/Value]))

(s/def
  :cognitect.aws.transfer/SecurityPolicyOptions
  (s/coll-of :cognitect.aws.transfer/SecurityPolicyOption))

(s/def
  :cognitect.aws.transfer/UserName
  (s/spec #(re-matches (re-pattern "^[\\w][\\w@.-]{2,99}$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.transfer/HomeDirectoryMappings
  (s/coll-of :cognitect.aws.transfer/HomeDirectoryMapEntry :min-count 1 :max-count 50))

(s/def
  :cognitect.aws.transfer/ListTagsForResourceRequest
  (s/keys
    :req-un
    [:cognitect.aws.transfer.ListTagsForResourceRequest/Arn]
    :opt-un
    [:cognitect.aws.transfer.ListTagsForResourceRequest/NextToken
     :cognitect.aws.transfer.ListTagsForResourceRequest/MaxResults]))

(s/def
  :cognitect.aws.transfer/TagStepDetails
  (s/keys
    :opt-un
    [:cognitect.aws.transfer.TagStepDetails/Tags
     :cognitect.aws.transfer.TagStepDetails/SourceFileLocation
     :cognitect.aws.transfer.TagStepDetails/Name]))

(s/def
  :cognitect.aws.transfer/WorkflowDetails
  (s/keys :req-un [:cognitect.aws.transfer.WorkflowDetails/OnUpload]))

(s/def
  :cognitect.aws.transfer/EndpointType
  (s/spec string? :gen #(s/gen #{"VPC_ENDPOINT" "PUBLIC" "VPC"})))

(s/def
  :cognitect.aws.transfer/ListWorkflowsResponse
  (s/keys
    :req-un
    [:cognitect.aws.transfer.ListWorkflowsResponse/Workflows]
    :opt-un
    [:cognitect.aws.transfer.ListWorkflowsResponse/NextToken]))

(s/def
  :cognitect.aws.transfer/UpdateServerResponse
  (s/keys :req-un [:cognitect.aws.transfer.UpdateServerResponse/ServerId]))

(s/def
  :cognitect.aws.transfer/ExecutionStepResults
  (s/coll-of :cognitect.aws.transfer/ExecutionStepResult :min-count 1 :max-count 50))

(s/def
  :cognitect.aws.transfer/SecurityGroupId
  (s/spec #(re-matches (re-pattern "^sg-[0-9a-f]{8,17}$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.transfer/DescribeUserResponse
  (s/keys
    :req-un
    [:cognitect.aws.transfer.DescribeUserResponse/ServerId
     :cognitect.aws.transfer.DescribeUserResponse/User]))

(s/def
  :cognitect.aws.transfer/TestIdentityProviderResponse
  (s/keys
    :req-un
    [:cognitect.aws.transfer.TestIdentityProviderResponse/StatusCode
     :cognitect.aws.transfer.TestIdentityProviderResponse/Url]
    :opt-un
    [:cognitect.aws.transfer.TestIdentityProviderResponse/Response
     :cognitect.aws.transfer.TestIdentityProviderResponse/Message]))

(s/def
  :cognitect.aws.transfer/DeleteUserRequest
  (s/keys
    :req-un
    [:cognitect.aws.transfer.DeleteUserRequest/ServerId
     :cognitect.aws.transfer.DeleteUserRequest/UserName]))

(s/def
  :cognitect.aws.transfer/SshPublicKey
  (s/keys
    :req-un
    [:cognitect.aws.transfer.SshPublicKey/DateImported
     :cognitect.aws.transfer.SshPublicKey/SshPublicKeyBody
     :cognitect.aws.transfer.SshPublicKey/SshPublicKeyId]))

(s/def :cognitect.aws.transfer/SubnetId string?)

(s/def
  :cognitect.aws.transfer/Certificate
  (s/spec
    (s/and string? #(>= 1600 (count %)))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 0 1600) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.transfer/StatusCode
  (s/spec int? :gen #(gen/choose Long/MIN_VALUE Long/MAX_VALUE)))

(s/def
  :cognitect.aws.transfer/CreateUserRequest
  (s/keys
    :req-un
    [:cognitect.aws.transfer.CreateUserRequest/Role
     :cognitect.aws.transfer.CreateUserRequest/ServerId
     :cognitect.aws.transfer.CreateUserRequest/UserName]
    :opt-un
    [:cognitect.aws.transfer.CreateUserRequest/SshPublicKeyBody
     :cognitect.aws.transfer.CreateUserRequest/Tags
     :cognitect.aws.transfer.CreateUserRequest/HomeDirectory
     :cognitect.aws.transfer.CreateUserRequest/PosixProfile
     :cognitect.aws.transfer.CreateUserRequest/HomeDirectoryType
     :cognitect.aws.transfer.CreateUserRequest/HomeDirectoryMappings
     :cognitect.aws.transfer.CreateUserRequest/Policy]))

(s/def
  :cognitect.aws.transfer/PreAuthenticationLoginBanner
  (s/spec #(re-matches (re-pattern "[\\x09-\\x0D\\x20-\\x7E]*") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.transfer/AddressAllocationIds
  (s/coll-of :cognitect.aws.transfer/AddressAllocationId))

(s/def
  :cognitect.aws.transfer/ListAccessesRequest
  (s/keys
    :req-un
    [:cognitect.aws.transfer.ListAccessesRequest/ServerId]
    :opt-un
    [:cognitect.aws.transfer.ListAccessesRequest/NextToken
     :cognitect.aws.transfer.ListAccessesRequest/MaxResults]))

(s/def
  :cognitect.aws.transfer/ListSecurityPoliciesResponse
  (s/keys
    :req-un
    [:cognitect.aws.transfer.ListSecurityPoliciesResponse/SecurityPolicyNames]
    :opt-un
    [:cognitect.aws.transfer.ListSecurityPoliciesResponse/NextToken]))

(s/def
  :cognitect.aws.transfer/UpdateUserResponse
  (s/keys
    :req-un
    [:cognitect.aws.transfer.UpdateUserResponse/ServerId
     :cognitect.aws.transfer.UpdateUserResponse/UserName]))

(s/def
  :cognitect.aws.transfer/DescribeServerRequest
  (s/keys :req-un [:cognitect.aws.transfer.DescribeServerRequest/ServerId]))

(s/def
  :cognitect.aws.transfer/ListedUser
  (s/keys
    :req-un
    [:cognitect.aws.transfer.ListedUser/Arn]
    :opt-un
    [:cognitect.aws.transfer.ListedUser/SshPublicKeyCount
     :cognitect.aws.transfer.ListedUser/HomeDirectory
     :cognitect.aws.transfer.ListedUser/UserName
     :cognitect.aws.transfer.ListedUser/Role
     :cognitect.aws.transfer.ListedUser/HomeDirectoryType]))

(s/def
  :cognitect.aws.transfer/ListServersRequest
  (s/keys
    :opt-un
    [:cognitect.aws.transfer.ListServersRequest/NextToken
     :cognitect.aws.transfer.ListServersRequest/MaxResults]))

(s/def
  :cognitect.aws.transfer/DescribedSecurityPolicy
  (s/keys
    :req-un
    [:cognitect.aws.transfer.DescribedSecurityPolicy/SecurityPolicyName]
    :opt-un
    [:cognitect.aws.transfer.DescribedSecurityPolicy/SshKexs
     :cognitect.aws.transfer.DescribedSecurityPolicy/SshCiphers
     :cognitect.aws.transfer.DescribedSecurityPolicy/TlsCiphers
     :cognitect.aws.transfer.DescribedSecurityPolicy/Fips
     :cognitect.aws.transfer.DescribedSecurityPolicy/SshMacs]))

(s/def
  :cognitect.aws.transfer/SessionId
  (s/spec #(re-matches (re-pattern "^[\\w-]*$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.transfer/SecurityPolicyNames
  (s/coll-of :cognitect.aws.transfer/SecurityPolicyName))

(s/def :cognitect.aws.transfer/AddressAllocationId string?)

(s/def
  :cognitect.aws.transfer/CopyStepDetails
  (s/keys
    :opt-un
    [:cognitect.aws.transfer.CopyStepDetails/OverwriteExisting
     :cognitect.aws.transfer.CopyStepDetails/SourceFileLocation
     :cognitect.aws.transfer.CopyStepDetails/DestinationFileLocation
     :cognitect.aws.transfer.CopyStepDetails/Name]))

(s/def
  :cognitect.aws.transfer/MapTarget
  (s/spec #(re-matches (re-pattern "^/.*") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.transfer/UserDetails
  (s/keys
    :req-un
    [:cognitect.aws.transfer.UserDetails/UserName :cognitect.aws.transfer.UserDetails/ServerId]
    :opt-un
    [:cognitect.aws.transfer.UserDetails/SessionId]))

(s/def
  :cognitect.aws.transfer/VpcEndpointId
  (s/spec #(re-matches (re-pattern "^vpce-[0-9a-f]{17}$") %) :gen #(gen/string)))

(s/def :cognitect.aws.transfer/ListedAccesses (s/coll-of :cognitect.aws.transfer/ListedAccess))

(s/def
  :cognitect.aws.transfer/UpdateUserRequest
  (s/keys
    :req-un
    [:cognitect.aws.transfer.UpdateUserRequest/ServerId
     :cognitect.aws.transfer.UpdateUserRequest/UserName]
    :opt-un
    [:cognitect.aws.transfer.UpdateUserRequest/HomeDirectory
     :cognitect.aws.transfer.UpdateUserRequest/Role
     :cognitect.aws.transfer.UpdateUserRequest/PosixProfile
     :cognitect.aws.transfer.UpdateUserRequest/HomeDirectoryType
     :cognitect.aws.transfer.UpdateUserRequest/HomeDirectoryMappings
     :cognitect.aws.transfer.UpdateUserRequest/Policy]))

(s/def
  :cognitect.aws.transfer/Arn
  (s/spec #(re-matches (re-pattern "arn:.*") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.transfer/LoggingConfiguration
  (s/keys
    :opt-un
    [:cognitect.aws.transfer.LoggingConfiguration/LoggingRole
     :cognitect.aws.transfer.LoggingConfiguration/LogGroupName]))

(s/def
  :cognitect.aws.transfer/SecurityPolicyName
  (s/spec #(re-matches (re-pattern "TransferSecurityPolicy-.+") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.transfer/ListAccessesResponse
  (s/keys
    :req-un
    [:cognitect.aws.transfer.ListAccessesResponse/ServerId
     :cognitect.aws.transfer.ListAccessesResponse/Accesses]
    :opt-un
    [:cognitect.aws.transfer.ListAccessesResponse/NextToken]))

(s/def
  :cognitect.aws.transfer/DescribeServerResponse
  (s/keys :req-un [:cognitect.aws.transfer.DescribeServerResponse/Server]))

(s/def
  :cognitect.aws.transfer/MaxResults
  (s/spec (s/and int? #(<= 1 % 1000)) :gen #(gen/choose 1 1000)))

(s/def :cognitect.aws.transfer/DateImported inst?)

(s/def
  :cognitect.aws.transfer/UntagResourceRequest
  (s/keys
    :req-un
    [:cognitect.aws.transfer.UntagResourceRequest/Arn
     :cognitect.aws.transfer.UntagResourceRequest/TagKeys]))

(s/def
  :cognitect.aws.transfer/TagKey
  (s/spec
    (s/and string? #(>= 128 (count %)))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 0 128) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.transfer/NullableRole
  (s/spec #(re-matches (re-pattern "^$|arn:.*role/.*") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.transfer/FileLocation
  (s/keys
    :opt-un
    [:cognitect.aws.transfer.FileLocation/EfsFileLocation
     :cognitect.aws.transfer.FileLocation/S3FileLocation]))

(s/def :cognitect.aws.transfer/Response string?)

(s/def
  :cognitect.aws.transfer/DescribeWorkflowResponse
  (s/keys :req-un [:cognitect.aws.transfer.DescribeWorkflowResponse/Workflow]))

(s/def :cognitect.aws.transfer/Fips boolean?)

(s/def
  :cognitect.aws.transfer/WorkflowDetail
  (s/keys
    :req-un
    [:cognitect.aws.transfer.WorkflowDetail/WorkflowId
     :cognitect.aws.transfer.WorkflowDetail/ExecutionRole]))

(s/def
  :cognitect.aws.transfer/ImportSshPublicKeyResponse
  (s/keys
    :req-un
    [:cognitect.aws.transfer.ImportSshPublicKeyResponse/ServerId
     :cognitect.aws.transfer.ImportSshPublicKeyResponse/SshPublicKeyId
     :cognitect.aws.transfer.ImportSshPublicKeyResponse/UserName]))

(s/def
  :cognitect.aws.transfer/State
  (s/spec
    string?
    :gen
    #(s/gen #{"STARTING" "OFFLINE" "START_FAILED" "STOP_FAILED" "ONLINE" "STOPPING"})))

(s/def
  :cognitect.aws.transfer/DeleteStepDetails
  (s/keys
    :opt-un
    [:cognitect.aws.transfer.DeleteStepDetails/SourceFileLocation
     :cognitect.aws.transfer.DeleteStepDetails/Name]))

(s/def
  :cognitect.aws.transfer/S3Tags
  (s/coll-of :cognitect.aws.transfer/S3Tag :min-count 1 :max-count 10))

(s/def
  :cognitect.aws.transfer/S3VersionId
  (s/spec #(re-matches (re-pattern "^.+$") %) :gen #(gen/string)))

(s/def :cognitect.aws.transfer/Protocol (s/spec string? :gen #(s/gen #{"FTP" "SFTP" "FTPS"})))

(s/def
  :cognitect.aws.transfer/ListExecutionsResponse
  (s/keys
    :req-un
    [:cognitect.aws.transfer.ListExecutionsResponse/WorkflowId
     :cognitect.aws.transfer.ListExecutionsResponse/Executions]
    :opt-un
    [:cognitect.aws.transfer.ListExecutionsResponse/NextToken]))

(s/def
  :cognitect.aws.transfer/EfsFileLocation
  (s/keys
    :opt-un
    [:cognitect.aws.transfer.EfsFileLocation/Path
     :cognitect.aws.transfer.EfsFileLocation/FileSystemId]))

(s/def
  :cognitect.aws.transfer/ListUsersResponse
  (s/keys
    :req-un
    [:cognitect.aws.transfer.ListUsersResponse/ServerId
     :cognitect.aws.transfer.ListUsersResponse/Users]
    :opt-un
    [:cognitect.aws.transfer.ListUsersResponse/NextToken]))

(s/def
  :cognitect.aws.transfer/SendWorkflowStepStateRequest
  (s/keys
    :req-un
    [:cognitect.aws.transfer.SendWorkflowStepStateRequest/WorkflowId
     :cognitect.aws.transfer.SendWorkflowStepStateRequest/ExecutionId
     :cognitect.aws.transfer.SendWorkflowStepStateRequest/Token
     :cognitect.aws.transfer.SendWorkflowStepStateRequest/Status]))

(s/def :cognitect.aws.transfer/ListedWorkflows (s/coll-of :cognitect.aws.transfer/ListedWorkflow))

(s/def
  :cognitect.aws.transfer/UpdateServerRequest
  (s/keys
    :req-un
    [:cognitect.aws.transfer.UpdateServerRequest/ServerId]
    :opt-un
    [:cognitect.aws.transfer.UpdateServerRequest/WorkflowDetails
     :cognitect.aws.transfer.UpdateServerRequest/SecurityPolicyName
     :cognitect.aws.transfer.UpdateServerRequest/Protocols
     :cognitect.aws.transfer.UpdateServerRequest/HostKey
     :cognitect.aws.transfer.UpdateServerRequest/EndpointType
     :cognitect.aws.transfer.UpdateServerRequest/EndpointDetails
     :cognitect.aws.transfer.UpdateServerRequest/PostAuthenticationLoginBanner
     :cognitect.aws.transfer.UpdateServerRequest/ProtocolDetails
     :cognitect.aws.transfer.UpdateServerRequest/PreAuthenticationLoginBanner
     :cognitect.aws.transfer.UpdateServerRequest/IdentityProviderDetails
     :cognitect.aws.transfer.UpdateServerRequest/LoggingRole
     :cognitect.aws.transfer.UpdateServerRequest/Certificate]))

(s/def
  :cognitect.aws.transfer/DescribeSecurityPolicyResponse
  (s/keys :req-un [:cognitect.aws.transfer.DescribeSecurityPolicyResponse/SecurityPolicy]))

(s/def :cognitect.aws.transfer/HomeDirectoryType (s/spec string? :gen #(s/gen #{"PATH" "LOGICAL"})))

(s/def
  :cognitect.aws.transfer/ListUsersRequest
  (s/keys
    :req-un
    [:cognitect.aws.transfer.ListUsersRequest/ServerId]
    :opt-un
    [:cognitect.aws.transfer.ListUsersRequest/NextToken
     :cognitect.aws.transfer.ListUsersRequest/MaxResults]))

(s/def
  :cognitect.aws.transfer/Role
  (s/spec #(re-matches (re-pattern "arn:.*role/.*") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.transfer/S3FileLocation
  (s/keys
    :opt-un
    [:cognitect.aws.transfer.S3FileLocation/Bucket
     :cognitect.aws.transfer.S3FileLocation/VersionId
     :cognitect.aws.transfer.S3FileLocation/Key
     :cognitect.aws.transfer.S3FileLocation/Etag]))

(s/def
  :cognitect.aws.transfer/WorkflowDescription
  (s/spec #(re-matches (re-pattern "^[\\w- ]*$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.transfer/MapEntry
  (s/spec #(re-matches (re-pattern "^/.*") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.transfer/EfsFileSystemId
  (s/spec
    #(re-matches
      (re-pattern
        "^(arn:aws[-a-z]*:elasticfilesystem:[0-9a-z-:]+:(access-point/fsap|file-system/fs)-[0-9a-f]{8,40}|fs(ap)?-[0-9a-f]{8,40})$")
      %)
    :gen
    #(gen/string)))

(s/def
  :cognitect.aws.transfer/SourceIp
  (s/spec
    #(re-matches (re-pattern "^\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}$") %)
    :gen
    #(gen/string)))

(s/def :cognitect.aws.transfer/SecurityGroupIds (s/coll-of :cognitect.aws.transfer/SecurityGroupId))

(s/def
  :cognitect.aws.transfer/ListedServer
  (s/keys
    :req-un
    [:cognitect.aws.transfer.ListedServer/Arn]
    :opt-un
    [:cognitect.aws.transfer.ListedServer/EndpointType
     :cognitect.aws.transfer.ListedServer/Domain
     :cognitect.aws.transfer.ListedServer/LoggingRole
     :cognitect.aws.transfer.ListedServer/IdentityProviderType
     :cognitect.aws.transfer.ListedServer/State
     :cognitect.aws.transfer.ListedServer/UserCount
     :cognitect.aws.transfer.ListedServer/ServerId]))

(s/def
  :cognitect.aws.transfer/TagKeys
  (s/coll-of :cognitect.aws.transfer/TagKey :min-count 1 :max-count 50))

(s/def
  :cognitect.aws.transfer/DescribeAccessResponse
  (s/keys
    :req-un
    [:cognitect.aws.transfer.DescribeAccessResponse/ServerId
     :cognitect.aws.transfer.DescribeAccessResponse/Access]))

(s/def
  :cognitect.aws.transfer/CreateServerRequest
  (s/keys
    :opt-un
    [:cognitect.aws.transfer.CreateServerRequest/WorkflowDetails
     :cognitect.aws.transfer.CreateServerRequest/SecurityPolicyName
     :cognitect.aws.transfer.CreateServerRequest/Protocols
     :cognitect.aws.transfer.CreateServerRequest/HostKey
     :cognitect.aws.transfer.CreateServerRequest/EndpointType
     :cognitect.aws.transfer.CreateServerRequest/EndpointDetails
     :cognitect.aws.transfer.CreateServerRequest/Tags
     :cognitect.aws.transfer.CreateServerRequest/PostAuthenticationLoginBanner
     :cognitect.aws.transfer.CreateServerRequest/ProtocolDetails
     :cognitect.aws.transfer.CreateServerRequest/PreAuthenticationLoginBanner
     :cognitect.aws.transfer.CreateServerRequest/IdentityProviderDetails
     :cognitect.aws.transfer.CreateServerRequest/Domain
     :cognitect.aws.transfer.CreateServerRequest/LoggingRole
     :cognitect.aws.transfer.CreateServerRequest/IdentityProviderType
     :cognitect.aws.transfer.CreateServerRequest/Certificate]))

(s/def
  :cognitect.aws.transfer/ExecutionStepResult
  (s/keys
    :opt-un
    [:cognitect.aws.transfer.ExecutionStepResult/Outputs
     :cognitect.aws.transfer.ExecutionStepResult/Error
     :cognitect.aws.transfer.ExecutionStepResult/StepType]))

(s/def
  :cognitect.aws.transfer/ListServersResponse
  (s/keys
    :req-un
    [:cognitect.aws.transfer.ListServersResponse/Servers]
    :opt-un
    [:cognitect.aws.transfer.ListServersResponse/NextToken]))

(s/def
  :cognitect.aws.transfer/DeleteAccessRequest
  (s/keys
    :req-un
    [:cognitect.aws.transfer.DeleteAccessRequest/ServerId
     :cognitect.aws.transfer.DeleteAccessRequest/ExternalId]))

(s/def :cognitect.aws.transfer/HostKeyFingerprint string?)

(s/def
  :cognitect.aws.transfer/SshPublicKeyId
  (s/spec #(re-matches (re-pattern "^key-[0-9a-f]{17}$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.transfer/CustomStepStatus
  (s/spec string? :gen #(s/gen #{"SUCCESS" "FAILURE"})))

(s/def
  :cognitect.aws.transfer/ListTagsForResourceResponse
  (s/keys
    :opt-un
    [:cognitect.aws.transfer.ListTagsForResourceResponse/Tags
     :cognitect.aws.transfer.ListTagsForResourceResponse/NextToken
     :cognitect.aws.transfer.ListTagsForResourceResponse/Arn]))

(s/def
  :cognitect.aws.transfer/DeleteServerRequest
  (s/keys :req-un [:cognitect.aws.transfer.DeleteServerRequest/ServerId]))

(s/def :cognitect.aws.transfer/ListedUsers (s/coll-of :cognitect.aws.transfer/ListedUser))

(s/def
  :cognitect.aws.transfer/UpdateAccessRequest
  (s/keys
    :req-un
    [:cognitect.aws.transfer.UpdateAccessRequest/ServerId
     :cognitect.aws.transfer.UpdateAccessRequest/ExternalId]
    :opt-un
    [:cognitect.aws.transfer.UpdateAccessRequest/HomeDirectory
     :cognitect.aws.transfer.UpdateAccessRequest/Role
     :cognitect.aws.transfer.UpdateAccessRequest/PosixProfile
     :cognitect.aws.transfer.UpdateAccessRequest/HomeDirectoryType
     :cognitect.aws.transfer.UpdateAccessRequest/HomeDirectoryMappings
     :cognitect.aws.transfer.UpdateAccessRequest/Policy]))

(s/def :cognitect.aws.transfer/VpcId string?)

(s/def :cognitect.aws.transfer/OverwriteExisting (s/spec string? :gen #(s/gen #{"TRUE" "FALSE"})))

(s/def
  :cognitect.aws.transfer/S3Etag
  (s/spec #(re-matches (re-pattern "^.+$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.transfer/TestIdentityProviderRequest
  (s/keys
    :req-un
    [:cognitect.aws.transfer.TestIdentityProviderRequest/ServerId
     :cognitect.aws.transfer.TestIdentityProviderRequest/UserName]
    :opt-un
    [:cognitect.aws.transfer.TestIdentityProviderRequest/ServerProtocol
     :cognitect.aws.transfer.TestIdentityProviderRequest/SourceIp
     :cognitect.aws.transfer.TestIdentityProviderRequest/UserPassword]))

(s/def
  :cognitect.aws.transfer/SshPublicKeyCount
  (s/spec int? :gen #(gen/choose Long/MIN_VALUE Long/MAX_VALUE)))

(s/def
  :cognitect.aws.transfer/ProtocolDetails
  (s/keys
    :opt-un
    [:cognitect.aws.transfer.ProtocolDetails/TlsSessionResumptionMode
     :cognitect.aws.transfer.ProtocolDetails/PassiveIp]))

(s/def
  :cognitect.aws.transfer/Policy
  (s/spec
    (s/and string? #(>= 2048 (count %)))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 0 2048) #(gen/vector (gen/char-alpha) %))))))

(s/def :cognitect.aws.transfer.S3InputFileLocation/Bucket :cognitect.aws.transfer/S3Bucket)

(s/def :cognitect.aws.transfer.S3InputFileLocation/Key :cognitect.aws.transfer/S3Key)

(s/def :cognitect.aws.transfer.TagResourceRequest/Arn :cognitect.aws.transfer/Arn)

(s/def :cognitect.aws.transfer.TagResourceRequest/Tags :cognitect.aws.transfer/Tags)

(s/def :cognitect.aws.transfer.PosixProfile/Uid :cognitect.aws.transfer/PosixId)

(s/def :cognitect.aws.transfer.PosixProfile/Gid :cognitect.aws.transfer/PosixId)

(s/def :cognitect.aws.transfer.PosixProfile/SecondaryGids :cognitect.aws.transfer/SecondaryGids)

(s/def :cognitect.aws.transfer.CustomStepDetails/Name :cognitect.aws.transfer/WorkflowStepName)

(s/def :cognitect.aws.transfer.CustomStepDetails/Target :cognitect.aws.transfer/CustomStepTarget)

(s/def
  :cognitect.aws.transfer.CustomStepDetails/TimeoutSeconds
  :cognitect.aws.transfer/CustomStepTimeoutSeconds)

(s/def
  :cognitect.aws.transfer.CustomStepDetails/SourceFileLocation
  :cognitect.aws.transfer/SourceFileLocation)

(s/def :cognitect.aws.transfer.StartServerRequest/ServerId :cognitect.aws.transfer/ServerId)

(s/def :cognitect.aws.transfer.ListedWorkflow/WorkflowId :cognitect.aws.transfer/WorkflowId)

(s/def
  :cognitect.aws.transfer.ListedWorkflow/Description
  :cognitect.aws.transfer/WorkflowDescription)

(s/def :cognitect.aws.transfer.ListedWorkflow/Arn :cognitect.aws.transfer/Arn)

(s/def :cognitect.aws.transfer.ListExecutionsRequest/MaxResults :cognitect.aws.transfer/MaxResults)

(s/def :cognitect.aws.transfer.ListExecutionsRequest/NextToken :cognitect.aws.transfer/NextToken)

(s/def :cognitect.aws.transfer.ListExecutionsRequest/WorkflowId :cognitect.aws.transfer/WorkflowId)

(s/def :cognitect.aws.transfer.CreateUserResponse/ServerId :cognitect.aws.transfer/ServerId)

(s/def :cognitect.aws.transfer.CreateUserResponse/UserName :cognitect.aws.transfer/UserName)

(s/def :cognitect.aws.transfer.ExecutionResults/Steps :cognitect.aws.transfer/ExecutionStepResults)

(s/def
  :cognitect.aws.transfer.ExecutionResults/OnExceptionSteps
  :cognitect.aws.transfer/ExecutionStepResults)

(s/def :cognitect.aws.transfer.ImportSshPublicKeyRequest/ServerId :cognitect.aws.transfer/ServerId)

(s/def
  :cognitect.aws.transfer.ImportSshPublicKeyRequest/SshPublicKeyBody
  :cognitect.aws.transfer/SshPublicKeyBody)

(s/def :cognitect.aws.transfer.ImportSshPublicKeyRequest/UserName :cognitect.aws.transfer/UserName)

(s/def :cognitect.aws.transfer.Tag/Key :cognitect.aws.transfer/TagKey)

(s/def :cognitect.aws.transfer.Tag/Value :cognitect.aws.transfer/TagValue)

(s/def :cognitect.aws.transfer.DescribeAccessRequest/ServerId :cognitect.aws.transfer/ServerId)

(s/def :cognitect.aws.transfer.DescribeAccessRequest/ExternalId :cognitect.aws.transfer/ExternalId)

(s/def
  :cognitect.aws.transfer.DescribedServer/IdentityProviderType
  :cognitect.aws.transfer/IdentityProviderType)

(s/def :cognitect.aws.transfer.DescribedServer/Domain :cognitect.aws.transfer/Domain)

(s/def :cognitect.aws.transfer.DescribedServer/Tags :cognitect.aws.transfer/Tags)

(s/def
  :cognitect.aws.transfer.DescribedServer/EndpointDetails
  :cognitect.aws.transfer/EndpointDetails)

(s/def :cognitect.aws.transfer.DescribedServer/UserCount :cognitect.aws.transfer/UserCount)

(s/def
  :cognitect.aws.transfer.DescribedServer/IdentityProviderDetails
  :cognitect.aws.transfer/IdentityProviderDetails)

(s/def :cognitect.aws.transfer.DescribedServer/LoggingRole :cognitect.aws.transfer/Role)

(s/def
  :cognitect.aws.transfer.DescribedServer/PostAuthenticationLoginBanner
  :cognitect.aws.transfer/PostAuthenticationLoginBanner)

(s/def :cognitect.aws.transfer.DescribedServer/Protocols :cognitect.aws.transfer/Protocols)

(s/def :cognitect.aws.transfer.DescribedServer/ServerId :cognitect.aws.transfer/ServerId)

(s/def
  :cognitect.aws.transfer.DescribedServer/WorkflowDetails
  :cognitect.aws.transfer/WorkflowDetails)

(s/def :cognitect.aws.transfer.DescribedServer/EndpointType :cognitect.aws.transfer/EndpointType)

(s/def :cognitect.aws.transfer.DescribedServer/Certificate :cognitect.aws.transfer/Certificate)

(s/def
  :cognitect.aws.transfer.DescribedServer/PreAuthenticationLoginBanner
  :cognitect.aws.transfer/PreAuthenticationLoginBanner)

(s/def :cognitect.aws.transfer.DescribedServer/Arn :cognitect.aws.transfer/Arn)

(s/def
  :cognitect.aws.transfer.DescribedServer/SecurityPolicyName
  :cognitect.aws.transfer/SecurityPolicyName)

(s/def :cognitect.aws.transfer.DescribedServer/State :cognitect.aws.transfer/State)

(s/def
  :cognitect.aws.transfer.DescribedServer/HostKeyFingerprint
  :cognitect.aws.transfer/HostKeyFingerprint)

(s/def
  :cognitect.aws.transfer.DescribedServer/ProtocolDetails
  :cognitect.aws.transfer/ProtocolDetails)

(s/def
  :cognitect.aws.transfer.CreateWorkflowRequest/Description
  :cognitect.aws.transfer/WorkflowDescription)

(s/def :cognitect.aws.transfer.CreateWorkflowRequest/Steps :cognitect.aws.transfer/WorkflowSteps)

(s/def
  :cognitect.aws.transfer.CreateWorkflowRequest/OnExceptionSteps
  :cognitect.aws.transfer/WorkflowSteps)

(s/def :cognitect.aws.transfer.CreateWorkflowRequest/Tags :cognitect.aws.transfer/Tags)

(s/def :cognitect.aws.transfer.ExecutionError/Type :cognitect.aws.transfer/ExecutionErrorType)

(s/def :cognitect.aws.transfer.ExecutionError/Message :cognitect.aws.transfer/ExecutionErrorMessage)

(s/def
  :cognitect.aws.transfer.InputFileLocation/S3FileLocation
  :cognitect.aws.transfer/S3InputFileLocation)

(s/def
  :cognitect.aws.transfer.InputFileLocation/EfsFileLocation
  :cognitect.aws.transfer/EfsFileLocation)

(s/def
  :cognitect.aws.transfer.DescribeSecurityPolicyRequest/SecurityPolicyName
  :cognitect.aws.transfer/SecurityPolicyName)

(s/def
  :cognitect.aws.transfer.DescribeExecutionResponse/WorkflowId
  :cognitect.aws.transfer/WorkflowId)

(s/def
  :cognitect.aws.transfer.DescribeExecutionResponse/Execution
  :cognitect.aws.transfer/DescribedExecution)

(s/def :cognitect.aws.transfer.DeleteWorkflowRequest/WorkflowId :cognitect.aws.transfer/WorkflowId)

(s/def :cognitect.aws.transfer.CreateWorkflowResponse/WorkflowId :cognitect.aws.transfer/WorkflowId)

(s/def :cognitect.aws.transfer.DescribedWorkflow/Arn :cognitect.aws.transfer/Arn)

(s/def
  :cognitect.aws.transfer.DescribedWorkflow/Description
  :cognitect.aws.transfer/WorkflowDescription)

(s/def :cognitect.aws.transfer.DescribedWorkflow/Steps :cognitect.aws.transfer/WorkflowSteps)

(s/def
  :cognitect.aws.transfer.DescribedWorkflow/OnExceptionSteps
  :cognitect.aws.transfer/WorkflowSteps)

(s/def :cognitect.aws.transfer.DescribedWorkflow/WorkflowId :cognitect.aws.transfer/WorkflowId)

(s/def :cognitect.aws.transfer.DescribedWorkflow/Tags :cognitect.aws.transfer/Tags)

(s/def
  :cognitect.aws.transfer.DescribeWorkflowRequest/WorkflowId
  :cognitect.aws.transfer/WorkflowId)

(s/def :cognitect.aws.transfer.DescribedUser/PosixProfile :cognitect.aws.transfer/PosixProfile)

(s/def :cognitect.aws.transfer.DescribedUser/SshPublicKeys :cognitect.aws.transfer/SshPublicKeys)

(s/def :cognitect.aws.transfer.DescribedUser/HomeDirectory :cognitect.aws.transfer/HomeDirectory)

(s/def :cognitect.aws.transfer.DescribedUser/Tags :cognitect.aws.transfer/Tags)

(s/def :cognitect.aws.transfer.DescribedUser/UserName :cognitect.aws.transfer/UserName)

(s/def
  :cognitect.aws.transfer.DescribedUser/HomeDirectoryMappings
  :cognitect.aws.transfer/HomeDirectoryMappings)

(s/def :cognitect.aws.transfer.DescribedUser/Arn :cognitect.aws.transfer/Arn)

(s/def
  :cognitect.aws.transfer.DescribedUser/HomeDirectoryType
  :cognitect.aws.transfer/HomeDirectoryType)

(s/def :cognitect.aws.transfer.DescribedUser/Role :cognitect.aws.transfer/Role)

(s/def :cognitect.aws.transfer.DescribedUser/Policy :cognitect.aws.transfer/Policy)

(s/def :cognitect.aws.transfer.CreateAccessResponse/ServerId :cognitect.aws.transfer/ServerId)

(s/def :cognitect.aws.transfer.CreateAccessResponse/ExternalId :cognitect.aws.transfer/ExternalId)

(s/def
  :cognitect.aws.transfer.CreateAccessRequest/HomeDirectory
  :cognitect.aws.transfer/HomeDirectory)

(s/def
  :cognitect.aws.transfer.CreateAccessRequest/HomeDirectoryType
  :cognitect.aws.transfer/HomeDirectoryType)

(s/def
  :cognitect.aws.transfer.CreateAccessRequest/HomeDirectoryMappings
  :cognitect.aws.transfer/HomeDirectoryMappings)

(s/def :cognitect.aws.transfer.CreateAccessRequest/Policy :cognitect.aws.transfer/Policy)

(s/def
  :cognitect.aws.transfer.CreateAccessRequest/PosixProfile
  :cognitect.aws.transfer/PosixProfile)

(s/def :cognitect.aws.transfer.CreateAccessRequest/Role :cognitect.aws.transfer/Role)

(s/def :cognitect.aws.transfer.CreateAccessRequest/ServerId :cognitect.aws.transfer/ServerId)

(s/def :cognitect.aws.transfer.CreateAccessRequest/ExternalId :cognitect.aws.transfer/ExternalId)

(s/def :cognitect.aws.transfer.WorkflowStep/Type :cognitect.aws.transfer/WorkflowStepType)

(s/def :cognitect.aws.transfer.WorkflowStep/CopyStepDetails :cognitect.aws.transfer/CopyStepDetails)

(s/def
  :cognitect.aws.transfer.WorkflowStep/CustomStepDetails
  :cognitect.aws.transfer/CustomStepDetails)

(s/def
  :cognitect.aws.transfer.WorkflowStep/DeleteStepDetails
  :cognitect.aws.transfer/DeleteStepDetails)

(s/def :cognitect.aws.transfer.WorkflowStep/TagStepDetails :cognitect.aws.transfer/TagStepDetails)

(s/def
  :cognitect.aws.transfer.EndpointDetails/AddressAllocationIds
  :cognitect.aws.transfer/AddressAllocationIds)

(s/def :cognitect.aws.transfer.EndpointDetails/SubnetIds :cognitect.aws.transfer/SubnetIds)

(s/def :cognitect.aws.transfer.EndpointDetails/VpcEndpointId :cognitect.aws.transfer/VpcEndpointId)

(s/def :cognitect.aws.transfer.EndpointDetails/VpcId :cognitect.aws.transfer/VpcId)

(s/def
  :cognitect.aws.transfer.EndpointDetails/SecurityGroupIds
  :cognitect.aws.transfer/SecurityGroupIds)

(s/def :cognitect.aws.transfer.ServiceMetadata/UserDetails :cognitect.aws.transfer/UserDetails)

(s/def
  :cognitect.aws.transfer.DescribeExecutionRequest/ExecutionId
  :cognitect.aws.transfer/ExecutionId)

(s/def
  :cognitect.aws.transfer.DescribeExecutionRequest/WorkflowId
  :cognitect.aws.transfer/WorkflowId)

(s/def :cognitect.aws.transfer.StopServerRequest/ServerId :cognitect.aws.transfer/ServerId)

(s/def
  :cognitect.aws.transfer.ListSecurityPoliciesRequest/MaxResults
  :cognitect.aws.transfer/MaxResults)

(s/def
  :cognitect.aws.transfer.ListSecurityPoliciesRequest/NextToken
  :cognitect.aws.transfer/NextToken)

(s/def :cognitect.aws.transfer.IdentityProviderDetails/Url :cognitect.aws.transfer/Url)

(s/def :cognitect.aws.transfer.IdentityProviderDetails/InvocationRole :cognitect.aws.transfer/Role)

(s/def
  :cognitect.aws.transfer.IdentityProviderDetails/DirectoryId
  :cognitect.aws.transfer/DirectoryId)

(s/def :cognitect.aws.transfer.IdentityProviderDetails/Function :cognitect.aws.transfer/Function)

(s/def :cognitect.aws.transfer.ListedAccess/HomeDirectory :cognitect.aws.transfer/HomeDirectory)

(s/def
  :cognitect.aws.transfer.ListedAccess/HomeDirectoryType
  :cognitect.aws.transfer/HomeDirectoryType)

(s/def :cognitect.aws.transfer.ListedAccess/Role :cognitect.aws.transfer/Role)

(s/def :cognitect.aws.transfer.ListedAccess/ExternalId :cognitect.aws.transfer/ExternalId)

(s/def :cognitect.aws.transfer.DeleteSshPublicKeyRequest/ServerId :cognitect.aws.transfer/ServerId)

(s/def
  :cognitect.aws.transfer.DeleteSshPublicKeyRequest/SshPublicKeyId
  :cognitect.aws.transfer/SshPublicKeyId)

(s/def :cognitect.aws.transfer.DeleteSshPublicKeyRequest/UserName :cognitect.aws.transfer/UserName)

(s/def :cognitect.aws.transfer.DescribeUserRequest/ServerId :cognitect.aws.transfer/ServerId)

(s/def :cognitect.aws.transfer.DescribeUserRequest/UserName :cognitect.aws.transfer/UserName)

(s/def :cognitect.aws.transfer.CreateServerResponse/ServerId :cognitect.aws.transfer/ServerId)

(s/def :cognitect.aws.transfer.UpdateAccessResponse/ServerId :cognitect.aws.transfer/ServerId)

(s/def :cognitect.aws.transfer.UpdateAccessResponse/ExternalId :cognitect.aws.transfer/ExternalId)

(s/def :cognitect.aws.transfer.ListedExecution/ExecutionId :cognitect.aws.transfer/ExecutionId)

(s/def
  :cognitect.aws.transfer.ListedExecution/InitialFileLocation
  :cognitect.aws.transfer/FileLocation)

(s/def
  :cognitect.aws.transfer.ListedExecution/ServiceMetadata
  :cognitect.aws.transfer/ServiceMetadata)

(s/def :cognitect.aws.transfer.ListedExecution/Status :cognitect.aws.transfer/ExecutionStatus)

(s/def :cognitect.aws.transfer.ListWorkflowsRequest/MaxResults :cognitect.aws.transfer/MaxResults)

(s/def :cognitect.aws.transfer.ListWorkflowsRequest/NextToken :cognitect.aws.transfer/NextToken)

(s/def :cognitect.aws.transfer.DescribedAccess/HomeDirectory :cognitect.aws.transfer/HomeDirectory)

(s/def
  :cognitect.aws.transfer.DescribedAccess/HomeDirectoryMappings
  :cognitect.aws.transfer/HomeDirectoryMappings)

(s/def
  :cognitect.aws.transfer.DescribedAccess/HomeDirectoryType
  :cognitect.aws.transfer/HomeDirectoryType)

(s/def :cognitect.aws.transfer.DescribedAccess/Policy :cognitect.aws.transfer/Policy)

(s/def :cognitect.aws.transfer.DescribedAccess/PosixProfile :cognitect.aws.transfer/PosixProfile)

(s/def :cognitect.aws.transfer.DescribedAccess/Role :cognitect.aws.transfer/Role)

(s/def :cognitect.aws.transfer.DescribedAccess/ExternalId :cognitect.aws.transfer/ExternalId)

(s/def :cognitect.aws.transfer.DescribedExecution/ExecutionId :cognitect.aws.transfer/ExecutionId)

(s/def
  :cognitect.aws.transfer.DescribedExecution/InitialFileLocation
  :cognitect.aws.transfer/FileLocation)

(s/def
  :cognitect.aws.transfer.DescribedExecution/ServiceMetadata
  :cognitect.aws.transfer/ServiceMetadata)

(s/def :cognitect.aws.transfer.DescribedExecution/ExecutionRole :cognitect.aws.transfer/Role)

(s/def
  :cognitect.aws.transfer.DescribedExecution/LoggingConfiguration
  :cognitect.aws.transfer/LoggingConfiguration)

(s/def :cognitect.aws.transfer.DescribedExecution/PosixProfile :cognitect.aws.transfer/PosixProfile)

(s/def :cognitect.aws.transfer.DescribedExecution/Status :cognitect.aws.transfer/ExecutionStatus)

(s/def :cognitect.aws.transfer.DescribedExecution/Results :cognitect.aws.transfer/ExecutionResults)

(s/def :cognitect.aws.transfer.HomeDirectoryMapEntry/Entry :cognitect.aws.transfer/MapEntry)

(s/def :cognitect.aws.transfer.HomeDirectoryMapEntry/Target :cognitect.aws.transfer/MapTarget)

(s/def :cognitect.aws.transfer.S3Tag/Key :cognitect.aws.transfer/S3TagKey)

(s/def :cognitect.aws.transfer.S3Tag/Value :cognitect.aws.transfer/S3TagValue)

(s/def :cognitect.aws.transfer.ListTagsForResourceRequest/Arn :cognitect.aws.transfer/Arn)

(s/def
  :cognitect.aws.transfer.ListTagsForResourceRequest/MaxResults
  :cognitect.aws.transfer/MaxResults)

(s/def
  :cognitect.aws.transfer.ListTagsForResourceRequest/NextToken
  :cognitect.aws.transfer/NextToken)

(s/def :cognitect.aws.transfer.TagStepDetails/Name :cognitect.aws.transfer/WorkflowStepName)

(s/def :cognitect.aws.transfer.TagStepDetails/Tags :cognitect.aws.transfer/S3Tags)

(s/def
  :cognitect.aws.transfer.TagStepDetails/SourceFileLocation
  :cognitect.aws.transfer/SourceFileLocation)

(s/def
  :cognitect.aws.transfer.WorkflowDetails/OnUpload
  :cognitect.aws.transfer/OnUploadWorkflowDetails)

(s/def :cognitect.aws.transfer.ListWorkflowsResponse/NextToken :cognitect.aws.transfer/NextToken)

(s/def
  :cognitect.aws.transfer.ListWorkflowsResponse/Workflows
  :cognitect.aws.transfer/ListedWorkflows)

(s/def :cognitect.aws.transfer.UpdateServerResponse/ServerId :cognitect.aws.transfer/ServerId)

(s/def :cognitect.aws.transfer.DescribeUserResponse/ServerId :cognitect.aws.transfer/ServerId)

(s/def :cognitect.aws.transfer.DescribeUserResponse/User :cognitect.aws.transfer/DescribedUser)

(s/def
  :cognitect.aws.transfer.TestIdentityProviderResponse/Response
  :cognitect.aws.transfer/Response)

(s/def
  :cognitect.aws.transfer.TestIdentityProviderResponse/StatusCode
  :cognitect.aws.transfer/StatusCode)

(s/def :cognitect.aws.transfer.TestIdentityProviderResponse/Message :cognitect.aws.transfer/Message)

(s/def :cognitect.aws.transfer.TestIdentityProviderResponse/Url :cognitect.aws.transfer/Url)

(s/def :cognitect.aws.transfer.DeleteUserRequest/ServerId :cognitect.aws.transfer/ServerId)

(s/def :cognitect.aws.transfer.DeleteUserRequest/UserName :cognitect.aws.transfer/UserName)

(s/def :cognitect.aws.transfer.SshPublicKey/DateImported :cognitect.aws.transfer/DateImported)

(s/def
  :cognitect.aws.transfer.SshPublicKey/SshPublicKeyBody
  :cognitect.aws.transfer/SshPublicKeyBody)

(s/def :cognitect.aws.transfer.SshPublicKey/SshPublicKeyId :cognitect.aws.transfer/SshPublicKeyId)

(s/def :cognitect.aws.transfer.CreateUserRequest/PosixProfile :cognitect.aws.transfer/PosixProfile)

(s/def
  :cognitect.aws.transfer.CreateUserRequest/SshPublicKeyBody
  :cognitect.aws.transfer/SshPublicKeyBody)

(s/def
  :cognitect.aws.transfer.CreateUserRequest/HomeDirectory
  :cognitect.aws.transfer/HomeDirectory)

(s/def :cognitect.aws.transfer.CreateUserRequest/Tags :cognitect.aws.transfer/Tags)

(s/def :cognitect.aws.transfer.CreateUserRequest/ServerId :cognitect.aws.transfer/ServerId)

(s/def :cognitect.aws.transfer.CreateUserRequest/UserName :cognitect.aws.transfer/UserName)

(s/def
  :cognitect.aws.transfer.CreateUserRequest/HomeDirectoryMappings
  :cognitect.aws.transfer/HomeDirectoryMappings)

(s/def
  :cognitect.aws.transfer.CreateUserRequest/HomeDirectoryType
  :cognitect.aws.transfer/HomeDirectoryType)

(s/def :cognitect.aws.transfer.CreateUserRequest/Role :cognitect.aws.transfer/Role)

(s/def :cognitect.aws.transfer.CreateUserRequest/Policy :cognitect.aws.transfer/Policy)

(s/def :cognitect.aws.transfer.ListAccessesRequest/MaxResults :cognitect.aws.transfer/MaxResults)

(s/def :cognitect.aws.transfer.ListAccessesRequest/NextToken :cognitect.aws.transfer/NextToken)

(s/def :cognitect.aws.transfer.ListAccessesRequest/ServerId :cognitect.aws.transfer/ServerId)

(s/def
  :cognitect.aws.transfer.ListSecurityPoliciesResponse/NextToken
  :cognitect.aws.transfer/NextToken)

(s/def
  :cognitect.aws.transfer.ListSecurityPoliciesResponse/SecurityPolicyNames
  :cognitect.aws.transfer/SecurityPolicyNames)

(s/def :cognitect.aws.transfer.UpdateUserResponse/ServerId :cognitect.aws.transfer/ServerId)

(s/def :cognitect.aws.transfer.UpdateUserResponse/UserName :cognitect.aws.transfer/UserName)

(s/def :cognitect.aws.transfer.DescribeServerRequest/ServerId :cognitect.aws.transfer/ServerId)

(s/def :cognitect.aws.transfer.ListedUser/Arn :cognitect.aws.transfer/Arn)

(s/def :cognitect.aws.transfer.ListedUser/HomeDirectory :cognitect.aws.transfer/HomeDirectory)

(s/def
  :cognitect.aws.transfer.ListedUser/HomeDirectoryType
  :cognitect.aws.transfer/HomeDirectoryType)

(s/def :cognitect.aws.transfer.ListedUser/Role :cognitect.aws.transfer/Role)

(s/def
  :cognitect.aws.transfer.ListedUser/SshPublicKeyCount
  :cognitect.aws.transfer/SshPublicKeyCount)

(s/def :cognitect.aws.transfer.ListedUser/UserName :cognitect.aws.transfer/UserName)

(s/def :cognitect.aws.transfer.ListServersRequest/MaxResults :cognitect.aws.transfer/MaxResults)

(s/def :cognitect.aws.transfer.ListServersRequest/NextToken :cognitect.aws.transfer/NextToken)

(s/def :cognitect.aws.transfer.DescribedSecurityPolicy/Fips :cognitect.aws.transfer/Fips)

(s/def
  :cognitect.aws.transfer.DescribedSecurityPolicy/SecurityPolicyName
  :cognitect.aws.transfer/SecurityPolicyName)

(s/def
  :cognitect.aws.transfer.DescribedSecurityPolicy/SshCiphers
  :cognitect.aws.transfer/SecurityPolicyOptions)

(s/def
  :cognitect.aws.transfer.DescribedSecurityPolicy/SshKexs
  :cognitect.aws.transfer/SecurityPolicyOptions)

(s/def
  :cognitect.aws.transfer.DescribedSecurityPolicy/SshMacs
  :cognitect.aws.transfer/SecurityPolicyOptions)

(s/def
  :cognitect.aws.transfer.DescribedSecurityPolicy/TlsCiphers
  :cognitect.aws.transfer/SecurityPolicyOptions)

(s/def :cognitect.aws.transfer.CopyStepDetails/Name :cognitect.aws.transfer/WorkflowStepName)

(s/def
  :cognitect.aws.transfer.CopyStepDetails/DestinationFileLocation
  :cognitect.aws.transfer/InputFileLocation)

(s/def
  :cognitect.aws.transfer.CopyStepDetails/OverwriteExisting
  :cognitect.aws.transfer/OverwriteExisting)

(s/def
  :cognitect.aws.transfer.CopyStepDetails/SourceFileLocation
  :cognitect.aws.transfer/SourceFileLocation)

(s/def :cognitect.aws.transfer.UserDetails/UserName :cognitect.aws.transfer/UserName)

(s/def :cognitect.aws.transfer.UserDetails/ServerId :cognitect.aws.transfer/ServerId)

(s/def :cognitect.aws.transfer.UserDetails/SessionId :cognitect.aws.transfer/SessionId)

(s/def
  :cognitect.aws.transfer.UpdateUserRequest/HomeDirectory
  :cognitect.aws.transfer/HomeDirectory)

(s/def
  :cognitect.aws.transfer.UpdateUserRequest/HomeDirectoryType
  :cognitect.aws.transfer/HomeDirectoryType)

(s/def
  :cognitect.aws.transfer.UpdateUserRequest/HomeDirectoryMappings
  :cognitect.aws.transfer/HomeDirectoryMappings)

(s/def :cognitect.aws.transfer.UpdateUserRequest/Policy :cognitect.aws.transfer/Policy)

(s/def :cognitect.aws.transfer.UpdateUserRequest/PosixProfile :cognitect.aws.transfer/PosixProfile)

(s/def :cognitect.aws.transfer.UpdateUserRequest/Role :cognitect.aws.transfer/Role)

(s/def :cognitect.aws.transfer.UpdateUserRequest/ServerId :cognitect.aws.transfer/ServerId)

(s/def :cognitect.aws.transfer.UpdateUserRequest/UserName :cognitect.aws.transfer/UserName)

(s/def :cognitect.aws.transfer.LoggingConfiguration/LoggingRole :cognitect.aws.transfer/Role)

(s/def
  :cognitect.aws.transfer.LoggingConfiguration/LogGroupName
  :cognitect.aws.transfer/LogGroupName)

(s/def :cognitect.aws.transfer.ListAccessesResponse/NextToken :cognitect.aws.transfer/NextToken)

(s/def :cognitect.aws.transfer.ListAccessesResponse/ServerId :cognitect.aws.transfer/ServerId)

(s/def :cognitect.aws.transfer.ListAccessesResponse/Accesses :cognitect.aws.transfer/ListedAccesses)

(s/def
  :cognitect.aws.transfer.DescribeServerResponse/Server
  :cognitect.aws.transfer/DescribedServer)

(s/def :cognitect.aws.transfer.UntagResourceRequest/Arn :cognitect.aws.transfer/Arn)

(s/def :cognitect.aws.transfer.UntagResourceRequest/TagKeys :cognitect.aws.transfer/TagKeys)

(s/def :cognitect.aws.transfer.FileLocation/S3FileLocation :cognitect.aws.transfer/S3FileLocation)

(s/def :cognitect.aws.transfer.FileLocation/EfsFileLocation :cognitect.aws.transfer/EfsFileLocation)

(s/def
  :cognitect.aws.transfer.DescribeWorkflowResponse/Workflow
  :cognitect.aws.transfer/DescribedWorkflow)

(s/def :cognitect.aws.transfer.WorkflowDetail/WorkflowId :cognitect.aws.transfer/WorkflowId)

(s/def :cognitect.aws.transfer.WorkflowDetail/ExecutionRole :cognitect.aws.transfer/Role)

(s/def :cognitect.aws.transfer.ImportSshPublicKeyResponse/ServerId :cognitect.aws.transfer/ServerId)

(s/def
  :cognitect.aws.transfer.ImportSshPublicKeyResponse/SshPublicKeyId
  :cognitect.aws.transfer/SshPublicKeyId)

(s/def :cognitect.aws.transfer.ImportSshPublicKeyResponse/UserName :cognitect.aws.transfer/UserName)

(s/def :cognitect.aws.transfer.DeleteStepDetails/Name :cognitect.aws.transfer/WorkflowStepName)

(s/def
  :cognitect.aws.transfer.DeleteStepDetails/SourceFileLocation
  :cognitect.aws.transfer/SourceFileLocation)

(s/def :cognitect.aws.transfer.ListExecutionsResponse/NextToken :cognitect.aws.transfer/NextToken)

(s/def :cognitect.aws.transfer.ListExecutionsResponse/WorkflowId :cognitect.aws.transfer/WorkflowId)

(s/def
  :cognitect.aws.transfer.ListExecutionsResponse/Executions
  :cognitect.aws.transfer/ListedExecutions)

(s/def :cognitect.aws.transfer.EfsFileLocation/FileSystemId :cognitect.aws.transfer/EfsFileSystemId)

(s/def :cognitect.aws.transfer.EfsFileLocation/Path :cognitect.aws.transfer/EfsPath)

(s/def :cognitect.aws.transfer.ListUsersResponse/NextToken :cognitect.aws.transfer/NextToken)

(s/def :cognitect.aws.transfer.ListUsersResponse/ServerId :cognitect.aws.transfer/ServerId)

(s/def :cognitect.aws.transfer.ListUsersResponse/Users :cognitect.aws.transfer/ListedUsers)

(s/def
  :cognitect.aws.transfer.SendWorkflowStepStateRequest/WorkflowId
  :cognitect.aws.transfer/WorkflowId)

(s/def
  :cognitect.aws.transfer.SendWorkflowStepStateRequest/ExecutionId
  :cognitect.aws.transfer/ExecutionId)

(s/def
  :cognitect.aws.transfer.SendWorkflowStepStateRequest/Token
  :cognitect.aws.transfer/CallbackToken)

(s/def
  :cognitect.aws.transfer.SendWorkflowStepStateRequest/Status
  :cognitect.aws.transfer/CustomStepStatus)

(s/def :cognitect.aws.transfer.UpdateServerRequest/HostKey :cognitect.aws.transfer/HostKey)

(s/def
  :cognitect.aws.transfer.UpdateServerRequest/EndpointDetails
  :cognitect.aws.transfer/EndpointDetails)

(s/def
  :cognitect.aws.transfer.UpdateServerRequest/IdentityProviderDetails
  :cognitect.aws.transfer/IdentityProviderDetails)

(s/def :cognitect.aws.transfer.UpdateServerRequest/LoggingRole :cognitect.aws.transfer/NullableRole)

(s/def
  :cognitect.aws.transfer.UpdateServerRequest/PostAuthenticationLoginBanner
  :cognitect.aws.transfer/PostAuthenticationLoginBanner)

(s/def :cognitect.aws.transfer.UpdateServerRequest/Protocols :cognitect.aws.transfer/Protocols)

(s/def :cognitect.aws.transfer.UpdateServerRequest/ServerId :cognitect.aws.transfer/ServerId)

(s/def
  :cognitect.aws.transfer.UpdateServerRequest/WorkflowDetails
  :cognitect.aws.transfer/WorkflowDetails)

(s/def
  :cognitect.aws.transfer.UpdateServerRequest/EndpointType
  :cognitect.aws.transfer/EndpointType)

(s/def :cognitect.aws.transfer.UpdateServerRequest/Certificate :cognitect.aws.transfer/Certificate)

(s/def
  :cognitect.aws.transfer.UpdateServerRequest/PreAuthenticationLoginBanner
  :cognitect.aws.transfer/PreAuthenticationLoginBanner)

(s/def
  :cognitect.aws.transfer.UpdateServerRequest/SecurityPolicyName
  :cognitect.aws.transfer/SecurityPolicyName)

(s/def
  :cognitect.aws.transfer.UpdateServerRequest/ProtocolDetails
  :cognitect.aws.transfer/ProtocolDetails)

(s/def
  :cognitect.aws.transfer.DescribeSecurityPolicyResponse/SecurityPolicy
  :cognitect.aws.transfer/DescribedSecurityPolicy)

(s/def :cognitect.aws.transfer.ListUsersRequest/MaxResults :cognitect.aws.transfer/MaxResults)

(s/def :cognitect.aws.transfer.ListUsersRequest/NextToken :cognitect.aws.transfer/NextToken)

(s/def :cognitect.aws.transfer.ListUsersRequest/ServerId :cognitect.aws.transfer/ServerId)

(s/def :cognitect.aws.transfer.S3FileLocation/Bucket :cognitect.aws.transfer/S3Bucket)

(s/def :cognitect.aws.transfer.S3FileLocation/Key :cognitect.aws.transfer/S3Key)

(s/def :cognitect.aws.transfer.S3FileLocation/VersionId :cognitect.aws.transfer/S3VersionId)

(s/def :cognitect.aws.transfer.S3FileLocation/Etag :cognitect.aws.transfer/S3Etag)

(s/def :cognitect.aws.transfer.ListedServer/Arn :cognitect.aws.transfer/Arn)

(s/def :cognitect.aws.transfer.ListedServer/Domain :cognitect.aws.transfer/Domain)

(s/def
  :cognitect.aws.transfer.ListedServer/IdentityProviderType
  :cognitect.aws.transfer/IdentityProviderType)

(s/def :cognitect.aws.transfer.ListedServer/EndpointType :cognitect.aws.transfer/EndpointType)

(s/def :cognitect.aws.transfer.ListedServer/LoggingRole :cognitect.aws.transfer/Role)

(s/def :cognitect.aws.transfer.ListedServer/ServerId :cognitect.aws.transfer/ServerId)

(s/def :cognitect.aws.transfer.ListedServer/State :cognitect.aws.transfer/State)

(s/def :cognitect.aws.transfer.ListedServer/UserCount :cognitect.aws.transfer/UserCount)

(s/def :cognitect.aws.transfer.DescribeAccessResponse/ServerId :cognitect.aws.transfer/ServerId)

(s/def
  :cognitect.aws.transfer.DescribeAccessResponse/Access
  :cognitect.aws.transfer/DescribedAccess)

(s/def
  :cognitect.aws.transfer.CreateServerRequest/IdentityProviderType
  :cognitect.aws.transfer/IdentityProviderType)

(s/def :cognitect.aws.transfer.CreateServerRequest/Domain :cognitect.aws.transfer/Domain)

(s/def :cognitect.aws.transfer.CreateServerRequest/HostKey :cognitect.aws.transfer/HostKey)

(s/def :cognitect.aws.transfer.CreateServerRequest/Tags :cognitect.aws.transfer/Tags)

(s/def
  :cognitect.aws.transfer.CreateServerRequest/EndpointDetails
  :cognitect.aws.transfer/EndpointDetails)

(s/def
  :cognitect.aws.transfer.CreateServerRequest/IdentityProviderDetails
  :cognitect.aws.transfer/IdentityProviderDetails)

(s/def :cognitect.aws.transfer.CreateServerRequest/LoggingRole :cognitect.aws.transfer/Role)

(s/def
  :cognitect.aws.transfer.CreateServerRequest/PostAuthenticationLoginBanner
  :cognitect.aws.transfer/PostAuthenticationLoginBanner)

(s/def :cognitect.aws.transfer.CreateServerRequest/Protocols :cognitect.aws.transfer/Protocols)

(s/def
  :cognitect.aws.transfer.CreateServerRequest/WorkflowDetails
  :cognitect.aws.transfer/WorkflowDetails)

(s/def
  :cognitect.aws.transfer.CreateServerRequest/EndpointType
  :cognitect.aws.transfer/EndpointType)

(s/def :cognitect.aws.transfer.CreateServerRequest/Certificate :cognitect.aws.transfer/Certificate)

(s/def
  :cognitect.aws.transfer.CreateServerRequest/PreAuthenticationLoginBanner
  :cognitect.aws.transfer/PreAuthenticationLoginBanner)

(s/def
  :cognitect.aws.transfer.CreateServerRequest/SecurityPolicyName
  :cognitect.aws.transfer/SecurityPolicyName)

(s/def
  :cognitect.aws.transfer.CreateServerRequest/ProtocolDetails
  :cognitect.aws.transfer/ProtocolDetails)

(s/def
  :cognitect.aws.transfer.ExecutionStepResult/StepType
  :cognitect.aws.transfer/WorkflowStepType)

(s/def
  :cognitect.aws.transfer.ExecutionStepResult/Outputs
  :cognitect.aws.transfer/StepResultOutputsJson)

(s/def :cognitect.aws.transfer.ExecutionStepResult/Error :cognitect.aws.transfer/ExecutionError)

(s/def :cognitect.aws.transfer.ListServersResponse/NextToken :cognitect.aws.transfer/NextToken)

(s/def :cognitect.aws.transfer.ListServersResponse/Servers :cognitect.aws.transfer/ListedServers)

(s/def :cognitect.aws.transfer.DeleteAccessRequest/ServerId :cognitect.aws.transfer/ServerId)

(s/def :cognitect.aws.transfer.DeleteAccessRequest/ExternalId :cognitect.aws.transfer/ExternalId)

(s/def :cognitect.aws.transfer.ListTagsForResourceResponse/Arn :cognitect.aws.transfer/Arn)

(s/def
  :cognitect.aws.transfer.ListTagsForResourceResponse/NextToken
  :cognitect.aws.transfer/NextToken)

(s/def :cognitect.aws.transfer.ListTagsForResourceResponse/Tags :cognitect.aws.transfer/Tags)

(s/def :cognitect.aws.transfer.DeleteServerRequest/ServerId :cognitect.aws.transfer/ServerId)

(s/def
  :cognitect.aws.transfer.UpdateAccessRequest/HomeDirectory
  :cognitect.aws.transfer/HomeDirectory)

(s/def
  :cognitect.aws.transfer.UpdateAccessRequest/HomeDirectoryType
  :cognitect.aws.transfer/HomeDirectoryType)

(s/def
  :cognitect.aws.transfer.UpdateAccessRequest/HomeDirectoryMappings
  :cognitect.aws.transfer/HomeDirectoryMappings)

(s/def :cognitect.aws.transfer.UpdateAccessRequest/Policy :cognitect.aws.transfer/Policy)

(s/def
  :cognitect.aws.transfer.UpdateAccessRequest/PosixProfile
  :cognitect.aws.transfer/PosixProfile)

(s/def :cognitect.aws.transfer.UpdateAccessRequest/Role :cognitect.aws.transfer/Role)

(s/def :cognitect.aws.transfer.UpdateAccessRequest/ServerId :cognitect.aws.transfer/ServerId)

(s/def :cognitect.aws.transfer.UpdateAccessRequest/ExternalId :cognitect.aws.transfer/ExternalId)

(s/def
  :cognitect.aws.transfer.TestIdentityProviderRequest/ServerId
  :cognitect.aws.transfer/ServerId)

(s/def
  :cognitect.aws.transfer.TestIdentityProviderRequest/ServerProtocol
  :cognitect.aws.transfer/Protocol)

(s/def
  :cognitect.aws.transfer.TestIdentityProviderRequest/SourceIp
  :cognitect.aws.transfer/SourceIp)

(s/def
  :cognitect.aws.transfer.TestIdentityProviderRequest/UserName
  :cognitect.aws.transfer/UserName)

(s/def
  :cognitect.aws.transfer.TestIdentityProviderRequest/UserPassword
  :cognitect.aws.transfer/UserPassword)

(s/def :cognitect.aws.transfer.ProtocolDetails/PassiveIp :cognitect.aws.transfer/PassiveIp)

(s/def
  :cognitect.aws.transfer.ProtocolDetails/TlsSessionResumptionMode
  :cognitect.aws.transfer/TlsSessionResumptionMode)

