;; Copyright (c) Cognitect, Inc.
;; All rights reserved.

(ns cognitect.aws.translate.specs
  (:require [clojure.spec.alpha :as s] [clojure.spec.gen.alpha :as gen]))

(s/def :cognitect.aws/client map?)

(s/def :core.async/channel any?)

(s/def
  :cognitect.aws.translate/UpdateParallelDataRequest
  (s/keys
    :req-un
    [:cognitect.aws.translate.UpdateParallelDataRequest/Name
     :cognitect.aws.translate.UpdateParallelDataRequest/ParallelDataConfig
     :cognitect.aws.translate.UpdateParallelDataRequest/ClientToken]
    :opt-un
    [:cognitect.aws.translate.UpdateParallelDataRequest/Description]))

(s/def
  :cognitect.aws.translate/JobName
  (s/spec #(re-matches (re-pattern "^([\\p{L}\\p{Z}\\p{N}_.:/=+\\-%@]*)$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.translate/CreateParallelDataRequest
  (s/keys
    :req-un
    [:cognitect.aws.translate.CreateParallelDataRequest/Name
     :cognitect.aws.translate.CreateParallelDataRequest/ParallelDataConfig
     :cognitect.aws.translate.CreateParallelDataRequest/ClientToken]
    :opt-un
    [:cognitect.aws.translate.CreateParallelDataRequest/Description
     :cognitect.aws.translate.CreateParallelDataRequest/EncryptionKey]))

(s/def
  :cognitect.aws.translate/ParallelDataFormat
  (s/spec string? :gen #(s/gen #{"CSV" "TSV" "TMX"})))

(s/def
  :cognitect.aws.translate/LanguageCodeStringList
  (s/coll-of :cognitect.aws.translate/LanguageCodeString))

(s/def
  :cognitect.aws.translate/TerminologyArn
  (s/spec
    (s/and string? #(<= 1 (count %) 512))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 1 512) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.translate/StartTextTranslationJobResponse
  (s/keys
    :opt-un
    [:cognitect.aws.translate.StartTextTranslationJobResponse/JobStatus
     :cognitect.aws.translate.StartTextTranslationJobResponse/JobId]))

(s/def
  :cognitect.aws.translate/TextTranslationJobFilter
  (s/keys
    :opt-un
    [:cognitect.aws.translate.TextTranslationJobFilter/JobStatus
     :cognitect.aws.translate.TextTranslationJobFilter/JobName
     :cognitect.aws.translate.TextTranslationJobFilter/SubmittedBeforeTime
     :cognitect.aws.translate.TextTranslationJobFilter/SubmittedAfterTime]))

(s/def
  :cognitect.aws.translate/DeleteParallelDataResponse
  (s/keys
    :opt-un
    [:cognitect.aws.translate.DeleteParallelDataResponse/Status
     :cognitect.aws.translate.DeleteParallelDataResponse/Name]))

(s/def
  :cognitect.aws.translate/ParallelDataPropertiesList
  (s/coll-of :cognitect.aws.translate/ParallelDataProperties))

(s/def :cognitect.aws.translate/Formality (s/spec string? :gen #(s/gen #{"INFORMAL" "FORMAL"})))

(s/def
  :cognitect.aws.translate/BoundedLengthString
  (s/spec #(re-matches (re-pattern "[\\P{M}\\p{M}]{1,5000}") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.translate/TextTranslationJobProperties
  (s/keys
    :opt-un
    [:cognitect.aws.translate.TextTranslationJobProperties/InputDataConfig
     :cognitect.aws.translate.TextTranslationJobProperties/JobDetails
     :cognitect.aws.translate.TextTranslationJobProperties/OutputDataConfig
     :cognitect.aws.translate.TextTranslationJobProperties/EndTime
     :cognitect.aws.translate.TextTranslationJobProperties/JobStatus
     :cognitect.aws.translate.TextTranslationJobProperties/TerminologyNames
     :cognitect.aws.translate.TextTranslationJobProperties/SourceLanguageCode
     :cognitect.aws.translate.TextTranslationJobProperties/JobName
     :cognitect.aws.translate.TextTranslationJobProperties/ParallelDataNames
     :cognitect.aws.translate.TextTranslationJobProperties/TargetLanguageCodes
     :cognitect.aws.translate.TextTranslationJobProperties/Message
     :cognitect.aws.translate.TextTranslationJobProperties/Settings
     :cognitect.aws.translate.TextTranslationJobProperties/DataAccessRoleArn
     :cognitect.aws.translate.TextTranslationJobProperties/SubmittedTime
     :cognitect.aws.translate.TextTranslationJobProperties/JobId]))

(s/def
  :cognitect.aws.translate/GetTerminologyRequest
  (s/keys
    :req-un
    [:cognitect.aws.translate.GetTerminologyRequest/Name]
    :opt-un
    [:cognitect.aws.translate.GetTerminologyRequest/TerminologyDataFormat]))

(s/def
  :cognitect.aws.translate/ListTextTranslationJobsRequest
  (s/keys
    :opt-un
    [:cognitect.aws.translate.ListTextTranslationJobsRequest/Filter
     :cognitect.aws.translate.ListTextTranslationJobsRequest/NextToken
     :cognitect.aws.translate.ListTextTranslationJobsRequest/MaxResults]))

(s/def
  :cognitect.aws.translate/JobId
  (s/spec #(re-matches (re-pattern "^([\\p{L}\\p{Z}\\p{N}_.:/=+\\-%@]*)$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.translate/GetTerminologyResponse
  (s/keys
    :opt-un
    [:cognitect.aws.translate.GetTerminologyResponse/TerminologyDataLocation
     :cognitect.aws.translate.GetTerminologyResponse/TerminologyProperties
     :cognitect.aws.translate.GetTerminologyResponse/AuxiliaryDataLocation]))

(s/def
  :cognitect.aws.translate/S3Uri
  (s/spec
    #(re-matches (re-pattern "s3://[a-z0-9][\\.\\-a-z0-9]{1,61}[a-z0-9](/.*)?") %)
    :gen
    #(gen/string)))

(s/def
  :cognitect.aws.translate/CreateParallelDataResponse
  (s/keys
    :opt-un
    [:cognitect.aws.translate.CreateParallelDataResponse/Status
     :cognitect.aws.translate.CreateParallelDataResponse/Name]))

(s/def
  :cognitect.aws.translate/ParallelDataConfig
  (s/keys
    :req-un
    [:cognitect.aws.translate.ParallelDataConfig/S3Uri
     :cognitect.aws.translate.ParallelDataConfig/Format]))

(s/def
  :cognitect.aws.translate/DeleteParallelDataRequest
  (s/keys :req-un [:cognitect.aws.translate.DeleteParallelDataRequest/Name]))

(s/def
  :cognitect.aws.translate/JobStatus
  (s/spec
    string?
    :gen
    #(s/gen
      #{"COMPLETED" "COMPLETED_WITH_ERROR" "IN_PROGRESS" "STOP_REQUESTED" "SUBMITTED" "STOPPED"
        "FAILED"})))

(s/def
  :cognitect.aws.translate/TerminologyDataFormat
  (s/spec string? :gen #(s/gen #{"CSV" "TSV" "TMX"})))

(s/def
  :cognitect.aws.translate/ParallelDataProperties
  (s/keys
    :opt-un
    [:cognitect.aws.translate.ParallelDataProperties/ImportedDataSize
     :cognitect.aws.translate.ParallelDataProperties/ParallelDataConfig
     :cognitect.aws.translate.ParallelDataProperties/SourceLanguageCode
     :cognitect.aws.translate.ParallelDataProperties/Status
     :cognitect.aws.translate.ParallelDataProperties/LastUpdatedAt
     :cognitect.aws.translate.ParallelDataProperties/TargetLanguageCodes
     :cognitect.aws.translate.ParallelDataProperties/SkippedRecordCount
     :cognitect.aws.translate.ParallelDataProperties/Description
     :cognitect.aws.translate.ParallelDataProperties/LatestUpdateAttemptAt
     :cognitect.aws.translate.ParallelDataProperties/Message
     :cognitect.aws.translate.ParallelDataProperties/FailedRecordCount
     :cognitect.aws.translate.ParallelDataProperties/LatestUpdateAttemptStatus
     :cognitect.aws.translate.ParallelDataProperties/Arn
     :cognitect.aws.translate.ParallelDataProperties/CreatedAt
     :cognitect.aws.translate.ParallelDataProperties/Name
     :cognitect.aws.translate.ParallelDataProperties/EncryptionKey
     :cognitect.aws.translate.ParallelDataProperties/ImportedRecordCount]))

(s/def
  :cognitect.aws.translate/TerminologyDataLocation
  (s/keys
    :req-un
    [:cognitect.aws.translate.TerminologyDataLocation/RepositoryType
     :cognitect.aws.translate.TerminologyDataLocation/Location]))

(s/def
  :cognitect.aws.translate/TranslationSettings
  (s/keys
    :opt-un
    [:cognitect.aws.translate.TranslationSettings/Profanity
     :cognitect.aws.translate.TranslationSettings/Formality]))

(s/def
  :cognitect.aws.translate/TranslateTextRequest
  (s/keys
    :req-un
    [:cognitect.aws.translate.TranslateTextRequest/Text
     :cognitect.aws.translate.TranslateTextRequest/SourceLanguageCode
     :cognitect.aws.translate.TranslateTextRequest/TargetLanguageCode]
    :opt-un
    [:cognitect.aws.translate.TranslateTextRequest/TerminologyNames
     :cognitect.aws.translate.TranslateTextRequest/Settings]))

(s/def
  :cognitect.aws.translate/ParallelDataDataLocation
  (s/keys
    :req-un
    [:cognitect.aws.translate.ParallelDataDataLocation/RepositoryType
     :cognitect.aws.translate.ParallelDataDataLocation/Location]))

(s/def
  :cognitect.aws.translate/EncryptionKey
  (s/keys
    :req-un
    [:cognitect.aws.translate.EncryptionKey/Type :cognitect.aws.translate.EncryptionKey/Id]))

(s/def
  :cognitect.aws.translate/MaxResultsInteger
  (s/spec (s/and int? #(<= 1 % 500)) :gen #(gen/choose 1 500)))

(s/def
  :cognitect.aws.translate/IamRoleArn
  (s/spec
    #(re-matches (re-pattern "arn:aws(-[^:]+)?:iam::[0-9]{12}:role/.+") %)
    :gen
    #(gen/string)))

(s/def
  :cognitect.aws.translate/GetParallelDataRequest
  (s/keys :req-un [:cognitect.aws.translate.GetParallelDataRequest/Name]))

(s/def
  :cognitect.aws.translate/NextToken
  (s/spec #(re-matches (re-pattern "\\p{ASCII}{0,8192}") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.translate/OutputDataConfig
  (s/keys
    :req-un
    [:cognitect.aws.translate.OutputDataConfig/S3Uri]
    :opt-un
    [:cognitect.aws.translate.OutputDataConfig/EncryptionKey]))

(s/def
  :cognitect.aws.translate/TerminologyProperties
  (s/keys
    :opt-un
    [:cognitect.aws.translate.TerminologyProperties/SourceLanguageCode
     :cognitect.aws.translate.TerminologyProperties/TermCount
     :cognitect.aws.translate.TerminologyProperties/Format
     :cognitect.aws.translate.TerminologyProperties/SizeBytes
     :cognitect.aws.translate.TerminologyProperties/LastUpdatedAt
     :cognitect.aws.translate.TerminologyProperties/SkippedTermCount
     :cognitect.aws.translate.TerminologyProperties/TargetLanguageCodes
     :cognitect.aws.translate.TerminologyProperties/Description
     :cognitect.aws.translate.TerminologyProperties/Message
     :cognitect.aws.translate.TerminologyProperties/Directionality
     :cognitect.aws.translate.TerminologyProperties/Arn
     :cognitect.aws.translate.TerminologyProperties/CreatedAt
     :cognitect.aws.translate.TerminologyProperties/Name
     :cognitect.aws.translate.TerminologyProperties/EncryptionKey]))

(s/def
  :cognitect.aws.translate/Integer
  (s/spec int? :gen #(gen/choose Long/MIN_VALUE Long/MAX_VALUE)))

(s/def
  :cognitect.aws.translate/ListTerminologiesRequest
  (s/keys
    :opt-un
    [:cognitect.aws.translate.ListTerminologiesRequest/NextToken
     :cognitect.aws.translate.ListTerminologiesRequest/MaxResults]))

(s/def
  :cognitect.aws.translate/ParallelDataStatus
  (s/spec string? :gen #(s/gen #{"UPDATING" "DELETING" "CREATING" "ACTIVE" "FAILED"})))

(s/def :cognitect.aws.translate/Profanity (s/spec string? :gen #(s/gen #{"MASK"})))

(s/def
  :cognitect.aws.translate/ClientTokenString
  (s/spec #(re-matches (re-pattern "^[a-zA-Z0-9-]+$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.translate/TerminologyFile
  (s/spec
    (s/or
      :byte-array
      (s/and bytes? #(>= 10485760 (count %)))
      :input-stream
      #(instance? java.io.InputStream %))
    :gen
    (fn []
      (gen/bind
        (gen/choose (or nil 0) (or 10485760 100))
        #(gen/return (byte-array % (repeatedly (fn [] (rand-int 256)))))))))

(s/def
  :cognitect.aws.translate/GetParallelDataResponse
  (s/keys
    :opt-un
    [:cognitect.aws.translate.GetParallelDataResponse/DataLocation
     :cognitect.aws.translate.GetParallelDataResponse/AuxiliaryDataLocation
     :cognitect.aws.translate.GetParallelDataResponse/LatestUpdateAttemptAuxiliaryDataLocation
     :cognitect.aws.translate.GetParallelDataResponse/ParallelDataProperties]))

(s/def :cognitect.aws.translate/EncryptionKeyType (s/spec string? :gen #(s/gen #{"KMS"})))

(s/def
  :cognitect.aws.translate/StopTextTranslationJobResponse
  (s/keys
    :opt-un
    [:cognitect.aws.translate.StopTextTranslationJobResponse/JobStatus
     :cognitect.aws.translate.StopTextTranslationJobResponse/JobId]))

(s/def
  :cognitect.aws.translate/TerminologyData
  (s/keys
    :req-un
    [:cognitect.aws.translate.TerminologyData/File :cognitect.aws.translate.TerminologyData/Format]
    :opt-un
    [:cognitect.aws.translate.TerminologyData/Directionality]))

(s/def
  :cognitect.aws.translate/TerminologyPropertiesList
  (s/coll-of :cognitect.aws.translate/TerminologyProperties))

(s/def
  :cognitect.aws.translate/EncryptionKeyID
  (s/spec
    #(re-matches
      (re-pattern
        "(arn:aws((-us-gov)|(-iso)|(-iso-b)|(-cn))?:kms:)?([a-z]{2}-[a-z]+(-[a-z]+)?-\\d:)?(\\d{12}:)?(((key/)?[a-zA-Z0-9-_]+)|(alias/[a-zA-Z0-9:/_-]+))")
      %)
    :gen
    #(gen/string)))

(s/def
  :cognitect.aws.translate/ImportTerminologyResponse
  (s/keys
    :opt-un
    [:cognitect.aws.translate.ImportTerminologyResponse/TerminologyProperties
     :cognitect.aws.translate.ImportTerminologyResponse/AuxiliaryDataLocation]))

(s/def
  :cognitect.aws.translate/DeleteTerminologyRequest
  (s/keys :req-un [:cognitect.aws.translate.DeleteTerminologyRequest/Name]))

(s/def
  :cognitect.aws.translate/StopTextTranslationJobRequest
  (s/keys :req-un [:cognitect.aws.translate.StopTextTranslationJobRequest/JobId]))

(s/def
  :cognitect.aws.translate/TranslateTextResponse
  (s/keys
    :req-un
    [:cognitect.aws.translate.TranslateTextResponse/TranslatedText
     :cognitect.aws.translate.TranslateTextResponse/SourceLanguageCode
     :cognitect.aws.translate.TranslateTextResponse/TargetLanguageCode]
    :opt-un
    [:cognitect.aws.translate.TranslateTextResponse/AppliedSettings
     :cognitect.aws.translate.TranslateTextResponse/AppliedTerminologies]))

(s/def
  :cognitect.aws.translate/ListTextTranslationJobsResponse
  (s/keys
    :opt-un
    [:cognitect.aws.translate.ListTextTranslationJobsResponse/NextToken
     :cognitect.aws.translate.ListTextTranslationJobsResponse/TextTranslationJobPropertiesList]))

(s/def
  :cognitect.aws.translate/ListTerminologiesResponse
  (s/keys
    :opt-un
    [:cognitect.aws.translate.ListTerminologiesResponse/TerminologyPropertiesList
     :cognitect.aws.translate.ListTerminologiesResponse/NextToken]))

(s/def
  :cognitect.aws.translate/JobDetails
  (s/keys
    :opt-un
    [:cognitect.aws.translate.JobDetails/DocumentsWithErrorsCount
     :cognitect.aws.translate.JobDetails/TranslatedDocumentsCount
     :cognitect.aws.translate.JobDetails/InputDocumentsCount]))

(s/def :cognitect.aws.translate/TermList (s/coll-of :cognitect.aws.translate/Term))

(s/def
  :cognitect.aws.translate/ParallelDataArn
  (s/spec
    (s/and string? #(<= 1 (count %) 512))
    :gen
    (fn []
      (gen/fmap #(apply str %) (gen/bind (gen/choose 1 512) #(gen/vector (gen/char-alpha) %))))))

(s/def :cognitect.aws.translate/Long (s/spec int? :gen #(gen/choose Long/MIN_VALUE Long/MAX_VALUE)))

(s/def :cognitect.aws.translate/MergeStrategy (s/spec string? :gen #(s/gen #{"OVERWRITE"})))

(s/def
  :cognitect.aws.translate/ListParallelDataResponse
  (s/keys
    :opt-un
    [:cognitect.aws.translate.ListParallelDataResponse/NextToken
     :cognitect.aws.translate.ListParallelDataResponse/ParallelDataPropertiesList]))

(s/def
  :cognitect.aws.translate/AppliedTerminology
  (s/keys
    :opt-un
    [:cognitect.aws.translate.AppliedTerminology/Terms
     :cognitect.aws.translate.AppliedTerminology/Name]))

(s/def
  :cognitect.aws.translate/Description
  (s/spec #(re-matches (re-pattern "[\\P{M}\\p{M}]{0,256}") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.translate/AppliedTerminologyList
  (s/coll-of :cognitect.aws.translate/AppliedTerminology))

(s/def :cognitect.aws.translate/ResourceNameList (s/coll-of :cognitect.aws.translate/ResourceName))

(s/def :cognitect.aws.translate/Timestamp inst?)

(s/def
  :cognitect.aws.translate/ContentType
  (s/spec #(re-matches (re-pattern "^[-\\w.]+\\/[-\\w.+]+$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.translate/ImportTerminologyRequest
  (s/keys
    :req-un
    [:cognitect.aws.translate.ImportTerminologyRequest/Name
     :cognitect.aws.translate.ImportTerminologyRequest/MergeStrategy
     :cognitect.aws.translate.ImportTerminologyRequest/TerminologyData]
    :opt-un
    [:cognitect.aws.translate.ImportTerminologyRequest/Description
     :cognitect.aws.translate.ImportTerminologyRequest/EncryptionKey]))

(s/def :cognitect.aws.translate/UnboundedLengthString string?)

(s/def
  :cognitect.aws.translate/TextTranslationJobPropertiesList
  (s/coll-of :cognitect.aws.translate/TextTranslationJobProperties))

(s/def
  :cognitect.aws.translate/LanguageCodeString
  (s/spec
    (s/and string? #(<= 2 (count %) 5))
    :gen
    (fn [] (gen/fmap #(apply str %) (gen/bind (gen/choose 2 5) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.translate/Term
  (s/keys
    :opt-un
    [:cognitect.aws.translate.Term/TargetText :cognitect.aws.translate.Term/SourceText]))

(s/def :cognitect.aws.translate/Directionality (s/spec string? :gen #(s/gen #{"MULTI" "UNI"})))

(s/def
  :cognitect.aws.translate/InputDataConfig
  (s/keys
    :req-un
    [:cognitect.aws.translate.InputDataConfig/S3Uri
     :cognitect.aws.translate.InputDataConfig/ContentType]))

(s/def
  :cognitect.aws.translate/ResourceName
  (s/spec #(re-matches (re-pattern "^([A-Za-z0-9-]_?)+$") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.translate/DescribeTextTranslationJobRequest
  (s/keys :req-un [:cognitect.aws.translate.DescribeTextTranslationJobRequest/JobId]))

(s/def
  :cognitect.aws.translate/StartTextTranslationJobRequest
  (s/keys
    :req-un
    [:cognitect.aws.translate.StartTextTranslationJobRequest/InputDataConfig
     :cognitect.aws.translate.StartTextTranslationJobRequest/OutputDataConfig
     :cognitect.aws.translate.StartTextTranslationJobRequest/DataAccessRoleArn
     :cognitect.aws.translate.StartTextTranslationJobRequest/SourceLanguageCode
     :cognitect.aws.translate.StartTextTranslationJobRequest/TargetLanguageCodes
     :cognitect.aws.translate.StartTextTranslationJobRequest/ClientToken]
    :opt-un
    [:cognitect.aws.translate.StartTextTranslationJobRequest/TerminologyNames
     :cognitect.aws.translate.StartTextTranslationJobRequest/JobName
     :cognitect.aws.translate.StartTextTranslationJobRequest/ParallelDataNames
     :cognitect.aws.translate.StartTextTranslationJobRequest/Settings]))

(s/def
  :cognitect.aws.translate/TargetLanguageCodeStringList
  (s/coll-of :cognitect.aws.translate/LanguageCodeString :min-count 1 :max-count 1))

(s/def
  :cognitect.aws.translate/UpdateParallelDataResponse
  (s/keys
    :opt-un
    [:cognitect.aws.translate.UpdateParallelDataResponse/Status
     :cognitect.aws.translate.UpdateParallelDataResponse/LatestUpdateAttemptAt
     :cognitect.aws.translate.UpdateParallelDataResponse/LatestUpdateAttemptStatus
     :cognitect.aws.translate.UpdateParallelDataResponse/Name]))

(s/def
  :cognitect.aws.translate/String
  (s/spec #(re-matches (re-pattern "[\\P{M}\\p{M}]{0,10000}") %) :gen #(gen/string)))

(s/def
  :cognitect.aws.translate/ListParallelDataRequest
  (s/keys
    :opt-un
    [:cognitect.aws.translate.ListParallelDataRequest/NextToken
     :cognitect.aws.translate.ListParallelDataRequest/MaxResults]))

(s/def
  :cognitect.aws.translate/DescribeTextTranslationJobResponse
  (s/keys
    :opt-un
    [:cognitect.aws.translate.DescribeTextTranslationJobResponse/TextTranslationJobProperties]))

(s/def
  :cognitect.aws.translate.UpdateParallelDataRequest/Name
  :cognitect.aws.translate/ResourceName)

(s/def
  :cognitect.aws.translate.UpdateParallelDataRequest/Description
  :cognitect.aws.translate/Description)

(s/def
  :cognitect.aws.translate.UpdateParallelDataRequest/ParallelDataConfig
  :cognitect.aws.translate/ParallelDataConfig)

(s/def
  :cognitect.aws.translate.UpdateParallelDataRequest/ClientToken
  :cognitect.aws.translate/ClientTokenString)

(s/def
  :cognitect.aws.translate.CreateParallelDataRequest/Name
  :cognitect.aws.translate/ResourceName)

(s/def
  :cognitect.aws.translate.CreateParallelDataRequest/Description
  :cognitect.aws.translate/Description)

(s/def
  :cognitect.aws.translate.CreateParallelDataRequest/ParallelDataConfig
  :cognitect.aws.translate/ParallelDataConfig)

(s/def
  :cognitect.aws.translate.CreateParallelDataRequest/EncryptionKey
  :cognitect.aws.translate/EncryptionKey)

(s/def
  :cognitect.aws.translate.CreateParallelDataRequest/ClientToken
  :cognitect.aws.translate/ClientTokenString)

(s/def
  :cognitect.aws.translate.StartTextTranslationJobResponse/JobId
  :cognitect.aws.translate/JobId)

(s/def
  :cognitect.aws.translate.StartTextTranslationJobResponse/JobStatus
  :cognitect.aws.translate/JobStatus)

(s/def :cognitect.aws.translate.TextTranslationJobFilter/JobName :cognitect.aws.translate/JobName)

(s/def
  :cognitect.aws.translate.TextTranslationJobFilter/JobStatus
  :cognitect.aws.translate/JobStatus)

(s/def
  :cognitect.aws.translate.TextTranslationJobFilter/SubmittedBeforeTime
  :cognitect.aws.translate/Timestamp)

(s/def
  :cognitect.aws.translate.TextTranslationJobFilter/SubmittedAfterTime
  :cognitect.aws.translate/Timestamp)

(s/def
  :cognitect.aws.translate.DeleteParallelDataResponse/Name
  :cognitect.aws.translate/ResourceName)

(s/def
  :cognitect.aws.translate.DeleteParallelDataResponse/Status
  :cognitect.aws.translate/ParallelDataStatus)

(s/def
  :cognitect.aws.translate.TextTranslationJobProperties/JobName
  :cognitect.aws.translate/JobName)

(s/def
  :cognitect.aws.translate.TextTranslationJobProperties/EndTime
  :cognitect.aws.translate/Timestamp)

(s/def
  :cognitect.aws.translate.TextTranslationJobProperties/Message
  :cognitect.aws.translate/UnboundedLengthString)

(s/def :cognitect.aws.translate.TextTranslationJobProperties/JobId :cognitect.aws.translate/JobId)

(s/def
  :cognitect.aws.translate.TextTranslationJobProperties/SubmittedTime
  :cognitect.aws.translate/Timestamp)

(s/def
  :cognitect.aws.translate.TextTranslationJobProperties/JobStatus
  :cognitect.aws.translate/JobStatus)

(s/def
  :cognitect.aws.translate.TextTranslationJobProperties/DataAccessRoleArn
  :cognitect.aws.translate/IamRoleArn)

(s/def
  :cognitect.aws.translate.TextTranslationJobProperties/SourceLanguageCode
  :cognitect.aws.translate/LanguageCodeString)

(s/def
  :cognitect.aws.translate.TextTranslationJobProperties/TargetLanguageCodes
  :cognitect.aws.translate/TargetLanguageCodeStringList)

(s/def
  :cognitect.aws.translate.TextTranslationJobProperties/OutputDataConfig
  :cognitect.aws.translate/OutputDataConfig)

(s/def
  :cognitect.aws.translate.TextTranslationJobProperties/JobDetails
  :cognitect.aws.translate/JobDetails)

(s/def
  :cognitect.aws.translate.TextTranslationJobProperties/Settings
  :cognitect.aws.translate/TranslationSettings)

(s/def
  :cognitect.aws.translate.TextTranslationJobProperties/InputDataConfig
  :cognitect.aws.translate/InputDataConfig)

(s/def
  :cognitect.aws.translate.TextTranslationJobProperties/TerminologyNames
  :cognitect.aws.translate/ResourceNameList)

(s/def
  :cognitect.aws.translate.TextTranslationJobProperties/ParallelDataNames
  :cognitect.aws.translate/ResourceNameList)

(s/def :cognitect.aws.translate.GetTerminologyRequest/Name :cognitect.aws.translate/ResourceName)

(s/def
  :cognitect.aws.translate.GetTerminologyRequest/TerminologyDataFormat
  :cognitect.aws.translate/TerminologyDataFormat)

(s/def
  :cognitect.aws.translate.ListTextTranslationJobsRequest/Filter
  :cognitect.aws.translate/TextTranslationJobFilter)

(s/def
  :cognitect.aws.translate.ListTextTranslationJobsRequest/NextToken
  :cognitect.aws.translate/NextToken)

(s/def
  :cognitect.aws.translate.ListTextTranslationJobsRequest/MaxResults
  :cognitect.aws.translate/MaxResultsInteger)

(s/def
  :cognitect.aws.translate.GetTerminologyResponse/TerminologyProperties
  :cognitect.aws.translate/TerminologyProperties)

(s/def
  :cognitect.aws.translate.GetTerminologyResponse/TerminologyDataLocation
  :cognitect.aws.translate/TerminologyDataLocation)

(s/def
  :cognitect.aws.translate.GetTerminologyResponse/AuxiliaryDataLocation
  :cognitect.aws.translate/TerminologyDataLocation)

(s/def
  :cognitect.aws.translate.CreateParallelDataResponse/Name
  :cognitect.aws.translate/ResourceName)

(s/def
  :cognitect.aws.translate.CreateParallelDataResponse/Status
  :cognitect.aws.translate/ParallelDataStatus)

(s/def :cognitect.aws.translate.ParallelDataConfig/S3Uri :cognitect.aws.translate/S3Uri)

(s/def
  :cognitect.aws.translate.ParallelDataConfig/Format
  :cognitect.aws.translate/ParallelDataFormat)

(s/def
  :cognitect.aws.translate.DeleteParallelDataRequest/Name
  :cognitect.aws.translate/ResourceName)

(s/def
  :cognitect.aws.translate.ParallelDataProperties/LatestUpdateAttemptAt
  :cognitect.aws.translate/Timestamp)

(s/def :cognitect.aws.translate.ParallelDataProperties/CreatedAt :cognitect.aws.translate/Timestamp)

(s/def
  :cognitect.aws.translate.ParallelDataProperties/SkippedRecordCount
  :cognitect.aws.translate/Long)

(s/def
  :cognitect.aws.translate.ParallelDataProperties/Message
  :cognitect.aws.translate/UnboundedLengthString)

(s/def
  :cognitect.aws.translate.ParallelDataProperties/ParallelDataConfig
  :cognitect.aws.translate/ParallelDataConfig)

(s/def
  :cognitect.aws.translate.ParallelDataProperties/LatestUpdateAttemptStatus
  :cognitect.aws.translate/ParallelDataStatus)

(s/def
  :cognitect.aws.translate.ParallelDataProperties/LastUpdatedAt
  :cognitect.aws.translate/Timestamp)

(s/def
  :cognitect.aws.translate.ParallelDataProperties/ImportedRecordCount
  :cognitect.aws.translate/Long)

(s/def
  :cognitect.aws.translate.ParallelDataProperties/SourceLanguageCode
  :cognitect.aws.translate/LanguageCodeString)

(s/def
  :cognitect.aws.translate.ParallelDataProperties/EncryptionKey
  :cognitect.aws.translate/EncryptionKey)

(s/def
  :cognitect.aws.translate.ParallelDataProperties/TargetLanguageCodes
  :cognitect.aws.translate/LanguageCodeStringList)

(s/def
  :cognitect.aws.translate.ParallelDataProperties/FailedRecordCount
  :cognitect.aws.translate/Long)

(s/def
  :cognitect.aws.translate.ParallelDataProperties/ImportedDataSize
  :cognitect.aws.translate/Long)

(s/def :cognitect.aws.translate.ParallelDataProperties/Arn :cognitect.aws.translate/ParallelDataArn)

(s/def :cognitect.aws.translate.ParallelDataProperties/Name :cognitect.aws.translate/ResourceName)

(s/def
  :cognitect.aws.translate.ParallelDataProperties/Description
  :cognitect.aws.translate/Description)

(s/def
  :cognitect.aws.translate.ParallelDataProperties/Status
  :cognitect.aws.translate/ParallelDataStatus)

(s/def
  :cognitect.aws.translate.TerminologyDataLocation/RepositoryType
  :cognitect.aws.translate/String)

(s/def :cognitect.aws.translate.TerminologyDataLocation/Location :cognitect.aws.translate/String)

(s/def :cognitect.aws.translate.TranslationSettings/Formality :cognitect.aws.translate/Formality)

(s/def :cognitect.aws.translate.TranslationSettings/Profanity :cognitect.aws.translate/Profanity)

(s/def
  :cognitect.aws.translate.TranslateTextRequest/Text
  :cognitect.aws.translate/BoundedLengthString)

(s/def
  :cognitect.aws.translate.TranslateTextRequest/TerminologyNames
  :cognitect.aws.translate/ResourceNameList)

(s/def
  :cognitect.aws.translate.TranslateTextRequest/SourceLanguageCode
  :cognitect.aws.translate/LanguageCodeString)

(s/def
  :cognitect.aws.translate.TranslateTextRequest/TargetLanguageCode
  :cognitect.aws.translate/LanguageCodeString)

(s/def
  :cognitect.aws.translate.TranslateTextRequest/Settings
  :cognitect.aws.translate/TranslationSettings)

(s/def
  :cognitect.aws.translate.ParallelDataDataLocation/RepositoryType
  :cognitect.aws.translate/String)

(s/def :cognitect.aws.translate.ParallelDataDataLocation/Location :cognitect.aws.translate/String)

(s/def :cognitect.aws.translate.EncryptionKey/Type :cognitect.aws.translate/EncryptionKeyType)

(s/def :cognitect.aws.translate.EncryptionKey/Id :cognitect.aws.translate/EncryptionKeyID)

(s/def :cognitect.aws.translate.GetParallelDataRequest/Name :cognitect.aws.translate/ResourceName)

(s/def :cognitect.aws.translate.OutputDataConfig/S3Uri :cognitect.aws.translate/S3Uri)

(s/def
  :cognitect.aws.translate.OutputDataConfig/EncryptionKey
  :cognitect.aws.translate/EncryptionKey)

(s/def :cognitect.aws.translate.TerminologyProperties/SizeBytes :cognitect.aws.translate/Integer)

(s/def :cognitect.aws.translate.TerminologyProperties/CreatedAt :cognitect.aws.translate/Timestamp)

(s/def
  :cognitect.aws.translate.TerminologyProperties/Message
  :cognitect.aws.translate/UnboundedLengthString)

(s/def
  :cognitect.aws.translate.TerminologyProperties/SkippedTermCount
  :cognitect.aws.translate/Integer)

(s/def
  :cognitect.aws.translate.TerminologyProperties/LastUpdatedAt
  :cognitect.aws.translate/Timestamp)

(s/def
  :cognitect.aws.translate.TerminologyProperties/SourceLanguageCode
  :cognitect.aws.translate/LanguageCodeString)

(s/def
  :cognitect.aws.translate.TerminologyProperties/EncryptionKey
  :cognitect.aws.translate/EncryptionKey)

(s/def
  :cognitect.aws.translate.TerminologyProperties/TargetLanguageCodes
  :cognitect.aws.translate/LanguageCodeStringList)

(s/def :cognitect.aws.translate.TerminologyProperties/TermCount :cognitect.aws.translate/Integer)

(s/def
  :cognitect.aws.translate.TerminologyProperties/Format
  :cognitect.aws.translate/TerminologyDataFormat)

(s/def :cognitect.aws.translate.TerminologyProperties/Arn :cognitect.aws.translate/TerminologyArn)

(s/def :cognitect.aws.translate.TerminologyProperties/Name :cognitect.aws.translate/ResourceName)

(s/def
  :cognitect.aws.translate.TerminologyProperties/Description
  :cognitect.aws.translate/Description)

(s/def
  :cognitect.aws.translate.TerminologyProperties/Directionality
  :cognitect.aws.translate/Directionality)

(s/def
  :cognitect.aws.translate.ListTerminologiesRequest/NextToken
  :cognitect.aws.translate/NextToken)

(s/def
  :cognitect.aws.translate.ListTerminologiesRequest/MaxResults
  :cognitect.aws.translate/MaxResultsInteger)

(s/def
  :cognitect.aws.translate.GetParallelDataResponse/ParallelDataProperties
  :cognitect.aws.translate/ParallelDataProperties)

(s/def
  :cognitect.aws.translate.GetParallelDataResponse/DataLocation
  :cognitect.aws.translate/ParallelDataDataLocation)

(s/def
  :cognitect.aws.translate.GetParallelDataResponse/AuxiliaryDataLocation
  :cognitect.aws.translate/ParallelDataDataLocation)

(s/def
  :cognitect.aws.translate.GetParallelDataResponse/LatestUpdateAttemptAuxiliaryDataLocation
  :cognitect.aws.translate/ParallelDataDataLocation)

(s/def :cognitect.aws.translate.StopTextTranslationJobResponse/JobId :cognitect.aws.translate/JobId)

(s/def
  :cognitect.aws.translate.StopTextTranslationJobResponse/JobStatus
  :cognitect.aws.translate/JobStatus)

(s/def :cognitect.aws.translate.TerminologyData/File :cognitect.aws.translate/TerminologyFile)

(s/def
  :cognitect.aws.translate.TerminologyData/Format
  :cognitect.aws.translate/TerminologyDataFormat)

(s/def
  :cognitect.aws.translate.TerminologyData/Directionality
  :cognitect.aws.translate/Directionality)

(s/def
  :cognitect.aws.translate.ImportTerminologyResponse/TerminologyProperties
  :cognitect.aws.translate/TerminologyProperties)

(s/def
  :cognitect.aws.translate.ImportTerminologyResponse/AuxiliaryDataLocation
  :cognitect.aws.translate/TerminologyDataLocation)

(s/def :cognitect.aws.translate.DeleteTerminologyRequest/Name :cognitect.aws.translate/ResourceName)

(s/def :cognitect.aws.translate.StopTextTranslationJobRequest/JobId :cognitect.aws.translate/JobId)

(s/def
  :cognitect.aws.translate.TranslateTextResponse/TranslatedText
  :cognitect.aws.translate/String)

(s/def
  :cognitect.aws.translate.TranslateTextResponse/SourceLanguageCode
  :cognitect.aws.translate/LanguageCodeString)

(s/def
  :cognitect.aws.translate.TranslateTextResponse/TargetLanguageCode
  :cognitect.aws.translate/LanguageCodeString)

(s/def
  :cognitect.aws.translate.TranslateTextResponse/AppliedTerminologies
  :cognitect.aws.translate/AppliedTerminologyList)

(s/def
  :cognitect.aws.translate.TranslateTextResponse/AppliedSettings
  :cognitect.aws.translate/TranslationSettings)

(s/def
  :cognitect.aws.translate.ListTextTranslationJobsResponse/TextTranslationJobPropertiesList
  :cognitect.aws.translate/TextTranslationJobPropertiesList)

(s/def
  :cognitect.aws.translate.ListTextTranslationJobsResponse/NextToken
  :cognitect.aws.translate/NextToken)

(s/def
  :cognitect.aws.translate.ListTerminologiesResponse/TerminologyPropertiesList
  :cognitect.aws.translate/TerminologyPropertiesList)

(s/def
  :cognitect.aws.translate.ListTerminologiesResponse/NextToken
  :cognitect.aws.translate/NextToken)

(s/def
  :cognitect.aws.translate.JobDetails/TranslatedDocumentsCount
  :cognitect.aws.translate/Integer)

(s/def
  :cognitect.aws.translate.JobDetails/DocumentsWithErrorsCount
  :cognitect.aws.translate/Integer)

(s/def :cognitect.aws.translate.JobDetails/InputDocumentsCount :cognitect.aws.translate/Integer)

(s/def
  :cognitect.aws.translate.ListParallelDataResponse/ParallelDataPropertiesList
  :cognitect.aws.translate/ParallelDataPropertiesList)

(s/def
  :cognitect.aws.translate.ListParallelDataResponse/NextToken
  :cognitect.aws.translate/NextToken)

(s/def :cognitect.aws.translate.AppliedTerminology/Name :cognitect.aws.translate/ResourceName)

(s/def :cognitect.aws.translate.AppliedTerminology/Terms :cognitect.aws.translate/TermList)

(s/def :cognitect.aws.translate.ImportTerminologyRequest/Name :cognitect.aws.translate/ResourceName)

(s/def
  :cognitect.aws.translate.ImportTerminologyRequest/MergeStrategy
  :cognitect.aws.translate/MergeStrategy)

(s/def
  :cognitect.aws.translate.ImportTerminologyRequest/Description
  :cognitect.aws.translate/Description)

(s/def
  :cognitect.aws.translate.ImportTerminologyRequest/TerminologyData
  :cognitect.aws.translate/TerminologyData)

(s/def
  :cognitect.aws.translate.ImportTerminologyRequest/EncryptionKey
  :cognitect.aws.translate/EncryptionKey)

(s/def :cognitect.aws.translate.Term/SourceText :cognitect.aws.translate/String)

(s/def :cognitect.aws.translate.Term/TargetText :cognitect.aws.translate/String)

(s/def :cognitect.aws.translate.InputDataConfig/S3Uri :cognitect.aws.translate/S3Uri)

(s/def :cognitect.aws.translate.InputDataConfig/ContentType :cognitect.aws.translate/ContentType)

(s/def
  :cognitect.aws.translate.DescribeTextTranslationJobRequest/JobId
  :cognitect.aws.translate/JobId)

(s/def
  :cognitect.aws.translate.StartTextTranslationJobRequest/JobName
  :cognitect.aws.translate/JobName)

(s/def
  :cognitect.aws.translate.StartTextTranslationJobRequest/DataAccessRoleArn
  :cognitect.aws.translate/IamRoleArn)

(s/def
  :cognitect.aws.translate.StartTextTranslationJobRequest/SourceLanguageCode
  :cognitect.aws.translate/LanguageCodeString)

(s/def
  :cognitect.aws.translate.StartTextTranslationJobRequest/TargetLanguageCodes
  :cognitect.aws.translate/TargetLanguageCodeStringList)

(s/def
  :cognitect.aws.translate.StartTextTranslationJobRequest/OutputDataConfig
  :cognitect.aws.translate/OutputDataConfig)

(s/def
  :cognitect.aws.translate.StartTextTranslationJobRequest/Settings
  :cognitect.aws.translate/TranslationSettings)

(s/def
  :cognitect.aws.translate.StartTextTranslationJobRequest/ClientToken
  :cognitect.aws.translate/ClientTokenString)

(s/def
  :cognitect.aws.translate.StartTextTranslationJobRequest/InputDataConfig
  :cognitect.aws.translate/InputDataConfig)

(s/def
  :cognitect.aws.translate.StartTextTranslationJobRequest/TerminologyNames
  :cognitect.aws.translate/ResourceNameList)

(s/def
  :cognitect.aws.translate.StartTextTranslationJobRequest/ParallelDataNames
  :cognitect.aws.translate/ResourceNameList)

(s/def
  :cognitect.aws.translate.UpdateParallelDataResponse/Name
  :cognitect.aws.translate/ResourceName)

(s/def
  :cognitect.aws.translate.UpdateParallelDataResponse/Status
  :cognitect.aws.translate/ParallelDataStatus)

(s/def
  :cognitect.aws.translate.UpdateParallelDataResponse/LatestUpdateAttemptStatus
  :cognitect.aws.translate/ParallelDataStatus)

(s/def
  :cognitect.aws.translate.UpdateParallelDataResponse/LatestUpdateAttemptAt
  :cognitect.aws.translate/Timestamp)

(s/def
  :cognitect.aws.translate.ListParallelDataRequest/NextToken
  :cognitect.aws.translate/NextToken)

(s/def
  :cognitect.aws.translate.ListParallelDataRequest/MaxResults
  :cognitect.aws.translate/MaxResultsInteger)

(s/def
  :cognitect.aws.translate.DescribeTextTranslationJobResponse/TextTranslationJobProperties
  :cognitect.aws.translate/TextTranslationJobProperties)

