/*
 * Decompiled with CFR 0.152.
 */
package com.datorama.oss.timbermill;

import com.datorama.oss.timbermill.ElasticsearchClient;
import com.datorama.oss.timbermill.ElasticsearchParams;
import com.datorama.oss.timbermill.common.Constants;
import com.datorama.oss.timbermill.common.ElasticsearchUtil;
import com.datorama.oss.timbermill.plugins.PluginsConfig;
import com.datorama.oss.timbermill.plugins.TaskLogPlugin;
import com.datorama.oss.timbermill.unit.AdoptedEvent;
import com.datorama.oss.timbermill.unit.Event;
import com.datorama.oss.timbermill.unit.HeartbeatTask;
import com.datorama.oss.timbermill.unit.IndexEvent;
import com.datorama.oss.timbermill.unit.PluginApplierTask;
import com.datorama.oss.timbermill.unit.Task;
import com.datorama.oss.timbermill.unit.TaskStatus;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.TreeNode;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TaskIndexer {
    private static final Logger LOG = LoggerFactory.getLogger(TaskIndexer.class);
    private final ElasticsearchClient es;
    private final Collection<TaskLogPlugin> logPlugins;
    private final Cache<String, Queue<AdoptedEvent>> parentIdTORootOrphansEventsCache;
    private long daysRotation;

    public TaskIndexer(ElasticsearchParams elasticsearchParams, ElasticsearchClient es) {
        this.daysRotation = Math.max(elasticsearchParams.getDaysRotation(), 1);
        this.logPlugins = PluginsConfig.initPluginsFromJson(elasticsearchParams.getPluginsJson());
        this.es = es;
        CacheBuilder cacheBuilder = CacheBuilder.newBuilder().weigher((key, value) -> {
            int sum = value.stream().mapToInt(Event::estimatedSize).sum();
            return key.length() + sum;
        });
        this.parentIdTORootOrphansEventsCache = cacheBuilder.maximumWeight((long)elasticsearchParams.getMaximumCacheSize()).expireAfterWrite((long)elasticsearchParams.getMaximumCacheMinutesHold(), TimeUnit.MINUTES).removalListener(notification -> {
            if (notification.wasEvicted()) {
                LOG.warn("Event {} was evicted from the cache due to {}", notification.getKey(), (Object)notification.getCause());
            }
        }).build();
    }

    public void retrieveAndIndex(Collection<Event> events, String env) {
        LOG.info("------------------ Batch Start ------------------");
        ZonedDateTime taskIndexerStartTime = ZonedDateTime.now();
        HashSet<String> heartbeatEvents = new HashSet<String>();
        LinkedHashSet<Event> timbermillEvents = new LinkedHashSet<Event>();
        events.forEach(e -> {
            if (e.getName() != null && e.getName().equals("metadata_timbermill_client_heartbeat")) {
                String heartbeatJson = Constants.GSON.toJson((Object)new HeartbeatTask((Event)e, this.daysRotation));
                heartbeatEvents.add(heartbeatJson);
            } else if (e.getTaskId() == null) {
                LOG.warn("Task ID is null for event {}", (Object)Constants.GSON.toJson(e));
            } else {
                e.trimAllStrings();
                timbermillEvents.add((Event)e);
            }
        });
        if (!heartbeatEvents.isEmpty()) {
            this.es.indexMetaDataTasks(env, heartbeatEvents);
        }
        LOG.info("{} events to handle in current batch", (Object)timbermillEvents.size());
        if (!timbermillEvents.isEmpty()) {
            IndexEvent indexEvent = this.handleTimbermillEvents(env, taskIndexerStartTime, timbermillEvents);
            this.es.indexMetaDataTask(env, Constants.GSON.toJson((Object)indexEvent));
        }
        LOG.info("------------------ Batch End ------------------");
    }

    private IndexEvent handleTimbermillEvents(String env, ZonedDateTime taskIndexerStartTime, Collection<Event> timbermillEvents) {
        this.applyPlugins(timbermillEvents, env);
        HashMap nodesMap = Maps.newHashMap();
        HashSet startEventsIds = Sets.newHashSet();
        HashSet parentIds = Sets.newHashSet();
        HashMap eventsMap = Maps.newHashMap();
        this.populateCollections(timbermillEvents, nodesMap, startEventsIds, parentIds, eventsMap);
        Map<String, Task> previouslyIndexedParentTasks = this.getMissingParents(startEventsIds, parentIds);
        this.connectNodesByParentId(nodesMap);
        Map<String, Task> tasksMap = this.createEnrichedTasks(nodesMap, eventsMap, previouslyIndexedParentTasks);
        String index = this.es.createTimbermillAlias(env);
        this.es.index(tasksMap, index);
        this.es.rolloverIndex(index);
        LOG.info("{} tasks were indexed to elasticsearch", (Object)tasksMap.size());
        return new IndexEvent(env, previouslyIndexedParentTasks.size(), taskIndexerStartTime, ZonedDateTime.now(), timbermillEvents.size(), this.daysRotation);
    }

    private void populateCollections(Collection<Event> timbermillEvents, Map<String, DefaultMutableTreeNode> nodesMap, Set<String> startEventsIds, Set<String> parentIds, Map<String, List<Event>> eventsMap) {
        timbermillEvents.forEach(event -> {
            if (event.isStartEvent()) {
                startEventsIds.add(event.getTaskId());
                nodesMap.put(event.getTaskId(), new DefaultMutableTreeNode(event));
            }
            if (event.getParentId() != null) {
                parentIds.add(event.getParentId());
            }
            if (!eventsMap.containsKey(event.getTaskId())) {
                eventsMap.put(event.getTaskId(), Lists.newArrayList((Object[])new Event[]{event}));
            } else {
                List events = (List)eventsMap.get(event.getTaskId());
                events.add(event);
            }
        });
    }

    private Map<String, Task> getMissingParents(Set<String> startEventsIds, Set<String> parentIds) {
        parentIds.removeAll(startEventsIds);
        LOG.info("Fetching {} missing parents", (Object)parentIds.size());
        Map<Object, Object> previouslyIndexedParentTasks = Maps.newHashMap();
        try {
            previouslyIndexedParentTasks = this.es.fetchIndexedTasks(parentIds);
        }
        catch (Throwable t) {
            LOG.error("Error fetching indexed tasks from Elasticsearch", t);
        }
        LOG.info("Fetched {} parents", (Object)previouslyIndexedParentTasks.size());
        return previouslyIndexedParentTasks;
    }

    private void connectNodesByParentId(Map<String, DefaultMutableTreeNode> nodesMap) {
        for (DefaultMutableTreeNode treeNode : nodesMap.values()) {
            DefaultMutableTreeNode parentNode;
            Event startEvent = (Event)treeNode.getUserObject();
            String parentId = startEvent.getParentId();
            if (parentId == null || (parentNode = nodesMap.get(parentId)) == null) continue;
            parentNode.add(treeNode);
        }
    }

    private Map<String, Task> createEnrichedTasks(Map<String, DefaultMutableTreeNode> nodesMap, Map<String, List<Event>> eventsMap, Map<String, Task> previouslyIndexedParentTasks) {
        this.enrichStartEventsByOrder(nodesMap.values(), eventsMap, previouslyIndexedParentTasks);
        return this.getTasksFromEvents(eventsMap);
    }

    private Map<String, Task> getTasksFromEvents(Map<String, List<Event>> eventsMap) {
        HashMap<String, Task> tasksMap = new HashMap<String, Task>();
        for (Map.Entry<String, List<Event>> eventEntry : eventsMap.entrySet()) {
            Task task = new Task(eventEntry.getValue(), this.daysRotation);
            tasksMap.put(eventEntry.getKey(), task);
        }
        return tasksMap;
    }

    private void enrichStartEventsByOrder(Collection<DefaultMutableTreeNode> nodes, Map<String, List<Event>> eventsMap, Map<String, Task> previouslyIndexedTasks) {
        for (DefaultMutableTreeNode node : nodes) {
            if (!node.isRoot()) continue;
            Enumeration<TreeNode> enumeration = node.breadthFirstEnumeration();
            while (enumeration.hasMoreElements()) {
                DefaultMutableTreeNode curr = (DefaultMutableTreeNode)enumeration.nextElement();
                Event startEvent = (Event)curr.getUserObject();
                this.enrichStartEvent(eventsMap, previouslyIndexedTasks, startEvent);
            }
        }
    }

    public static void logErrorInEventsMap(Map<String, List<Event>> eventsMap, String where) {
        for (Map.Entry<String, List<Event>> stringListEntry : eventsMap.entrySet()) {
            List<Event> value = stringListEntry.getValue();
            if (value.stream().filter(Event::isStartEvent).count() <= 1L) continue;
            LOG.warn("Too many start events in {} events: {}", (Object)where, (Object)Constants.GSON.toJson(value));
        }
    }

    private void updateAdoptedOrphans(Map<String, List<Event>> eventsMap, String parentTaskId) {
        Queue adoptedEvents = (Queue)this.parentIdTORootOrphansEventsCache.getIfPresent((Object)parentTaskId);
        if (adoptedEvents != null) {
            this.parentIdTORootOrphansEventsCache.invalidate((Object)parentTaskId);
            for (AdoptedEvent adoptedEvent : adoptedEvents) {
                this.populateParentParams(adoptedEvent, null, (Collection<Event>)eventsMap.get(parentTaskId));
                String adoptedId = adoptedEvent.getTaskId();
                if (eventsMap.containsKey(adoptedId)) {
                    eventsMap.get(adoptedId).add(adoptedEvent);
                } else {
                    eventsMap.put(adoptedId, Lists.newArrayList((Object[])new Event[]{adoptedEvent}));
                }
                this.updateAdoptedOrphans(eventsMap, adoptedId);
            }
        }
    }

    private void enrichStartEvent(Map<String, List<Event>> eventsMap, Map<String, Task> previouslyIndexedTasks, Event startEvent) {
        String parentId = startEvent.getParentId();
        if (parentId != null) {
            if (this.isOrphan(startEvent, previouslyIndexedTasks, eventsMap)) {
                startEvent.setOrphan(Boolean.valueOf(true));
                startEvent.setPrimaryId(null);
                this.addOrphanToCache(startEvent, parentId);
            } else {
                this.populateParentParams(startEvent, previouslyIndexedTasks.get(parentId), (Collection<Event>)eventsMap.get(parentId));
            }
        } else {
            startEvent.setPrimaryId(startEvent.getTaskId());
        }
        if (this.hasAdoptedOrphans(startEvent)) {
            this.updateAdoptedOrphans(eventsMap, startEvent.getTaskId());
        }
    }

    private boolean hasAdoptedOrphans(Event event) {
        String taskId = event.getTaskId();
        Queue orphansEvents = (Queue)this.parentIdTORootOrphansEventsCache.getIfPresent((Object)taskId);
        return orphansEvents != null && (event.isOrphan() == null || event.isOrphan() == false);
    }

    private void populateParentParams(Event event, Task parentIndexedTask, Collection<Event> parentCurrentEvent) {
        String parentName;
        ParentProperties parentProperties = TaskIndexer.getParentProperties(parentIndexedTask, parentCurrentEvent);
        ArrayList<String> parentsPath = new ArrayList<String>();
        String primaryId = parentProperties.getPrimaryId();
        event.setPrimaryId(primaryId);
        if (event.getContext() == null) {
            event.setContext((Map)Maps.newHashMap());
        }
        for (Map.Entry<String, String> entry : parentProperties.getContext().entrySet()) {
            event.getContext().putIfAbsent(entry.getKey(), entry.getValue());
        }
        Collection<String> parentParentsPath = parentProperties.getParentPath();
        if (parentParentsPath != null && !parentParentsPath.isEmpty()) {
            parentsPath.addAll(parentParentsPath);
        }
        if ((parentName = parentProperties.getParentName()) != null) {
            parentsPath.add(parentName);
        }
        if (!parentsPath.isEmpty()) {
            event.setParentsPath(parentsPath);
        }
    }

    private void addOrphanToCache(Event startEvent, String parentId) {
        LinkedList<AdoptedEvent> eventList = (LinkedList<AdoptedEvent>)this.parentIdTORootOrphansEventsCache.getIfPresent((Object)parentId);
        AdoptedEvent orphanEvent = new AdoptedEvent(startEvent);
        if (eventList == null) {
            eventList = new LinkedList<AdoptedEvent>();
            eventList.add(orphanEvent);
            this.parentIdTORootOrphansEventsCache.put((Object)parentId, eventList);
        } else {
            eventList.add(orphanEvent);
        }
    }

    private boolean isOrphan(Event event, Map<String, Task> previouslyIndexedTasks, Map<String, List<Event>> eventsMap) {
        String parentId = event.getParentId();
        if (parentId == null) {
            return false;
        }
        if (previouslyIndexedTasks.containsKey(parentId)) {
            return false;
        }
        if (eventsMap.containsKey(parentId)) {
            if (eventsMap.get(parentId).stream().anyMatch(Event::isAdoptedEvent)) {
                return false;
            }
            if (eventsMap.get(parentId).stream().anyMatch(Event::isStartEvent)) {
                List parentEvents = eventsMap.get(parentId).stream().filter(Event::isStartEvent).collect(Collectors.toList());
                if (parentEvents.size() != 1) {
                    LOG.warn("Problem with parent events. Events: {}", (Object)Constants.GSON.toJson(parentEvents));
                }
                for (Event e : parentEvents) {
                    if (e.isOrphan() == null || !e.isOrphan().booleanValue()) continue;
                    return true;
                }
                return false;
            }
        }
        return true;
    }

    private void applyPlugins(Collection<Event> events, String env) {
        try {
            for (TaskLogPlugin plugin : this.logPlugins) {
                TaskStatus status;
                ZonedDateTime startTime = ZonedDateTime.now();
                String exception = null;
                try {
                    plugin.apply(events);
                    status = TaskStatus.SUCCESS;
                }
                catch (Exception ex) {
                    exception = ExceptionUtils.getStackTrace((Throwable)ex);
                    status = TaskStatus.ERROR;
                    LOG.error("error in plugin" + plugin, (Throwable)ex);
                }
                ZonedDateTime endTime = ZonedDateTime.now();
                long duration = ElasticsearchUtil.getTimesDuration(startTime, endTime);
                PluginApplierTask pluginApplierTask = new PluginApplierTask(env, plugin.getName(), plugin.getClass().getSimpleName(), status, exception, endTime, duration, startTime, this.daysRotation);
                this.es.indexMetaDataTask(env, Constants.GSON.toJson((Object)pluginApplierTask));
            }
        }
        catch (Throwable t) {
            LOG.error("Error running plugins", t);
        }
    }

    private static ParentProperties getParentProperties(Task parentIndexedTask, Collection<Event> parentCurrentEvent) {
        HashMap context = Maps.newHashMap();
        String primaryId = null;
        List parentPath = null;
        String parentName = null;
        if (parentCurrentEvent != null && !parentCurrentEvent.isEmpty()) {
            for (Event previousEvent : parentCurrentEvent) {
                String previousName;
                List previousPath;
                Map previousContext;
                String previousPrimaryId = previousEvent.getPrimaryId();
                if (previousPrimaryId != null) {
                    primaryId = previousPrimaryId;
                }
                if ((previousContext = previousEvent.getContext()) != null) {
                    context.putAll(previousContext);
                }
                if ((previousPath = previousEvent.getParentsPath()) != null) {
                    parentPath = previousPath;
                }
                if ((previousName = previousEvent.getName()) == null) continue;
                parentName = previousName;
            }
        }
        if (parentIndexedTask != null) {
            String indexedName;
            List indexedParentsPath;
            Map<String, String> indexedContext;
            String indexedPrimary = parentIndexedTask.getPrimaryId();
            if (indexedPrimary != null) {
                primaryId = indexedPrimary;
            }
            if ((indexedContext = parentIndexedTask.getCtx()) != null) {
                context.putAll(indexedContext);
            }
            if ((indexedParentsPath = parentIndexedTask.getParentsPath()) != null) {
                parentPath = indexedParentsPath;
            }
            if ((indexedName = parentIndexedTask.getName()) != null) {
                parentName = indexedName;
            }
        }
        return new ParentProperties(primaryId, context, parentPath, parentName);
    }

    public void close() {
        this.es.close();
    }

    public Cache<String, Queue<AdoptedEvent>> getParentIdTORootOrphansEventsCache() {
        return this.parentIdTORootOrphansEventsCache;
    }

    private static class ParentProperties {
        private final String primaryId;
        private final Map<String, String> context;
        private final Collection<String> parentPath;
        private final String parentName;

        ParentProperties(String primaryId, Map<String, String> context, Collection<String> parentPath, String parentName) {
            this.primaryId = primaryId;
            this.context = context;
            this.parentPath = parentPath;
            this.parentName = parentName;
        }

        String getPrimaryId() {
            return this.primaryId;
        }

        Map<String, String> getContext() {
            return this.context;
        }

        Collection<String> getParentPath() {
            return this.parentPath;
        }

        String getParentName() {
            return this.parentName;
        }
    }
}

