/*
 * Decompiled with CFR 0.152.
 */
package com.davfx.ninio.snmp;

import com.davfx.ninio.core.Address;
import com.davfx.ninio.core.Closeable;
import com.davfx.ninio.core.CloseableByteBufferHandler;
import com.davfx.ninio.core.FailableCloseableByteBufferHandler;
import com.davfx.ninio.core.Queue;
import com.davfx.ninio.core.Ready;
import com.davfx.ninio.core.ReadyConnection;
import com.davfx.ninio.core.ReadyFactory;
import com.davfx.ninio.snmp.AuthRemoteEngine;
import com.davfx.ninio.snmp.Oid;
import com.davfx.ninio.snmp.Result;
import com.davfx.ninio.snmp.SnmpClientHandler;
import com.davfx.ninio.snmp.Version2cPacketBuilder;
import com.davfx.ninio.snmp.Version2cPacketParser;
import com.davfx.ninio.snmp.Version3PacketBuilder;
import com.davfx.ninio.snmp.Version3PacketParser;
import com.davfx.ninio.util.QueueScheduled;
import com.davfx.util.ConfigUtils;
import com.davfx.util.DateUtils;
import com.typesafe.config.Config;
import com.typesafe.config.ConfigFactory;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class SnmpClient
implements AutoCloseable,
Closeable {
    private static final Logger LOGGER = LoggerFactory.getLogger(SnmpClient.class);
    private static final Config CONFIG = ConfigFactory.load((ClassLoader)SnmpClient.class.getClassLoader());
    private static final int BULK_SIZE = CONFIG.getInt("ninio.snmp.bulkSize");
    private static final double MIN_TIME_TO_REPEAT = ConfigUtils.getDuration((Config)CONFIG, (String)"ninio.snmp.minTimeToRepeat");
    private static final int GET_LIMIT = CONFIG.getInt("ninio.snmp.getLimit");
    private static final double REPEAT_TIME = ConfigUtils.getDuration((Config)CONFIG, (String)"ninio.snmp.repeatTime");
    private static final double REPEAT_RANDOMIZATION = ConfigUtils.getDuration((Config)CONFIG, (String)"ninio.snmp.repeatRandomization");
    private final Queue queue;
    private final ReadyFactory readyFactory;
    private final Address address;
    private final String community;
    private final AuthRemoteEngine authEngine;
    private final double timeoutFromBeginning;
    private final Closeable closeable;
    private final RequestIdProvider requestIdProvider = new RequestIdProvider();
    private final Set<InstanceMapper> instanceMappers = new HashSet<InstanceMapper>();
    private static final Random RANDOM = new Random(System.currentTimeMillis());

    private SnmpClient(Queue queue, ReadyFactory readyFactory, Address address, String community, AuthRemoteEngine authEngine, double timeoutFromBeginning) {
        this.queue = queue;
        this.readyFactory = readyFactory;
        this.address = address;
        this.community = community;
        this.authEngine = authEngine;
        this.timeoutFromBeginning = timeoutFromBeginning;
        this.closeable = QueueScheduled.schedule((Queue)queue, (double)REPEAT_TIME, (Runnable)new Runnable(){

            @Override
            public void run() {
                double now = DateUtils.now();
                for (InstanceMapper i : SnmpClient.this.instanceMappers) {
                    i.repeat(now);
                }
            }
        });
    }

    public SnmpClient(Queue queue, ReadyFactory readyFactory, Address address, String community, double timeoutFromBeginning) {
        this(queue, readyFactory, address, community, null, timeoutFromBeginning);
    }

    public SnmpClient(Queue queue, ReadyFactory readyFactory, Address address, AuthRemoteEngine authEngine, double timeoutFromBeginning) {
        this(queue, readyFactory, address, null, authEngine, timeoutFromBeginning);
    }

    @Override
    public void close() {
        this.closeable.close();
        this.queue.post(new Runnable(){

            @Override
            public void run() {
                for (InstanceMapper i : SnmpClient.this.instanceMappers) {
                    i.close();
                }
                SnmpClient.this.instanceMappers.clear();
            }
        });
    }

    public void connect(final SnmpClientHandler clientHandler) {
        this.queue.post(new Runnable(){

            @Override
            public void run() {
                Ready ready = SnmpClient.this.readyFactory.create(SnmpClient.this.queue);
                final InstanceMapper instanceMapper = new InstanceMapper(SnmpClient.this.address, SnmpClient.this.requestIdProvider);
                SnmpClient.this.instanceMappers.add(instanceMapper);
                ready.connect(SnmpClient.this.address, new ReadyConnection(){

                    public void handle(Address address, ByteBuffer buffer) {
                        Iterable<Result> results;
                        int errorIndex;
                        int errorStatus;
                        int instanceId;
                        try {
                            if (SnmpClient.this.authEngine == null) {
                                Version2cPacketParser parser = new Version2cPacketParser(buffer);
                                instanceId = parser.getRequestId();
                                errorStatus = parser.getErrorStatus();
                                errorIndex = parser.getErrorIndex();
                                results = parser.getResults();
                            } else {
                                Version3PacketParser parser = new Version3PacketParser(SnmpClient.this.authEngine, buffer);
                                instanceId = parser.getRequestId();
                                errorStatus = parser.getErrorStatus();
                                errorIndex = parser.getErrorIndex();
                                results = parser.getResults();
                            }
                        }
                        catch (Exception e) {
                            LOGGER.error("Invalid packet", (Throwable)e);
                            return;
                        }
                        instanceMapper.handle(instanceId, errorStatus, errorIndex, results);
                    }

                    public void failed(IOException e) {
                        if (SnmpClient.this.instanceMappers.remove(instanceMapper)) {
                            clientHandler.failed(e);
                        }
                    }

                    public void connected(final FailableCloseableByteBufferHandler write) {
                        final SnmpWriter w = new SnmpWriter((CloseableByteBufferHandler)write, SnmpClient.this.community, SnmpClient.this.authEngine);
                        clientHandler.launched(new SnmpClientHandler.Callback(){

                            public void close() {
                                if (SnmpClient.this.instanceMappers.remove(instanceMapper)) {
                                    // empty if block
                                }
                                write.close();
                            }

                            @Override
                            public void get(Oid oid, SnmpClientHandler.Callback.GetCallback callback) {
                                Instance i = new Instance(instanceMapper, callback, w, oid, SnmpClient.this.timeoutFromBeginning);
                                instanceMapper.map(i);
                                w.get(SnmpClient.this.address, i.instanceId, oid);
                            }
                        });
                    }

                    public void close() {
                        if (SnmpClient.this.instanceMappers.remove(instanceMapper)) {
                            clientHandler.close();
                        }
                    }
                });
            }
        });
    }

    private static final class Instance {
        private final InstanceMapper instanceMapper;
        private SnmpClientHandler.Callback.GetCallback callback;
        private final SnmpWriter write;
        private final Oid initialRequestOid;
        private Oid requestOid;
        private int countResults = 0;
        private final double timeoutFromBeginning;
        private final double beginningTimestamp = DateUtils.now();
        private double sendTimestamp = DateUtils.now();
        private int shouldRepeatWhat = 160;
        public int instanceId;
        private final double repeatRandomizationRandomized;

        public Instance(InstanceMapper instanceMapper, SnmpClientHandler.Callback.GetCallback callback, SnmpWriter write, Oid requestOid, double timeoutFromBeginning) {
            this.instanceMapper = instanceMapper;
            this.callback = callback;
            this.write = write;
            this.requestOid = requestOid;
            this.timeoutFromBeginning = timeoutFromBeginning;
            this.initialRequestOid = requestOid;
            this.repeatRandomizationRandomized = RANDOM.nextDouble() * REPEAT_RANDOMIZATION - 0.5;
        }

        public void close() {
            this.write.close();
            if (this.callback == null) {
                return;
            }
            if (this.requestOid == null) {
                return;
            }
            this.shouldRepeatWhat = 0;
            this.requestOid = null;
            SnmpClientHandler.Callback.GetCallback c = this.callback;
            this.callback = null;
            c.failed(new IOException("Closed"));
        }

        public void repeat(double now) {
            if (this.callback == null) {
                return;
            }
            if (this.requestOid == null) {
                return;
            }
            double t = now - this.beginningTimestamp;
            if (t >= this.timeoutFromBeginning) {
                this.shouldRepeatWhat = 0;
                this.requestOid = null;
                SnmpClientHandler.Callback.GetCallback c = this.callback;
                this.callback = null;
                c.failed(new IOException("Timeout from beginning [" + t + " seconds] requesting: " + this.instanceMapper.address + " " + this.initialRequestOid));
                return;
            }
            if (now - this.sendTimestamp >= MIN_TIME_TO_REPEAT + this.repeatRandomizationRandomized) {
                LOGGER.trace("Repeating {} {}", (Object)this.instanceMapper.address, (Object)this.requestOid);
                switch (this.shouldRepeatWhat) {
                    case 160: {
                        this.write.get(this.instanceMapper.address, this.instanceId, this.requestOid);
                        break;
                    }
                    case 161: {
                        this.write.getNext(this.instanceMapper.address, this.instanceId, this.requestOid);
                        break;
                    }
                    case 165: {
                        this.write.getBulk(this.instanceMapper.address, this.instanceId, this.requestOid, BULK_SIZE);
                        break;
                    }
                }
            }
        }

        private void handle(int errorStatus, int errorIndex, Iterable<Result> results) {
            if (this.callback == null) {
                LOGGER.trace("Received more but finished");
                return;
            }
            if (this.requestOid == null) {
                return;
            }
            if (errorStatus == -2) {
                this.requestOid = null;
                SnmpClientHandler.Callback.GetCallback c = this.callback;
                this.callback = null;
                c.failed(new IOException("Authentication failed"));
                return;
            }
            if (this.shouldRepeatWhat == 160) {
                if (errorStatus == -1) {
                    LOGGER.trace("Retrying GET after receiving auth engine completion message");
                    this.instanceMapper.map(this);
                    this.sendTimestamp = DateUtils.now();
                    this.write.get(this.instanceMapper.address, this.instanceId, this.requestOid);
                } else {
                    boolean fallback = false;
                    if (errorStatus != 0) {
                        LOGGER.trace("Fallbacking to GETNEXT/GETBULK after receiving error: {}/{}", (Object)errorStatus, (Object)errorIndex);
                        fallback = true;
                    } else {
                        Result found = null;
                        for (Result r : results) {
                            if (r.getValue() == null) {
                                LOGGER.trace(r.getOid() + " fallback to GETNEXT/GETBULK");
                                fallback = true;
                                break;
                            }
                            if (!this.requestOid.equals(r.getOid())) {
                                LOGGER.trace("{} not as expected: {}, fallbacking to GETNEXT/GETBULK", (Object)r.getOid(), (Object)this.requestOid);
                                fallback = true;
                                break;
                            }
                            LOGGER.trace("Scalar found: {}", (Object)r);
                            found = r;
                        }
                        if (found != null) {
                            this.requestOid = null;
                            SnmpClientHandler.Callback.GetCallback c = this.callback;
                            this.callback = null;
                            c.result(found);
                            c.close();
                            return;
                        }
                    }
                    if (fallback) {
                        this.instanceMapper.map(this);
                        this.sendTimestamp = DateUtils.now();
                        this.shouldRepeatWhat = 165;
                        this.write.getBulk(this.instanceMapper.address, this.instanceId, this.requestOid, BULK_SIZE);
                    }
                }
            } else if (errorStatus != 0) {
                this.requestOid = null;
                SnmpClientHandler.Callback.GetCallback c = this.callback;
                this.callback = null;
                c.close();
            } else {
                Oid lastOid = null;
                for (Result r : results) {
                    LOGGER.trace("Received in bulk: {}", (Object)r);
                }
                for (Result r : results) {
                    if (r.getValue() == null) continue;
                    if (!this.initialRequestOid.isPrefix(r.getOid())) {
                        LOGGER.trace("{} not prefixed by {}", (Object)r.getOid(), (Object)this.initialRequestOid);
                        lastOid = null;
                        break;
                    }
                    LOGGER.trace("Addind to results: {}", (Object)r);
                    if (GET_LIMIT > 0 && this.countResults >= GET_LIMIT) {
                        LOGGER.warn("{} reached limit", (Object)this.requestOid);
                        lastOid = null;
                        break;
                    }
                    ++this.countResults;
                    this.callback.result(r);
                    lastOid = r.getOid();
                }
                if (lastOid != null) {
                    LOGGER.trace("Continuing from: {}", lastOid);
                    this.requestOid = lastOid;
                    this.instanceMapper.map(this);
                    this.sendTimestamp = DateUtils.now();
                    this.shouldRepeatWhat = 165;
                    this.write.getBulk(this.instanceMapper.address, this.instanceId, this.requestOid, BULK_SIZE);
                } else {
                    this.requestOid = null;
                    SnmpClientHandler.Callback.GetCallback c = this.callback;
                    this.callback = null;
                    c.close();
                }
            }
        }
    }

    private static final class SnmpWriter {
        private final CloseableByteBufferHandler write;
        private final String community;
        private final AuthRemoteEngine authEngine;

        public SnmpWriter(CloseableByteBufferHandler write, String community, AuthRemoteEngine authEngine) {
            this.write = write;
            this.community = community;
            this.authEngine = authEngine;
        }

        public void close() {
            this.write.close();
        }

        public void get(Address to, int instanceId, Oid oid) {
            if (this.authEngine == null) {
                Version2cPacketBuilder builder = Version2cPacketBuilder.get(this.community, instanceId, oid);
                LOGGER.trace("Writing GET: {} #{} ({})", new Object[]{oid, instanceId, this.community});
                this.write.handle(to, builder.getBuffer());
            } else {
                Version3PacketBuilder builder = Version3PacketBuilder.get(this.authEngine, instanceId, oid);
                this.write.handle(to, builder.getBuffer());
            }
        }

        public void getNext(Address to, int instanceId, Oid oid) {
            if (this.authEngine == null) {
                Version2cPacketBuilder builder = Version2cPacketBuilder.getNext(this.community, instanceId, oid);
                LOGGER.trace("Writing GETNEXT: {} #{} ({})", new Object[]{oid, instanceId, this.community});
                this.write.handle(to, builder.getBuffer());
            } else {
                Version3PacketBuilder builder = Version3PacketBuilder.getNext(this.authEngine, instanceId, oid);
                this.write.handle(to, builder.getBuffer());
            }
        }

        public void getBulk(Address to, int instanceId, Oid oid, int bulkLength) {
            if (this.authEngine == null) {
                Version2cPacketBuilder builder = Version2cPacketBuilder.getBulk(this.community, instanceId, oid, bulkLength);
                LOGGER.trace("Writing GETBULK: {} #{} ({})", new Object[]{oid, instanceId, this.community});
                this.write.handle(to, builder.getBuffer());
            } else {
                Version3PacketBuilder builder = Version3PacketBuilder.getBulk(this.authEngine, instanceId, oid, bulkLength);
                this.write.handle(to, builder.getBuffer());
            }
        }
    }

    private static final class InstanceMapper {
        private final Address address;
        private final Map<Integer, Instance> instances = new HashMap<Integer, Instance>();
        private RequestIdProvider requestIdProvider;

        public InstanceMapper(Address address, RequestIdProvider requestIdProvider) {
            this.address = address;
            this.requestIdProvider = requestIdProvider;
        }

        public void map(Instance instance) {
            int instanceId = this.requestIdProvider.get();
            if (this.instances.containsKey(instanceId)) {
                LOGGER.warn("The maximum number of simultaneous request has been reached [{}]", (Object)this.address);
                return;
            }
            this.instances.put(instanceId, instance);
            instance.instanceId = instanceId;
        }

        public void close() {
            for (Instance i : this.instances.values()) {
                i.close();
            }
            this.instances.clear();
        }

        public void handle(int instanceId, int errorStatus, int errorIndex, Iterable<Result> results) {
            Instance i = this.instances.remove(instanceId);
            if (i == null) {
                return;
            }
            i.handle(errorStatus, errorIndex, results);
        }

        public void repeat(double now) {
            for (Instance i : this.instances.values()) {
                i.repeat(now);
            }
            Iterator<Instance> ii = this.instances.values().iterator();
            while (ii.hasNext()) {
                Instance i;
                i = ii.next();
                if (i.callback != null) continue;
                ii.remove();
            }
        }
    }

    private static final class RequestIdProvider {
        private static final int LOOP_REQUEST_ID = 65536;
        private static final AtomicInteger PREFIX = new AtomicInteger(SnmpClient.access$100().nextInt());
        private final int prefix = PREFIX.getAndIncrement();
        private int nextRequestId = 0;

        public int get() {
            int id = (this.prefix & 0xFFFF) << 16 | this.nextRequestId & 0xFFFF;
            ++this.nextRequestId;
            if (this.nextRequestId == 65536) {
                this.nextRequestId = 0;
            }
            return id;
        }
    }
}

