/*
 * Copyright 2020 Google LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.google.cloud.tools.opensource.dependencies;

import static com.google.common.collect.ImmutableSet.toImmutableSet;

import com.google.cloud.tools.opensource.classpath.ClassFile;
import com.google.cloud.tools.opensource.classpath.ClassPathBuilder;
import com.google.cloud.tools.opensource.classpath.ClassPathEntry;
import com.google.cloud.tools.opensource.classpath.ClassPathResult;
import com.google.cloud.tools.opensource.classpath.LinkageChecker;
import com.google.cloud.tools.opensource.classpath.LinkageProblem;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.ImmutableSetMultimap;
import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.logging.Logger;
import org.eclipse.aether.artifact.Artifact;

/**
 * A tool to find Java 8-incompatible references in the class files in a BOM. It checks the class
 * files in the following artifacts:
 *
 * <ul>
 *   <li>The artifacts in the {@code dependencyManagement} section of the BOM
 *   <li>The artifacts in the dependency graph of each of the artifacts above
 *   <li>The latest version of the artifacts in the dependency graphs
 * </ul>
 *
 * <p>If it finds references to classes in the core library (the {@code java} package) which are not
 * present in Java 8, it exits with status code 1; otherwise it exits with status code 0.
 *
 * @see <a
 *     href="https://github.com/GoogleCloudPlatform/cloud-opensource-java/wiki/Java-8-incompatible-references-of-java.nio.Buffer-classes-generated-by-Java-9--compilers"
 *     >Java 8 incompatible references of java.nio.Buffer classes generated by Java 9 compilers</a>
 */
public class Java8IncompatibleReferenceCheck {

  private static final Logger logger =
      Logger.getLogger(Java8IncompatibleReferenceCheck.class.getName());

  public static void main(String[] arguments) throws MavenRepositoryException, IOException {

    if (arguments.length < 2) {
      System.err.println("Specify a path to the BOM file and a path to the exclusion rule");
      System.exit(1);
    }

    String bomFileName = arguments[0];
    Path exclusionFile = Paths.get(arguments[1]).toAbsolutePath();

    Path bomFile = Paths.get(bomFileName);
    Bom bom = Bom.readBom(bomFile);

    ImmutableList<Artifact> managedDependencies = bom.getManagedDependencies();

    int count = 1;

    // The BOM member to problematic dependencies
    ImmutableSetMultimap.Builder<Artifact, Artifact> problematicDependencies =
        ImmutableSetMultimap.builder();
    for (Artifact managedDependency : managedDependencies) {
      logger.info(
          "Checking "
              + managedDependency
              + " ("
              + (count++)
              + "/"
              + managedDependencies.size()
              + ")");

      ClassPathBuilder classPathBuilder = new ClassPathBuilder();
      ClassPathResult result = classPathBuilder.resolve(ImmutableList.of(managedDependency), false);

      LinkageChecker linkageChecker =
          LinkageChecker.create(result.getClassPath(), result.getClassPath(), exclusionFile);

      ImmutableSet<LinkageProblem> linkageProblems = linkageChecker.findLinkageProblems();

      ImmutableSet<LinkageProblem> invalidReferencesToJavaCoreLibrary =
          linkageProblems.stream()
              .filter(problem -> problem.getSymbol().getClassBinaryName().startsWith("java."))
              .collect(toImmutableSet());

      if (!invalidReferencesToJavaCoreLibrary.isEmpty()) {
        invalidReferencesToJavaCoreLibrary.stream()
            .map(LinkageProblem::getSourceClass)
            .map(ClassFile::getClassPathEntry)
            .map(ClassPathEntry::getArtifact)
            .forEach(artifact -> problematicDependencies.put(managedDependency, artifact));

        // No need to supply classPath result as the artifact information is output below
        logger.severe(
            LinkageProblem.formatLinkageProblems(invalidReferencesToJavaCoreLibrary, null));
      }
    }

    ImmutableSetMultimap<Artifact, Artifact> bomMemberToProblematicDependencies =
        problematicDependencies.build();

    if (bomMemberToProblematicDependencies.isEmpty()) {
      logger.info("No problematic artifacts");
      return;
    }

    StringBuilder message = new StringBuilder();
    message.append(
        "The following artifacts contain references to classes in the core library,"
            + " which are not present in Java 8\n");
    for (Artifact artifact : bomMemberToProblematicDependencies.inverse().keySet()) {
      message.append("  " + artifact + "\n");
    }
    message.append(
        "The following artifacts in the BOM contain the artifacts in their dependencies\n");
    for (Artifact bomMember : bomMemberToProblematicDependencies.keySet()) {
      ImmutableSet<Artifact> dependencies = bomMemberToProblematicDependencies.get(bomMember);
      message.append("  " + bomMember + " due to " + dependencies + "\n");
    }
    logger.severe(message.toString());
    System.exit(1);
  }
}
