/**
 * Copyright © 2012, 2013 dr. ir. Jeroen M. Valk
 * 
 * This file is part of Badgerfish CPX. Badgerfish CPX is free software:
 * you can redistribute it and/or modify it under the terms of the GNU Lesser
 * General Public License as published by the Free Software Foundation, either
 * version 3 of the License, or (at your option) any later version. Badgerfish
 * CPX is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE. See the GNU Lesser General Public License for more details. You
 * should have received a copy of the GNU Lesser General Public License along
 * with Badgerfish CPX. If not, see <http://www.gnu.org/licenses/>.
 */

package com.googlecode.aaw.mojo;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileReader;

import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.w3c.dom.Element;
import org.xml.sax.InputSource;

import com.googlecode.aaw.arche.Arche;
import com.googlecode.aaw.badgerfish.Badgerfish;
import com.googlecode.aaw.badgerfish.NodeJS;

/**
 * Compile Javascript written in the Badgerfish framework.
 * 
 * @goal compile
 * 
 * @phase compile
 */
public class Compile extends AbstractMojo {

	/**
	 * Directory that contains the main Javascript source files.
	 * 
	 * @parameter expression="${basedir}/src/main/javascript"
	 * @required
	 */
	private File sourceMain;

	/**
	 * Directory that contains the unit test sources.
	 * 
	 * @parameter expression="${basedir}/src/test/javascript"
	 * @required
	 */
	private File sourceTest;

	/**
	 * Output directory where compiled Javascript files will be written.
	 * 
	 * @parameter expression="${target}"
	 * @required
	 */
	private File outputDirectory;

	/**
	 * Directory where unpacked dependencies are located.
	 * 
	 * @parameter expression="${project.build.directory}/dependency"
	 * @required
	 */
	private File dependencyDirectory;

	/**
	 * ArtifactId of the compiler to use.
	 * 
	 * @parameter default-value="badgerfish"
	 * @required
	 */
	private String compilerArtifactId;

	/**
	 * Version of the compiler to use.
	 * 
	 * @parameter expression="${version.badgerfish}"
	 * @required
	 */
	private String compilerVersion;

	/**
	 * Class to use as a bootstrap, e.g., for compiling the compiler itself.
	 * 
	 * @parameter
	 */
	private String bootstrap;

	public void execute() throws MojoFailureException, MojoExecutionException {
		Arche arche = Arche.getArche(Compile.class, new LoggerFactory(getLog()));
		Logger logger = (Logger) arche.getInstance();
		if (!sourceMain.isDirectory() && !sourceTest.isDirectory()) {
			logger.info("nothing to compile");
			return;
		}
		Badgerfish badgerfish = null;
		if (bootstrap != null) {
			badgerfish = new Badgerfish(null, dependencyDirectory);
			badgerfish.startDocument();
			badgerfish.startElement("badgerfish");
			badgerfish.startElement("config");
			badgerfish.element("outputDirectory", outputDirectory.getPath());
			badgerfish.endElement();

			String path = bootstrap.replaceAll("\\.", "/");
			File script = new File(sourceMain, path + ".js");
			try {
				badgerfish.element("script", new FileReader(script));
			} catch (FileNotFoundException e) {
				logger.error(e.getMessage());
			}
		} else {
			InputSource input = new InputSource();
			try {
				input.setByteStream(new FileInputStream(new File(
						dependencyDirectory, compilerArtifactId + "-"
								+ compilerVersion + "/classes.xml")));
			} catch (FileNotFoundException e) {
				logger.error(e.getMessage(), e);
			}
			badgerfish = new Badgerfish(input, dependencyDirectory);
			badgerfish.startDocument();
			badgerfish.startElement("badgerfish");
			badgerfish.startElement("config");
			badgerfish.element("sourceMain", sourceMain.getPath());
			badgerfish.element("sourceTest", sourceTest.getPath());
			badgerfish.element("outputDirectory", outputDirectory.getPath());
			badgerfish.endElement();
		}
		badgerfish.endElement();
		badgerfish.endDocument();
		if (!outputDirectory.isDirectory()) {
			outputDirectory.mkdirs();
		}
		Element config = badgerfish.selectElement(null, "badgerfish/config");
		logger.info("Main source directory: "
				+ badgerfish.selectText(config, "sourceMain"));
		logger.info("Test source directory: "
				+ badgerfish.selectText(config, "sourceTest"));
		NodeJS node = new NodeJS(badgerfish, 0);
		int status = node.waitForExitValue();
		logger.info("Compiling Javascript sources to: "
				+ badgerfish.toFile().getAbsolutePath());
		if (status != 0) {
			throw new MojoFailureException("NodeJS exited with status: "
					+ status);
		}
		if (logger.getErrorCount() > 0) {
			throw new MojoExecutionException(logger.getErrorCount()
					+ " ERRORS; " + logger.getWarnCount() + " WARNINGS");
		}
	}
}
