/*
 *    Copyright 2009-2010 The iBaGuice Team
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */
package com.googlecode.ibaguice.core;

import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

import javax.sql.DataSource;

import org.apache.ibatis.mapping.Environment;
import org.apache.ibatis.plugin.Interceptor;
import org.apache.ibatis.reflection.factory.ObjectFactory;
import org.apache.ibatis.session.Configuration;
import org.apache.ibatis.transaction.TransactionFactory;
import org.apache.ibatis.type.TypeHandler;

import com.google.inject.Provider;
import com.google.inject.Scopes;
import com.google.inject.TypeLiteral;
import com.google.inject.multibindings.MapBinder;
import com.google.inject.multibindings.Multibinder;
import com.googlecode.ibaguice.core.configuration.ConfigurationProvider;
import com.googlecode.ibaguice.core.configuration.Mappers;
import com.googlecode.ibaguice.core.configuration.TypeAliases;
import com.googlecode.ibaguice.core.datasource.PooledDataSourceProvider;
import com.googlecode.ibaguice.core.environment.EnvironmentProvider;
import com.googlecode.ibaguice.core.transactionfactory.JdbcTransactionFactoryProvider;

/**
 * Easy to use helper Module that alleviates users to write the boilerplate
 * google-guice bindings to create the SqlSessionFactory.
 *
 * @version $Id: SqlSessionFactoryModule.java 2098 2010-06-04 14:13:52Z simone.tripodi $
 */
public final class SqlSessionFactoryModule extends AbstractSqlSessionFactoryModule {

    /**
     * The DataSource Provider class reference.
     */
    private final Class<? extends Provider<DataSource>> dataSourceProviderClass;

    /**
     * The TransactionFactory Provider class reference.
     */
    private final Class<? extends Provider<TransactionFactory>> transactionFactoryProviderClass;

    /**
     * The user defined aliases.
     */
    private final Map<String, Class<?>> aliases = new HashMap<String, Class<?>>();

    /**
     * The user defined type handlers.
     */
    private final Map<Class<?>, Class<? extends TypeHandler>> handlers = new HashMap<Class<?>, Class<? extends TypeHandler>>();

    /**
     * The user defined Interceptor classes.
     */
    private Class<? extends Interceptor>[] interceptorsClasses;

    /**
     * The user defined mapper classes.
     */
    private Class<?>[] mapperClasses;

    /**
     * The ObjectFactory Provider class reference.
     */
    private Class<? extends Provider<ObjectFactory>> objectFactoryProviderClass;

    /**
     * Creates a new module that binds all the needed modules to create the
     * SqlSessionFactory, injecting all the required components.
     *
     * @param dataSourceProviderClass the DataSource Provider class reference.
     * @param transactionFactoryProviderClass the TransactionFactory Provider
     *        class reference.
     */
    public SqlSessionFactoryModule(final Class<? extends Provider<DataSource>> dataSourceProviderClass,
            final Class<? extends Provider<TransactionFactory>> transactionFactoryProviderClass) {
        super(SqlSessionFactoryProvider.class);

        if (dataSourceProviderClass == null) {
            throw new IllegalArgumentException("Data Source provider class mustn't be null");
        }
        this.dataSourceProviderClass = dataSourceProviderClass;

        if (transactionFactoryProviderClass == null) {
            throw new IllegalArgumentException("Transaction Factory provider class mustn't be null");
        }
        this.transactionFactoryProviderClass = transactionFactoryProviderClass;
    }

    /**
     * Adding simple aliases means that every specified class will be bound
     * using the simple class name, i.e.  {@code com.acme.Foo} becomes
     *  {@code Foo}.
     *
     * @param types he specified types have to be bind.
     */
    public void addSimpleAliases(final Class<?>...types) {
        for (Class<?> clazz : types) {
            this.addAlias(clazz.getSimpleName(), clazz);
        }
    }

    /**
     * Add a user defined binding.
     *
     * @param alias the string type alias
     * @param clazz the type has to be bound.
     */
    public void addAlias(final String alias, final Class<?> clazz) {
        this.aliases.put(alias, clazz);
    }

    /**
     * Add a user defined Type Handler letting google-guice creating it.
     *
     * @param type the specified type has to be handled.
     * @param handler the handler type.
     */
    public void addTypeHandler(final Class<?> type, final Class<? extends TypeHandler> handler) {
        this.handlers.put(type, handler);
    }

    /**
     * Sets the user defined iBatis interceptors plugins types, letting
     * google-guice creating them.
     *
     * @param interceptorsClasses the user defined iBatis interceptors plugins
     *        types.
     */
    public void setInterceptorsClasses(Class<? extends Interceptor>...interceptorsClasses) {
        this.interceptorsClasses = interceptorsClasses;
    }

    /**
     * Sets the user defined mapper classes.
     *
     * @param mapperClasses the user defined mapper classes.
     */
    public void setMapperClasses(Class<?>...mapperClasses) {
        this.mapperClasses = mapperClasses;
    }

    /**
     * Sets the ObjectFactory provider class.
     *
     * @param objectFactoryProviderClass the ObjectFactory provider class.
     */
    public void setObjectFactoryProviderClass(Class<? extends Provider<ObjectFactory>> objectFactoryProviderClass) {
        this.objectFactoryProviderClass = objectFactoryProviderClass;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected void configure() {
        this.bind(TransactionFactory.class).toProvider(this.transactionFactoryProviderClass).in(Scopes.SINGLETON);
        this.bind(DataSource.class).toProvider(this.dataSourceProviderClass).in(Scopes.SINGLETON);
        this.bind(Environment.class).toProvider(EnvironmentProvider.class).in(Scopes.SINGLETON);
        this.bind(Configuration.class).toProvider(ConfigurationProvider.class).in(Scopes.SINGLETON);

        super.configure();

        // optional bindings

        // aliases
        if (!this.aliases.isEmpty()) {
            this.bind(new TypeLiteral<Map<String, Class<?>>>(){}).annotatedWith(TypeAliases.class).toInstance(this.aliases);
        }

        // type handlers
        if (!this.handlers.isEmpty()) {
            MapBinder<Class<?>, TypeHandler> handlerBinder =
                MapBinder.newMapBinder(this.binder(), new TypeLiteral<Class<?>>(){}, new TypeLiteral<TypeHandler>(){});
            for (Entry<Class<?>, Class<? extends TypeHandler>> entry : this.handlers.entrySet()) {
                handlerBinder.addBinding(entry.getKey()).to(entry.getValue()).in(Scopes.SINGLETON);
            }
        }

        // interceptors plugin
        if (this.interceptorsClasses != null) {
            Multibinder<Interceptor> cacheMultibinder = Multibinder.newSetBinder(this.binder(), Interceptor.class);
            for (Class<? extends Interceptor> interceptorClass : this.interceptorsClasses) {
                cacheMultibinder.addBinding().to(interceptorClass).in(Scopes.SINGLETON);
            }
        }

        // mappers
        if (this.mapperClasses != null) {
            this.bind(new TypeLiteral<Class<?>[]>() {}).annotatedWith(Mappers.class).toInstance(this.mapperClasses);
        }

        // the object factory
        if (this.objectFactoryProviderClass != null) {
            this.bind(ObjectFactory.class).toProvider(this.objectFactoryProviderClass).in(Scopes.SINGLETON);
        }
    }

    /**
     * Simple {@link SqlSessionFactoryModule} builder through methods chaining.
     */
    public static class Builder {

        private final SqlSessionFactoryModule module;

        /**
         * Creates a new module that binds all the needed modules to create the
         * SqlSessionFactory, injecting all the required components using by default
         * the {@link PooledDataSourceProvider} as DataSourceProvider and the
         * {@link JdbcTransactionFactoryProvider} TransactionFactory provider.
         */
        public Builder() {
            this(PooledDataSourceProvider.class);
        }

        /**
         * Creates a new module that binds all the needed modules to create the
         * SqlSessionFactory, injecting all the required components using by default
         * the {@link JdbcTransactionFactoryProvider} as TransactionFactory provider.
         *
         * @param dataSourceProviderClass the DataSource Provider class reference.
         */
        public Builder(final Class<? extends Provider<DataSource>> dataSourceProviderClass) {
            this(dataSourceProviderClass, JdbcTransactionFactoryProvider.class);
        }

        /**
         * Creates a new module that binds all the needed modules to create the
         * SqlSessionFactory, injecting all the required components.
         *
         * @param dataSourceProviderClass the DataSource Provider class reference.
         * @param transactionFactoryProviderClass the TransactionFactory Provider
         *        class reference.
         */
        public Builder(final Class<? extends Provider<DataSource>> dataSourceProviderClass,
                final Class<? extends Provider<TransactionFactory>> transactionFactoryProviderClass) {
            this.module = new SqlSessionFactoryModule(dataSourceProviderClass, transactionFactoryProviderClass);
        }

        /**
         * Add a user defined binding.
         *
         * @param alias the string type alias
         * @param clazz the type has to be bound.
         * @return this Builder instance.
         */
        public Builder addAlias(String alias, Class<?> clazz) {
            this.module.addAlias(alias, clazz);
            return this;
        }

        /**
         * Adding simple aliases means that every specified class will be bound
         * using the simple class name, i.e.  {@code com.acme.Foo} becomes
         *  {@code Foo}.
         *
         * @param types he specified types have to be bind.
         * @return this Builder instance.
         */
        public Builder addSimpleAliases(Class<?>...types) {
            this.module.addSimpleAliases(types);
            return this;
        }

        /**
         * Add a user defined Type Handler letting google-guice creating it.
         *
         * @param type the specified type has to be handled.
         * @param handler the handler type.
         * @return this Builder instance.
         */
        public Builder addTypeHandler(Class<?> type, Class<? extends TypeHandler> handler) {
            this.module.addTypeHandler(type, handler);
            return this;
        }

        /**
         * Sets the user defined iBatis interceptors plugins types, letting
         * google-guice creating them.
         *
         * @param interceptorsClasses the user defined iBatis interceptors plugins
         *        types.
         * @return this Builder instance.
         */
        public Builder setInterceptorsClasses(Class<? extends Interceptor>...interceptorsClasses) {
            this.module.setInterceptorsClasses(interceptorsClasses);
            return this;
        }

        /**
         * Sets the user defined mapper classes.
         *
         * @param mapperClasses the user defined mapper classes.
         * @return this Builder instance.
         */
        public Builder setMapperClasses(Class<?>...mapperClasses) {
            this.module.setMapperClasses(mapperClasses);
            return this;
        }

        /**
         * Sets the ObjectFactory provider class.
         *
         * @param objectFactoryProviderClass the ObjectFactory provider class.
         * @return this Builder instance.
         */
        public Builder setObjectFactoryProviderClass(Class<? extends Provider<ObjectFactory>> objectFactoryProviderClass) {
            this.module.setObjectFactoryProviderClass(objectFactoryProviderClass);
            return this;
        }

        /**
         * Returns the {@link SqlSessionFactoryModule} instance.
         *
         * @return the {@link SqlSessionFactoryModule} instance.
         */
        public SqlSessionFactoryModule getModule() {
            return this.module;
        }

    }

}
