/*
 *    Copyright 2009-2010 The iBaGuice Team
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */
package com.googlecode.ibaguice.core.dao;

import java.util.List;

import org.apache.ibatis.session.ResultHandler;
import org.apache.ibatis.session.RowBounds;
import org.apache.ibatis.session.SqlSession;

/**
 *
 *
 * @version $Id: SqlSessionRegistry.java 2095 2010-06-04 13:56:24Z simone.tripodi $
 */
public final class SqlSessionRegistry {

    private final ThreadLocal<SqlSession> threadedSessions = new ThreadLocal<SqlSession>();

    /**
     * Return the current {@code SqlSession} handled for current thread.
     *
     * @return The {@link SqlSession} store for current thread.
     */
    protected SqlSession getSqlSession() {
        return this.threadedSessions.get();
    }

    /**
     * Set the {@code SqlSession} for current thread.
     *
     * @param sqlSession
     */
    protected void setSqlSession(SqlSession sqlSession) {
        this.threadedSessions.set(sqlSession);
    }

    /**
     * Remove the {@link SqlSession} stored for current thread.
     */
    protected void removeSqlSessionForCurrentThread(){
        this.threadedSessions.remove();
    }

    private SqlSession getCheckedSqlSession() {
        SqlSession session;
        if ((session = this.getSqlSession()) == null) {
            throw new RuntimeException("SqlSession not set for Thread "
                    + Thread.currentThread().getId()
                    + ", please the interested method is annotated with '"
                    + Transactional.class.getName()
                    + "'");
        }
        return session;
    }

    public Object selectOne(String statement) {
        return this.getCheckedSqlSession().selectOne(statement);
    }

    public Object selectOne(String statement, Object parameter) {
        return this.getCheckedSqlSession().selectOne(statement, parameter);
    }

    @SuppressWarnings("unchecked")
    public List selectList(String statement) {
        return this.getCheckedSqlSession().selectList(statement);
    }

    @SuppressWarnings("unchecked")
    public List selectList(String statement, Object parameter) {
        return this.getCheckedSqlSession().selectList(statement, parameter);
    }

    @SuppressWarnings("unchecked")
    public List selectList(String statement, Object parameter, RowBounds rowBounds) {
        return this.getCheckedSqlSession().selectList(statement, parameter, rowBounds);
    }

    public void select(String statement, Object parameter, ResultHandler handler) {
        this.getCheckedSqlSession().select(statement, parameter, handler);
    }

    public void select(String statement, Object parameter, RowBounds rowBounds, ResultHandler handler) {
        this.getCheckedSqlSession().select(statement, parameter, rowBounds, handler);
    }

    public int insert(String statement) {
        return this.getCheckedSqlSession().insert(statement);
    }

    public int insert(String statement, Object parameter) {
        return this.getCheckedSqlSession().insert(statement, parameter);
    }

    public int update(String statement) {
        return this.getCheckedSqlSession().update(statement);
    }

    public int update(String statement, Object parameter) {
        return this.getCheckedSqlSession().update(statement, parameter);
    }

    public int delete(String statement) {
        return this.getCheckedSqlSession().delete(statement);
    }

    public int delete(String statement, Object parameter) {
        return this.getCheckedSqlSession().delete(statement, parameter);
    }

    public <T> T getMapper(Class<T> type) {
        return this.getCheckedSqlSession().getMapper(type);
    }

}
