/*
 * Copyright 2011 Michael Bedward
 * 
 * This file is part of jai-tools.
 *
 * jai-tools is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 *
 * jai-tools is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public 
 * License along with jai-tools.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package jaitools.numeric;

/**
 * Provides static methods to compare floating point values, taking into account
 * an absolute or proportional tolerance. There are methods for both {@code float} and
 * {@code double} values.
 * The {@code acompare} and {@code aequal} methods use absolute tolerance while
 * the {@code pcompare} and {@code pequal} methods use proportional tolerance.
 * <p>
 * For the proportional tolerance methods, a corresponding absolute tolerance
 * is calculated as:
 * <pre><code>
 *     atol = |ptol| * MAX(|x1|,|x2|)
 * </code></pre>
 * <b>Note:</b> this class does not give any special consideration to the Float 
 * and Double constants {@code NEGATIVE_INFINITY}, {@code POSITIVE_INFINITY}
 * and {@code NaN} over that provided by Java itself.
 * 
 * @author Michael Bedward
 * @since 1.1
 * @version $Id: CompareOp.java 1610 2011-03-31 04:44:28Z michael.bedward $
 */
public class CompareOp {

    /** Default tolerance for double comparisons: 1.0e-8 */
    public static final double DTOL = 1.0e-8d;
    
    /** Default tolerance for float comparisons: 1.0e-4 */
    public static final float FTOL = 1.0e-4f;
    
    /**
     * Tests if the given {@code double} value is within the default tolerance
     * of zero.
     * 
     * @param x the value
     * @return {@code true} if zero; {@code false} otherwise
     */
    public static boolean isZero(double x) {
        return Math.abs(x) < DTOL;
    }
    
    /**
     * Tests if the given {@code float} value is within the default tolerance
     * of zero.
     * 
     * @param x the value
     * @return {@code true} if zero; {@code false} otherwise
     */
    public static boolean isZero(float x) {
        return Math.abs(x) < FTOL;
    }

    /**
     * Tests if the given {@code double} value is within the specified tolerance
     * of zero. Note that performance reasons, {@code tol} is <strong>assumed</strong>
     * to be positive, ie. this is not checked.
     * 
     * @param x the value
     * @param tol the tolerance
     * @return {@code true} if zero; {@code false} otherwise
     */
    public static boolean isZero(double x, double tol) {
        return Math.abs(x) < tol;
    }

    /**
     * Tests if the given {@code float} value is within the specified tolerance
     * of zero. Note that performance reasons, {@code tol} is <strong>assumed</strong>
     * to be positive, ie. this is not checked.
     * 
     * @param x the value
     * @param tol the tolerance
     * @return {@code true} if zero; {@code false} otherwise
     */
    public static boolean isZero(float x, float tol) {
        return Math.abs(x) < tol;
    }

    /**
     * Compares two {@code double} values using the default tolerance.
     * 
     * @param x1 first value
     * @param x2 second value
     * 
     * @return a value less than 0 if x1 is less than x2; 0 if x1 is equal to x2;
     * a value greater than 0 if x1 is greater than x2
     */
    public static int acompare(double x1, double x2) {
        if (isZero(x1 - x2)) {
            return 0;
        } else {
            return Double.compare(x1, x2);
        }
    }
    
    /**
     * Compares two {@code float} values using the default tolerance.
     * 
     * @param x1 first value
     * @param x2 second value
     * 
     * @return a value less than 0 if x1 is less than x2; 0 if x1 is equal to x2;
     * a value greater than 0 if x1 is greater than x2
     */
    public static int acompare(float x1, float x2) {
        if (isZero(x1 - x2)) {
            return 0;
        } else {
            return Float.compare(x1, x2);
        }
    }
    
    /**
     * Compares two {@code double} values using the specified tolerance.
     * Note that performance reasons, {@code tol} is <strong>assumed</strong>
     * to be positive, ie. this is not checked.
     * 
     * @param x1 first value
     * @param x2 second value
     * 
     * @return a value less than 0 if x1 is less than x2; 0 if x1 is equal to x2;
     * a value greater than 0 if x1 is greater than x2
     */
    public static int acompare(double x1, double x2, double tol) {
        if (isZero(x1 - x2, tol)) {
            return 0;
        } else {
            return Double.compare(x1, x2);
        }
    }

    /**
     * Compares two {@code float} values using the specified tolerance.
     * Note that performance reasons, {@code tol} is <strong>assumed</strong>
     * to be positive, ie. this is not checked.
     * 
     * @param x1 first value
     * @param x2 second value
     * 
     * @return a value less than 0 if x1 is less than x2; 0 if x1 is equal to x2;
     * a value greater than 0 if x1 is greater than x2
     */
    public static int acompare(float x1, float x2, float tol) {
        if (isZero(x1 - x2, tol)) {
            return 0;
        } else {
            return Float.compare(x1, x2);
        }
    }

    /**
     * Compares two {@code double} values using the specified proportional
     * tolerance. This is equivalent to:
     * <pre><code>
     *     double absoluteTol = Math.abs(propTol) * Math.max(Math.abs(x1), Math.abs(x2));
     *     int comp = acompare(x1, x2, absTol);
     * </code></pre>
     * 
     * @param x1 first value
     * @param x2 second value
     * @param propTol proportional tolerance between 0 and 1
     * 
     * @return a value less than 0 if x1 is less than x2; 0 if x1 is equal to x2;
     * a value greater than 0 if x1 is greater than x2
     */
    public static int pcompare(double x1, double x2, double propTol) {
        if (aequal(x1, x2)) {
            return 0;
        }
        
        int comp = acompare(Math.abs(x1), Math.abs(x2));
        double absTol = Math.abs(propTol) * (comp > 0 ? x1 : x2);
        return acompare(x1, x2, absTol);
    }

    /**
     * Compares two {@code float} values using the specified proportional
     * tolerance. This is equivalent to:
     * <pre><code>
     *     float absoluteTol = Math.abs(propTol) * Math.max(Math.abs(x1), Math.abs(x2));
     *     int comp = acompare(x1, x2, absTol);
     * </code></pre>
     * 
     * @param x1 first value
     * @param x2 second value
     * @param propTol proportional tolerance between 0 and 1
     * 
     * @return a value less than 0 if x1 is less than x2; 0 if x1 is equal to x2;
     * a value greater than 0 if x1 is greater than x2
     */
    public static int pcompare(float x1, float x2, float propTol) {
        if (aequal(x1, x2)) {
            return 0;
        }
        
        int comp = acompare(Math.abs(x1), Math.abs(x2));
        double absTol = Math.abs(propTol) * (comp > 0 ? x1 : x2);
        return acompare(x1, x2, absTol);
    }

    /**
     * Tests if two {@code double} values are equal within the default tolerance.
     * This is equivalent to {@code dzero(x1 - x2)}.
     * 
     * @param x1 first value
     * @param x2 second value
     * 
     * @return {@code true} if equal; {@code false} otherwise
     */
    public static boolean aequal(double x1, double x2) {
        return isZero(x1 - x2);
    }

    /**
     * Tests if two {@code float} values are equal within the default tolerance.
     * This is equivalent to {@code dzero(x1 - x2)}.
     * 
     * @param x1 first value
     * @param x2 second value
     * 
     * @return {@code true} if equal; {@code false} otherwise
     */
    public static boolean aequal(float x1, float x2) {
        return isZero(x1 - x2);
    }

    /**
     * Tests if two {@code double} values are equal within the specified tolerance.
     * This is equivalent to {@code dzero(x1 - x2, tol)}.
     * Note that performance reasons, {@code tol} is <strong>assumed</strong>
     * to be positive, ie. this is not checked.
     * 
     * @param x1 first value
     * @param x2 second value
     * 
     * @return {@code true} if equal; {@code false} otherwise
     */
    public static boolean aequal(double x1, double x2, double tol) {
        return isZero(x1 - x2, tol);
    }

    /**
     * Tests if two {@code float} values are equal within the specified tolerance.
     * This is equivalent to {@code dzero(x1 - x2, tol)}.
     * Note that performance reasons, {@code tol} is <strong>assumed</strong>
     * to be positive, ie. this is not checked.
     * 
     * @param x1 first value
     * @param x2 second value
     * 
     * @return {@code true} if equal; {@code false} otherwise
     */
    public static boolean aequal(float x1, float x2, float tol) {
        return isZero(x1 - x2, tol);
    }

    /**
     * Tests if two {@code double} values are equal within the specified 
     * proportional tolerance. This is equivalent to:
     * <pre><code>
     *     double absoluteTol = Math.abs(propTol) * Math.max(Math.abs(x1), Math.abs(x2));
     *     boolean b = aequal(x1, x2, absTol);
     * </code></pre>
     * 
     * @param x1 first value
     * @param x2 second value
     * @param propTol proportional tolerance between 0 and 1
     * 
     * @return {@code true} if equal; {@code false} otherwise
     */
    public static boolean pequal(double x1, double x2, double propTol) {
        if (aequal(x1, x2)) {
            return true;
        }
        
        int comp = acompare(Math.abs(x1), Math.abs(x2));
        double absTol = Math.abs(propTol) * (comp > 0 ? x1 : x2);
        return aequal(x1, x2, absTol);
    }

    /**
     * Tests if two {@code float} values are equal within the specified 
     * proportional tolerance. This is equivalent to:
     * <pre><code>
     *     float absoluteTol = Math.abs(propTol) * Math.max(Math.abs(x1), Math.abs(x2));
     *     boolean b = aequal(x1, x2, absTol);
     * </code></pre>
     * 
     * @param x1 first value
     * @param x2 second value
     * @param propTol proportional tolerance between 0 and 1
     * 
     * @return {@code true} if equal; {@code false} otherwise
     */
    public static boolean pequal(float x1, float x2, float propTol) {
        if (aequal(x1, x2)) {
            return true;
        }
        
        int comp = acompare(Math.abs(x1), Math.abs(x2));
        double absTol = Math.abs(propTol) * (comp > 0 ? x1 : x2);
        return aequal(x1, x2, absTol);
    }

}
