/*
 * Copyright 2013 Jin Kwon <onacit at gmail.com>.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package com.googlecode.jinahya.xml.bind;


import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.xml.bind.annotation.adapters.XmlAdapter;


/**
 * An {@code XmlAdapter} for {@link MapEntries} and {@link Map}.
 *
 * @author Jin Kwon <onacit at gmail.com>
 * @param <T> MepEntries type parameter
 * @param <K> Map key type parameter
 * @param <V> Map value type parameter
 */
public abstract class MapEntriesAdapter<T extends MapEntries<?, K, V>, K, V>
    extends XmlAdapter<T, Map<K, V>> {


    private static final Logger LOGGER =
        Logger.getLogger(MapEntriesAdapter.class.getName());


    static {
        LOGGER.setLevel(Level.INFO);
    }


    /**
     * Creates a new instance.
     *
     * @param entriesType wrapper type.
     */
    public MapEntriesAdapter(final Class<T> entriesType) {

        super();

        if (entriesType == null) {
            throw new NullPointerException("entriesType");
        }

        this.entriesType = entriesType;
    }


    @Override
    public Map<K, V> unmarshal(final T value) throws Exception {

        LOGGER.log(Level.INFO, "unmarshal({0})", value);

        if (value == null) {
            return null;
        }

        final List<? extends MapEntry<K, V>> ventries = value.getEntries();

        final Map<K, V> bound = new HashMap<K, V>(ventries.size());

        for (MapEntry<K, V> ventry : ventries) {
            bound.put(ventry.getKey(), ventry.getValue());
        }

        LOGGER.log(Level.INFO, "bound: {0}", bound);
        return bound;
    }


    @Override
    public T marshal(final Map<K, V> bound) throws Exception {

        if (bound == null) {
            return null;
        }

        final T value = entriesType.newInstance();

        value.addEntries(bound);

        /*
         //final List<? extends MapEntry<K, V>> entries = value.getEntries();
         @SuppressWarnings("unchecked")
         final List<? extends MapEntry<K, V>> entries =
         (List<MapEntry<K, V>>) value.getEntries();
         ((ArrayList) entries).ensureCapacity(bound.size());

         for (Entry<K, V> bentry : bound.entrySet()) {
         final MapEntry<K, V> ventry = value.getEntryType().newInstance();
         ventry.setKey(bentry.getKey());
         ventry.setValue(bentry.getValue());
         entries.add(ventry);
         }
         */

        return value;
    }


    /**
     * entries type.
     */
    private final Class<T> entriesType;


}

