package com.googlecode.juffrou.util.reflect;

import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.lang.reflect.TypeVariable;
import java.security.InvalidParameterException;
import java.util.HashMap;
import java.util.Map;

import com.googlecode.juffrou.error.ReflectionException;

/**
 * Utility class for bean handling.<br>
 * Allows access to the wrapped bean's properties and also to the properties of beans referenced by them. For example
 * the property path <code>"pro1.prop2"</code> will access the property prop2 from the nested bean referenced by prop1.<br>
 * This bean wrapper auto grows nested paths, so for each nested bean referenced in this manner, a nested bean wrapper is automatically created. In the previous
 * example, a bean wrapper would be created for the bean referenced by property prop1. The nested bean wrappers can be
 * obtained by calling the <code>getNestesWrappers()</code> method.<br>
 * You can reference nested properties as deep as you like as long as the bean path exists.
 * 
 * @author cem
 */
public class BeanWrapper {

	private Object instance;
	private final Class clazz;
	private Map<Type, Type> typeArgumentsMap;
	private Map<String, BeanFieldHandler> fields;
	private Map<String, BeanWrapper> nestesWrappers;

	/**
	 * Construct a bean wrapper around an existing bean object
	 * 
	 * @param instance
	 */
	public BeanWrapper(Object instance) {
		this.instance = instance;
		this.clazz = instance.getClass();
		init();
	}

	/**
	 * Construct a bean wrapper around an object that is an instance of the class
	 * 
	 * @param clazz
	 *            class to instantiate the wrapped bean
	 */
	public BeanWrapper(Class clazz) {
		this.clazz = clazz;
		init();
		try {
			this.instance = clazz.newInstance();
		} catch (InstantiationException e) {
			this.instance = null;
		} catch (IllegalAccessException e) {
			this.instance = null;
		}
	}

	private void init() {
		this.typeArgumentsMap = ReflectionUtil.getTypeArgumentsMap(Class.class, clazz);
		this.fields = new HashMap<String, BeanFieldHandler>();
		initFieldInfo(this.clazz, this.fields);
		nestesWrappers = new HashMap<String, BeanWrapper>();
	}
	
	private void initFieldInfo(Class<?> clazz, Map<String, BeanFieldHandler> fs) {
		Class<?> superclass = clazz.getSuperclass();
		if(superclass != Object.class) {
			initFieldInfo(superclass, fs);
		}
		for(Field f : clazz.getDeclaredFields()) {
			if( !Modifier.isStatic(f.getModifiers()) )
				fs.put(f.getName(), new BeanFieldHandler(f));
		}
	}

	/**
	 * Get the wrapped bean
	 * 
	 * @return the wrapped bean
	 */
	public Object getBean() {
		return instance;
	}
	
	/**
	 * Replaces the wrapped bean with another instance of the same type
	 * @param bean instance of the new bean to wrap
	 * @throws InvalidParameterException if the new bean is not of the same type of the initially wrapped bean.
	 */
	public void setBean(Object bean) {
		if(! clazz.equals(bean.getClass())) {
			throw new InvalidParameterException("Bean must be of type " + clazz.getSimpleName());
		}
		instance = bean;
	}

	/**
	 * Get the wrapped bean class
	 * @return
	 */
	public Class<?> getBeanClass() {
		return clazz;
	}

	/**
	 * Returns all the nested bean wrappers that have been created inside this bean wrapper.<br>
	 * Nested bean wrappers are created when you access a nested property (i.e. getValue("prop1.prop2"))
	 * @return a Map where the keys are property names and the values are bean wrappers
	 */
	public Map<String, BeanWrapper> getNestesWrappers() {
		return nestesWrappers;
	}
	
	/**
	 * sets all properties to null in this instance and in all nested bean instances
	 */
	public void reset() {
		for(BeanWrapper bw : nestesWrappers.values()) {
			bw.reset();
		}
		try {
			this.instance = clazz.newInstance();
		} catch (InstantiationException e) {
			this.instance = null;
		} catch (IllegalAccessException e) {
			this.instance = null;
		}
	}
	
	/**
	 * Checks whether a property exists in the wrapped bean. If that property references another bean (a nested bean) 
	 * the method verifies if the property of the nested bean also exists.<br>
	 * For example <code>hasProperty("pro1.prop2")</code> returns true only if prop1 exists is this bean and prop2 exists in the bean referenced by prop1.<br>
	 * For each nested bean referenced in this manner, a nested bean wrapper is automatically created. In the previous
	 * example, a bean wrapper would be created for the bean referenced by property prop1.<br>
	 * @param propertyName
	 * @return
	 */
	public boolean hasProperty(String propertyName) {
		
		int nestedIndex = propertyName.indexOf('.');
		if (nestedIndex == -1) {
			// not a nested property
			return fields.containsKey(propertyName);
		} else {
			// its a nested property
			String thisProperty = propertyName.substring(0, nestedIndex);
			String nestedProperty = propertyName.substring(nestedIndex + 1);
			BeanWrapper nestedWrapper = getNestedWrapper(thisProperty);
			return nestedWrapper.hasProperty(nestedProperty);
		}

	}

	/**
	 * Gets the value of a property in the wrapped bean. If that property references another bean (a nested bean) Its
	 * property values can also be obtained by specifying a property path.<br>
	 * For example <code>getValue("pro1.prop2")</code> will get the value of prop2 from the nested bean referenced by
	 * prop1.<br>
	 * For each nested bean referenced in this manner, a nested bean wrapper is automatically created. In the previous
	 * example, a bean wrapper would be created for the bean referenced by property prop1.<br>
	 * 
	 * @param propertyName
	 * @return the value held in the bean property
	 */
	public Object getValue(String propertyName) {
		int nestedIndex = propertyName.indexOf('.');
		if (nestedIndex == -1) {
			return getBeanFieldHandler(propertyName).getValue();
		}
		 else {
				// its a nested property
				String thisProperty = propertyName.substring(0, nestedIndex);
				String nestedProperty = propertyName.substring(nestedIndex + 1);
				BeanWrapper nestedWrapper = getNestedWrapper(thisProperty);
				return nestedWrapper.getValue(nestedProperty);
			}
	}

	/**
	 * Gets the class of a property in the wrapped bean. If that property references another bean (a nested bean) Its
	 * property types can also be obtained by specifying a property path.<br>
	 * For example <code>getType("pro1.prop2")</code> will get the type of prop2 from the nested bean referenced by
	 * prop1.<br>
	 * For each nested bean referenced in this manner, a nested bean wrapper is automatically created. In the previous
	 * example, a bean wrapper would be created for the bean referenced by property prop1.<br>
	 * 
	 * @param propertyName
	 * @return
	 */
	public Class<?> getClazz(String propertyName) {
		return ReflectionUtil.getClass(getType(propertyName));
	}

	/**
	 * Gets the type of a property in the wrapped bean. If that property references another bean (a nested bean) Its
	 * property types can also be obtained by specifying a property path.<br>
	 * For example <code>getType("pro1.prop2")</code> will get the type of prop2 from the nested bean referenced by
	 * prop1.<br>
	 * For each nested bean referenced in this manner, a nested bean wrapper is automatically created. In the previous
	 * example, a bean wrapper would be created for the bean referenced by property prop1.<br>
	 * 
	 * @param propertyName
	 * @return
	 */
	public Type getType(String propertyName) {
		int nestedIndex = propertyName.indexOf('.');
		if (nestedIndex == -1) {
			return getBeanFieldHandler(propertyName).getType();
		} else {
			// its a nested property
			String thisProperty = propertyName.substring(0, nestedIndex);
			String nestedProperty = propertyName.substring(nestedIndex + 1);
			BeanWrapper nestedWrapper = getNestedWrapper(thisProperty);
			return nestedWrapper.getType(nestedProperty);
		}
	}

	/**
	 * Gets the type of a property in the wrapped bean. If that property references another bean (a nested bean) Its
	 * property types can also be obtained by specifying a property path.<br>
	 * For example <code>getType("pro1.prop2")</code> will get the type of prop2 from the nested bean referenced by
	 * prop1.<br>
	 * For each nested bean referenced in this manner, a nested bean wrapper is automatically created. In the previous
	 * example, a bean wrapper would be created for the bean referenced by property prop1.<br>
	 * 
	 * @param propertyName
	 * @return
	 */
	public Type[] getTypeArguments(String propertyName) {
		int nestedIndex = propertyName.indexOf('.');
		if (nestedIndex == -1) {
			return getBeanFieldHandler(propertyName).getTypeArguments();
		} else {
			// its a nested property
			String thisProperty = propertyName.substring(0, nestedIndex);
			String nestedProperty = propertyName.substring(nestedIndex + 1);
			BeanWrapper nestedWrapper = getNestedWrapper(thisProperty);
			return nestedWrapper.getTypeArguments(nestedProperty);
		}
	}

	/**
	 * Gets the type of a property in the wrapped bean. If that property references another bean (a nested bean) Its
	 * property types can also be obtained by specifying a property path.<br>
	 * For example <code>getType("pro1.prop2")</code> will get the type of prop2 from the nested bean referenced by
	 * prop1.<br>
	 * For each nested bean referenced in this manner, a nested bean wrapper is automatically created. In the previous
	 * example, a bean wrapper would be created for the bean referenced by property prop1.<br>
	 * 
	 * @param propertyName
	 * @return
	 */
	public Field getField(String propertyName) {
		int nestedIndex = propertyName.indexOf('.');
		if (nestedIndex == -1) {
			return getBeanFieldHandler(propertyName).getField();
		} else {
			// its a nested property
			String thisProperty = propertyName.substring(0, nestedIndex);
			String nestedProperty = propertyName.substring(nestedIndex + 1);
			BeanWrapper nestedWrapper = getNestedWrapper(thisProperty);
			return nestedWrapper.getField(nestedProperty);
		}
	}

	/**
	 * Same as <code>setValue(String propertyName, Object value)</code> but the value will be converted from String to
	 * whatever type the property referenced by propertyName is.
	 * 
	 * @param propertyName
	 * @param value
	 *            String representation of the value to be set
	 */
	public void setValueOfString(String propertyName, String value) {
		int nestedIndex = propertyName.indexOf('.');
		if (nestedIndex == -1) {
			// not a nested property
			BeanFieldHandler beanFieldHandler = getBeanFieldHandler(propertyName);
			Class<?> paramType = (Class<?>) beanFieldHandler.getType();
			try {
				if (paramType.equals(String.class)) {
					beanFieldHandler.setValue(value);
				} else {
					Constructor<?> constructor = paramType.getConstructor(new Class<?>[] { String.class });
					Object convertedValue = constructor.newInstance(value);
					beanFieldHandler.setValue(convertedValue);
				}
			} catch (IllegalArgumentException e) {
				throw new ReflectionException(e);
			} catch (IllegalAccessException e) {
				throw new ReflectionException(e);
			} catch (InvocationTargetException e) {
				throw new ReflectionException(e);
			} catch (SecurityException e) {
				throw new ReflectionException(e);
			} catch (NoSuchMethodException e) {
				throw new ReflectionException(clazz.getName() + "." + propertyName + ": Cannot convert from String to " + paramType.getSimpleName() + ". Trying to convert " + value);
			} catch (InstantiationException e) {
				throw new ReflectionException(e);
			}
		} else {
			// its a nested property
			String thisProperty = propertyName.substring(0, nestedIndex);
			String nestedProperty = propertyName.substring(nestedIndex + 1);
			BeanWrapper nestedWrapper = getNestedWrapper(thisProperty);
			nestedWrapper.setValueOfString(nestedProperty, value);
		}
	}

	
	/**
	 * Sets the value of a property in the wrapped bean. If that property references another bean (a nested bean) Its
	 * property values can also be set by specifying a property path.<br>
	 * For example <code>setValue("pro1.prop2", Boolean.TRUE)</code> will set the value of prop2 from the nested bean
	 * referenced by prop1. If the value of prop1 was originally null, it would also be set to reference the new bean
	 * holding the value of prop2<br>
	 * For each nested bean referenced in this manner, a nested bean wrapper is automatically created. In the previous
	 * example, a bean wrapper would be created for the bean referenced by property prop1.<br>
	 * 
	 * @param propertyName
	 * @param value
	 *            value to be set
	 */
	public void setValue(String propertyName, Object value) {
		int nestedIndex = propertyName.indexOf('.');
		if (nestedIndex == -1) {
			// not a nested property
			getBeanFieldHandler(propertyName).setValue(value);
		} else {
			// its a nested property
			String thisProperty = propertyName.substring(0, nestedIndex);
			String nestedProperty = propertyName.substring(nestedIndex + 1);
			BeanWrapper nestedWrapper = getNestedWrapper(thisProperty);
			nestedWrapper.setValue(nestedProperty, value);
		}
	}

	private BeanWrapper getNestedWrapper(String thisProperty) {
		BeanWrapper nestedWrapper = nestesWrappers.get(thisProperty);
		if (nestedWrapper == null) {
			Object value = getValue(thisProperty);
			if (value != null) {
				nestedWrapper = new BeanWrapper(value);
			} else {
				Class<?> propertyType = (Class<?>) getType(thisProperty);
				nestedWrapper = new BeanWrapper(propertyType);
				setValue(thisProperty, nestedWrapper.getBean());
			}
			nestesWrappers.put(thisProperty, nestedWrapper);
		}
		return nestedWrapper;
	}
	
	private BeanFieldHandler getBeanFieldHandler(String propertyName) {
		BeanFieldHandler bfh = fields.get(propertyName);
		if (bfh == null) {
			throw new ReflectionException("The class " + clazz.getName() + " does not have a field with name "
					+ propertyName);
		}
		return bfh;

	}
	
	private class BeanFieldHandler {
		private final Field field;
		private final Type ftype;
		private final Type[] ftypeArguments;
		private Method getter = null;
		private Method setter = null;
		
		public BeanFieldHandler(Field field) {
			this.field = field;
			Type t = field.getGenericType();
			if(t instanceof TypeVariable) {
				t = typeArgumentsMap.get(t);
			}
			if(t instanceof ParameterizedType) {
				ParameterizedType pt = (ParameterizedType) t;
				this.ftypeArguments = pt.getActualTypeArguments();
			}
			else {
				this.ftypeArguments = null;
			}
			this.ftype = t;

		}
		
		public Field getField() {
			return this.field;
		}
		
		public Type getType() {
			return ftype;
		}
		
		public Type[] getTypeArguments() {
			return ftypeArguments;
		}
		
		public Object getValue() {
			
			try {
				if(getter == null) {
					String name = field.getName();
					String methodName = "get" + name.substring(0, 1).toUpperCase() + name.substring(1);
					getter = clazz.getMethod(methodName, null);
				}
				return getter.invoke(instance, null);
			} catch (IllegalArgumentException e) {
				throw new ReflectionException(e);
			} catch (IllegalAccessException e) {
				throw new ReflectionException(e);
			} catch (InvocationTargetException e) {
				throw new ReflectionException(e);
			} catch (SecurityException e) {
				throw new ReflectionException(e);
			} catch (NoSuchMethodException e) {
				throw new ReflectionException("The class " + clazz.getSimpleName() + " does not have a getter method for the field "
						+ field.getName());
			}

		}
		
		public void setValue(Object value) {
			try {
				if(setter == null) {
					String name = field.getName();
					String methodName = "set" + name.substring(0, 1).toUpperCase() + name.substring(1);
					setter = clazz.getMethod(methodName, (Class<?>) this.field.getType());
				}
				setter.invoke(instance, value);
			} catch (IllegalArgumentException e) {
				throw new ReflectionException(e);
			} catch (IllegalAccessException e) {
				throw new ReflectionException(e);
			} catch (InvocationTargetException e) {
				throw new ReflectionException(e);
			} catch (SecurityException e) {
				throw new ReflectionException(e);
			} catch (NoSuchMethodException e) {
				throw new ReflectionException("The class " + clazz.getSimpleName() + " does not have a setter method for the field "
						+ field.getName());
			}
		}
		
		public void setValueIfBeanField(Object value) {
			if(getter != null || setter != null) {
				try {
					setValue(value);
				}
				catch(ReflectionException e) {}
			}
		}
		
	}
}
