package com.googlecode.lucastody.javatojson.phase;

import java.io.IOException;
import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.List;

import com.googlecode.lucastody.javatojson.annotation.AjaxInterceptor;
import com.googlecode.lucastody.javatojson.application.AjaxApplication;
import com.googlecode.lucastody.javatojson.application.AjaxContext;
import com.googlecode.lucastody.javatojson.exception.AjaxException;
import com.googlecode.lucastody.javatojson.exception.AjaxInterceptorException;
import com.googlecode.lucastody.javatojson.exception.CallUserMethodException;
import com.googlecode.lucastody.javatojson.exception.ExecuteException;
import com.googlecode.lucastody.javatojson.http.MutableRequest;
import com.googlecode.lucastody.javatojson.http.MutableResponse;
import com.googlecode.lucastody.javatojson.interceptor.Interceptor;
import com.googlecode.lucastody.javatojson.phase.typeHandler.TypeHandlerImpl;

import net.sf.json.JSONArray;
import net.sf.json.JSONObject;

public class ExecutePhase extends PhaseContainer implements Phase {

	public ExecutePhase(AjaxContext context) {
		super(context);
	}
	
	@Override
	public void start() throws ExecuteException {
		MutableRequest request = context.getRequest();
		MutableResponse response = context.getResponse();
		
		try {
			/*
			 * Obtém os parâmetros da requisição
			 */
			String beanName = request.getParameter("beanName");
			String methodName = request.getParameter("methodName");
			String parametersTypes = request.getParameter("parametersTypes");
			String arguments = request.getParameter("arguments");
			
			if(beanName == null || methodName == null || parametersTypes == null || arguments == null) {
				throw new AjaxException("Invalid request parameters.");
			}
			
			/*
			 * Obtém os tipos dos parâmetros do método a ser invocado
			 */
			JSONArray arrayParametersTypes = JSONArray.fromObject(parametersTypes);
			List<?> parametersTypesList = (List<?>) JSONArray.toCollection(arrayParametersTypes);
			Class<?>[] typesClassList = convertParametersTypes(parametersTypesList);
			
			/*
			 * Carrega a classe a ser invocada
			 */
			Class<?> beanClass = AjaxApplication.ANNOTATED_CLASSES.get(beanName);
			
			/*
			 * Instancia o controller
			 */
			Object beanObject = beanClass.newInstance();
			
			/* 
			 * Obtém o método a ser invocado
			 */
			Method method = beanObject.getClass().getDeclaredMethod(methodName, typesClassList);
			Class<?> returnType = method.getReturnType();
			
			/*
			 * Obtém os argumentos do método a ser invocado
			 */
			JSONArray arrayArguments = JSONArray.fromObject(arguments);
			List<?> argumentsList = (List<?>) JSONArray.toCollection(arrayArguments);
			Object[] objectsArgumentsList = convertToArrayObjects(argumentsList, typesClassList, method);
			
			/*
			 * Carrega o interceptador de método, se existir
			 */
			Interceptor methodInterceptor = null;
			if(method.isAnnotationPresent(AjaxInterceptor.class)) {
				AjaxInterceptor tmp = method.getAnnotation(AjaxInterceptor.class);
				methodInterceptor = (Interceptor) tmp.value().newInstance();
			}
			
			/*
			 * Executa o interceptador de método antes de invocá-lo
			 */
			if(methodInterceptor != null) {
				methodInterceptor.beforeHandler(context, beanObject, objectsArgumentsList);
			}
			
			/*
			 * Invoca o método
			 */
			Object methodReturn = callMethod(beanObject, method, objectsArgumentsList, methodInterceptor);
			
			/*
			 * Trata o retorno
			 */
			if(returnType != Void.TYPE && methodReturn != null) {
				TypeHandlerImpl handleReturn = TypeHandlerImpl.getHandler(returnType);
				handleReturn.returnHandler(context, methodReturn);
			}
		} catch (InstantiationException e) {
			Throwable cause = e.getCause();
			throw new ExecuteException(cause);
		} catch (IllegalAccessException e) {
			Throwable cause = e.getCause();
			throw new ExecuteException(cause);
		} catch (ClassNotFoundException e) {
			Throwable cause = e.getCause();
			throw new ExecuteException(cause);
		} catch (SecurityException e) {
			Throwable cause = e.getCause();
			throw new ExecuteException(cause);
		} catch (NoSuchMethodException e) {
			Throwable cause = e.getCause();
			throw new ExecuteException(cause);
		} catch (IllegalArgumentException e) {
			Throwable cause = e.getCause();
			throw new ExecuteException(cause);
		} catch (CallUserMethodException e) {
			try {
				response.setStatus(500);
				Throwable cause = e.getCause();
				response.getOutputStream().write(JSONObject.fromObject(cause).toString().getBytes());
			} catch (IOException e1) {
				throw new ExecuteException(e1);
			}
		} catch (AjaxInterceptorException e) {
			try {
				response.setStatus(500);
				Throwable cause = e.getCause();
				response.getOutputStream().write(JSONObject.fromObject(cause).toString().getBytes());
			} catch (IOException e1) {
				throw new ExecuteException(e1);
			}
		}
	}
 
	private Object callMethod(Object beanObject, Method method, Object[] objectsArgumentsList, Interceptor methodInterceptor) throws CallUserMethodException, AjaxInterceptorException {
		Object methodReturn = null;
		
		try {
			methodReturn = method.invoke(beanObject, objectsArgumentsList);
			return methodReturn;
		} catch (Exception e) {
			Throwable cause = e.getCause();
			throw new CallUserMethodException(cause);
		} finally {
			// Executa o interceptador de m�todo depois de invoc�-lo
			if(methodInterceptor != null) {
				try {
					methodInterceptor.afterHandler(context, beanObject, objectsArgumentsList, methodReturn);
				} catch(RuntimeException e1) {
					Throwable cause = e1.getCause();
					throw new AjaxInterceptorException(cause);
				}
			}
		}
	}
	
	private Class<?>[] convertParametersTypes(List<?> types) throws ClassNotFoundException {
		Class<?>[] clazzTypes = new Class[types.size()];
		
		for(int i = 0; i < types.size(); i++) {
			if("char".equalsIgnoreCase(types.get(i).toString())) {
				clazzTypes[i] = Character.TYPE;
			} else if("long".equalsIgnoreCase(types.get(i).toString())) {
				clazzTypes[i] = Long.TYPE;
			} else if("int".equalsIgnoreCase(types.get(i).toString())) {
				clazzTypes[i] = Integer.TYPE;
			} else if("short".equalsIgnoreCase(types.get(i).toString())) {
				clazzTypes[i] = Short.TYPE;
			} else if("double".equalsIgnoreCase(types.get(i).toString())) {
				clazzTypes[i] = Double.TYPE;
			} else if("float".equalsIgnoreCase(types.get(i).toString())) {
				clazzTypes[i] = Float.TYPE;
			} else if("boolean".equalsIgnoreCase(types.get(i).toString())) {
				clazzTypes[i] = Boolean.TYPE;
			} else {
				clazzTypes[i] = Class.forName(types.get(i).toString());
			}
		}
		
		return clazzTypes;
	}
	
	private Object[] convertToArrayObjects(List<?> arrayArgs, Class<?>[] types, Method method) {
		Object[] arguments = new Object[arrayArgs.size()];
		
		for(int i = 0; i < arrayArgs.size(); i++) {
			arguments[i] = castObject(arrayArgs.get(i), types[i], method, i);
		}
		
		return arguments;
	}
	
	@SuppressWarnings({ "unchecked", "rawtypes" })
	private Object castObject(Object obj, Class<?> type, Method method, Integer index) {
		if(type == List.class || type == ArrayList.class || type == Collection.class) {
			Class<?> tmp = recuperaTipoParametrizado(method, index);
			return JSONArray.toCollection(JSONArray.fromObject(obj), tmp);
		} else {
			if(type == String.class) {
				return (String) obj;
			} else if(!isEmpty(obj) && (type == Character.class || type == Character.TYPE)) {
				return new Character(obj.toString().charAt(0));
			} else if(!isEmpty(obj) && (type == Long.class || type == Long.TYPE)) {
				return new Long(obj.toString());
			} else if(!isEmpty(obj) && (type == Integer.class || type == Integer.TYPE)) {
				return new Integer(obj.toString());
			} else if(!isEmpty(obj) && (type == Short.class || type == Short.TYPE)) {
				return new Short(obj.toString());
			} else if(!isEmpty(obj) && (type == BigDecimal.class)) {
				return new BigDecimal(obj.toString());
			} else if(!isEmpty(obj) && (type == Double.class || type == Double.TYPE)) {
				return new Double(obj.toString());
			} else if(!isEmpty(obj) && (type == Float.class || type == Float.TYPE)) {
				return new Float(obj.toString());
			} else if(!isEmpty(obj) && (type == Boolean.class || type == Boolean.TYPE)) {
				return new Boolean(obj.toString());
			} else if(!isEmpty(obj) && (type == Date.class)) {
				return new Date(Long.parseLong(obj.toString()));
			} else if(!isEmpty(obj) && (type.isEnum())) {
				return Enum.valueOf((Class<Enum>) type, obj.toString());
			} else if(!isEmpty(obj)) {
				return JSONObject.toBean(JSONObject.fromObject(obj), type);
			} else {
				return null;
			}
		}
	}
	
	/*
	 * Recupera o class do objeto parametrizado de cole��es
	 */
	public Class<?> recuperaTipoParametrizado(Method method, int index) {
		Class<?> tmp = null;

		Type[] genericParameterTypes = method.getGenericParameterTypes();
		
	    if(genericParameterTypes[index] instanceof ParameterizedType) {
	        ParameterizedType aType = (ParameterizedType) genericParameterTypes[index];
	        Type[] parameterArgTypes = aType.getActualTypeArguments();
	        for(Type parameterArgType : parameterArgTypes){
	            Class<?> parameterArgClass = (Class<?>) parameterArgType;
	            tmp = parameterArgClass;
	        }
	    }		
		return tmp;
	}
	
	public static Boolean isEmpty(Object obj) {
		return obj == null || "".equals(obj.toString());
	}
}