/*
 * Decompiled with CFR 0.152.
 */
package com.googlecode.fascinator.harvester.filesystem;

import com.googlecode.fascinator.common.JsonSimpleConfig;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Timestamp;
import java.util.HashSet;
import java.util.Properties;
import java.util.Set;
import org.apache.commons.codec.binary.Hex;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DerbyCache {
    private final Logger log = LoggerFactory.getLogger(DerbyCache.class);
    private static String DERBY_DRIVER = "org.apache.derby.jdbc.EmbeddedDriver";
    private static String DERBY_PROTOCOL = "jdbc:derby:";
    private static String DATABASE_NAME = "fsHarvestCache";
    private static String BASIC_TABLE = "basic";
    private static String HASH_TABLE = "hashed";
    private String derbyHome;
    private Connection connection;
    private boolean useCache;
    private String cacheType;
    private String cacheId;
    private static final int BUFFER_SIZE = 1024;

    public DerbyCache(JsonSimpleConfig config) throws Exception {
        this.cacheType = config.getString(null, new Object[]{"harvester", "file-system", "caching"});
        this.cacheId = config.getString(null, new Object[]{"harvester", "file-system", "cacheId"});
        if (this.cacheType != null && this.cacheId != null && (this.cacheType.equals("basic") || this.cacheType.equals("hashed"))) {
            this.useCache = true;
            this.startDatabase(config);
        } else {
            this.log.error("Caching is either disabled or not configured properly:");
            this.log.error("Cache Type: '{}'", (Object)this.cacheType);
            this.log.error("Cache ID: '{}'", (Object)this.cacheId);
        }
    }

    private void startDatabase(JsonSimpleConfig config) throws Exception {
        this.derbyHome = config.getString(null, new Object[]{"database-service", "derbyHome"});
        String oldHome = System.getProperty("derby.system.home");
        if (oldHome != null) {
            if (this.derbyHome != null) {
                this.log.warn("Using previously specified data directory: '{}', provided value has been ignored: '{}'", (Object)oldHome, (Object)this.derbyHome);
            } else {
                this.log.info("Using existing data directory: '{}'", (Object)oldHome);
            }
        } else {
            if (this.derbyHome == null) {
                this.log.error("No database home directory configured!");
                return;
            }
            File file = new File(this.derbyHome);
            if (file.exists()) {
                if (!file.isDirectory()) {
                    throw new Exception("Database home '" + this.derbyHome + "' is not a directory!");
                }
            } else {
                file.mkdirs();
                if (!file.exists()) {
                    throw new Exception("Database home '" + this.derbyHome + "' does not exist and could not be created!");
                }
            }
            System.setProperty("derby.system.home", this.derbyHome);
        }
        try {
            this.checkTable(BASIC_TABLE);
            this.checkTable(HASH_TABLE);
        }
        catch (SQLException ex) {
            this.log.error("Error during database preparation:", (Throwable)ex);
            throw new Exception("Error during database preparation:", ex);
        }
    }

    private Connection connection() throws SQLException {
        if (this.connection == null || !this.connection.isValid(1)) {
            if (this.connection != null) {
                this.log.error("!!! Database connection has failed, recreating.");
                try {
                    this.connection.close();
                }
                catch (SQLException ex) {
                    this.log.error("Error closing invalid connection, ignoring: {}", (Object)ex.getMessage());
                }
            }
            Properties props = new Properties();
            try {
                Class.forName(DERBY_DRIVER).newInstance();
            }
            catch (Exception ex) {
                this.log.error("Driver load failed: ", (Throwable)ex);
                throw new SQLException("Driver load failed: ", ex);
            }
            this.connection = DriverManager.getConnection(DERBY_PROTOCOL + DATABASE_NAME + ";create=true", props);
        }
        return this.connection;
    }

    public boolean hasChanged(String oid, File file) {
        if (file == null || !file.exists()) {
            return false;
        }
        if (this.useCache) {
            try {
                if (this.cacheType.equals("basic")) {
                    return this.checkBasicCache(oid, file);
                }
                if (this.cacheType.equals("hashed")) {
                    return this.checkHashedCache(oid, file);
                }
            }
            catch (Exception ex) {
                this.log.error("Error during cache process: ", (Throwable)ex);
                return false;
            }
        }
        return true;
    }

    public void resetFlags() {
        if (this.useCache) {
            try {
                PreparedStatement sql = null;
                if (this.cacheType.equals("basic")) {
                    sql = this.connection().prepareStatement("UPDATE " + BASIC_TABLE + " SET changeFlag = 0" + " WHERE cacheId = '" + this.cacheId + "'");
                }
                if (this.cacheType.equals("hashed")) {
                    sql = this.connection().prepareStatement("UPDATE " + HASH_TABLE + " SET changeFlag = 0" + " WHERE cacheId = '" + this.cacheId + "'");
                }
                sql.executeUpdate();
                this.close(sql);
            }
            catch (Exception ex) {
                this.log.error("Error updating cache to reset flags: ", (Throwable)ex);
            }
        }
    }

    public Set<String> getUnsetFlags() {
        HashSet<String> response = null;
        if (this.useCache) {
            try {
                ResultSet result = null;
                PreparedStatement sql = null;
                if (this.cacheType.equals("basic")) {
                    sql = this.connection().prepareStatement("SELECT oid FROM " + BASIC_TABLE + " WHERE changeFlag = 0" + " AND cacheId = '" + this.cacheId + "'");
                }
                if (this.cacheType.equals("hashed")) {
                    sql = this.connection().prepareStatement("SELECT oid FROM " + HASH_TABLE + " WHERE changeFlag = 0" + " AND cacheId = '" + this.cacheId + "'");
                }
                response = new HashSet<String>();
                result = sql.executeQuery();
                while (result.next()) {
                    String oid = result.getString("oid");
                    if (oid == null) continue;
                    response.add(oid);
                }
                this.close(result);
                this.close(sql);
            }
            catch (Exception ex) {
                this.log.error("Error updating cache to reset flags: ", (Throwable)ex);
            }
        }
        return response;
    }

    public void purgeUnsetFlags() {
        if (this.useCache) {
            try {
                PreparedStatement sql = null;
                if (this.cacheType.equals("basic")) {
                    sql = this.connection().prepareStatement("DELETE FROM " + BASIC_TABLE + " WHERE changeFlag = 0" + " AND cacheId = '" + this.cacheId + "'");
                }
                if (this.cacheType.equals("hashed")) {
                    sql = this.connection().prepareStatement("DELETE FROM " + HASH_TABLE + " WHERE changeFlag = 0" + " AND cacheId = '" + this.cacheId + "'");
                }
                sql.executeUpdate();
                this.close(sql);
            }
            catch (Exception ex) {
                this.log.error("Error updating cache to delete data: ", (Throwable)ex);
            }
        }
    }

    private boolean checkBasicCache(String oid, File file) throws Exception {
        long lastCached = this.getLastModified(oid);
        long lastModified = file.lastModified();
        if (lastCached == -1L) {
            this.insertLastModified(oid, file.lastModified());
            return true;
        }
        this.updateLastModified(oid, file.lastModified());
        return lastModified > lastCached;
    }

    private boolean checkHashedCache(String oid, File file) throws Exception {
        String cachedHash = this.getHash(oid);
        String currentHash = this.hashFile(file);
        if (cachedHash == null) {
            this.insertHash(oid, currentHash);
            return true;
        }
        this.updateHash(oid, currentHash);
        return !currentHash.equals(cachedHash);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void shutdown() throws Exception {
        String threadedShutdownMessage = DERBY_DRIVER + " is not registered with the JDBC driver manager";
        try {
            DriverManager.getConnection(DERBY_PROTOCOL + DATABASE_NAME + ";shutdown=true");
        }
        catch (SQLException ex) {
            if (ex.getErrorCode() == 45000 && ex.getSQLState().equals("08006")) {
                // empty if block
            }
        }
        finally {
            try {
                if (this.connection != null) {
                    this.connection.close();
                    this.connection = null;
                }
            }
            catch (SQLException ex) {
                throw new Exception("Error closing connection:", ex);
            }
        }
    }

    private long getLastModified(String oid) {
        try {
            PreparedStatement sql = this.connection().prepareStatement("SELECT lastModified FROM " + BASIC_TABLE + " WHERE oid = ? AND cacheId = ?");
            sql.setString(1, oid);
            sql.setString(2, this.cacheId);
            ResultSet result = sql.executeQuery();
            Timestamp ts = null;
            if (result.next()) {
                ts = result.getTimestamp("lastModified");
            }
            this.close(result);
            this.close(sql);
            if (ts == null) {
                return -1L;
            }
            return ts.getTime();
        }
        catch (SQLException ex) {
            this.log.error("Error querying last modified date: ", (Throwable)ex);
            return -1L;
        }
    }

    private void insertLastModified(String oid, long lastModified) throws Exception {
        PreparedStatement sql = this.connection().prepareStatement("INSERT INTO " + BASIC_TABLE + " (oid, cacheId, lastModified, changeFlag)" + " VALUES (?, ?, ?, 1)");
        sql.setString(1, oid);
        sql.setString(2, this.cacheId);
        sql.setTimestamp(3, new Timestamp(lastModified));
        sql.executeUpdate();
        this.close(sql);
    }

    private void updateLastModified(String oid, long lastModified) throws Exception {
        PreparedStatement sql = this.connection().prepareStatement("UPDATE " + BASIC_TABLE + " SET lastModified = ?, changeFlag = 1" + " WHERE oid = ? and cacheId = ?");
        sql.setTimestamp(1, new Timestamp(lastModified));
        sql.setString(2, oid);
        sql.setString(3, this.cacheId);
        sql.executeUpdate();
        this.close(sql);
    }

    private String getHash(String oid) {
        try {
            PreparedStatement sql = this.connection().prepareStatement("SELECT hash FROM " + HASH_TABLE + " WHERE oid = ? AND cacheId = ?");
            sql.setString(1, oid);
            sql.setString(2, this.cacheId);
            ResultSet result = sql.executeQuery();
            String response = null;
            if (result.next()) {
                response = result.getString("hash");
            }
            this.close(result);
            this.close(sql);
            return response;
        }
        catch (SQLException ex) {
            this.log.error("Error querying last hash: ", (Throwable)ex);
            return null;
        }
    }

    private void insertHash(String oid, String hash) throws Exception {
        PreparedStatement sql = this.connection().prepareStatement("INSERT INTO " + HASH_TABLE + " (oid, cacheId, hash, changeFlag) VALUES (?, ?, ?, 1)");
        sql.setString(1, oid);
        sql.setString(2, this.cacheId);
        sql.setString(3, hash);
        sql.executeUpdate();
        this.close(sql);
    }

    private void updateHash(String oid, String hash) throws Exception {
        PreparedStatement sql = this.connection().prepareStatement("UPDATE " + HASH_TABLE + " SET hash = ?, changeFlag = 1" + " WHERE oid = ? AND cacheId = ?");
        sql.setString(1, hash);
        sql.setString(2, oid);
        sql.setString(3, this.cacheId);
        sql.executeUpdate();
        this.close(sql);
    }

    private void checkTable(String table) throws SQLException {
        boolean tableFound = this.findTable(table);
        if (!tableFound) {
            this.log.debug("Table '{}' not found, creating now!", (Object)table);
            this.createTable(table);
            if (!this.findTable(table)) {
                this.log.error("Unknown error creating table '{}'", (Object)table);
                throw new SQLException("Could not find or create table '" + table + "'");
            }
        }
    }

    private boolean findTable(String table) throws SQLException {
        boolean tableFound = false;
        DatabaseMetaData meta = this.connection().getMetaData();
        ResultSet result = meta.getTables(null, null, null, null);
        while (result.next() && !tableFound) {
            if (!result.getString("TABLE_NAME").equalsIgnoreCase(table)) continue;
            tableFound = true;
        }
        this.close(result);
        return tableFound;
    }

    private void createTable(String table) throws SQLException {
        if (table.equals(BASIC_TABLE)) {
            Statement sql = this.connection().createStatement();
            sql.execute("CREATE TABLE " + BASIC_TABLE + "(oid VARCHAR(255) NOT NULL, " + "cacheId VARCHAR(255) NOT NULL, " + "lastModified TIMESTAMP NOT NULL, " + "changeFlag SMALLINT NOT NULL, " + "PRIMARY KEY (oid, cacheId))");
            this.close(sql);
            return;
        }
        if (table.equals(HASH_TABLE)) {
            Statement sql = this.connection().createStatement();
            sql.execute("CREATE TABLE " + HASH_TABLE + "(oid VARCHAR(255) NOT NULL, " + "cacheId VARCHAR(255) NOT NULL, " + "hash VARCHAR(50) NOT NULL, " + "changeFlag SMALLINT NOT NULL, " + "PRIMARY KEY (oid, cacheId))");
            this.close(sql);
            return;
        }
        throw new SQLException("Unknown table '" + table + "' requested!");
    }

    private void close(ResultSet resultSet) {
        if (resultSet != null) {
            try {
                resultSet.close();
            }
            catch (SQLException ex) {
                this.log.error("Error closing result set: ", (Throwable)ex);
            }
        }
        resultSet = null;
    }

    private void close(Statement statement) {
        if (statement != null) {
            try {
                statement.close();
            }
            catch (SQLException ex) {
                this.log.error("Error closing statement: ", (Throwable)ex);
            }
        }
        statement = null;
    }

    private String hashFile(File file) throws IOException {
        FileInputStream data = new FileInputStream(file);
        try {
            MessageDigest digest = MessageDigest.getInstance("SHA");
            byte[] buffer = new byte[1024];
            int read = ((InputStream)data).read(buffer, 0, 1024);
            while (read > -1) {
                digest.update(buffer, 0, read);
                read = ((InputStream)data).read(buffer, 0, 1024);
            }
            return new String(Hex.encodeHex((byte[])digest.digest()));
        }
        catch (NoSuchAlgorithmException nsae) {
            throw new RuntimeException(nsae.getMessage());
        }
    }
}

