/*
 * Decompiled with CFR 0.152.
 */
package com.helger.masterdata.swift;

import com.helger.commons.ValueEnforcer;
import com.helger.commons.annotation.Nonempty;
import com.helger.commons.annotation.ReturnsMutableCopy;
import com.helger.commons.collection.impl.CommonsHashMap;
import com.helger.commons.collection.impl.ICommonsMap;
import com.helger.commons.collection.impl.ICommonsSet;
import com.helger.commons.datetime.PDTFromString;
import com.helger.commons.exception.InitializationException;
import com.helger.commons.io.resource.ClassPathResource;
import com.helger.commons.io.resource.IReadableResource;
import com.helger.commons.locale.country.CountryCache;
import com.helger.commons.regex.RegExHelper;
import com.helger.commons.string.StringHelper;
import com.helger.commons.string.StringParser;
import com.helger.masterdata.iso.ISO7064;
import com.helger.masterdata.swift.IBANCountryData;
import com.helger.xml.microdom.IMicroDocument;
import com.helger.xml.microdom.IMicroElement;
import com.helger.xml.microdom.serialize.MicroReader;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.Locale;
import javax.annotation.Nonnegative;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class IBANManager {
    private static final String ELEMENT_COUNTRY = "country";
    private static final String ATTR_VALIDFROM = "validfrom";
    private static final String ATTR_VALIDUNTIL = "validuntil";
    private static final String ATTR_LEN = "len";
    private static final String ATTR_CHECKDIGITS = "checkdigits";
    private static final String ATTR_LAYOUT = "layout";
    private static final Logger LOGGER = LoggerFactory.getLogger(IBANManager.class);
    private static final ICommonsMap<String, IBANCountryData> s_aIBANData = new CommonsHashMap();

    private IBANManager() {
    }

    private static void _readIBANDataFromXML() {
        IMicroDocument iMicroDocument = MicroReader.readMicroXML((IReadableResource)new ClassPathResource("codelists/iban-country-data.xml"));
        if (iMicroDocument == null) {
            throw new InitializationException("Failed to read IBAN country data [1]");
        }
        if (iMicroDocument.getDocumentElement() == null) {
            throw new InitializationException("Failed to read IBAN country data [2]");
        }
        DateTimeFormatter dateTimeFormatter = DateTimeFormatter.ISO_DATE;
        for (IMicroElement iMicroElement : iMicroDocument.getDocumentElement().getAllChildElements(ELEMENT_COUNTRY)) {
            String string = iMicroElement.getTextContent();
            String string2 = string.substring(0, 2);
            if (CountryCache.getInstance().getCountry(string2) == null && LOGGER.isWarnEnabled()) {
                LOGGER.warn("IBAN country data: no such country code '" + string2 + "' - be careful");
            }
            LocalDate localDate = null;
            if (iMicroElement.hasAttribute(ATTR_VALIDFROM)) {
                localDate = PDTFromString.getLocalDateFromString((String)iMicroElement.getAttributeValue(ATTR_VALIDFROM), (DateTimeFormatter)dateTimeFormatter);
            }
            LocalDate localDate2 = null;
            if (iMicroElement.hasAttribute(ATTR_VALIDUNTIL)) {
                localDate2 = PDTFromString.getLocalDateFromString((String)iMicroElement.getAttributeValue(ATTR_VALIDUNTIL), (DateTimeFormatter)dateTimeFormatter);
            }
            String string3 = iMicroElement.getAttributeValue(ATTR_LAYOUT);
            String string4 = iMicroElement.getAttributeValue(ATTR_CHECKDIGITS);
            String string5 = iMicroElement.getAttributeValue(ATTR_LEN);
            int n = StringParser.parseInt((String)string5, (int)-1);
            if (n == -1) {
                throw new InitializationException("Failed to convert length '" + string5 + "' to int!");
            }
            if (s_aIBANData.containsKey((Object)string2)) {
                throw new IllegalArgumentException("Country " + string2 + " is already contained!");
            }
            s_aIBANData.put((Object)string2, (Object)IBANCountryData.createFromString(string2, n, string3, string4, localDate, localDate2, string));
        }
    }

    @Nullable
    public static IBANCountryData getCountryData(@Nonnull String string) {
        ValueEnforcer.notNull((Object)string, (String)"CountryCode");
        return (IBANCountryData)s_aIBANData.get((Object)string.toUpperCase(Locale.US));
    }

    @Nonnull
    @ReturnsMutableCopy
    public static ICommonsSet<String> getAllSupportedCountries() {
        return s_aIBANData.copyOfKeySet();
    }

    private static boolean _isValidChecksumChar(char c) {
        return c >= '0' && c <= '9';
    }

    @Nullable
    public static String unifyIBAN(@Nullable String string) {
        if (string == null) {
            return null;
        }
        String string2 = string.toUpperCase(Locale.US);
        if ((string2 = RegExHelper.stringReplacePattern((String)"[^0-9A-Z]", (String)string2, (String)"")).length() < 4) {
            return null;
        }
        return string2;
    }

    public static boolean isValidIBAN(@Nullable String string) {
        return IBANManager.isValidIBAN(string, false);
    }

    private static int _calculateChecksum(@Nonnull String string) {
        String string2 = string.substring(4) + string.substring(0, 4);
        return ISO7064.Mod97.getChecksum(string2);
    }

    public static boolean isValidIBAN(@Nullable String string, boolean bl) {
        String string2 = IBANManager.unifyIBAN(string);
        if (string2 == null) {
            return false;
        }
        IBANCountryData iBANCountryData = (IBANCountryData)s_aIBANData.get((Object)string2.substring(0, 2));
        if (iBANCountryData == null) {
            return bl;
        }
        if (iBANCountryData.getExpectedLength() != string2.length()) {
            return false;
        }
        if (!IBANManager._isValidChecksumChar(string2.charAt(2)) || !IBANManager._isValidChecksumChar(string2.charAt(3))) {
            return false;
        }
        if (IBANManager._calculateChecksum(string2) != 1) {
            return false;
        }
        return iBANCountryData.matchesPattern(string2);
    }

    public static int createChecksumOfNewIBAN(@Nonnull String string, @Nonnull String string2) {
        ValueEnforcer.notNull((Object)string, (String)"CountryCode");
        if (string.length() != 2) {
            throw new IllegalArgumentException("Country code does not have exactly 2 characters!");
        }
        ValueEnforcer.notNull((Object)string2, (String)"BBAN");
        String string3 = IBANManager.unifyIBAN(string + "00" + string2);
        if (string3 == null) {
            throw new IllegalArgumentException("The passed data does not resemble an IBAN!");
        }
        int n = IBANManager._calculateChecksum(string3);
        if (n == -1) {
            throw new IllegalArgumentException("The passed data does not resemble an IBAN!");
        }
        return 98 - ((n - 1) % 97 + 1);
    }

    @Nonnull
    public static String createIBANWithValidChecksum(@Nonnull String string, @Nonnull String string2) {
        int n = IBANManager.createChecksumOfNewIBAN(string, string2);
        return IBANManager.unifyIBAN(string + StringHelper.getLeadingZero((int)n, (int)2) + string2);
    }

    @Nullable
    public static String getFormattedIBAN(@Nullable String string) {
        return IBANManager.getFormattedIBAN(string, 4, " ");
    }

    @Nullable
    public static String getFormattedIBAN(@Nullable String string, @Nonnegative int n, @Nonnull @Nonempty String string2) {
        ValueEnforcer.isGT0((int)n, (String)"GroupSize");
        ValueEnforcer.notEmpty((CharSequence)string2, (String)"Delimiter");
        if (StringHelper.hasNoText((String)string)) {
            return string;
        }
        StringBuilder stringBuilder = new StringBuilder();
        String string3 = IBANManager.unifyIBAN(string);
        while (string3.length() >= n) {
            stringBuilder.append(string3, 0, n);
            if ((string3 = string3.substring(n)).length() <= 0) continue;
            stringBuilder.append(string2);
        }
        stringBuilder.append(string3);
        return stringBuilder.toString();
    }

    static {
        IBANManager._readIBANDataFromXML();
    }
}

