[okio](../../index.md) / [okio](../index.md) / [Source](./index.md)

# Source

`expect interface Source`

**Platform and version requirements:** JVM, js, native

Supplies a stream of bytes. Use this interface to read data from wherever it's located: from the
network, storage, or a buffer in memory. Sources may be layered to transform supplied data, such
as to decompress, decrypt, or remove protocol framing.

Most applications shouldn't operate on a source directly, but rather on a [BufferedSource](../-buffered-source/index.md) which
is both more efficient and more convenient. Use [buffer](../buffer.md) to wrap any source with a buffer.

Sources are easy to test: just use a [Buffer](../-buffer/index.md) in your tests, and fill it with the data your
application is to read.

### Comparison with InputStream

This interface is functionally equivalent to [java.io.InputStream](https://docs.oracle.com/javase/8/docs/api/java/io/InputStream.html).

`InputStream` requires multiple layers when consumed data is heterogeneous: a `DataInputStream`
for primitive values, a `BufferedInputStream` for buffering, and `InputStreamReader` for strings.
This library uses `BufferedSource` for all of the above.

Source avoids the impossible-to-implement [available()](https://docs.oracle.com/javase/8/docs/api/java/io/InputStream.html#available()) method.
Instead callers specify how many bytes they [require](../-buffered-source/require.md).

Source omits the unsafe-to-compose [mark and reset](https://docs.oracle.com/javase/8/docs/api/java/io/InputStream.html#mark(int)) state that's
tracked by `InputStream`; instead, callers just buffer what they need.

When implementing a source, you don't need to worry about the [read()](https://docs.oracle.com/javase/8/docs/api/java/io/InputStream.html#read())
method that is awkward to implement efficiently and returns one of 257 possible values.

And source has a stronger `skip` method: [BufferedSource.skip](../-buffered-source/skip.md) won't return prematurely.

### Interop with InputStream

Use [source](#) to adapt an `InputStream` to a source. Use [BufferedSource.inputStream](#) to adapt a
source to an `InputStream`.

### Functions

| Name | Summary |
|---|---|
| [close](close.md) | `abstract fun close(): `[`Unit`](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-unit/index.html)<br>Closes this source and releases the resources held by this source. It is an error to read a closed source. It is safe to close a source more than once. |
| [read](read.md) | `abstract fun read(sink: `[`Buffer`](../-buffer/index.md)`, byteCount: `[`Long`](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-long/index.html)`): `[`Long`](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-long/index.html)<br>Removes at least 1, and up to `byteCount` bytes from this and appends them to `sink`. Returns the number of bytes read, or -1 if this source is exhausted. |
| [timeout](timeout.md) | `abstract fun timeout(): `[`Timeout`](../-timeout/index.md)<br>Returns the timeout for this source. |

### Extension Functions

| Name | Summary |
|---|---|
| [buffer](../buffer.md)<br>(JVM, js, native) | `expect fun `[`Source`](./index.md)`.buffer(): `[`BufferedSource`](../-buffered-source/index.md)<br>Returns a new source that buffers reads from `source`. The returned source will perform bulk reads into its in-memory buffer. Use this wherever you read a source to get an ergonomic and efficient access to data. |
| [gzip](../gzip.md)<br>(JVM) | `fun `[`Source`](./index.md)`.gzip(): `[`GzipSource`](../-gzip-source/index.md)<br>Returns a [GzipSource](../-gzip-source/index.md) that gzip-decompresses this [Source](./index.md) while reading. |
| [inflate](../inflate.md)<br>(JVM) | `fun `[`Source`](./index.md)`.inflate(inflater: `[`Inflater`](https://docs.oracle.com/javase/8/docs/api/java/util/zip/Inflater.html)` = Inflater()): `[`InflaterSource`](../-inflater-source/index.md)<br>Returns an [InflaterSource](../-inflater-source/index.md) that DEFLATE-decompresses this [Source](./index.md) while reading. |

### Inheritors

| Name | Summary |
|---|---|
| [BufferedSource](../-buffered-source/index.md)<br>(JVM, js, native) | `expect interface BufferedSource : `[`Source`](./index.md)<br>A source that keeps a buffer internally so that callers can do small reads without a performance penalty. It also allows clients to read ahead, buffering as much as necessary before consuming input. |
| [ForwardingSource](../-forwarding-source/index.md)<br>(JVM) | `abstract class ForwardingSource : `[`Source`](./index.md)<br>A [Source](./index.md) which forwards calls to another. Useful for subclassing. |
| [GzipSource](../-gzip-source/index.md)<br>(JVM) | `class GzipSource : `[`Source`](./index.md)<br>A source that uses [GZIP](http://www.ietf.org/rfc/rfc1952.txt) to decompress data read from another source. |
| [InflaterSource](../-inflater-source/index.md)<br>(JVM) | `class InflaterSource : `[`Source`](./index.md)<br>A source that uses [DEFLATE](http://tools.ietf.org/html/rfc1951) to decompress data read from another source. |
