//[okio](../../../index.md)/[okio](../index.md)/[ForwardingFileSystem](index.md)

# ForwardingFileSystem

[common]\
abstract class [ForwardingFileSystem](index.md)(**delegate**: [FileSystem](../-file-system/index.md)) : [FileSystem](../-file-system/index.md)

A [FileSystem](../-file-system/index.md) that forwards calls to another, intended for subclassing.

###  Fault Injection

You can use this to deterministically trigger file system failures in tests. This is useful to confirm that your program behaves correctly even if its file system operations fail. For example, this subclass fails every access of files named unlucky.txt:

val faultyFileSystem = object : ForwardingFileSystem(FileSystem.SYSTEM) {\
  override fun onPathParameter(path: Path, functionName: String, parameterName: String): Path {\
    if (path.name == "unlucky.txt") throw IOException("synthetic failure!")\
    return path\
  }\
}

You can fail specific operations by overriding them directly:

val faultyFileSystem = object : ForwardingFileSystem(FileSystem.SYSTEM) {\
  override fun delete(path: Path) {\
    throw IOException("synthetic failure!")\
  }\
}

###  Observability

You can extend this to verify which files your program accesses. This is a testing file system that records accesses as they happen:

class LoggingFileSystem : ForwardingFileSystem(FileSystem.SYSTEM) {\
  val log = mutableListOf<String>()\
\
  override fun onPathParameter(path: Path, functionName: String, parameterName: String): Path {\
    log += "$functionName($parameterName=$path)"\
    return path\
  }\
}

This makes it easy for tests to assert exactly which files were accessed.

@Test\
fun testMergeJsonReports() {\
  createSampleJsonReports()\
  loggingFileSystem.log.clear()\
\
  mergeJsonReports()\
\
  assertThat(loggingFileSystem.log).containsExactly(\
    "list(dir=json_reports)",\
    "source(file=json_reports/2020-10.json)",\
    "source(file=json_reports/2020-12.json)",\
    "source(file=json_reports/2020-11.json)",\
    "sink(file=json_reports/2020-all.json)"\
  )\
}

###  Transformations

Subclasses can transform file names and content.

For example, your program may be written to operate on a well-known directory like /etc/ or /System. You can rewrite paths to make such operations safer to test.

You may also transform file content to apply application-layer encryption or compression. This is particularly useful in situations where it's difficult or impossible to enable those features in the underlying file system.

###  Abstract Functions Only

Some file system functions like copy are implemented by using other features. These are the non-abstract functions in the [FileSystem](../-file-system/index.md) interface.

**This class forwards only the abstract functions;** non-abstract functions delegate to the other functions of this class. If desired, subclasses may override non-abstract functions to forward them.

## Constructors

| | |
|---|---|
| [ForwardingFileSystem](-forwarding-file-system.md) | [common]<br>fun [ForwardingFileSystem](-forwarding-file-system.md)(delegate: [FileSystem](../-file-system/index.md)) |

## Functions

| Name | Summary |
|---|---|
| [appendingSink](appending-sink.md) | [common]<br>open override fun [appendingSink](appending-sink.md)(file: [Path](../-path/index.md), mustExist: [Boolean](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-boolean/index.html)): [Sink](../-sink/index.md)<br>Returns a sink that appends bytes to the end of [file](appending-sink.md), creating it if it doesn't already exist. |
| [atomicMove](atomic-move.md) | [common]<br>open override fun [atomicMove](atomic-move.md)(source: [Path](../-path/index.md), target: [Path](../-path/index.md))<br>Moves [source](atomic-move.md) to [target](atomic-move.md) in-place if the underlying file system supports it. |
| [canonicalize](canonicalize.md) | [common]<br>open override fun [canonicalize](canonicalize.md)(path: [Path](../-path/index.md)): [Path](../-path/index.md)<br>Resolves [path](canonicalize.md) against the current working directory and symlinks in this file system. |
| [copy](../-file-system/copy.md) | [common]<br>open fun [copy](../-file-system/copy.md)(source: [Path](../-path/index.md), target: [Path](../-path/index.md))<br>Copies all of the bytes from the file at [source](../-file-system/copy.md) to the file at [target](../-file-system/copy.md). |
| [createDirectories](../-file-system/create-directories.md) | [common]<br>fun [createDirectories](../-file-system/create-directories.md)(dir: [Path](../-path/index.md))<br>Creates a directory at the path identified by [dir](../-file-system/create-directories.md), and any enclosing parent path directories, recursively. |
| [createDirectory](create-directory.md) | [common]<br>open override fun [createDirectory](create-directory.md)(dir: [Path](../-path/index.md))<br>Creates a directory at the path identified by [dir](create-directory.md). |
| [createSymlink](create-symlink.md) | [common]<br>open override fun [createSymlink](create-symlink.md)(source: [Path](../-path/index.md), target: [Path](../-path/index.md))<br>Creates a symbolic link at [source](create-symlink.md) that resolves to [target](create-symlink.md). |
| [delete](delete.md) | [common]<br>open override fun [delete](delete.md)(path: [Path](../-path/index.md))<br>Deletes the file or directory at [path](delete.md). |
| [deleteRecursively](../-file-system/delete-recursively.md) | [common]<br>open fun [deleteRecursively](../-file-system/delete-recursively.md)(fileOrDirectory: [Path](../-path/index.md))<br>Recursively deletes all children of [fileOrDirectory](../-file-system/delete-recursively.md) if it is a directory, then deletes [fileOrDirectory](../-file-system/delete-recursively.md) itself. |
| [exists](../-file-system/exists.md) | [common]<br>fun [exists](../-file-system/exists.md)(path: [Path](../-path/index.md)): [Boolean](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-boolean/index.html)<br>Returns true if [path](../-file-system/exists.md) identifies an object on this file system. |
| [list](list.md) | [common]<br>open override fun [list](list.md)(dir: [Path](../-path/index.md)): [List](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin.collections/-list/index.html)<[Path](../-path/index.md)><br>Returns the children of the directory identified by [dir](list.md). |
| [listOrNull](list-or-null.md) | [common]<br>open override fun [listOrNull](list-or-null.md)(dir: [Path](../-path/index.md)): [List](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin.collections/-list/index.html)<[Path](../-path/index.md)>?<br>Returns the children of the directory identified by [dir](list-or-null.md). |
| [listRecursively](list-recursively.md) | [common]<br>open override fun [listRecursively](list-recursively.md)(dir: [Path](../-path/index.md), followSymlinks: [Boolean](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-boolean/index.html)): [Sequence](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin.sequences/-sequence/index.html)<[Path](../-path/index.md)><br>Returns a sequence that **lazily** traverses the children of [dir](list-recursively.md) using repeated calls to [list](list.md). |
| [metadata](../-file-system/metadata.md) | [common]<br>fun [metadata](../-file-system/metadata.md)(path: [Path](../-path/index.md)): [FileMetadata](../-file-metadata/index.md)<br>Returns metadata of the file, directory, or object identified by [path](../-file-system/metadata.md). |
| [metadataOrNull](metadata-or-null.md) | [common]<br>open override fun [metadataOrNull](metadata-or-null.md)(path: [Path](../-path/index.md)): [FileMetadata](../-file-metadata/index.md)?<br>Returns metadata of the file, directory, or object identified by [path](metadata-or-null.md). |
| [onPathParameter](on-path-parameter.md) | [common]<br>open fun [onPathParameter](on-path-parameter.md)(path: [Path](../-path/index.md), functionName: [String](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-string/index.html), parameterName: [String](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-string/index.html)): [Path](../-path/index.md)<br>Invoked each time a path is passed as a parameter to this file system. |
| [onPathResult](on-path-result.md) | [common]<br>open fun [onPathResult](on-path-result.md)(path: [Path](../-path/index.md), functionName: [String](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-string/index.html)): [Path](../-path/index.md)<br>Invoked each time a path is returned by [delegate](delegate.md). |
| [openReadOnly](open-read-only.md) | [common]<br>open override fun [openReadOnly](open-read-only.md)(file: [Path](../-path/index.md)): [FileHandle](../-file-handle/index.md)<br>Returns a handle to read [file](open-read-only.md). |
| [openReadWrite](open-read-write.md) | [common]<br>open override fun [openReadWrite](open-read-write.md)(file: [Path](../-path/index.md), mustCreate: [Boolean](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-boolean/index.html), mustExist: [Boolean](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-boolean/index.html)): [FileHandle](../-file-handle/index.md)<br>Returns a handle to read and write [file](open-read-write.md). |
| [read](../-file-system/read.md) | [common]<br>inline fun <[T](../-file-system/read.md)> [read](../-file-system/read.md)(file: [Path](../-path/index.md), readerAction: [BufferedSource](../-buffered-source/index.md).() -> [T](../-file-system/read.md)): [T](../-file-system/read.md)<br>Creates a source to read [file](../-file-system/read.md), executes [readerAction](../-file-system/read.md) to read it, and then closes the source. |
| [sink](sink.md) | [common]<br>open override fun [sink](sink.md)(file: [Path](../-path/index.md), mustCreate: [Boolean](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-boolean/index.html)): [Sink](../-sink/index.md)<br>Returns a sink that writes bytes to [file](sink.md) from beginning to end. |
| [source](source.md) | [common]<br>open override fun [source](source.md)(file: [Path](../-path/index.md)): [Source](../-source/index.md)<br>Returns a source that reads the bytes of [file](source.md) from beginning to end. |
| [toString](to-string.md) | [common]<br>open override fun [toString](to-string.md)(): [String](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-string/index.html) |
| [write](../-file-system/write.md) | [common]<br>inline fun <[T](../-file-system/write.md)> [write](../-file-system/write.md)(file: [Path](../-path/index.md), mustCreate: [Boolean](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-boolean/index.html) = false, writerAction: [BufferedSink](../-buffered-sink/index.md).() -> [T](../-file-system/write.md)): [T](../-file-system/write.md)<br>Creates a sink to write [file](../-file-system/write.md), executes [writerAction](../-file-system/write.md) to write it, and then closes the sink. |

## Properties

| Name | Summary |
|---|---|
| [delegate](delegate.md) | [common]<br>val [delegate](delegate.md): [FileSystem](../-file-system/index.md)<br>[FileSystem](../-file-system/index.md) to which this instance is delegating. |
