//[okio](../../index.md)/[okio](../index.md)/[ForwardingFilesystem](index.md)



# ForwardingFilesystem  
 [common] abstract class [ForwardingFilesystem](index.md)(**delegate**: [Filesystem](../-filesystem/index.md)) : [Filesystem](../-filesystem/index.md)

A [Filesystem](../-filesystem/index.md) that forwards calls to another, intended for subclassing.



###  Fault Injection  


You can use this to deterministically trigger filesystem failures in tests. This is useful to confirm that your program behaves correctly even if its filesystem operations fail. For example, this subclass fails every access of files named unlucky.txt:

val faultyFilesystem = object : ForwardingFilesystem(FileSystem.SYSTEM) {  
  override fun onPathParameter(path: Path, functionName: String, parameterName: String): Path {  
    if (path.name == "unlucky.txt") throw IOException("synthetic failure!")  
    return path  
  }  
}

You can fail specific operations by overriding them directly:

val faultyFilesystem = object : ForwardingFilesystem(FileSystem.SYSTEM) {  
  override fun delete(path: Path) {  
    throw IOException("synthetic failure!")  
  }  
}

###  Observability  


You can extend this to verify which files your program accesses. This is a testing filesystem that records accesses as they happen:

class LoggingFilesystem : ForwardingFilesystem(Filesystem.SYSTEM) {  
  val log = mutableListOf<String>()  
  
  override fun onPathParameter(path: Path, functionName: String, parameterName: String): Path {  
    log += "$functionName($parameterName=$path)"  
    return path  
  }  
}

This makes it easy for tests to assert exactly which files were accessed.

@Test  
fun testMergeJsonReports() {  
  createSampleJsonReports()  
  loggingFilesystem.log.clear()  
  
  mergeJsonReports()  
  
  assertThat(loggingFilesystem.log).containsExactly(  
    "list(dir=json_reports)",  
    "source(file=json_reports/2020-10.json)",  
    "source(file=json_reports/2020-12.json)",  
    "source(file=json_reports/2020-11.json)",  
    "sink(file=json_reports/2020-all.json)"  
  )  
}

###  Transformations  


Subclasses can transform file names and content.



For example, your program may be written to operate on a well-known directory like /etc/ or /System. You can rewrite paths to make such operations safer to test.



You may also transform file content to apply application-layer encryption or compression. This is particularly useful in situations where it's difficult or impossible to enable those features in the underlying filesystem.



###  Abstract Functions Only  


Some filesystem functions like copy are implemented by using other features. These are the non-abstract functions in the [Filesystem](../-filesystem/index.md) interface.



**This class forwards only the abstract functions;** non-abstract functions delegate to the other functions of this class. If desired, subclasses may override non-abstract functions to forward them.

   


## Constructors  
  
|  Name|  Summary| 
|---|---|
| <a name="okio/ForwardingFilesystem/ForwardingFilesystem/#okio.Filesystem/PointingToDeclaration/"></a>[ForwardingFilesystem](-forwarding-filesystem.md)| <a name="okio/ForwardingFilesystem/ForwardingFilesystem/#okio.Filesystem/PointingToDeclaration/"></a> [common] fun [ForwardingFilesystem](-forwarding-filesystem.md)(delegate: [Filesystem](../-filesystem/index.md))   <br>


## Functions  
  
|  Name|  Summary| 
|---|---|
| <a name="okio/ForwardingFilesystem/appendingSink/#okio.Path/PointingToDeclaration/"></a>[appendingSink](appending-sink.md)| <a name="okio/ForwardingFilesystem/appendingSink/#okio.Path/PointingToDeclaration/"></a>[common]  <br>Content  <br>open override fun [appendingSink](appending-sink.md)(file: [Path](../-path/index.md)): [Sink](../-sink/index.md)  <br>More info  <br>Returns a sink that appends bytes to the end of file, creating it if it doesn't already exist.  <br><br><br>
| <a name="okio/ForwardingFilesystem/atomicMove/#okio.Path#okio.Path/PointingToDeclaration/"></a>[atomicMove](atomic-move.md)| <a name="okio/ForwardingFilesystem/atomicMove/#okio.Path#okio.Path/PointingToDeclaration/"></a>[common]  <br>Content  <br>open override fun [atomicMove](atomic-move.md)(source: [Path](../-path/index.md), target: [Path](../-path/index.md))  <br>More info  <br>Moves source to target in-place if the underlying file system supports it.  <br><br><br>
| <a name="okio/ForwardingFilesystem/canonicalize/#okio.Path/PointingToDeclaration/"></a>[canonicalize](canonicalize.md)| <a name="okio/ForwardingFilesystem/canonicalize/#okio.Path/PointingToDeclaration/"></a>[common]  <br>Content  <br>open override fun [canonicalize](canonicalize.md)(path: [Path](../-path/index.md)): [Path](../-path/index.md)  <br>More info  <br>Resolves path against the current working directory and symlinks in this filesystem.  <br><br><br>
| <a name="okio/Filesystem/copy/#okio.Path#okio.Path/PointingToDeclaration/"></a>[copy](../-filesystem/copy.md)| <a name="okio/Filesystem/copy/#okio.Path#okio.Path/PointingToDeclaration/"></a>[common]  <br>Content  <br>open fun [copy](../-filesystem/copy.md)(source: [Path](../-path/index.md), target: [Path](../-path/index.md))  <br>More info  <br>Copies all of the bytes from the file at source to the file at target.  <br><br><br>
| <a name="okio/Filesystem/createDirectories/#okio.Path/PointingToDeclaration/"></a>[createDirectories](../-filesystem/create-directories.md)| <a name="okio/Filesystem/createDirectories/#okio.Path/PointingToDeclaration/"></a>[common]  <br>Content  <br>fun [createDirectories](../-filesystem/create-directories.md)(dir: [Path](../-path/index.md))  <br>More info  <br>Creates a directory at the path identified by dir, and any enclosing parent path directories, recursively.  <br><br><br>
| <a name="okio/ForwardingFilesystem/createDirectory/#okio.Path/PointingToDeclaration/"></a>[createDirectory](create-directory.md)| <a name="okio/ForwardingFilesystem/createDirectory/#okio.Path/PointingToDeclaration/"></a>[common]  <br>Content  <br>open override fun [createDirectory](create-directory.md)(dir: [Path](../-path/index.md))  <br>More info  <br>Creates a directory at the path identified by dir.  <br><br><br>
| <a name="okio/ForwardingFilesystem/delete/#okio.Path/PointingToDeclaration/"></a>[delete](delete.md)| <a name="okio/ForwardingFilesystem/delete/#okio.Path/PointingToDeclaration/"></a>[common]  <br>Content  <br>open override fun [delete](delete.md)(path: [Path](../-path/index.md))  <br>More info  <br>Deletes the file or directory at path.  <br><br><br>
| <a name="okio/Filesystem/deleteRecursively/#okio.Path/PointingToDeclaration/"></a>[deleteRecursively](../-filesystem/delete-recursively.md)| <a name="okio/Filesystem/deleteRecursively/#okio.Path/PointingToDeclaration/"></a>[common]  <br>Content  <br>open fun [deleteRecursively](../-filesystem/delete-recursively.md)(fileOrDirectory: [Path](../-path/index.md))  <br>More info  <br>Recursively deletes all children of fileOrDirectory if it is a directory, then deletes fileOrDirectory itself.  <br><br><br>
| <a name="kotlin/Any/equals/#kotlin.Any?/PointingToDeclaration/"></a>[equals](../-timeout/-companion/index.md#%5Bkotlin%2FAny%2Fequals%2F%23kotlin.Any%3F%2FPointingToDeclaration%2F%5D%2FFunctions%2F-792583948)| <a name="kotlin/Any/equals/#kotlin.Any?/PointingToDeclaration/"></a>[common]  <br>Content  <br>open operator fun [equals](../-timeout/-companion/index.md#%5Bkotlin%2FAny%2Fequals%2F%23kotlin.Any%3F%2FPointingToDeclaration%2F%5D%2FFunctions%2F-792583948)(other: [Any](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-any/index.html)?): [Boolean](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-boolean/index.html)  <br><br><br>
| <a name="okio/Filesystem/exists/#okio.Path/PointingToDeclaration/"></a>[exists](../-filesystem/exists.md)| <a name="okio/Filesystem/exists/#okio.Path/PointingToDeclaration/"></a>[common]  <br>Content  <br>fun [exists](../-filesystem/exists.md)(path: [Path](../-path/index.md)): [Boolean](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-boolean/index.html)  <br>More info  <br>Returns true if path identifies an object on this filesystem.  <br><br><br>
| <a name="kotlin/Any/hashCode/#/PointingToDeclaration/"></a>[hashCode](../-timeout/-companion/index.md#%5Bkotlin%2FAny%2FhashCode%2F%23%2FPointingToDeclaration%2F%5D%2FFunctions%2F-792583948)| <a name="kotlin/Any/hashCode/#/PointingToDeclaration/"></a>[common]  <br>Content  <br>open fun [hashCode](../-timeout/-companion/index.md#%5Bkotlin%2FAny%2FhashCode%2F%23%2FPointingToDeclaration%2F%5D%2FFunctions%2F-792583948)(): [Int](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-int/index.html)  <br><br><br>
| <a name="okio/ForwardingFilesystem/list/#okio.Path/PointingToDeclaration/"></a>[list](list.md)| <a name="okio/ForwardingFilesystem/list/#okio.Path/PointingToDeclaration/"></a>[common]  <br>Content  <br>open override fun [list](list.md)(dir: [Path](../-path/index.md)): [List](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin.collections/-list/index.html)<[Path](../-path/index.md)>  <br>More info  <br>Returns the children of the directory identified by dir.  <br><br><br>
| <a name="okio/Filesystem/metadata/#okio.Path/PointingToDeclaration/"></a>[metadata](../-filesystem/metadata.md)| <a name="okio/Filesystem/metadata/#okio.Path/PointingToDeclaration/"></a>[common]  <br>Content  <br>fun [metadata](../-filesystem/metadata.md)(path: [Path](../-path/index.md)): [FileMetadata](../-file-metadata/index.md)  <br>More info  <br>Returns metadata of the file, directory, or object identified by path.  <br><br><br>
| <a name="okio/ForwardingFilesystem/metadataOrNull/#okio.Path/PointingToDeclaration/"></a>[metadataOrNull](metadata-or-null.md)| <a name="okio/ForwardingFilesystem/metadataOrNull/#okio.Path/PointingToDeclaration/"></a>[common]  <br>Content  <br>open override fun [metadataOrNull](metadata-or-null.md)(path: [Path](../-path/index.md)): [FileMetadata](../-file-metadata/index.md)?  <br>More info  <br>Returns metadata of the file, directory, or object identified by path.  <br><br><br>
| <a name="okio/ForwardingFilesystem/onPathParameter/#okio.Path#kotlin.String#kotlin.String/PointingToDeclaration/"></a>[onPathParameter](on-path-parameter.md)| <a name="okio/ForwardingFilesystem/onPathParameter/#okio.Path#kotlin.String#kotlin.String/PointingToDeclaration/"></a>[common]  <br>Content  <br>open fun [onPathParameter](on-path-parameter.md)(path: [Path](../-path/index.md), functionName: [String](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-string/index.html), parameterName: [String](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-string/index.html)): [Path](../-path/index.md)  <br>More info  <br>Invoked each time a path is passed as a parameter to this filesystem.  <br><br><br>
| <a name="okio/ForwardingFilesystem/onPathResult/#okio.Path#kotlin.String/PointingToDeclaration/"></a>[onPathResult](on-path-result.md)| <a name="okio/ForwardingFilesystem/onPathResult/#okio.Path#kotlin.String/PointingToDeclaration/"></a>[common]  <br>Content  <br>open fun [onPathResult](on-path-result.md)(path: [Path](../-path/index.md), functionName: [String](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-string/index.html)): [Path](../-path/index.md)  <br>More info  <br>Invoked each time a path is returned by [delegate](delegate.md).  <br><br><br>
| <a name="okio/ForwardingFilesystem/sink/#okio.Path/PointingToDeclaration/"></a>[sink](sink.md)| <a name="okio/ForwardingFilesystem/sink/#okio.Path/PointingToDeclaration/"></a>[common]  <br>Content  <br>open override fun [sink](sink.md)(file: [Path](../-path/index.md)): [Sink](../-sink/index.md)  <br>More info  <br>Returns a sink that writes bytes to file from beginning to end.  <br><br><br>
| <a name="okio/ForwardingFilesystem/source/#okio.Path/PointingToDeclaration/"></a>[source](source.md)| <a name="okio/ForwardingFilesystem/source/#okio.Path/PointingToDeclaration/"></a>[common]  <br>Content  <br>open override fun [source](source.md)(file: [Path](../-path/index.md)): [Source](../-source/index.md)  <br>More info  <br>Returns a source that reads the bytes of file from beginning to end.  <br><br><br>
| <a name="okio/ForwardingFilesystem/toString/#/PointingToDeclaration/"></a>[toString](to-string.md)| <a name="okio/ForwardingFilesystem/toString/#/PointingToDeclaration/"></a>[common]  <br>Content  <br>open override fun [toString](to-string.md)(): [String](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-string/index.html)  <br><br><br>


## Properties  
  
|  Name|  Summary| 
|---|---|
| <a name="okio/ForwardingFilesystem/delegate/#/PointingToDeclaration/"></a>[delegate](delegate.md)| <a name="okio/ForwardingFilesystem/delegate/#/PointingToDeclaration/"></a> [common] val [delegate](delegate.md): [Filesystem](../-filesystem/index.md)[Filesystem](../-filesystem/index.md) to which this instance is delegating.   <br>

