/*
 * Copyright (c) 2024 Liang Wenjian
 * magicall is licensed under Mulan PSL v2.
 * You can use this software according to the terms and conditions of the Mulan PSL v2.
 * You may obtain a copy of Mulan PSL v2 at:
 *          http://license.coscl.org.cn/MulanPSL2
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PSL v2 for more details.
 */

package cool.document.qtmd;

import org.jsoup.nodes.Element;

/**
 * 正格式化文本的状态。
 * 注：不能嵌套hr。即比如：
 * <code>
 * \b(
 * ---
 * )
 * </code>
 * 中的三横线不会变成hr。
 */
public class FormattingState extends NotStartState {
	protected final boolean isSingleline;
	protected final String flag;
	protected final Element dom;

	public FormattingState(final State preState, final CharSequence flag, final boolean isSingleline) {
		this(preState, preState.dom(), flag, isSingleline);
	}

	public FormattingState(final State preState, final Element parentDom, final CharSequence flag,
			final boolean isSingleline) {
		this(preState, createDom(preState, flag.toString(), isSingleline, parentDom), isSingleline);
	}

	public FormattingState(final State preState, final Element dom, final boolean isSingleline) {
		super(preState);
		flag = dom.tagName();
		this.isSingleline = isSingleline;
		this.dom = dom;
	}

	private static Element createDom(final State preState, final String flag, final boolean isSingleline,
			final Element parentDom) {
		preState.flushBuffer();
		if (!flag.isEmpty()) {
			return Helper.newDom(flag, isSingleline).appendTo(parentDom);
		}
		return parentDom;
	}

	@Override
	public Element dom() {
		return dom;
	}

	@Override
	public State metActivateChar(final char activateChar) {
		return new InFormattingActivateState(this, activateChar);
	}

	@Override
	public State metNewline(final char newlineChar) {
		if (isSingleline) {
			return finish(newlineChar);
		}
		//多行，todo：可能需要检查hr
		return super.metNewline(newlineChar);
	}

	@Override
	public boolean isFinishChar(final char c) {
		return isSingleline && isNewline(c);
	}

	@Override
	public State finish(final char finishChar) {
		flushBuffer();
		return handleFinishChar(finishChar);
	}

	protected State handleFinishChar(final char finishChar) {
		if (isNewline(finishChar)) {//换行符要冒泡传递
			return preState().metChar(finishChar);
		}
		return preState();
	}
}
