/*
 * Copyright (c) 2024 Liang Wenjian
 * magicall is licensed under Mulan PSL v2.
 * You can use this software according to the terms and conditions of the Mulan PSL v2.
 * You may obtain a copy of Mulan PSL v2 at:
 *          http://license.coscl.org.cn/MulanPSL2
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PSL v2 for more details.
 */

package cool.document.qtmd;

import me.magicall.贵阳DearSun.exception.UnknownException;
import org.jsoup.nodes.Element;

public class LiState extends FormattingState {
	protected final int indentCount;
	protected final State preState;//本li结束后返回的状态。不是父类上那个
	protected final Element dom;//即liDom

	public LiState(final State preState, final CharSequence activateChars, final CharSequence flag,
			final boolean isSingleline) {
		super(preState, flag, isSingleline);

		final var liParentDom = super.dom;//父类创建的ol或ul
		//前置状态有3种可能：
		// 1，LiContentFinishedState，说明之前也是一个li，且内容已结束。
		// 2，LiState，说明之前也是一个li，且内容已结束，且本li确定是之前li的子级
		// 3，其他，说明之前不是li。
		if (preState instanceof final LiContentFinishedState liContentFinishedState) {
			//前置状态的buffer里应该只有空白字符，在多层li中，只用来算缩进，不作为内容的一部分。
			indentCount = calIndent(liContentFinishedState.clearBuffer() + activateChars);
			final var delta = indentCount - liContentFinishedState.liState().indentCount;
			if (delta > 0) {//本li缩进较大，视为之前li的下级（忽略缩进数差距）
				this.preState = liContentFinishedState;
				dom = liParentDom.appendElement("li");
			} else {//delta<=0
				//无论如何，本li不是之前li的下级，要计算真正的parent dom。
				final var curLiGrandparentDom = preState.dom();// liParentDom.parent();//即之前li的dom：preState.dom()
				final var preLiParentDom = curLiGrandparentDom.parent();//之前li的dom的父dom，ol或ul。
				if (preLiParentDom == null) {
					throw new UnknownException();
				}
				liParentDom.remove();

				if (delta == 0) {//缩进相同，是同级。
					// 令所有同级li具有相同的前置状态。
					final var preLi = liContentFinishedState.preState;
					this.preState = preLi.preState();
					dom = preLiParentDom.appendElement("li");
				} else {//本li缩进较小，可能是前一个li的祖先层级的下级。
					//下面的代码看着挫，可能可以优化
					var preLiState = liContentFinishedState.preState();//liContentFinishedState对应的LiState
					State preStateOfFirstSiblingLi = preLiState.preState();
					while (preStateOfFirstSiblingLi instanceof final LiContentFinishedState s) {
						preLiState = s.preState;
						if (((LiState) preLiState).indentCount < indentCount) {
							break;
						}
						preStateOfFirstSiblingLi = preLiState.preState();
					}
					this.preState = preStateOfFirstSiblingLi;
					final var realParentDom = preLiState.dom().parent();//一个ol或ul
					if (realParentDom == null) {
						throw new UnknownException();
					}
					dom = realParentDom.appendElement("li");
				}
			}
		} else {//preState不是LiContentFinishedState，可能是Li、文本。
			this.preState = preState;
			dom = liParentDom.appendElement("li");
			//用前一个文本dom最后一行算缩进
			final var siblingNodes = liParentDom.siblingNodes();
			if (siblingNodes.isEmpty()) {
				indentCount = calIndent(activateChars);
			} else {
				final var lastTextNode = siblingNodes.get(siblingNodes.size() - 1);
				indentCount = calIndent(lastTextNode.toString() + activateChars);
			}
		}
	}

	protected static int calIndent(final CharSequence activateChars) {
		//最后一个字符是\
		return Helper.calIndent(activateChars.subSequence(0, activateChars.length() - 1).toString());
	}

	@Override
	public State preState() {
		return preState;
	}

	@Override
	public Element dom() {
		return dom;
	}

	//--------------------------------------------------------------------------

	@Override
	public State metActivateChar(final char activateChar) {
		return new InLiActivatingState(this, activateChar);
	}

	@Override
	public State metNewline(final char newlineChar) {
		if (isSingleline) {
			flushBuffer();//若为单行，遇到换行符则意味着列表项的文本部分已经结束了。
			return new UnderLiState(this, newlineChar);
		}
		return super.metNewline(newlineChar);
	}

	@Override
	public boolean isFinishChar(final char c) {
		//Li状态下不会自行结束。
		return false;
	}

	//========================================================

	public static class InLiActivatingState extends InFormattingActivateState {
		private final LiState liState;

		public InLiActivatingState(final LiState preState, final char activateChar) {
			super(preState, activateChar);
			liState = preState;
		}

		@Override
		public State finish(final char finishChar) {
			if (willConsumeFinishChar(finishChar)) {
				return preState();
			}
			return preState().metChar(finishChar);
		}

		@Override
		protected State finishFormatting(final char c) {
			return new LiContentFinishedState(liState, c);
		}

		@Override
		protected HashState metHash(final char c) {
			//buffer里应当只有一个\
			return new HashState(liState, buffer, c);
		}
	}

	//========================================================

	public interface MaybeFollowingNextLiState extends MiddleState {
		default LiState liState() {
			final var preState = preState();
			if (preState instanceof final LiState liState) {
				return liState;
			}
			if (preState instanceof final MaybeFollowingNextLiState maybeFollowingNextLiState) {
				return maybeFollowingNextLiState.liState();
			}
			throw new UnknownException();
		}

		@Override
		default State metActivateChar(final char activateChar) {
			return finishLi(activateChar);
		}

		@Override
		default State metNewline(final char newlineChar) {
			return finishLi(newlineChar);
		}

		@Override
		default State finish(final char finishChar) {
			//与父类不同：其他中间状态，其前置状态（内容状态）已经完成了，所以结束（回滚）时，所缓存字符可以直接加到前置状态的缓存字符。
			//本类状态相应的li只能算内容结束，结构未必结束。
			// 若把本类状态视为对应的li状态的一种子状态，本类状态的结束（回滚）意味着对应的li状态真正结束。
			return finishLi(finishChar);
		}

		/**
		 * 默认实现：相关（持有/代理/代表）li状态结束，回退至其前置状态
		 * 且处理本状态的buffer（对应li内容结束之后遇到的内容，通常是空白字符），以及当前所遇字符。
		 *
		 * @return
		 */
		default State finishLi(final char c) {
			return finishLi0().metChars(buffer()).metChar(c);
		}

		/**
		 * 相关（持有/代理/代表）li状态结束，回退至其前置状态。
		 *
		 * @return
		 */
		default State finishLi0() {
			return liState().finish(liContentFinishChar());
		}

		char liContentFinishChar();

		@Override
		default State metOtherChar(final char c) {
			if (Character.isWhitespace(c)) {
				return addChar(c);
			}
			return finishLi(c);
		}

		@Override
		default State flushBuffer() {//本状态代表li内容已完成，所以flush buffer时不需要刷。
			return this;
		}
	}

	//========================================================

	public static class LiContentFinishedState extends NotStartState implements MaybeFollowingNextLiState {
		protected final char liContentFinishChar;

		public LiContentFinishedState(final State preState, final char liContentFinishChar,
				final CharSequence leadingChars) {
			super(preState);
			this.liContentFinishChar = liContentFinishChar;
			buffer.append(leadingChars);
			preState.flushBuffer();
		}

		public LiContentFinishedState(final State preState, final char liContentFinishChar) {
			super(preState);
			this.liContentFinishChar = liContentFinishChar;
			preState.flushBuffer();
		}

		@Override
		public State metNewline(final char newlineChar) {
			return new UnderLiState(preState, liContentFinishChar, buffer.toString(), newlineChar);
		}

		@Override
		public char liContentFinishChar() {
			return liContentFinishChar;
		}
	}

	//==============================================================

	public static class UnderLiState extends LiContentFinishedState implements MaybeFollowingNextLiState {
		public UnderLiState(final State preState, final char liContentFinishChar, final CharSequence leadingChars,
				final char newlineChar) {
			super(preState, liContentFinishChar, leadingChars);
			buffer.append(newlineChar);
		}

		public UnderLiState(final State preState, final char liContentFinishChar) {
			super(preState, liContentFinishChar);
		}

		@Override
		public State metNewline(final char newlineChar) {
			return finishLi(newlineChar);
		}

		@Override
		public ActivateState metActivateChar(final char activateChar) {
			return new UnderLiActivateState(this, activateChar);
		}
	}

	//======================================================

	public static class UnderLiActivateState extends ActivateState implements MaybeFollowingNextLiState {
		public UnderLiActivateState(final UnderLiState preState, final char activateChar) {
			super(preState, activateChar);
		}

		@Override
		public State metOtherChar(final char c) {
			if (c == '#' || c == '.') {
				return super.metOtherChar(c);
			}
			return finish(c);
		}

		@Override
		public char liContentFinishChar() {
			return ((MaybeFollowingNextLiState) preState).liContentFinishChar();
		}
	}
}
