/*
 * Copyright (c) 2024 Liang Wenjian
 * magicall is licensed under Mulan PSL v2.
 * You can use this software according to the terms and conditions of the Mulan PSL v2.
 * You may obtain a copy of Mulan PSL v2 at:
 *          http://license.coscl.org.cn/MulanPSL2
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PSL v2 for more details.
 */

package cool.document.qtmd;

import org.jsoup.nodes.Element;

/**
 * 中间状态，是一个处理文本的状态到下一个处理文本的状态中间的临时状态。
 * 可能回滚到前置状态：表示本状态内所遇字符作为普通文本刷回前置状态的所遇字符中。
 * 不应当产生dom。
 * 自然结束逻辑上就是回滚。
 */
public interface MiddleState extends State {
	@Override
	State preState();

	default State stateToRollback() {
		return preState();
	}

	@Override
	default Element dom() {
		return preState().dom();
	}

	/**
	 * 中间状态的finish其实是回滚：把buffer刷到前置状态的buffer里，然后由前置状态处理结束符。
	 *
	 * @return
	 */
	@Override
	default State finish(final char finishChar) {
		flushBufferToPreState();
		if (willConsumeFinishChar(finishChar)) {
			return preState();
		}
		return preState().metChar(finishChar);
	}

	/**
	 * 把本状态内所遇字符刷到前置状态。
	 * 便利工具方法，通常不需要重写。
	 */
	default void flushBufferToPreState() {
		preState().addChars(buffer());
		clearBuffer();
	}

	/**
	 * 当遇上指定的回滚符号时，是否会消费它（即不向回滚后的状态传递）。默认皆否。
	 *
	 * @return
	 */
	default boolean willConsumeFinishChar(final char rollbackChar) {
		return false;
	}

	/**
	 * 结束时，把文本刷到前置状态，然后前置状态结束。
	 */
	@Override
	default void end() {
		flushBufferToPreState();
		preState().end();
	}
}
