/*
 * Copyright (c) 2024 Liang Wenjian
 * magicall is licensed under Mulan PSL v2.
 * You can use this software according to the terms and conditions of the Mulan PSL v2.
 * You may obtain a copy of Mulan PSL v2 at:
 *          http://license.coscl.org.cn/MulanPSL2
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PSL v2 for more details.
 */

package cool.document.qtmd;

import me.magicall.贵阳DearSun.exception.UnknownException;
import org.jsoup.nodes.Element;

public class TableState extends FormattingState {
	protected final Element tableDom;
	protected Element tdDom;

	public TableState(final State preState, final char flag) {
		super(preState, String.valueOf(flag), false);
		//父类会创建table dom
		tableDom = super.dom();
		Helper.attr(tableDom, "border", "1");
		final var tr = Helper.newTr(tableDom);
		tdDom = Helper.newTd(tr);
	}

	@Override
	public Element dom() {
		return tdDom;
	}

	@Override
	public ActivateState metActivateChar(final char activateChar) {
		return new InTdActivateState(this, activateChar);
	}

	//===============================================================

	/**
	 * 在表格的格子里激活的状态。接下来可能是格子结束，也可能是行结束。
	 */
	public static class InTdActivateState extends ActivateState {
		public InTdActivateState(final TableState preState, final char activateChar) {
			super(preState, activateChar);
		}

		@Override
		public boolean isFinishChar(final char c) {
			return !isTdEnd(c) && !isTrEnd(c) && super.isFinishChar(c);
		}

		@Override
		public State metOtherChar(final char c) {
			if (isTrEnd(c)) {
				return endTr(c);
			}
			if (isTdEnd(c)) {
				return newTd();
			}
			return super.metOtherChar(c);
		}

		private State endTr(final char c) {
			endTd();
			return new TableRowEndedState((TableState) preState, c);
		}

		private State newTd() {
			final var tr = endTd();
			tableState().tdDom = Helper.newTd(tr);
			return preState;
		}

		private Element endTd() {
			preState.flushBuffer();
			final var tr = tableState().tdDom.parent();
			if (tr == null) {
				throw new UnknownException();
			}
			return tr;
		}

		private TableState tableState() {
			return (TableState) preState();
		}

		private static boolean isTdEnd(final char c) {
			return c == '|';
		}

		private static boolean isTrEnd(final char c) {
			return c == ']';
		}
	}

	//============================================================

	/**
	 * 在表格行结束之后的状态。可以接一些空白字符。
	 */
	public interface TrFinishedState extends MiddleState {
		default TableState tableState() {
			return (TableState) preState();
		}

		@Override
		default State metActivateChar(final char activateChar) {
			return finishTable(activateChar);
		}

		@Override
		default State metNewline(final char newlineChar) {
			return finishTable(newlineChar);
		}

		@Override
		default State metOtherChar(final char c) {
			if (Character.isWhitespace(c)) {
				return addChar(c);
			}
			return finishTable(c);
		}

		/**
		 * 1，把buffer加到表格的前置状态。实现类自行决定哪些字符将会保存到buffer
		 * 2，表格结束
		 *
		 * @return
		 */
		default State finishTable(final char finishChar) {
			return tableState().finish(finishChar()).metChars(buffer()).metChar(finishChar);
		}

		char finishChar();
	}

	//=============================================================

	public static class TableRowEndedState extends NotStartState implements TrFinishedState {
		private final char endTrChar;

		public TableRowEndedState(final TableState preState, final char endTrChar) {
			super(preState);
			this.endTrChar = endTrChar;
		}

		@Override
		public State metNewline(final char newlineChar) {
			return new UnderTrState(tableState(), endTrChar, buffer.toString() + newlineChar);
		}

		@Override
		public char finishChar() {
			return endTrChar;
		}
	}

	//===============================================================

	/**
	 * 表格行结束后的状态。接下来可能是激活-另一行，也可能是表格结束。
	 */
	public static class UnderTrState extends NotStartState implements TrFinishedState {
		private final char endTrChar;

		public UnderTrState(final TableState preState, final char endTrChar, final Object enterChars) {
			super(preState);
			this.endTrChar = endTrChar;
			buffer.append(enterChars);
		}

		@Override
		public ActivateState metActivateChar(final char activateChar) {
			return new UnderTrActivateState(tableState(), endTrChar, buffer.toString() + activateChar);
		}

		@Override
		public char finishChar() {
			return endTrChar;
		}
	}

	//===============================================================

	/**
	 * 表格行结束之后激活的状态。接下来可能是另一行，也可能是表格结束。
	 */
	public static class UnderTrActivateState extends ActivateState implements TrFinishedState {
		private final char endTrChar;

		public UnderTrActivateState(final TableState preState, final char endTrChar, final String activateChars) {
			//activateChars可能包含前面的空白字符
			super(preState, activateChars);
			this.endTrChar = endTrChar;
		}

		@Override
		public State metOtherChar(final char c) {
			final var tableState = tableState();
			if (isTrStart(c)) {
				final var tr = Helper.newTr(tableState.tableDom);
				tableState.tdDom = Helper.newTd(tr);
				return tableState;
			}
			return finishTable(c);
		}

		@Override
		public char finishChar() {
			return endTrChar;
		}

		private static boolean isTrStart(final char c) {
			return c == '[';
		}
	}
}
