/*
 * Copyright (c) 2024 Document.Cool
 * QTMD is licensed under Mulan PSL v2.
 * You can use this software according to the terms and conditions of the Mulan PSL v2.
 * You may obtain a copy of Mulan PSL v2 at:
 *          http://license.coscl.org.cn/MulanPSL2
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PSL v2 for more details.
 */

package cool.document.qtmd;

import org.jsoup.nodes.Element;

/**
 * 文本状态。
 */
public class ContentState extends BaseState {
	private final Element dom;

	public ContentState(final ContentState stateAfterFinish, final Element dom) {
		super(stateAfterFinish);
		this.dom = dom;
	}

	@Override
	public ContentState stateAfterFinish() {
		return stateAfterFinish == null ? this : stateAfterFinish;
	}

	public Element dom() {
		return dom;
	}

	@Override
	public State metActivateChar(final char activateChar) {
		return new ActivateState(this, activateChar);
	}

	@Override
	public State metNewline(final char newlineChar) {
		checkHr();
		return super.metNewline(newlineChar);
	}

	@Override
	public void end() {
		checkHr();
		flushBuffer();
		final var stateAfterFinish = stateAfterFinish();
		if (stateAfterFinish != null && !equals(stateAfterFinish)) {
			stateAfterFinish.end();
		}
	}

	/**
	 * 刷缓冲区：把缓冲区字符刷到当前dom。
	 * 便利工具方法，通常不需要重写。
	 *
	 * @return
	 */
	@Override
	public State flushBuffer() {
		Helper.flushToDom(buffer(), dom());
		return this;
	}

	protected boolean checkHr() {
		return Helper.canHaveHr(dom.tagName()) && checkHr(this);
	}

	static boolean checkHr(final ContentState state) {
		final var s = state.buffer();
		final var index = s.lastIndexOf("\n");
		final var afterLastNewlineIndex = index + 1;
		final String lastLine;
		if (index == -1) {
			lastLine = s.toString();
		} else {
			lastLine = s.substring(afterLastNewlineIndex);
		}
		if (countHyphens(lastLine) > 2) {
			s.delete(afterLastNewlineIndex, s.length());
			state.flushBuffer();
			state.dom().appendElement("hr");
			state.clearBuffer();
			return true;
		}
		return false;
	}

	static int countHyphens(final CharSequence s) {
		int count = 0;
		final int bound = s.length();
		for (int i = 0; i < bound; i++) {
			final var c = s.charAt(i);
			if (c == '-') {
				count++;
			} else if (!Character.isWhitespace(c)) {
				return 0;
			}
		}
		return count;
	}
}
