/*
 * Copyright (c) 2024 Document.Cool
 * QTMD is licensed under Mulan PSL v2.
 * You can use this software according to the terms and conditions of the Mulan PSL v2.
 * You may obtain a copy of Mulan PSL v2 at:
 *          http://license.coscl.org.cn/MulanPSL2
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PSL v2 for more details.
 */

package cool.document.qtmd;

import me.magicall.贵阳DearSun.exception.UnknownException;
import org.jsoup.nodes.Element;

public class LiState extends FormattingState {
	public final int indentCount;
	private final Element liParentDom;
	private Element curLiDom;
	private boolean isCurLiContentFinished;
	private boolean isCurLiSingleline;

	public LiState(final ContentState stateAfterFinish, final CharSequence flag, final boolean isSingleline,
			final int indentCount) {
		super(stateAfterFinish, flag, isSingleline);
		this.indentCount = indentCount;
		liParentDom = super.dom();
		newLi(isSingleline);
	}

	public final LiState newLi(final boolean isCurLiSingleline) {
		curLiDom = liParentDom.appendElement("li");
		isCurLiContentFinished = false;
		this.isCurLiSingleline = isCurLiSingleline;
		return this;
	}

	@Override
	public Element dom() {
		return curLiDom;
	}

	@Override
	public boolean isSingleline() {
		return isCurLiSingleline;
	}

	@Override
	public State flushBuffer() {
		if (isCurLiContentFinished) {
			return this;
		}
		return super.flushBuffer();
	}

	@Override
	public State metActivateChar(final char activateChar) {
		return new InLiActivatingState(this, activateChar);
	}

	@Override
	public State metNewline(final char newlineChar) {
		if (isCurLiSingleline) {
			curLiFinishContent();
			return new UnderLiState(this, newlineChar);
		}
		if (isCurLiContentFinished) {
			return new UnderLiState(this, newlineChar);
		}
		return super.metNewline(newlineChar);
	}

	@Override
	public boolean isFinishChar(final char c) {
		return false;
	}

	public LiState curLiFinishContent() {
		flushBuffer();
		isCurLiContentFinished = true;
		return this;
	}

	public boolean isCurLiContentFinished() {
		return isCurLiContentFinished;
	}

	//==================================================================

	public static class InLiActivatingState extends InFormattingActivateState {
		private final LiState liState;

		public InLiActivatingState(final LiState stateAfterFinish, final char activateChar) {
			super(stateAfterFinish, activateChar);
			liState = stateAfterFinish;
		}

		@Override
		protected State finishFormatting(final char c) {
			liState.curLiFinishContent();
			return new LiContentFinishedState(liState, c);
		}
	}

	//=================================================================

	public interface MaybeFollowingNextLiState extends MiddleState {
		default LiState liState() {
			final var stateAfterFinish = stateAfterFinish();
			if (stateAfterFinish instanceof final LiState liState) {
				return liState;
			}
			if (stateAfterFinish instanceof final MaybeFollowingNextLiState maybeFollowingNextLiState) {
				return maybeFollowingNextLiState.liState();
			}
			throw new UnknownException();
		}

		/**
		 * 默认实现：相关（持有/代理/代表）li状态结束，回退至其前置状态
		 * 且处理本状态的buffer（对应li内容结束之后遇到的内容，通常是空白字符），以及当前所遇字符。
		 *
		 * @return
		 */
		default State finishLi(final char c) {
			return liState().finish(liContentFinishChar()).metChars(buffer()).metChar(c);
		}

		char liContentFinishChar();

		@Override
		default State metActivateChar(final char activateChar) {
			return finishLi(activateChar);
		}

		@Override
		default State metNewline(final char newlineChar) {
			return finishLi(newlineChar);
		}

		@Override
		default State finish(final char finishChar) {
			return finishLi(finishChar);
		}

		@Override
		default State metOtherChar(final char c) {
			if (Character.isWhitespace(c)) {
				return addChar(c);
			}
			return finishLi(c);
		}

		@Override
		default State flushBuffer() {//本状态代表li内容已完成，所以flush buffer时不需要刷。
			return this;
		}
	}

	//---------------------------------------------------------

	public static class LiContentFinishedState extends BaseMiddleState implements MaybeFollowingNextLiState {
		private final char liContentFinishChar;

		public LiContentFinishedState(final LiState liState, final char liContentFinishChar) {
			super(liState);
			this.liContentFinishChar = liContentFinishChar;
			liState.curLiFinishContent();
		}

		@Override
		public char liContentFinishChar() {
			return liContentFinishChar;
		}

		@Override
		public State metNewline(final char newlineChar) {
			return new UnderLiState(liState(), liContentFinishChar, buffer, newlineChar);
		}
	}

	//---------------------------------------------------------

	public static class UnderLiState extends BaseMiddleState implements MaybeFollowingNextLiState {
		protected final char liContentFinishChar;

		public UnderLiState(final LiState stateAfterFinish, final char liContentFinishChar, final CharSequence leadingChars,
				final char newlineChar) {
			this(stateAfterFinish, liContentFinishChar);
			buffer.append(leadingChars).append(newlineChar);
		}

		public UnderLiState(final LiState stateAfterFinish, final char liContentFinishChar) {
			super(stateAfterFinish);
			this.liContentFinishChar = liContentFinishChar;
			stateAfterFinish.curLiFinishContent();
		}

		@Override
		public ActivateState metActivateChar(final char activateChar) {
			return new UnderLiActivateState(liState(), buffer.toString() + activateChar, liContentFinishChar);
		}

		@Override
		public char liContentFinishChar() {
			return liContentFinishChar;
		}
	}

	//---------------------------------------------------------

	public static class UnderLiActivateState extends ActivateState implements MaybeFollowingNextLiState {
		private final char liContentFinishChar;

		public UnderLiActivateState(final LiState stateAfterFinish, final Object activateChars,
				final char liContentFinishChar) {
			super(stateAfterFinish, activateChars);
			this.liContentFinishChar = liContentFinishChar;
		}

		@Override
		public boolean isFinishChar(final char c) {
			return c != '#' && c != '.';
		}

		@Override
		public char liContentFinishChar() {
			return liContentFinishChar;
		}
	}
}


