/*
 * Copyright (c) 2024 Document.Cool
 * QTMD is licensed under Mulan PSL v2.
 * You can use this software according to the terms and conditions of the Mulan PSL v2.
 * You may obtain a copy of Mulan PSL v2 at:
 *          http://license.coscl.org.cn/MulanPSL2
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PSL v2 for more details.
 */

package cool.document.qtmd;

public abstract class MultimediaState extends FormattingState {
	private boolean contentFinished;

	public MultimediaState(final ContentState stateAfterFinish, final CharSequence flag, final boolean isSingleline) {
		super(stateAfterFinish, flag, isSingleline);
	}

	@Override
	public State metActivateChar(final char activateChar) {
		//内容完成后，来的应该是url的一部分。把\视为url的一部分。
		if (contentFinished) {
			return addChar(activateChar);
		}
		return new InMultimediaActivatingState(this, activateChar);
	}

	private static class InMultimediaActivatingState extends InFormattingActivateState {
		private final MultimediaState multimediaState;

		public InMultimediaActivatingState(final MultimediaState multimediaState, final char activateChar) {
			super(multimediaState, activateChar);
			this.multimediaState = multimediaState;
		}

		@Override
		public State metNewline(final char newlineChar) {
			return super.metNewline(newlineChar);
		}

		@Override
		protected State finishFormatting(final char c) {
			return multimediaState.isSingleline() ? multimediaState.finish(c) : multimediaState.contentFinished();
		}
	}

	@Override
	public State metNewline(final char newlineChar) {
		if (isSingleline() || contentFinished) {
			return finish(newlineChar);
		}
		return super.metNewline(newlineChar);
	}

	@Override
	public boolean isFinishChar(final char c) {
		return Character.isWhitespace(c) && (isSingleline() || contentFinished);
	}

	@Override
	public State finish(final char finishChar) {
		ensureAttrs();
		return super.finish(finishChar);
	}

	@Override
	protected State handleFinishChar(final char finishChar) {
		if (Character.isWhitespace(finishChar)) {//用空白字符断开url的，空白字符要冒泡传递
			return stateAfterFinish().metChar(finishChar);
		}
		return stateAfterFinish();
	}

	@Override
	public void end() {
		ensureAttrs();
		super.end();
	}

	private void ensureAttrs() {
		if (isSingleline()) {
			singlelineSetAttrs();
		} else {
			if (contentFinished) {
				setUrl();
			} else {
				contentFinished();
			}
		}
	}

	private MultimediaState contentFinished() {
		setContent();
		clearBuffer();
		contentFinished = true;
		return this;
	}

	private void singlelineSetAttrs() {
		setContent();
		setUrl();
	}

	protected void setUrl() {
		Helper.attr(dom(), urlAttr(), url());
		clearBuffer();
	}

	protected abstract String urlAttr();

	protected CharSequence url() {
		return buffer;
	}

	protected void setContent() {
		dom().appendText(buffer.toString());
	}

	//==========================================================

	public static class ImgState extends MultimediaState {
		public ImgState(final ContentState stateAfterFinish, final StringBuilder flag, final boolean isSingleline) {
			super(stateAfterFinish, flag, isSingleline);
		}

		@Override
		protected String urlAttr() {
			return "src";
		}

		@Override
		protected void setContent() {
			Helper.attr(dom(), "alt", buffer);
		}
	}

	//==========================================================

	public static class LinkState extends MultimediaState {
		public LinkState(final ContentState stateAfterFinish, final CharSequence flag, final boolean isSingleline) {
			super(stateAfterFinish, flag, isSingleline);
		}

		@Override
		protected String urlAttr() {
			return "href";
		}

		@Override
		protected CharSequence url() {
			return Helper.isEmail(buffer) ? "mailto:" + buffer : buffer;
		}
	}
}
