/*
 * Copyright (c) 2024 Document.Cool
 * QTMD is licensed under Mulan PSL v2.
 * You can use this software according to the terms and conditions of the Mulan PSL v2.
 * You may obtain a copy of Mulan PSL v2 at:
 *          http://license.coscl.org.cn/MulanPSL2
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PSL v2 for more details.
 */

package cool.document.qtmd;

import org.jsoup.Jsoup;
import org.jsoup.nodes.Document;
import org.jsoup.nodes.Document.OutputSettings;
import org.jsoup.nodes.Element;

public class Qtmd {

	private static final String HTML_TEMPLATE = "<!DOCTYPE html>"//
			+ "<html>"//
			+ "<head>"//
			+ "<meta charset=\"UTF-8\">"//
			+ "<meta name=\"viewport\" content=\"width=device-width, initial-scale=1.0\">"//
			+ "<title>QTMD预览</title>"//
			+ "</head>"//
			+ "<body>" + "<div style=\"white-space:pre-wrap\"></div>" + "</body>"//
			+ "</html>";

	/**
	 * 把内容转换成html。仅包含内容本身。
	 *
	 * @return
	 */
	public static String toHtml(final String content) {
		return toDom(content).html();
	}

	/**
	 * 把内容转换成html，包含一个包装div。
	 *
	 * @return
	 */
	public static String toWrapperDomHtml(final String content) {
		return toDom(content).outerHtml();
	}

	/**
	 * 把内容转换成完整的html文档。
	 *
	 * @return
	 */
	public static Document toDoc(final String content) {
		return toDom(content).ownerDocument();
	}

	/**
	 * 把内容转换成一个dom（html片段）
	 *
	 * @return
	 */
	public static Element toDom(final String content) {
		final var doc = Jsoup.parse(HTML_TEMPLATE);
		final OutputSettings outputSettings = new OutputSettings();
		outputSettings.prettyPrint(false); // 禁用格式化输出。jsoup默认会合并html里连续的空白字符。
		doc.outputSettings(outputSettings);
		final var container = doc.select("div").get(0);

		final String s = content.replace("\r\n", "\n").replace('\r', '\n');

		final var finalState = new ContentState(null, container).metChars(s);

		finalState.end();
		return container;
	}
}
