/*
 * Copyright (c) 2024 Document.Cool
 * QTMD is licensed under Mulan PSL v2.
 * You can use this software according to the terms and conditions of the Mulan PSL v2.
 * You may obtain a copy of Mulan PSL v2 at:
 *          http://license.coscl.org.cn/MulanPSL2
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PSL v2 for more details.
 */

package cool.document.qtmd;

import org.jsoup.nodes.Element;

/**
 * 正格式化文本的状态。
 * 注：不能嵌套hr。即比如：
 * <code>
 * \b(
 * ---
 * )
 * </code>
 * 中的三横线不会变成hr。
 */
public class FormattingState extends ContentState {
	protected final boolean isSingleline;
	protected final String flag;

	public FormattingState(final ContentState stateAfterFinish, final CharSequence flag, final boolean isSingleline) {
		this(stateAfterFinish, createDom(stateAfterFinish, flag.toString(), isSingleline, stateAfterFinish.dom()),
				isSingleline);
	}

	public FormattingState(final ContentState stateAfterFinish, final Element dom, final boolean isSingleline) {
		super(stateAfterFinish, dom);
		flag = dom.tagName();
		this.isSingleline = isSingleline;
	}

	private static Element createDom(final State stateAfterFinish, final String flag, final boolean isSingleline,
			final Element parentDom) {
		stateAfterFinish.flushBuffer();
		if (!flag.isEmpty()) {
			return Helper.newDom(flag, isSingleline).appendTo(parentDom);
		}
		return parentDom;
	}

	public boolean isSingleline() {
		return isSingleline;
	}

	@Override
	public State metActivateChar(final char activateChar) {
		return new InFormattingActivateState(this, activateChar);
	}

	@Override
	public State metNewline(final char newlineChar) {
		if (isSingleline()) {
			return finish(newlineChar);
		}
		return super.metNewline(newlineChar);
	}

	@Override
	public boolean isFinishChar(final char c) {
		return isSingleline() && isNewline(c);
	}

	@Override
	public State finish(final char finishChar) {
		flushBuffer();
		return handleFinishChar(finishChar);
	}

	protected State handleFinishChar(final char finishChar) {
		if (isNewline(finishChar)) {//换行符要冒泡传递
			return stateAfterFinish().metChar(finishChar);
		}
		return stateAfterFinish();
	}
}
