/*
 * Copyright (c) 2024 Document.Cool
 * QTMD is licensed under Mulan PSL v2.
 * You can use this software according to the terms and conditions of the Mulan PSL v2.
 * You may obtain a copy of Mulan PSL v2 at:
 *          http://license.coscl.org.cn/MulanPSL2
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PSL v2 for more details.
 */

package cool.document.qtmd;

import com.google.common.collect.Maps;
import me.magicall.贵阳DearSun.exception.NullValException;
import me.magicall.贵阳DearSun.exception.UnknownException;
import org.jsoup.nodes.Document;
import org.jsoup.nodes.Document.OutputSettings;
import org.jsoup.nodes.Element;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.regex.Pattern;
import java.util.stream.IntStream;
import java.util.stream.Stream;

public class Helper {
	public static final String[] NAMED_COLORS = {"aliceblue", "antiquewhite", "aqua", "aquamarine", "azure", "beige",
			"bisque", "black", "blanchedalmond", "blue", "blueviolet", "brown", "burlywood", "cadetblue", "chartreuse",
			"chocolate", "coral", "cornflowerblue", "cornsilk", "crimson", "cyan", "darkblue", "darkcyan", "darkgoldenrod",
			"darkgray", "darkgreen", "darkkhaki", "darkmagenta", "darkolivegreen", "darkorange", "darkorchid", "darkred",
			"darksalmon", "darkseagreen", "darkslateblue", "darkslategray", "darkturquoise", "darkviolet", "deeppink",
			"deepskyblue", "dimgray", "dodgerblue", "firebrick", "floralwhite", "forestgreen", "fuchsia", "gainsboro",
			"ghostwhite", "gold", "goldenrod", "gray", "green", "greenyellow", "honeydew", "hotpink", "indianred", "indigo",
			"ivory", "khaki", "lavender", "lavenderblush", "lawngreen", "lemonchiffon", "lightblue", "lightcoral",
			"lightcyan", "lightgoldenrodyellow", "lightgreen", "lightgrey", "lightpink", "lightsalmon", "lightseagreen",
			"lightskyblue", "lightslategray", "lightsteelblue", "lightyellow", "lime", "limegreen", "linen", "magenta",
			"maroon", "mediumaquamarine", "mediumblue", "mediumorchid", "mediumpurple", "mediumseagreen", "mediumslateblue",
			"mediumspringgreen", "mediumturquoise", "mediumvioletred", "midnightblue", "mintcream", "mistyrose", "moccasin",
			"navajowhite", "navy", "oldlace", "olive", "olivedrab", "orange", "orangered", "orchid", "palegoldenrod",
			"palegreen", "paleturquoise", "palevioletred", "papayawhip", "peachpuff", "peru", "pink", "plum", "powderblue",
			"purple", "red", "rosybrown", "royalblue", "saddlebrown", "salmon", "sandybrown", "seagreen", "seashell",
			"sienna", "silver", "skyblue", "slateblue", "slategray", "snow", "springgreen", "steelblue", "tan", "teal",
			"thistle", "tomato", "turquoise", "violet", "wheat", "white", "whitesmoke", "yellow", "yellowgreen"};

	public static String colorOf(final String name) {
		final var n = name.toLowerCase();
		return Stream.of(NAMED_COLORS).filter(e -> e.equals(n)).findFirst().orElse(null);
	}

	public static boolean isColorName(final String s) {
		return colorOf(s) != null;
	}

	public static String parseColor(final String maybeColor) {
		if (isColorName(maybeColor)) {
			return maybeColor;
		}
		if (isHex(maybeColor)) {
			final var color = switch (maybeColor.length()) {
				case 1, 2 -> maybeColor.repeat(3);
				case 3, 4, 6, 8 -> maybeColor;
				default -> null;
			};
			return color == null ? null : '#' + color;
		}
		return null;
	}

	public static Element setCssColor(final String maybeColorFlag) {
		final var index = maybeColorFlag.indexOf('#');
		if (index < 0) {
			throw new UnknownException();
		}
		final String cssName;
		if (index == 0) {
			cssName = "color";
		} else if ("bg".equals(maybeColorFlag.substring(0, index))) {
			cssName = "background-color";
		} else {
			throw new UnknownException();
		}
		final var color = parseColor(maybeColorFlag.substring(index + 1));
		if (color == null) {
			return null;
		}
		final var span = new Element("span");
		css(span, cssName, color);
		return span;
	}

	public static boolean isHexCharacter(final char c) {
		if (c >= '0' && c <= '9') {
			return true;
		}
		if (c >= 'a' && c <= 'f') {
			return true;
		}
		return c >= 'A' && c <= 'F';
	}

	public static boolean isHex(final String s) {
		return IntStream.range(0, s.length()).allMatch(i -> isHexCharacter(s.charAt(i)));
	}

	private static final List<String> TAGS_CAN_HAVE_HR = Arrays.asList(//
			"body", "div", "section", "article", "aside", "header", "footer", "nav",//
			"h1", "h2", "h3", "h4", "h5", "h6", "blockquote", "p", "li", "th", "td", "fieldset");

	public static boolean canHaveHr(final String tagName) {
		return TAGS_CAN_HAVE_HR.contains(tagName.toLowerCase());
	}

	public static final String EMAIL_PATTERN = "^[\\p{L}\\p{N}._%+-]+@[\\p{L}\\p{N}.-]+\\.\\p{L}{2,}$";

	public static boolean isEmail(final CharSequence s) {
		return Pattern.matches(EMAIL_PATTERN, s);
	}

	public static URL url(final String s) {
		try {
			return new URL(s);
		} catch (final MalformedURLException e) {
			return null;
		}
	}

	public static Element newDom(final String flagSymbol, final boolean isSingleline) {
		final var flag = flagSymbol.toLowerCase();
		final var domPattern = FLAG_SYMBOL_TO_TAG.get(flag);
		if (domPattern instanceof final String tagName) {
			if (tagName.contains(" ")) {
				final var split = tagName.split(" ");
				return new Element(split[0]).attr("style", split[1]);
			}
			return new Element(tagName);
		} else if (domPattern instanceof final Function function) {
			return toElement(function.apply(isSingleline));
		}

		throw new UnknownException();
	}

	public static Element toElement(final Object o) {
		if (o instanceof final Element element) {
			return element;
		}
		if (o instanceof final String tagName) {
			return new Element(tagName);
		}
		throw new UnknownException();
	}

	private static final Map<String, Object> FLAG_SYMBOL_TO_TAG = Maps.newHashMap();

	static {
		FLAG_SYMBOL_TO_TAG.put("!", "b");
		FLAG_SYMBOL_TO_TAG.put("b", "b");
		FLAG_SYMBOL_TO_TAG.put("bold", "b");
		FLAG_SYMBOL_TO_TAG.put("/", "i");
		FLAG_SYMBOL_TO_TAG.put("i", "i");
		FLAG_SYMBOL_TO_TAG.put("italic", "i");
		FLAG_SYMBOL_TO_TAG.put("_", "u");
		FLAG_SYMBOL_TO_TAG.put("u", "u");
		FLAG_SYMBOL_TO_TAG.put("underline", "u");
		FLAG_SYMBOL_TO_TAG.put("-", "s");
		FLAG_SYMBOL_TO_TAG.put("s", "s");
		FLAG_SYMBOL_TO_TAG.put("strike", "s");
		FLAG_SYMBOL_TO_TAG.put("^", "sup");
		FLAG_SYMBOL_TO_TAG.put("sup", "sup");
		FLAG_SYMBOL_TO_TAG.put("v", "sub");
		FLAG_SYMBOL_TO_TAG.put("sub", "sub");
		FLAG_SYMBOL_TO_TAG.put(">", "code");
		FLAG_SYMBOL_TO_TAG.put("code", "code");
		FLAG_SYMBOL_TO_TAG.put("hide", "span display:none");
		FLAG_SYMBOL_TO_TAG.put("link", "a");
		FLAG_SYMBOL_TO_TAG.put("img", "img");
		FLAG_SYMBOL_TO_TAG.put("h1", "h1");
		FLAG_SYMBOL_TO_TAG.put("h2", "h2");
		FLAG_SYMBOL_TO_TAG.put("h3", "h3");
		FLAG_SYMBOL_TO_TAG.put("h4", "h4");
		FLAG_SYMBOL_TO_TAG.put("h5", "h5");
		FLAG_SYMBOL_TO_TAG.put("h6", "h6");

		final var q = (Function<Boolean, String>) isSingleline -> isSingleline ? "q" : "blockquote";
		FLAG_SYMBOL_TO_TAG.put("'", q);
		FLAG_SYMBOL_TO_TAG.put("\"", q);
		FLAG_SYMBOL_TO_TAG.put("q", q);

		FLAG_SYMBOL_TO_TAG.put("#", "ol");
		FLAG_SYMBOL_TO_TAG.put(".", "ul");
		FLAG_SYMBOL_TO_TAG.put("[", "table");

		FLAG_SYMBOL_TO_TAG.put("link", "a");
		FLAG_SYMBOL_TO_TAG.put("img", "img");
	}

	public static boolean maybeFlag(final String flag) {
		final var lowerCase = flag.toLowerCase();
		return FLAG_SYMBOL_TO_TAG.keySet().stream().anyMatch(s -> s.startsWith(lowerCase));
	}

	public static boolean isFlagReady(final String flag) {
		final var lowerCase = flag.toLowerCase();
		return FLAG_SYMBOL_TO_TAG.keySet().stream().anyMatch(s -> s.equals(lowerCase));
	}

	public static void flushToDom(final StringBuilder sb, final Element dom) {
		if (!sb.isEmpty()) {
			dom.appendText(clear(sb));
		}
	}

	public static String clear(final StringBuilder sb) {
		final var rt = sb.toString();
		sb.setLength(0);
		return rt;
	}

	public static ContentState checkStateAfterFinish(final ContentState stateAfterFinish) {
		if (stateAfterFinish == null) {
			throw new NullValException("stateAfterFinish");
		}
		return stateAfterFinish;
	}

	public static void attr(final Element dom, final String attrName, final Object val) {
		dom.attr(attrName, String.valueOf(val));
	}

	public static void css(final Element dom, final String cssName, final Object val) {
		attr(dom, "style", cssName + ':' + val);
	}

	/**
	 * 计算一个字符串中“最后一行（即最后一个换行符之后的部分）”的缩进。
	 * 计算缩进的算法：检查字符串开头的tab和空格，一个tab算一个缩进，[0,2)个空格算0个缩进，[2,6)个空格算1个缩进，[6,10)个空格算2个缩进，以此类推（空格代表的缩进=(空格数+2/4)）。
	 *
	 * @return
	 */
	public static int calIndent(final CharSequence s) {
		final var withoutActivateChar = s.charAt(s.length() - 1) == '\\' ? s.subSequence(0, s.length() - 1) : s;
		final var lastNewlineIndex = lastIndexOfNewline(withoutActivateChar);
		final var lastLine = withoutActivateChar.subSequence(lastNewlineIndex, withoutActivateChar.length());
		final var len = lastLine.length();
		int tabCount = 0;
		int spaceCount = 0;
		for (int i = len - 1; i >= 0; i--) {
			final var c = lastLine.charAt(i);
			if (c == '\t') {
				tabCount += 1;
			} else if (c == ' ') {
				spaceCount += 1;
			} else {
				return 0;
			}
		}
		return tabCount + (spaceCount + 2) / 4;
	}

	public static int lastIndexOfNewline(final CharSequence s) {
		for (int i = s.length() - 1; i >= 0; i--) {
			final var c = s.charAt(i);
			if (isNewline(c)) {
				return i + 1;
			}
		}
		return 0;
	}

	static boolean isNewline(final char c) {
		return c == '\n' || c == '\r';
	}

	static void keepFormatting(final Document doc) {
		final OutputSettings outputSettings = new OutputSettings();
		outputSettings.prettyPrint(false); // 禁用格式化输出。jsoup默认会合并html里连续的空白字符。
		doc.outputSettings(outputSettings);
	}
}
