/*
 * Copyright (c) 2024 Document.Cool
 * QTMD is licensed under Mulan PSL v2.
 * You can use this software according to the terms and conditions of the Mulan PSL v2.
 * You may obtain a copy of Mulan PSL v2 at:
 *          http://license.coscl.org.cn/MulanPSL2
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PSL v2 for more details.
 */

package cool.document.qtmd;

/**
 * 中间状态，是一个处理文本的状态到下一个处理文本的状态中间的临时状态。
 * 可能回滚到前置状态：表示本状态内所遇字符作为普通文本刷回前置状态的所遇字符中。
 * 不应当产生dom。
 * 自然结束逻辑上就是回滚。
 */
public interface MiddleState extends State {
	/**
	 * 遇见激活字符的处理方法。通常是遇到激活字符（{@link #isActivateChar(char)}）时调用的方法。默认创建并返回一个“激活状态（{@link ActivateState}）”。
	 *
	 * @return
	 */
	@Override
	default State metActivateChar(final char activateChar) {
		return finish(activateChar);
	}

	/**
	 * 中间状态的finish其实是回滚：把buffer刷到前置状态的buffer里，然后由前置状态处理结束符。
	 *
	 * @return
	 */
	@Override
	default State finish(final char finishChar) {
		flushBuffer();
		if (willConsumeFinishChar(finishChar)) {
			return stateAfterFinish();
		}
		return stateAfterFinish().metChar(finishChar);
	}

	/**
	 * 与父类不同：中间态把buffer直接刷到结束后状态的buffer里。
	 *
	 * @return
	 */
	@Override
	default State flushBuffer() {
		stateAfterFinish().addChars(buffer());
		return State.super.flushBuffer();
	}

	/**
	 * 当遇上指定的回滚符号时，是否会消费它（即不向回滚后的状态传递）。默认皆否。
	 *
	 * @return
	 */
	default boolean willConsumeFinishChar(final char finishChar) {
		return false;
	}
}
