/*
 * Copyright (c) 2024 Document.Cool
 * QTMD is licensed under Mulan PSL v2.
 * You can use this software according to the terms and conditions of the Mulan PSL v2.
 * You may obtain a copy of Mulan PSL v2 at:
 *          http://license.coscl.org.cn/MulanPSL2
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PSL v2 for more details.
 */

package cool.document.qtmd;

public interface State {
	/**
	 * 结束后返回的状态。
	 *
	 * @return
	 */
	ContentState stateAfterFinish();

	/**
	 * 在本状态内遇到的字符的缓冲。
	 * 注意：有些状态会决定丢弃一些字符。具体取决于该状态本身。
	 *
	 * @return
	 */
	StringBuilder buffer();

	/**
	 * 遇到字符序列。
	 * 默认为依次调用{@link #metChar(char)}，算是它的简便方法。
	 *
	 * @return
	 */
	default State metChars(final CharSequence s) {
		State curState = this;
		for (int i = 0; i < s.length(); i++) {
			final var c = s.charAt(i);
			curState = curState.metChar(c);
		}
		return curState;
	}

	/**
	 * 遇到单个字符，返回结果状态。
	 * 默认处理：
	 * 1，若是激活字符（{@link #isActivateChar(char)}），则激活（{@link #metActivateChar(char)}）。
	 * 2，若是换行符（{@link #isNewline(char)}），则换行（{@link #metNewline(char)}）。
	 * 3，若是结束字符（{@link #isFinishChar(char)}），则结束（{@link #finish(char)}）。
	 * 4，否则视为其他字符来处理（{@link #metOtherChar(char)}）。
	 *
	 * @return
	 */
	default State metChar(final char c) {
		if (isActivateChar(c)) {
			return metActivateChar(c);
		}
		if (isNewline(c)) {
			return metNewline(c);
		}
		if (isFinishChar(c)) {
			return finish(c);
		}
		return metOtherChar(c);
	}

	/**
	 * 在当前上下文中，判断指定字符对本状态是否激活字符。默认只有\是。
	 *
	 * @return
	 */
	default boolean isActivateChar(final char c) {
		return c == '\\';
	}

	/**
	 * 遇见激活字符的处理方法。通常是遇到激活字符（{@link #isActivateChar(char)}）时调用的方法。默认创建并返回一个“激活状态（{@link ActivateState}）”。
	 *
	 * @return
	 */
	default State metActivateChar(final char activateChar) {
		return addChar(activateChar);
	}

	/**
	 * 在当前上下文中，判断指定字符对本状态是否激活字符。默认只有\n、\r是。
	 *
	 * @return
	 */
	default boolean isNewline(final char c) {
		return Helper.isNewline(c);
	}

	/**
	 * 遇见换行符的处理方法。通常是遇到换行符（{@link #isNewline(char)}）时调用的方法。默认加到缓冲中。
	 *
	 * @return
	 */
	default State metNewline(final char newlineChar) {
		return addChar(newlineChar);
	}

	/**
	 * 在当前上下文中，判断指定字符是否本状态的结束字符。默认皆否。
	 *
	 * @return
	 */
	default boolean isFinishChar(final char c) {
		return false;
	}

	/**
	 * 结束。通常是遇到结束字符时调用的方法。
	 * 默认处理：
	 * 1，刷本状态所遇内容字符（{@link #flushBuffer()}）。
	 * 2，返回前置状态（{@link #stateAfterFinish()}）。
	 *
	 * @return
	 */
	default State finish(final char finishChar) {
		flushBuffer();
		return stateAfterFinish();
	}

	/**
	 * 遇见其他字符（非激活字符（{@link #isActivateChar(char)}}）、换行符（{@link #isNewline(char)}}）、结束字符（{@link #isFinishChar(char)}}））的处理方法。
	 *
	 * @return
	 */
	default State metOtherChar(final char c) {
		return addChar(c);
	}

	/**
	 * 增加字符到缓冲区。
	 * 便利工具方法，通常不需要重写。
	 *
	 * @return
	 */
	default State addChar(final char c) {
		buffer().append(c);
		return this;
	}

	/**
	 * 增加字符序列到缓冲区。
	 * 便利工具方法，通常不需要重写。
	 *
	 * @return
	 */
	default State addChars(final CharSequence s) {
		buffer().append(s);
		return this;
	}

	/**
	 * 终结。通常是不再有更多字符需要处理时，本状态的退出/结束/终结机制。
	 * 默认先刷缓冲区字符（{@link #flushBuffer()}），再结束前置状态。
	 */
	default void end() {
		flushBuffer();
		stateAfterFinish().end();
	}

	/**
	 * 刷缓冲区。默认实现只是清掉缓冲区。
	 *
	 * @return
	 */
	default State flushBuffer() {
		clearBuffer();
		return this;
	}

	/**
	 * 清除缓冲区。
	 * 便利工具方法，通常不需要重写。
	 */
	default String clearBuffer() {
		return Helper.clear(buffer());
	}
}
