package cool.doudou.doudada.mq.core.helper;

import cool.doudou.doudada.mq.core.factory.ProducerMapFactory;
import lombok.extern.slf4j.Slf4j;
import org.apache.pulsar.client.api.MessageId;
import org.apache.pulsar.client.api.Producer;
import org.apache.pulsar.client.api.PulsarClientException;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;

/**
 * MqHelper
 *
 * @author jiangcs
 * @since 2022/2/19
 */
@Slf4j
public class MqHelper {
    /**
     * 发送消息
     *
     * @param topic 主题
     * @param msg   消息
     * @return MessageId
     */
    public String send(String topic, String msg) {
        try {
            Producer<String> producer = ProducerMapFactory.get(topic);
            if (producer != null) {
                return producer
                        .send(msg)
                        .toString();
            }
        } catch (PulsarClientException e) {
            log.error("send exception: ", e);
        }
        return null;
    }

    /**
     * 发送消息
     *
     * @param topic 主题
     * @param key   关键字
     * @param msg   消息
     * @return MessageId
     */
    public String send(String topic, String key, String msg) {
        try {
            Producer<String> producer = ProducerMapFactory.get(topic);
            if (producer != null) {
                return producer.newMessage()
                        .key(key)
                        .value(msg)
                        .send()
                        .toString();
            }
        } catch (PulsarClientException e) {
            log.error("send exception: ", e);
        }
        return null;
    }

    /**
     * 发送消息
     *
     * @param topic 主题
     * @param msg   消息
     * @param delay 延迟时间，单位 毫秒
     * @return MessageId
     */
    public String send(String topic, String msg, long delay) {
        try {
            Producer<String> producer = ProducerMapFactory.get(topic);
            if (producer != null) {
                return producer.newMessage()
                        .value(msg)
                        .deliverAfter(delay, TimeUnit.MILLISECONDS)
                        .send()
                        .toString();
            }
        } catch (PulsarClientException e) {
            log.error("send exception: ", e);
        }
        return null;
    }

    /**
     * 发送消息
     *
     * @param topic 主题
     * @param key   关键字
     * @param msg   消息
     * @param delay 延迟时间，单位 毫秒
     * @return MessageId
     */
    public String send(String topic, String key, String msg, long delay) {
        try {
            Producer<String> producer = ProducerMapFactory.get(topic);
            if (producer != null) {
                return producer.newMessage()
                        .key(key)
                        .value(msg)
                        .deliverAfter(delay, TimeUnit.MILLISECONDS)
                        .send()
                        .toString();
            }
        } catch (PulsarClientException e) {
            log.error("send exception: ", e);
        }
        return null;
    }

    /**
     * 异步发送消息
     *
     * @param topic  主题
     * @param msg    消息
     * @param action 回执动作，参数为MessageId
     */
    public void sendAsync(String topic, String msg, Consumer<String> action) {
        Producer<String> producer = ProducerMapFactory.get(topic);
        if (producer != null) {
            CompletableFuture<MessageId> completableFuture = producer.sendAsync(msg)
                    .exceptionally((e) -> {
                        log.error("sendAsync exception: ", e);
                        return null;
                    });
            if (action != null) {
                completableFuture.thenAccept((messageId -> action.accept(messageId.toString())));
            }
        }
    }

    /**
     * 异步发送消息
     *
     * @param topic  主题
     * @param key    关键字
     * @param msg    消息
     * @param action 回执动作，参数为MessageId
     */
    public void sendAsync(String topic, String key, String msg, Consumer<String> action) {
        Producer<String> producer = ProducerMapFactory.get(topic);
        if (producer != null) {
            CompletableFuture<MessageId> completableFuture = producer.newMessage()
                    .key(key)
                    .value(msg)
                    .sendAsync()
                    .exceptionally((e) -> {
                        log.error("sendAsync exception: ", e);
                        return null;
                    });
            if (action != null) {
                completableFuture.thenAccept((messageId -> action.accept(messageId.toString())));
            }
        }
    }

    /**
     * 异步发送消息
     *
     * @param topic  主题
     * @param msg    消息
     * @param delay  延迟时间，单位 毫秒
     * @param action 回执动作，参数为MessageId
     */
    public void sendAsync(String topic, String msg, long delay, Consumer<String> action) {
        Producer<String> producer = ProducerMapFactory.get(topic);
        if (producer != null) {
            CompletableFuture<MessageId> completableFuture = producer.newMessage()
                    .value(msg)
                    .deliverAfter(delay, TimeUnit.MILLISECONDS)
                    .sendAsync()
                    .exceptionally((e) -> {
                        log.error("sendAsync exception: ", e);
                        return null;
                    });
            if (action != null) {
                completableFuture.thenAccept((messageId -> action.accept(messageId.toString())));
            }
        }
    }

    /**
     * 异步发送消息
     *
     * @param topic  主题
     * @param key    关键字
     * @param msg    消息
     * @param delay  延迟时间，单位 毫秒
     * @param action 回执动作，参数为MessageId
     */
    public void sendAsync(String topic, String key, String msg, long delay, Consumer<String> action) {
        Producer<String> producer = ProducerMapFactory.get(topic);
        if (producer != null) {
            CompletableFuture<MessageId> completableFuture = producer.newMessage()
                    .key(key)
                    .value(msg)
                    .deliverAfter(delay, TimeUnit.MILLISECONDS)
                    .sendAsync()
                    .exceptionally((e) -> {
                        log.error("sendAsync exception: ", e);
                        return null;
                    });
            if (action != null) {
                completableFuture.thenAccept((messageId -> action.accept(messageId.toString())));
            }
        }
    }
}
