package cool.doudou.doudada.pay.core.api;

import com.alibaba.fastjson2.JSONObject;
import cool.doudou.doudada.pay.core.entity.PlaceOrderParam;
import cool.doudou.doudada.pay.core.entity.RefundParam;
import cool.doudou.doudada.pay.core.enums.ZfbTrade;
import cool.doudou.doudada.pay.core.util.HttpUtil;
import cool.doudou.doudada.pay.properties.ZfbPayProperties;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.ObjectUtils;

import java.util.HashMap;
import java.util.Map;

/**
 * ZfbPayApi
 *
 * @author jiangcs
 * @since 2022/06/23
 */
public class ZfbPayApi {
    private ZfbPayProperties zfbPayProperties;

    /**
     * 下单
     *
     * @param placeOrderParam 下单参数
     * @return 前端调用支付参数
     */
    public Map<String, String> place(PlaceOrderParam placeOrderParam) {
        if (ObjectUtils.isEmpty(placeOrderParam.getOutTradeNo())) {
            throw new RuntimeException("商户订单号不能为空");
        }
        if (placeOrderParam.getOutTradeNo().length() > 64) {
            throw new RuntimeException("商户订单号长度不能超多64位");
        }
        if (!ObjectUtils.isEmpty(placeOrderParam.getTimeExpire()) && placeOrderParam.getTimeExpire().length() != 19) {
            throw new RuntimeException("交易超时时间格式错误");
        }

        // 业务参数
        Map<String, String> params = new HashMap<>(1);
        JSONObject apiParamObj = new JSONObject();
        apiParamObj.put("out_trade_no", placeOrderParam.getOutTradeNo());
        apiParamObj.put("total_amount", placeOrderParam.getMoney());
        apiParamObj.put("subject", placeOrderParam.getDescription());
        if (!ObjectUtils.isEmpty(placeOrderParam.getUid())) {
            apiParamObj.put("buyer_id", placeOrderParam.getUid());
        }
        if (!ObjectUtils.isEmpty(placeOrderParam.getAttach())) {
            apiParamObj.put("body", placeOrderParam.getAttach());
        }
        if (!ObjectUtils.isEmpty(placeOrderParam.getTimeExpire())) {
            apiParamObj.put("time_expire", placeOrderParam.getTimeExpire());
        }
        params.put("biz_content", apiParamObj.toJSONString());

        String result = HttpUtil.doPost4Zfb(zfbPayProperties.getServerAddress(), ZfbTrade.TRADE_CREATE.method(), params, zfbPayProperties.getAppId(), zfbPayProperties.getNotifyUrl());
        if (ObjectUtils.isEmpty(result)) {
            throw new RuntimeException("下单失败");
        }
        JSONObject resultObj = JSONObject.parseObject(result, JSONObject.class);

        Map<String, String> map = new HashMap<>(1);
        map.put("outTradeNo", resultObj.getString("out_trade_no"));
        return map;
    }

    /**
     * 查询
     *
     * @param outTradeNo 商户订单号
     * @return 订单信息
     */
    public String query(String outTradeNo) {
        if (ObjectUtils.isEmpty(outTradeNo)) {
            throw new RuntimeException("商户订单号不能为空");
        }

        // 业务参数
        Map<String, String> params = new HashMap<>(1);
        JSONObject apiParamObj = new JSONObject();
        apiParamObj.put("out_trade_no", outTradeNo);
        params.put("biz_content", apiParamObj.toJSONString());

        return HttpUtil.doPost4Zfb(zfbPayProperties.getServerAddress(), ZfbTrade.TRADE_QUERY.method(), params, zfbPayProperties.getAppId(), zfbPayProperties.getNotifyUrl());
    }

    /**
     * 关闭
     *
     * @param outTradeNo 商户订单号
     * @return 关闭结果
     */
    public String close(String outTradeNo) {
        if (ObjectUtils.isEmpty(outTradeNo)) {
            throw new RuntimeException("商户订单号不能为空");
        }

        // 业务参数
        Map<String, String> params = new HashMap<>(1);
        JSONObject apiParamObj = new JSONObject();
        apiParamObj.put("out_trade_no", outTradeNo);
        params.put("biz_content", apiParamObj.toJSONString());

        return HttpUtil.doPost4Zfb(zfbPayProperties.getServerAddress(), ZfbTrade.TRADE_CLOSE.method(), params, zfbPayProperties.getAppId(), zfbPayProperties.getNotifyUrl());
    }

    /**
     * 退款
     *
     * @param refundParam 退款参数
     * @return 退款信息
     */
    public String refund(RefundParam refundParam) {
        if (ObjectUtils.isEmpty(refundParam.getOutTradeNo())) {
            throw new RuntimeException("商户订单号不能为空");
        }

        // 业务参数
        Map<String, String> params = new HashMap<>(1);
        JSONObject apiParamObj = new JSONObject();
        apiParamObj.put("out_trade_no", refundParam.getOutTradeNo());
        apiParamObj.put("refund_amount", refundParam.getRefundMoney());
        if (!ObjectUtils.isEmpty(refundParam.getReason())) {
            apiParamObj.put("refund_reason", refundParam.getReason());
        }
        if (!ObjectUtils.isEmpty(refundParam.getOutRefundNo())) {
            apiParamObj.put("out_request_no", refundParam.getOutRefundNo());
        }
        params.put("biz_content", apiParamObj.toJSONString());

        return HttpUtil.doPost4Zfb(zfbPayProperties.getServerAddress(), ZfbTrade.TRADE_REFUND.method(), params, zfbPayProperties.getAppId(), zfbPayProperties.getNotifyUrl());
    }

    /**
     * 交易账单
     *
     * @param billDate 账单日期（日账单格式为yyyy-MM-dd；月账单格式为yyyy-MM）
     * @return 账单下载地址
     */
    public String tradeBill(String billDate) {
        if (ObjectUtils.isEmpty(billDate)) {
            throw new RuntimeException("账单日期不能为空");
        }

        Map<String, String> params = new HashMap<>(1);
        JSONObject apiParamObj = new JSONObject();
        apiParamObj.put("bill_type", "trade");
        apiParamObj.put("bill_date", billDate);
        params.put("biz_content", apiParamObj.toJSONString());

        return HttpUtil.doPost4Zfb(zfbPayProperties.getServerAddress(), ZfbTrade.TRADE_BILL_QUERY.method(), params, zfbPayProperties.getAppId(), zfbPayProperties.getNotifyUrl());
    }

    @Autowired
    public void setZfbPayProperties(ZfbPayProperties zfbPayProperties) {
        this.zfbPayProperties = zfbPayProperties;
    }
}
