package cool.doudou.doudada.pay.core.helper;

import cool.doudou.doudada.pay.core.api.WxPayApi;
import cool.doudou.doudada.pay.core.api.ZfbPayApi;
import cool.doudou.doudada.pay.core.entity.PlaceOrderParam;
import cool.doudou.doudada.pay.core.entity.RefundParam;
import cool.doudou.doudada.pay.core.enums.PayMode;
import org.springframework.beans.factory.annotation.Autowired;

import java.io.ByteArrayInputStream;
import java.util.Map;

/**
 * PayHelper
 *
 * @author jiangcs
 * @since 2022/2/19
 */
public class PayHelper {
    private WxPayApi wxPayApi;
    private ZfbPayApi zfbPayApi;

    /**
     * 下单
     *
     * @param PayMode         支付方式
     * @param placeOrderParam 订单参数
     * @return 前端调用支付参数
     * @throws Exception 业务异常
     */
    public Map<String, String> place(PayMode PayMode, PlaceOrderParam placeOrderParam) throws Exception {
        switch (PayMode) {
            case WX:
                return wxPayApi.place(placeOrderParam);
            case ZFB:
                return zfbPayApi.place(placeOrderParam);
            default:
                throw new RuntimeException("支付模式匹配失败");
        }
    }

    /**
     * 查询
     *
     * @param PayMode    支付方式
     * @param outTradeNo 商户订单号
     * @return 订单信息
     * @throws Exception 业务异常
     */
    public String query(PayMode PayMode, String outTradeNo) throws Exception {
        switch (PayMode) {
            case WX:
                return wxPayApi.query(outTradeNo);
            case ZFB:
                return zfbPayApi.query(outTradeNo);
            default:
                throw new RuntimeException("支付模式匹配失败");
        }
    }

    /**
     * 关闭
     *
     * @param PayMode    支付方式
     * @param outTradeNo 商户订单号
     * @return 关闭结果
     * @throws Exception 业务异常
     */
    public String close(PayMode PayMode, String outTradeNo) throws Exception {
        switch (PayMode) {
            case WX:
                return wxPayApi.close(outTradeNo);
            case ZFB:
                return zfbPayApi.close(outTradeNo);
            default:
                throw new RuntimeException("支付模式匹配失败");
        }
    }

    /**
     * 退款
     *
     * @param PayMode     支付方式
     * @param refundParam 退款参数
     * @return 退款信息
     * @throws Exception 业务异常
     */
    public String refund(PayMode PayMode, RefundParam refundParam) throws Exception {
        switch (PayMode) {
            case WX:
                return wxPayApi.refund(refundParam);
            case ZFB:
                return zfbPayApi.refund(refundParam);
            default:
                throw new RuntimeException("支付模式匹配失败");
        }
    }

    /**
     * 交易账单
     *
     * @param PayMode  支付方式
     * @param billDate 账单日期
     * @return 账单下载地址
     * @throws Exception 业务异常
     */
    public String tradeBill(PayMode PayMode, String billDate) throws Exception {
        switch (PayMode) {
            case WX:
                return wxPayApi.tradeBill(billDate);
            case ZFB:
                return zfbPayApi.tradeBill(billDate);
            default:
                throw new RuntimeException("支付模式匹配失败");
        }
    }

    /**
     * 下载账单
     *
     * @param PayMode 支付方式
     * @param billUrl 账单地址
     * @return 账单信息字节数组流
     * @throws Exception 业务异常
     */
    public ByteArrayInputStream downloadBill(PayMode PayMode, String billUrl) throws Exception {
        switch (PayMode) {
            case WX:
                return wxPayApi.downloadBill(billUrl);
            default:
                throw new RuntimeException("支付模式匹配失败");
        }
    }

    @Autowired
    public void setWxPayApi(WxPayApi wxPayApi) {
        this.wxPayApi = wxPayApi;
    }

    @Autowired
    public void setZfbPayApi(ZfbPayApi zfbPayApi) {
        this.zfbPayApi = zfbPayApi;
    }
}
