package cool.doudou.doudada.pay.core.util;

import cool.doudou.doudada.pay.core.factory.OkHttpFactory;
import cool.doudou.doudada.pay.core.signer.WxSigner;
import cool.doudou.doudada.pay.core.signer.ZfbSigner;
import lombok.extern.slf4j.Slf4j;
import okhttp3.*;
import org.springframework.http.HttpStatus;
import org.springframework.util.ObjectUtils;

import java.io.ByteArrayInputStream;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

/**
 * HttpUtil
 *
 * @author jiangcs
 * @since 2022/06/30
 */
@Slf4j
public class HttpUtil {
    private static final OkHttpClient okHttpClient;

    static {
        okHttpClient = OkHttpFactory.get();
    }

    /**
     * get 请求 微信
     *
     * @param serverAddress          服务器地址
     * @param reqAbsoluteUrl         请求API地址
     * @param params                 Query参数
     * @param mchId                  商户ID
     * @param privateKeySerialNumber 商户密钥证书序列号
     * @return 结果
     */
    public static String doGet4Wx(String serverAddress, String reqAbsoluteUrl, Map<String, Object> params, String mchId, String privateKeySerialNumber) {
        String url = serverAddress + reqAbsoluteUrl;

        Request.Builder builder = new Request.Builder();
        // Header
        builder.addHeader("Content-Type", "application/json");
        builder.addHeader("Accept", "application/json");
        builder.addHeader("Authorization", WxSigner.getAuthorization(mchId, privateKeySerialNumber, "GET", reqAbsoluteUrl, params, ""));
        // Query
        StringBuilder sbUrlParam = new StringBuilder();
        if (params != null && params.keySet().size() > 0) {
            for (String key : params.keySet()) {
                if (sbUrlParam.length() <= 0) {
                    sbUrlParam.append("?");
                } else {
                    sbUrlParam.append("&");
                }
                sbUrlParam.append(key).append("=").append(params.get(key));
            }
        }
        url += sbUrlParam;
        Request request = builder.url(url).build();

        log.info("[wx] url => GET {}", url);
        log.info("[wx] authorization => {}", request.header("Authorization"));
        log.info("[wx] params => {}", params);

        return execute(request);
    }

    /**
     * getInputStream 请求 微信
     *
     * @param serverAddress          服务器地址
     * @param reqAbsoluteUrl         请求API地址
     * @param params                 Query参数
     * @param mchId                  商户ID
     * @param privateKeySerialNumber 商户密钥证书序列号
     * @return 字节数组输入流
     */
    public static ByteArrayInputStream doGetInputStream4Wx(String serverAddress, String reqAbsoluteUrl, Map<String, Object> params, String mchId, String privateKeySerialNumber) {
        String url = serverAddress + reqAbsoluteUrl;

        Request.Builder builder = new Request.Builder();
        // Header
        builder.addHeader("Content-Type", "application/json");
        builder.addHeader("Accept", "application/json");
        builder.addHeader("Authorization", WxSigner.getAuthorization(mchId, privateKeySerialNumber, "GET", reqAbsoluteUrl, params, ""));
        // Query
        StringBuilder sbUrlParam = new StringBuilder();
        if (params != null && params.keySet().size() > 0) {
            for (String key : params.keySet()) {
                if (sbUrlParam.length() <= 0) {
                    sbUrlParam.append("?");
                } else {
                    sbUrlParam.append("&");
                }
                sbUrlParam.append(key).append("=").append(params.get(key));
            }
        }
        url += sbUrlParam;
        Request request = builder.url(url).build();

        log.info("[wx] url => GET {}", url);
        log.info("[wx] authorization => {}", request.header("Authorization"));
        log.info("[wx] params => {}", params);

        return executeInputStream(request);
    }

    /**
     * Post 请求 微信
     *
     * @param serverAddress           服务器地址
     * @param reqAbsoluteUrl          请求API地址
     * @param jsonBody                Body参数
     * @param mchId                   商户ID
     * @param certificateSerialNumber 商户证书序列号
     * @return 结果
     */
    public static boolean doPostNoContent4Wx(String serverAddress, String reqAbsoluteUrl, String jsonBody, String mchId, String certificateSerialNumber) {
        String url = serverAddress + reqAbsoluteUrl;

        Request.Builder builder = new Request.Builder();
        // Header
        builder.addHeader("Content-Type", "application/json");
        builder.addHeader("Accept", "application/json");
        builder.addHeader("Authorization", WxSigner.getAuthorization(mchId, certificateSerialNumber, "POST", reqAbsoluteUrl, null, jsonBody));
        // Body
        if (jsonBody == null) {
            jsonBody = "";
        }
        RequestBody requestBody = RequestBody.create(MediaType.parse("application/json; charset=utf-8"), jsonBody);
        Request request = builder.url(url).post(requestBody).build();

        log.info("[wx] url => POST {}", url);
        log.info("[wx] authorization => {}", request.header("Authorization"));
        log.info("[wx] body => {}", jsonBody);

        return executeNoContent(request);
    }

    /**
     * Post 请求 微信
     *
     * @param serverAddress           服务器地址
     * @param reqAbsoluteUrl          请求API地址
     * @param jsonBody                Body参数
     * @param mchId                   商户ID
     * @param certificateSerialNumber 商户证书序列号
     * @return 结果
     */
    public static String doPost4Wx(String serverAddress, String reqAbsoluteUrl, String jsonBody, String mchId, String certificateSerialNumber) {
        String url = serverAddress + reqAbsoluteUrl;

        Request.Builder builder = new Request.Builder();
        // Header
        builder.addHeader("Content-Type", "application/json");
        builder.addHeader("Accept", "application/json");
        builder.addHeader("Authorization", WxSigner.getAuthorization(mchId, certificateSerialNumber, "POST", reqAbsoluteUrl, null, jsonBody));
        // Body
        if (jsonBody == null) {
            jsonBody = "";
        }
        RequestBody requestBody = RequestBody.create(jsonBody, MediaType.parse("application/json; charset=utf-8"));
        Request request = builder.url(url).post(requestBody).build();

        log.info("[wx] url => POST {}", url);
        log.info("[wx] authorization => {}", request.header("Authorization"));
        log.info("[wx] body => {}", jsonBody);

        return execute(request);
    }

    /**
     * Post 请求 支付宝
     *
     * @param url       网关地址
     * @param method    接口名称
     * @param params    业务参数
     * @param appId     应用ID
     * @param notifyUrl 异步通知地址
     * @return 结果
     */
    public static String doPost4Zfb(String url, String method, Map<String, String> params, String appId, String notifyUrl) {
        // 公共参数
        Map<String, String> publicParamMap = new HashMap<>(8);
        publicParamMap.put("app_id", appId);
        publicParamMap.put("method", method);
        publicParamMap.put("charset", "utf-8");
        publicParamMap.put("sign_type", "RSA2");
        publicParamMap.put("timestamp", DateUtil.formatDate(new Date()));
        publicParamMap.put("version", "1.0");
        publicParamMap.put("notify_url", notifyUrl);
        // 签名
        publicParamMap.put("sign", ZfbSigner.get(publicParamMap, params));

        Request.Builder builder = new Request.Builder();
        // Query
        StringBuilder sbPublicParam = new StringBuilder();
        for (String key : publicParamMap.keySet()) {
            if (sbPublicParam.length() <= 0) {
                sbPublicParam.append("?");
            } else {
                sbPublicParam.append("&");
            }
            sbPublicParam.append(key).append("=").append(URLEncoder.encode(publicParamMap.get(key), StandardCharsets.UTF_8));
        }
        url += sbPublicParam;
        // Body
        FormBody.Builder formBuilder = new FormBody.Builder();
        if (params != null && params.keySet().size() > 0) {
            for (String key : params.keySet()) {
                formBuilder.add(key, params.get(key));
            }
        }
        RequestBody requestBody = formBuilder.build();
        Request request = builder.url(url).post(requestBody).build();

        log.info("[zfb] url => POST {}", url);
        log.info("[zfb] method => {}", method);
        log.info("[zfb] params => {}", params);

        return execute(request);
    }

    private static String execute(Request request) {
        try (Response response = okHttpClient.newCall(request).execute()) {
            if (response.isSuccessful()) {
                ResponseBody responseBody = response.body();
                if (!ObjectUtils.isEmpty(responseBody)) {
                    String result = responseBody.string();
                    log.info("execute => success: {}", result);
                    return result;
                }
            }
            log.error("execute => fail: {}", response);
        } catch (Exception e) {
            log.error("execute => exception: ", e);
        }
        return null;
    }

    private static boolean executeNoContent(Request request) {
        try (Response response = okHttpClient.newCall(request).execute()) {
            if (response.isSuccessful() && response.code() == HttpStatus.NO_CONTENT.value()) {
                return true;
            }
            log.error("execute => fail: {}", response);
        } catch (Exception e) {
            log.error("execute => exception: ", e);
        }
        return false;
    }

    private static ByteArrayInputStream executeInputStream(Request request) {
        try (Response response = okHttpClient.newCall(request).execute()) {
            if (response.isSuccessful()) {
                ResponseBody responseBody = response.body();
                if (!ObjectUtils.isEmpty(responseBody)) {
                    log.info("execute => success: {}[{}]", responseBody.contentType(), responseBody.contentLength());
                    return new ByteArrayInputStream(responseBody.bytes());
                }
            }
            log.error("execute => fail: {}", response);
        } catch (Exception e) {
            log.error("execute => exception: ", e);
        }
        return null;
    }
}
