/*
 * Copyright 2024 Craig Motlin
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package cool.klass.generator.reladomo.mergehook;

import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.PrintStream;
import java.nio.file.Path;
import java.util.Objects;

import javax.annotation.Nonnull;

import com.google.common.base.CaseFormat;
import com.google.common.base.Converter;
import cool.klass.model.meta.domain.api.DomainModel;
import cool.klass.model.meta.domain.api.Klass;
import cool.klass.model.meta.domain.api.PackageableElement;
import cool.klass.model.meta.domain.api.PrimitiveType;
import cool.klass.model.meta.domain.api.property.DataTypeProperty;

// TODO: Consider moving this into its own module.
public class ReladomoMergeHookGenerator
{
    private static final Converter<String, String> LOWER_CAMEL_TO_UPPER_CAMEL =
            CaseFormat.LOWER_CAMEL.converterTo(CaseFormat.UPPER_CAMEL);

    @Nonnull
    private final DomainModel domainModel;

    public ReladomoMergeHookGenerator(@Nonnull DomainModel domainModel)
    {
        this.domainModel = Objects.requireNonNull(domainModel);
    }

    public void writeMergeHookFiles(@Nonnull Path outputPath)
    {
        this.domainModel
                .getClasses()
                .reject(Klass::isAbstract)
                .select(Klass::isVersioned)
                .forEachWith(this::writeMergeHookFile, outputPath);
    }

    private void writeMergeHookFile(@Nonnull Klass klass, @Nonnull Path outputPath)
    {
        Path   mergeHookOutputPath = this.getMergeHookOutputPath(outputPath, klass);
        String classSourceCode     = this.getMergeHookSourceCode(klass);
        this.printStringToFile(mergeHookOutputPath, classSourceCode);
    }

    @Nonnull
    private Path getMergeHookOutputPath(
            @Nonnull Path outputPath,
            @Nonnull PackageableElement packageableElement)
    {
        String packageRelativePath = packageableElement.getPackageName()
                .replaceAll("\\.", "/");
        Path mergeHookDirectory = outputPath
                .resolve(packageRelativePath)
                .resolve("reladomo")
                .resolve("merge")
                .resolve("hook");
        mergeHookDirectory.toFile().mkdirs();
        String fileName = packageableElement.getName() + "MergeHook.java";
        return mergeHookDirectory.resolve(fileName);
    }

    private void printStringToFile(@Nonnull Path path, String contents)
    {
        try (PrintStream printStream = new PrintStream(new FileOutputStream(path.toFile())))
        {
            printStream.print(contents);
        }
        catch (FileNotFoundException e)
        {
            throw new RuntimeException(e);
        }
    }

    @Nonnull
    private String getMergeHookSourceCode(@Nonnull Klass klass)
    {
        Klass versionClass = klass.getVersionProperty().get().getType();

        String setKeyPropertiesSourceCode = klass
                .getKeyProperties()
                .collect(this::getKeyPropertySourceCode)
                .makeString("");

        String setAuditPropertiesOnCreateSourceCode = klass.isAudited()
                ? """
                        version.setCreatedOn(newObject.getCreatedOn());
                        version.setCreatedById(newObject.getCreatedById());
                        version.setLastUpdatedById(newObject.getLastUpdatedById());
                """
                : "";

        String setAuditPropertiesOnUpdateSourceCode = klass.isAudited()
                ? "        existingVersion.setLastUpdatedById(incoming.getLastUpdatedById());\n"
                : "";

        // @formatter:off
        // language=JAVA
        String sourceCode = ""
                + "package " + klass.getPackageName() + ".reladomo.merge.hook;\n"
                + "\n"
                + "import javax.annotation.*;\n"
                + "\n"
                + "import " + klass.getFullyQualifiedName() + ";\n"
                + "import " + versionClass.getFullyQualifiedName() + ";\n"
                + "import com.gs.fw.common.mithra.list.merge.MergeBuffer;\n"
                + "import com.gs.fw.common.mithra.list.merge.MergeHook;\n"
                + "\n"
                + "/**\n"
                + " * Auto-generated by {@link " + this.getClass().getCanonicalName() + "}\n"
                + " */\n"
                + "public class " + klass.getName() + "MergeHook extends MergeHook<" + klass.getName() + ">\n"
                + "{\n"
                + "    @Override\n"
                + "    public InsertInstruction beforeInsertOfNew(@Nonnull " + klass.getName() + " newObject)\n"
                + "    {\n"
                + "        " + versionClass.getName() + " version = new " + versionClass.getName() + "();\n"
                + "        version.setNumber(1);\n"
                + setKeyPropertiesSourceCode
                + setAuditPropertiesOnCreateSourceCode
                + "        version.insert();\n"
                + "        return super.beforeInsertOfNew(newObject);\n"
                + "    }\n"
                + "\n"
                + "    @Override\n"
                + "    public UpdateInstruction matchedWithDifferenceBeforeAttributeCopy(\n"
                + "            @Nonnull " + klass.getName() + " existing,\n"
                + "            " + klass.getName() + " incoming)\n"
                + "    {\n"
                + "        " + versionClass.getName() + " existingVersion = existing.getVersion();\n"
                + "        existingVersion.setNumber(existingVersion.getNumber() + 1);\n"
                + setAuditPropertiesOnUpdateSourceCode
                + "        return super.matchedWithDifferenceBeforeAttributeCopy(existing, incoming);\n"
                + "    }\n"
                + "\n"
                + "    @Override\n"
                + "    public DeleteOrTerminateInstruction beforeDeleteOrTerminate(\n"
                + "            @Nonnull " + klass.getName() + " existing,\n"
                + "            MergeBuffer<" + klass.getName() + "> mergeBuffer)\n"
                + "    {\n"
                + "        existing.getVersion().terminate();\n"
                + "        return super.beforeDeleteOrTerminate(existing, mergeBuffer);\n"
                + "    }\n"
                + "}\n"
                + "\n";
        // @formatter:on

        return sourceCode;
    }

    private String getKeyPropertySourceCode(DataTypeProperty keyProperty)
    {
        String name   = LOWER_CAMEL_TO_UPPER_CAMEL.convert(keyProperty.getName());
        String prefix = keyProperty.getType() == PrimitiveType.BOOLEAN ? "is" : "get";
        return "        version.set" + name + "(newObject." + prefix + name + "());\n";
    }
}
