package cool.mtc.security.auth.password;

import cool.mtc.core.result.ResultConstant;
import cool.mtc.security.constant.AuthTypeConstant;
import cool.mtc.security.data.model.UserDetails;
import cool.mtc.security.exception.AuthException;
import cool.mtc.security.service.SecurityService;
import lombok.Setter;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.authentication.AuthenticationProvider;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.util.Assert;

import java.util.Optional;

/**
 * @author 明河
 */
@Setter(onMethod = @__(@Autowired))
public class PasswordAuthProvider implements AuthenticationProvider, InitializingBean {
    private SecurityService securityService;
    private PasswordEncoder passwordEncoder;

    @Override
    public void afterPropertiesSet() {
        Assert.notNull(this.securityService, "A SecurityService Bean Must Be Set");
    }

    @Override
    public Authentication authenticate(Authentication authentication) throws AuthenticationException {
        String username = authentication.getPrincipal().toString();
        String password = authentication.getCredentials().toString();

        UserDetails user = Optional.ofNullable(securityService.loadByUsernameAndAuthType(username, AuthTypeConstant.PASSWORD))
                .orElseThrow(() -> new AuthException(ResultConstant.A0201, "login.username_or_password.error"));
        this.handleCheckPasswordCorrect(user, password);
        this.handleCheckUserStatus(user);

        PasswordAuthToken token = new PasswordAuthToken(user.getUsername(), user.getPassword(), user.getAuthorities());
        token.setDetails(user);
        return token;
    }

    @Override
    public boolean supports(Class<?> aClass) {
        return PasswordAuthToken.class.isAssignableFrom(aClass);
    }

    /**
     * 检查密码是否正确
     */
    private void handleCheckPasswordCorrect(UserDetails user, String inputPassword) {
        if (passwordEncoder.matches(inputPassword, user.getPassword())) {
            return;
        }
        throw new AuthException(ResultConstant.A0210, "login.username_or_password.error");
    }

    /**
     * 检查用户状态以验证是否允许登录
     */
    private void handleCheckUserStatus(UserDetails user) {
        if (user.isEnabled()) {
            return;
        }
        throw new AuthException(ResultConstant.A0200, "login.status.error");
    }
}
