package cool.mtc.security.auth.jwt;

import cool.mtc.core.result.ResultConstant;
import cool.mtc.security.data.model.UserDetails;
import cool.mtc.security.exception.AuthException;
import cool.mtc.security.plugin.jwt.JwtConstant;
import cool.mtc.security.plugin.jwt.JwtTemplate;
import cool.mtc.security.service.SecurityService;
import io.jsonwebtoken.Claims;
import io.jsonwebtoken.ExpiredJwtException;
import lombok.Setter;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.authentication.AuthenticationProvider;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.util.Assert;

import java.util.Optional;

/**
 * @author 明河
 */
@Setter(onMethod = @__(@Autowired))
public class JwtAuthProvider implements AuthenticationProvider, InitializingBean {
    private SecurityService securityService;
    private JwtTemplate jwtTemplate;

    @Override
    public void afterPropertiesSet() {
        Assert.notNull(this.securityService, "A SecurityService Bean Must Be Set");
    }

    @Override
    public Authentication authenticate(Authentication authentication) throws AuthenticationException {
        String token = ((JwtAuthToken) authentication).getToken();
        boolean isTokenExpired = false;
        // 解析token
        Claims claims;
        try {
            claims = this.parseToken(token);
        } catch (ExpiredJwtException ex) {
            claims = ex.getClaims();
            isTokenExpired = true;
        }

        this.handleCheckTokenInWhiteList(token);
        UserDetails user = Optional.ofNullable(this.getUserDetailsByToken(claims))
                .orElseThrow(() -> new AuthException(ResultConstant.A0301));
        this.handleCheckUserStatus(user);

        JwtAuthToken authToken = new JwtAuthToken(token, isTokenExpired, user.getAuthorities());
        authToken.setDetails(user);
        return authToken;
    }

    @Override
    public boolean supports(Class<?> aClass) {
        return JwtAuthToken.class.isAssignableFrom(aClass);
    }

    /**
     * 解析token
     *
     * @throws ExpiredJwtException token过期异常
     */
    private Claims parseToken(String token) throws ExpiredJwtException {
        try {
            return jwtTemplate.parse(token);
        } catch (ExpiredJwtException ex) {
            throw ex;
        } catch (Exception ex) {
            throw new AuthException(ResultConstant.A0301);
        }
    }

    /**
     * 从token参数中获取用户信息
     *
     * @throws NullPointerException 参数错误
     */
    private UserDetails getUserDetailsByToken(Claims claims) {
        // 获取token中的参数
        try {
            Object userId = claims.get(JwtConstant.PARAM_KEY_USER_ID);
            Object orgId = claims.get(JwtConstant.PARAM_KEY_ORG_ID);
            Object authType = claims.get(JwtConstant.PARAM_KEY_AUTH_TYPE);
            return securityService.loadByUserIdAndOrgIdAndAuthType(userId, orgId, authType.toString());
        } catch (NullPointerException ex) {
            throw new AuthException(ResultConstant.A0301);
        }
    }

    private void handleCheckTokenInWhiteList(String token) {
        if (securityService.isTokenInAllowList(token)) {
            return;
        }
        throw new AuthException(ResultConstant.A0311);
    }

    /**
     * 检查用户状态以验证是否允许登录
     */
    private void handleCheckUserStatus(UserDetails user) {
        if (user.isEnabled()) {
            return;
        }
        throw new AuthException(ResultConstant.A0200.newInstance(), "login.status.error");
    }
}
