package cool.mtc.security.auth.password;

import com.fasterxml.jackson.databind.ObjectMapper;
import cool.mtc.core.result.ResultConstant;
import cool.mtc.core.util.StringUtil;
import cool.mtc.security.SecurityProperties;
import cool.mtc.security.exception.AuthException;
import cool.mtc.security.service.SecurityService;
import org.springframework.http.HttpMethod;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.web.authentication.AbstractAuthenticationProcessingFilter;
import org.springframework.security.web.util.matcher.AntPathRequestMatcher;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;

/**
 * @author 明河
 */
public class PasswordAuthFilter extends AbstractAuthenticationProcessingFilter {
    private static final ObjectMapper MAPPER = new ObjectMapper();

    private final SecurityService securityService;

    public PasswordAuthFilter(SecurityProperties securityProperties, SecurityService securityService) {
        super(new AntPathRequestMatcher(securityProperties.getLoginUrlByPassword(), HttpMethod.POST.name()));
        this.securityService = securityService;
    }

    @Override
    public Authentication attemptAuthentication(HttpServletRequest request, HttpServletResponse response) throws AuthenticationException {
        PasswordForm form = this.handleCheckParams(request);
        if (securityService.isVerifyCodeError(form.getVerifyCode())) {
            throw new AuthException(ResultConstant.A0130, "login.verify_code.error");
        }
        PasswordAuthToken token = new PasswordAuthToken(form.getUsername(), form.getPassword());
        this.setDetails(request, token);
        return super.getAuthenticationManager().authenticate(token);
    }

    private PasswordForm handleCheckParams(HttpServletRequest request) {
        PasswordForm form;
        try {
            form = MAPPER.readValue(request.getInputStream(), PasswordForm.class);
        } catch (IOException ex) {
            throw new AuthException(ResultConstant.A0402, "login.param.error");
        }
        if (StringUtil.isEmpty(form.getUsername())) {
            throw new AuthException(ResultConstant.A0410, "login.username.required");
        }
        if (StringUtil.isEmpty(form.getPassword())) {
            throw new AuthException(ResultConstant.A0410, "login.password.required");
        }
        return form;
    }

    private void setDetails(HttpServletRequest request, PasswordAuthToken token) {
        token.setDetails(authenticationDetailsSource.buildDetails(request));
    }
}
