package cool.mtc.security;

import cool.mtc.security.auth.jwt.JwtAuthAdapter;
import cool.mtc.security.auth.password.PasswordAuthAdapter;
import cool.mtc.security.config.WebSecurityConfigurer;
import cool.mtc.security.handler.auth.AuthFailureHandler;
import cool.mtc.security.handler.auth.jwt.JwtAuthFailureHandler;
import cool.mtc.security.handler.auth.jwt.JwtAuthSuccessHandler;
import cool.mtc.security.handler.auth.password.PasswordAuthFailureHandler;
import cool.mtc.security.handler.auth.password.PasswordAuthSuccessHandler;
import cool.mtc.security.handler.deny.AccessDeniedHandler;
import cool.mtc.security.handler.deny.AuthenticationEntryPoint;
import cool.mtc.security.handler.logout.LogoutHandler;
import cool.mtc.security.handler.logout.LogoutSuccessHandler;
import cool.mtc.security.plugin.jwt.JwtProperties;
import cool.mtc.security.plugin.jwt.JwtTemplate;
import cool.mtc.security.service.PermissionService;
import cool.mtc.security.service.SecurityService;
import cool.mtc.security.service.impl.DefaultSecurityServiceImpl;
import cool.mtc.security.service.impl.PermissionServiceImpl;
import cool.mtc.web.component.I18nHandler;
import lombok.RequiredArgsConstructor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.security.config.annotation.method.configuration.EnableGlobalMethodSecurity;
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity;

/**
 * @author 明河
 */
@Configuration
@EnableWebSecurity
@EnableGlobalMethodSecurity(prePostEnabled = true)
@ConditionalOnProperty(value = "mtc.security.enabled", havingValue = "true", matchIfMissing = true)
@EnableConfigurationProperties({
        SecurityProperties.class,
        JwtProperties.class
})
@Import({
        WebSecurityConfigurer.class,
})
@RequiredArgsConstructor(onConstructor = @__(@Autowired))
public class SecurityAutoConfiguration {
    private final SecurityProperties securityProperties;
    private final JwtProperties jwtProperties;
    private final I18nHandler i18nHandler;

    @Bean
    public JwtTemplate jwtTemplate() {
        return new JwtTemplate(jwtProperties);
    }

    @Bean
    @ConditionalOnMissingBean(value = SecurityService.class)
    public SecurityService securityService() {
        return new DefaultSecurityServiceImpl();
    }

    @Bean(name = "ps")
    public PermissionService permissionService() {
        return new PermissionServiceImpl();
    }

    @Bean
    public AccessDeniedHandler accessDeniedHandler() {
        return new AccessDeniedHandler(i18nHandler);
    }

    @Bean
    public AuthenticationEntryPoint authenticationEntryPoint() {
        return new AuthenticationEntryPoint(i18nHandler);
    }

    @Bean
    public LogoutHandler logoutHandler() {
        return new LogoutHandler(securityService());
    }

    @Bean
    public LogoutSuccessHandler logoutSuccessHandler() {
        return new LogoutSuccessHandler(i18nHandler);
    }

    @Bean
    public AuthFailureHandler authFailureHandler() {
        return new AuthFailureHandler(i18nHandler);
    }

    @Bean
    public PasswordAuthAdapter passwordAuthAdapter() {
        return new PasswordAuthAdapter(securityProperties, securityService(), passwordAuthSuccessHandler(), passwordAuthFailureHandler());
    }

    @Bean
    public PasswordAuthSuccessHandler passwordAuthSuccessHandler() {
        return new PasswordAuthSuccessHandler(securityService(), jwtTemplate());
    }

    @Bean
    public PasswordAuthFailureHandler passwordAuthFailureHandler() {
        return new PasswordAuthFailureHandler(i18nHandler);
    }

    @Bean
    public JwtAuthAdapter jwtAuthAdapter() {
        return new JwtAuthAdapter(securityService(), jwtTemplate(), jwtAuthSuccessHandler(), jwtAuthFailureHandler());
    }

    @Bean
    public JwtAuthSuccessHandler jwtAuthSuccessHandler() {
        return new JwtAuthSuccessHandler(securityService(), jwtTemplate());
    }

    @Bean
    public JwtAuthFailureHandler jwtAuthFailureHandler() {
        return new JwtAuthFailureHandler(i18nHandler);
    }
}
