package cool.mtc.security.config;

import cool.mtc.security.SecurityProperties;
import cool.mtc.security.auth.jwt.JwtAuthAdapter;
import cool.mtc.security.auth.password.PasswordAuthAdapter;
import cool.mtc.security.handler.deny.AccessDeniedHandler;
import cool.mtc.security.handler.deny.AuthenticationEntryPoint;
import cool.mtc.security.handler.logout.LogoutHandler;
import cool.mtc.security.handler.logout.LogoutSuccessHandler;
import cool.mtc.security.service.SecurityService;
import lombok.RequiredArgsConstructor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.builders.WebSecurity;
import org.springframework.security.config.annotation.web.configuration.WebSecurityConfigurerAdapter;

/**
 * @author 明河
 */
@Configuration
@RequiredArgsConstructor(onConstructor = @__(@Autowired))
public class WebSecurityConfigurer extends WebSecurityConfigurerAdapter {
    private final SecurityProperties securityProperties;
    private final LogoutHandler logoutHandler;
    private final LogoutSuccessHandler logoutSuccessHandler;
    private final AuthenticationEntryPoint authenticationEntryPoint;
    private final AccessDeniedHandler accessDeniedHandler;
    private final PasswordAuthAdapter passwordAuthAdapter;
    private final JwtAuthAdapter jwtAuthAdapter;

    @Override
    protected void configure(HttpSecurity http) throws Exception {
        http.csrf().disable();
        // 跨域
        if (securityProperties.isCors()) {
            http.cors();
        } else {
            http.cors().disable();
        }
        // 用户名密码登录适配器
        http.apply(passwordAuthAdapter);
        // Jwt认证适配器
        http.apply(jwtAuthAdapter);
        // 需要认证的地址
        http.authorizeRequests()
                .antMatchers(securityProperties.getAuthUrls())
                .authenticated();
        // 登出
        http.logout()
                .logoutUrl(securityProperties.getLogoutUrl())
                .addLogoutHandler(logoutHandler)
                .logoutSuccessHandler(logoutSuccessHandler);
        // 异常处理
        http.exceptionHandling()
                .authenticationEntryPoint(authenticationEntryPoint)
                .accessDeniedHandler(accessDeniedHandler);

        // 各项目自定义配置
        securityConfigurer().configure(http);
    }

    @Override
    public void configure(WebSecurity web) {
        // 忽略认证检查的
        if (securityProperties.getIgnoreUrls().length > 0) {
            web.ignoring().antMatchers(securityProperties.getIgnoreUrls());
        }
    }

    @Bean
    @ConditionalOnMissingBean(value = SecurityConfigurerSupport.class)
    public SecurityConfigurerSupport securityConfigurer() {
        return new DefaultSecurityConfigurer();
    }
}
