package cool.mtc.security.plugin.jwt;

import cool.mtc.core.util.DateUtil;
import cool.mtc.security.data.model.UserDetailsSupport;
import io.jsonwebtoken.*;
import io.jsonwebtoken.impl.DefaultClaims;
import lombok.RequiredArgsConstructor;

import javax.crypto.spec.SecretKeySpec;
import java.nio.charset.StandardCharsets;
import java.security.Key;
import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.Map;

/**
 * @author 明河
 */
@RequiredArgsConstructor
public class JwtTemplate {
    private final JwtProperties jwtProperties;

    public String generate(UserDetailsSupport user) {
        Map<String, Object> data = new HashMap<>();
        data.put(JwtConstant.PARAM_KEY_USER_ID, user.getUserId());
        data.put(JwtConstant.PARAM_KEY_ORG_ID, user.getOrgId());
        data.put(JwtConstant.PARAM_KEY_AUTH_TYPE, user.getAuthType());
        return this.generate(data);
    }

    public String generate(Map<String, Object> data) {
        return Jwts.builder()
                .addClaims(this.handleClaimsData(data))
                .signWith(jwtProperties.getSignatureAlgorithm(), this.getKey())
                .compact();
    }

    public Claims parse(String token) throws ExpiredJwtException, UnsupportedJwtException, MalformedJwtException, SignatureException, IllegalArgumentException {
        return Jwts.parser().setSigningKey(this.getKey()).parseClaimsJws(token).getBody();
    }

    /**
     * 处理载荷数据
     */
    private Claims handleClaimsData(Map<String, Object> data) {
        Claims claims = new DefaultClaims();
        claims.putAll(data);
        LocalDateTime now = LocalDateTime.now();
        // 签发时间
        claims.setIssuedAt(DateUtil.localDateTimeToDate(now));
        // 过期时间
        LocalDateTime expiredDateTime = now.plusMinutes(jwtProperties.getValidDuration().toMinutes());
        claims.setExpiration(DateUtil.localDateTimeToDate(expiredDateTime));
        return claims;
    }


    /**
     * 根据设置的秘钥及签名算法获取key
     */
    private Key getKey() {
        byte[] key_bytes = jwtProperties.getKey().getBytes(StandardCharsets.UTF_8);
        return new SecretKeySpec(key_bytes, jwtProperties.getSignatureAlgorithm().getJcaName());
    }
}
