package cool.scx.util;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

/**
 * 简化 zip 的创建
 *
 * @author scx567888
 * @version 0.1.4
 */
public class ZipBuilder {

    private final List<ZipItem> items = new ArrayList<>();

    /**
     * <p>put.</p>
     *
     * @param path a {@link java.lang.String} object
     * @return a
     */
    public ZipBuilder put(String path) {
        items.add(new ZipItem(path, null, null, ZipItemType.DIR));
        return this;
    }

    /**
     * <p>put.</p>
     *
     * @param path a {@link java.lang.String} object
     * @param f    a {@link java.io.File} object
     * @return a
     */
    public ZipBuilder put(String path, File f) {
        items.add(new ZipItem(path, f, null, ZipItemType.FILE));
        return this;
    }

    /**
     * <p>put.</p>
     *
     * @param path  a {@link java.lang.String} object
     * @param bytes an array of {@link byte} objects
     * @return a
     */
    public ZipBuilder put(String path, byte[] bytes) {
        items.add(new ZipItem(path, null, bytes, ZipItemType.BYTES));
        return this;
    }

    /**
     * <p>remove.</p>
     *
     * @param path a {@link java.lang.String} object
     * @return a
     */
    public ZipBuilder remove(String path) {
        if (StringUtils.notBlank(path)) {
            items.removeIf(c -> c.path.startsWith(path));
        }
        return this;
    }

    /**
     * <p>writeToZipOutputStream.</p>
     *
     * @param zos a {@link java.util.zip.ZipOutputStream} object
     * @throws java.lang.Exception if any.
     */
    public void writeToZipOutputStream(ZipOutputStream zos) throws Exception {
        for (var i : items) {
            switch (i.type()) {
                case DIR -> zos.putNextEntry(new ZipEntry(i.path + "/"));
                case FILE -> {
                    if (i.file.isDirectory()) {
                        ZipUtils.writeToZipOutputStream(i.file.toPath(), zos, i.path + "/");
                    } else {
                        zos.putNextEntry(new ZipEntry(i.path));
                        Files.copy(i.file.toPath(), zos);
                    }
                }
                case BYTES -> {
                    zos.putNextEntry(new ZipEntry(i.path));
                    zos.write(i.bytes);
                }
            }
            zos.closeEntry();
        }
    }

    /**
     * 将 virtualFile 转换为 byte 数组 方便前台用户下载使用
     *
     * @return a
     * @throws java.lang.Exception a
     */
    public byte[] toZipBytes() throws Exception {
        var bo = new ByteArrayOutputStream();
        try (var zos = new ZipOutputStream(bo)) {
            //遍历目录
            writeToZipOutputStream(zos);
        }
        return bo.toByteArray();
    }

    /**
     * 将一个虚拟文件压缩
     *
     * @param outputPath a
     * @throws java.lang.Exception a
     */
    public void toZipFile(Path outputPath) throws Exception {
        // 创建一个新的空的输出文件的临时文件
        Files.createDirectories(outputPath.getParent());
        try (var zos = new ZipOutputStream(Files.newOutputStream(outputPath))) {
            //遍历目录
            writeToZipOutputStream(zos);
        }
    }

    enum ZipItemType {
        DIR,
        FILE,
        BYTES,
    }

    /**
     * @param path 总路径 包括自己的名称
     */
    record ZipItem(String path, File file, byte[] bytes, ZipItemType type) {

    }

}
