package cool.scx.common.util.reflect;

import com.fasterxml.jackson.databind.JavaType;

import java.lang.annotation.Annotation;
import java.lang.reflect.Modifier;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.stream.Stream;

import static cool.scx.common.util.ObjectUtils.resolveMemberType;
import static cool.scx.common.util.reflect.AccessModifier.*;
import static java.util.Collections.addAll;

final class InitHelper {

    public static boolean initIsRecord(ClassInfo classInfo) {
        return classInfo._class().isRecord();
    }

    public static boolean initIsInterface(ClassInfo classInfo) {
        return classInfo._class().isInterface();
    }

    public static ClassInfo initSuperClass(ClassInfo classInfo) {
        return classInfo.type().getSuperClass() != null ? new ClassInfo(classInfo.type().getSuperClass()) : null;
    }

    public static String initName(FieldInfo fieldInfo) {
        return fieldInfo._field().getName();
    }

    public static String initName(MethodInfo methodInfo) {
        return methodInfo._method().getName();
    }

    public static String initName(ParameterInfo parameterInfo) {
        return parameterInfo._parameter().getName();
    }

    public static AccessModifier initAccessModifier(FieldInfo fieldInfo) {
        return initAccessModifier(fieldInfo._field().getModifiers());
    }

    public static AccessModifier initAccessModifier(MethodInfo methodInfo) {
        return initAccessModifier(methodInfo._method().getModifiers());
    }

    public static AccessModifier initAccessModifier(ConstructorInfo constructorInfo) {
        return initAccessModifier(constructorInfo._constructor().getModifiers());
    }

    public static AccessModifier initAccessModifier(int m) {
        if (Modifier.isPublic(m)) {
            return PUBLIC;
        } else if (Modifier.isProtected(m)) {
            return PROTECTED;
        } else if (Modifier.isPrivate(m)) {
            return PRIVATE;
        } else {
            return DEFAULT;
        }
    }

    public static JavaType initType(FieldInfo fieldInfo) {
        return initType(fieldInfo._field().getGenericType(), fieldInfo.classInfo());
    }

    public static JavaType initType(ParameterInfo parameterInfo) {
        return initType(parameterInfo._parameter().getParameterizedType(), parameterInfo.executableInfo().classInfo());
    }

    public static JavaType initReturnType(MethodInfo methodInfo) {
        return initType(methodInfo._method().getGenericReturnType(), methodInfo.classInfo());
    }

    public static JavaType initType(Type type, ClassInfo classInfo) {
        return resolveMemberType(type, classInfo.type().getBindings());
    }

    public static Annotation[] initAnnotations(ClassInfo classInfo) {
        return classInfo._class().getDeclaredAnnotations();
    }

    public static Annotation[] initAnnotations(FieldInfo fieldInfo) {
        return fieldInfo._field().getDeclaredAnnotations();
    }

    public static Annotation[] initAnnotations(MethodInfo methodInfo) {
        return methodInfo._method().getDeclaredAnnotations();
    }

    /**
     * 获取当前方法的注解 同时包含 重写方法的注解
     *
     * @return a
     */
    public static Annotation[] initAllAnnotations(MethodInfo methodInfo) {
        var allAnnotations = new ArrayList<Annotation>();
        while (methodInfo != null) {
            addAll(allAnnotations, methodInfo.annotations());
            methodInfo = methodInfo.superMethod();
        }
        return allAnnotations.toArray(Annotation[]::new);
    }

    /**
     * 获取当前方法的注解 同时包含 重写方法的注解
     *
     * @return a
     */
    public static Annotation[] initAllAnnotations(ClassInfo classInfo) {
        var allAnnotations = new ArrayList<Annotation>();
        while (classInfo != null) {
            addAll(allAnnotations, classInfo.annotations());
            classInfo = classInfo.superClass();
        }
        return allAnnotations.toArray(Annotation[]::new);
    }

    public static ConstructorInfo[] initConstructorInfos(ClassInfo classInfo) {
        return Stream.of(classInfo._class().getDeclaredConstructors()).map(c -> new ConstructorInfo(c, classInfo)).toArray(ConstructorInfo[]::new);
    }

    /**
     * 寻找 无参构造函数 (不支持成员类)
     *
     * @param classInfo c
     * @return a
     */
    public static ConstructorInfo initNoArgsConstructor(ClassInfo classInfo) {
        for (var constructor : classInfo.constructors()) {
            if (constructor.parameters().length == 0) {
                return constructor;
            }
        }
        return null;
    }

    /**
     * 寻找 Record 规范构造参数
     */
    public static ConstructorInfo initRecordConstructor(ClassInfo classInfo) {
        if (!classInfo.isRecord()) {
            return null;
        }
        var recordComponentTypes = getRecordComponentsTypes(classInfo);
        for (var constructor : classInfo.constructors()) {
            // 判断参数类型是否匹配
            var matched = hasSameParameterTypes(constructor, recordComponentTypes);
            if (matched) {
                return constructor;
            }
        }
        return null;
    }

    public static FieldInfo[] initFieldInfos(ClassInfo classInfo) {
        return Arrays.stream(classInfo._class().getDeclaredFields()).map(field -> new FieldInfo(field, classInfo)).toArray(FieldInfo[]::new);
    }

    public static FieldInfo[] initAllFieldInfos(ClassInfo classInfo) {
        var allFieldInfos = new ArrayList<FieldInfo>();
        while (classInfo != null) {
            addAll(allFieldInfos, classInfo.fields());
            classInfo = classInfo.superClass();
        }
        return allFieldInfos.toArray(FieldInfo[]::new);
    }

    /**
     * 获取当前 ClassInfo 的所有方法 (不包括父类方法 不包括桥接方法)
     *
     * @param classInfo c
     * @return c
     */
    public static MethodInfo[] initMethodInfos(ClassInfo classInfo) {
        return Arrays.stream(classInfo._class().getDeclaredMethods()).filter(method -> !method.isBridge()).map(method -> new MethodInfo(method, classInfo)).toArray(MethodInfo[]::new);
    }

    public static MethodInfo initSuperMethod(MethodInfo methodInfo) {
        var superClass = methodInfo.classInfo().superClass();
        while (superClass != null) {
            var superMethods = superClass.methods();
            for (var superMethod : superMethods) {
                var b = isOverride(methodInfo, superMethod);
                // 只查找第一次匹配的方法 
                if (b) {
                    return superMethod;
                }
            }
            superClass = superClass.superClass();
        }
        return null;
    }

    public static MethodInfo[] initAllMethodInfos(ClassInfo classInfo) {
        //存储所有出现过的父类方法 用于过滤
        var filter = new HashSet<MethodInfo>();
        var allMethodInfo = new ArrayList<MethodInfo>();
        //这里 排除 Object 的所有方法
        while (classInfo != null) {
            var methods = classInfo.methods();
            for (var method : methods) {
                //如果有上层的重写方法 添加到 superMethodSet 中 在下一次循环中以便过滤
                if (method.superMethod() != null) {
                    filter.add(method.superMethod());
                }
                var b = filter.contains(method);
                if (!b) {
                    allMethodInfo.add(method);
                }
            }
            classInfo = classInfo.superClass();
        }
        return allMethodInfo.toArray(MethodInfo[]::new);
    }

    public static ParameterInfo[] initParameterInfos(MethodInfo methodInfo) {
        return Arrays.stream(methodInfo._method().getParameters()).map(parameter -> new ParameterInfo(parameter, methodInfo)).toArray(ParameterInfo[]::new);
    }

    public static ParameterInfo[] initParameterInfos(ConstructorInfo constructorInfo) {
        return Arrays.stream(constructorInfo._constructor().getParameters()).map(parameter -> new ParameterInfo(parameter, constructorInfo)).toArray(ParameterInfo[]::new);
    }

    /**
     * 判断是否为重写方法
     *
     * @param rootMethod      a
     * @param candidateMethod a
     * @return a
     */
    private static boolean isOverride(MethodInfo rootMethod, MethodInfo candidateMethod) {
        return PRIVATE != candidateMethod.accessModifier() &&
               candidateMethod.name().equals(rootMethod.name()) &&
               hasSameParameterTypes(rootMethod, candidateMethod);
    }

    private static JavaType[] getRecordComponentsTypes(ClassInfo classInfo) {
        return Arrays.stream(classInfo._class().getRecordComponents()).map(r -> initType(r.getGenericType(), classInfo)).toArray(JavaType[]::new);
    }

    private static boolean hasSameParameterTypes(MethodInfo rootMethod, MethodInfo candidateMethod) {
        if (candidateMethod.parameters().length != rootMethod.parameters().length) {
            return false;
        }
        var p1 = rootMethod.parameters();
        var p2 = candidateMethod.parameters();
        for (int i = 0; i < p1.length; i++) {
            var p1Type = p1[i].type().getRawClass();
            var p2Type = p2[i].type().getRawClass();
            if (p1Type != p2Type) {
                return false;
            }
        }
        return true;
    }

    private static boolean hasSameParameterTypes(ConstructorInfo constructorInfo, JavaType[] types) {
        if (constructorInfo.parameters().length != types.length) {
            return false;
        }
        var p1 = constructorInfo.parameters();
        for (int i = 0; i < p1.length; i++) {
            var p1Type = p1[i].type().getRawClass();
            var p2Type = types[i].getRawClass();
            if (p1Type != p2Type) {
                return false;
            }
        }
        return true;
    }

}
