package cool.scx.common.util.reflect;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;

import static cool.scx.common.util.reflect.AccessModifier.PRIVATE;

public final class MethodUtils {

    /**
     * 获取当前 ClassInfo 的所有方法 (不包括父类方法 不包括桥接方法)
     *
     * @param classInfo c
     * @return c
     */
    public static MethodInfo[] findMethodInfos(ClassInfo classInfo) {
        return Arrays.stream(classInfo._class().getDeclaredMethods()).filter(method -> !method.isBridge()).map(method -> new MethodInfo(method, classInfo)).toArray(MethodInfo[]::new);
    }

    public static MethodInfo findSuperMethod(MethodInfo methodInfo) {
        var superClass = methodInfo.classInfo().superClass();
        while (superClass != null) {
            var superMethods = superClass.methods();
            for (var superMethod : superMethods) {
                var b = isOverride(methodInfo, superMethod);
                // 只查找第一次匹配的方法 
                if (b) {
                    return superMethod;
                }
            }
            superClass = superClass.superClass();
        }
        return null;
    }

    public static MethodInfo[] findAllMethodInfos(ClassInfo classInfo) {
        //存储所有出现过的父类方法 用于过滤
        var filter = new HashSet<MethodInfo>();
        var allMethodInfo = new ArrayList<MethodInfo>();
        //这里 排除 Object 的所有方法
        while (classInfo != null) {
            var methods = classInfo.methods();
            for (var method : methods) {
                //如果有上层的重写方法 添加到 superMethodSet 中 在下一次循环中以便过滤
                if (method.superMethod() != null) {
                    filter.add(method.superMethod());
                }
                var b = filter.contains(method);
                if (!b) {
                    allMethodInfo.add(method);
                }
            }
            classInfo = classInfo.superClass();
        }
        return allMethodInfo.toArray(MethodInfo[]::new);
    }

    public static String findName(MethodInfo methodInfo) {
        return methodInfo._method().getName();
    }

    /**
     * 判断是否为重写方法
     *
     * @param rootMethod      a
     * @param candidateMethod a
     * @return a
     */
    private static boolean isOverride(MethodInfo rootMethod, MethodInfo candidateMethod) {
        return PRIVATE != candidateMethod.accessModifier() &&
               candidateMethod.name().equals(rootMethod.name()) &&
               hasSameParameterTypes(rootMethod, candidateMethod);
    }

    private static boolean hasSameParameterTypes(MethodInfo rootMethod, MethodInfo candidateMethod) {
        if (candidateMethod.parameters().length != rootMethod.parameters().length) {
            return false;
        }
        var p1 = rootMethod.parameters();
        var p2 = candidateMethod.parameters();
        for (int i = 0; i < p1.length; i++) {
            var p1Type = p1[i].type().getRawClass();
            var p2Type = p2[i].type().getRawClass();
            if (p1Type != p2Type) {
                return false;
            }
        }
        return true;
    }

}
