package cool.scx.ext.crud;

import cool.scx.Scx;
import cool.scx.ScxContext;
import cool.scx.annotation.FromBody;
import cool.scx.annotation.FromPath;
import cool.scx.annotation.ScxMapping;
import cool.scx.enumeration.Method;
import cool.scx.exception.HttpRequestException;
import cool.scx.ext.organization.auth.OrganizationPerms;
import cool.scx.vo.Json;

import java.sql.SQLException;
import java.util.List;
import java.util.Map;

/**
 * 通用 Crud的 controller
 *
 * @author scx567888
 * @version 1.0.10
 */
@ScxMapping("api/crud")
public class CRUDController {

    private final CRUDHandler crudHandler;

    public CRUDController() {
        var crudHandlerClass = Scx.getScxModuleByBaseModuleClass(CRUDModule.class).baseModuleExample().crudHandlerClass();
        this.crudHandler = ScxContext.getBean(crudHandlerClass);
    }

    /**
     * 列表查询
     *
     * @param modelName     a {@link java.lang.String} object.
     * @param limit         a {@link java.lang.Integer} object.
     * @param page          a {@link java.lang.Integer} object.
     * @param orderByColumn a {@link java.lang.String} object.
     * @param sortType      a {@link java.lang.String} object.
     * @param whereBodyList a {@link java.util.Map} object.
     * @return a {@link cool.scx.vo.Json} object.
     * @throws cool.scx.exception.HttpRequestException if any.
     * @throws SQLException                            if any.
     */
    @OrganizationPerms(checkedLogin = true)
    @ScxMapping(value = ":modelName/list", method = Method.POST)
    public Json list(@FromPath String modelName,
                     @FromBody(value = "pagination.limit", required = false) Integer limit,
                     @FromBody(value = "pagination.page", required = false) Integer page,
                     @FromBody(value = "orderBy.orderByColumn", required = false) String orderByColumn,
                     @FromBody(value = "orderBy.sortType", required = false) String sortType,
                     @FromBody(value = "whereBodyList", required = false) List<CrudWhereBody> whereBodyList
    ) throws HttpRequestException, SQLException {
        return crudHandler.list(modelName, limit, page, orderByColumn, sortType, whereBodyList);
    }

    /**
     * 获取详细信息
     *
     * @param modelName a {@link java.lang.String} object.
     * @param id        a {@link java.lang.Long} object.
     * @return a {@link cool.scx.vo.Json} object.
     * @throws cool.scx.exception.HttpRequestException if any.
     * @throws SQLException                            if any.
     */
    @OrganizationPerms(checkedLogin = true)
    @ScxMapping(value = ":modelName/:id", method = Method.GET)
    public Json info(String modelName, Long id) throws HttpRequestException, SQLException {
        return crudHandler.info(modelName, id);
    }

    /**
     * 保存
     *
     * @param modelName a {@link java.lang.String} object.
     * @param entityMap a {@link java.util.Map} object.
     * @return a {@link cool.scx.vo.Json} object.
     * @throws cool.scx.exception.HttpRequestException if any.
     * @throws SQLException                            if any.
     */
    @OrganizationPerms(checkedLogin = true)
    @ScxMapping(value = ":modelName", method = Method.POST)
    public Json save(String modelName, Map<String, Object> entityMap) throws HttpRequestException, SQLException {
        return crudHandler.save(modelName, entityMap);
    }

    /**
     * 更新
     *
     * @param modelName a {@link java.lang.String} object.
     * @param entityMap a {@link java.util.Map} object.
     * @return a {@link cool.scx.vo.Json} object.
     * @throws HttpRequestException if any.
     * @throws SQLException         if any.
     */
    @OrganizationPerms(checkedLogin = true)
    @ScxMapping(value = ":modelName", method = Method.PUT)
    public Json update(String modelName, Map<String, Object> entityMap) throws HttpRequestException, SQLException {
        return crudHandler.update(modelName, entityMap);
    }

    /**
     * 删除
     *
     * @param modelName a
     * @param id        a
     * @return j
     * @throws HttpRequestException if any.
     * @throws SQLException         if any.
     */
    @OrganizationPerms(checkedLogin = true)
    @ScxMapping(value = ":modelName/:id", method = Method.DELETE)
    public Json delete(String modelName, Integer id) throws HttpRequestException, SQLException {
        return crudHandler.delete(modelName, id);
    }

    /**
     * 批量删除
     *
     * @param modelName a {@link java.lang.String} object.
     * @param deleteIds a {@link java.util.Map} object.
     * @return a {@link cool.scx.vo.Json} object.
     * @throws cool.scx.exception.HttpRequestException if any.
     * @throws java.sql.SQLException                   SQLException
     */
    @OrganizationPerms(checkedLogin = true)
    @ScxMapping(value = ":modelName/batch-delete", method = Method.DELETE)
    public Json batchDelete(String modelName, @FromBody("deleteIds") long[] deleteIds) throws HttpRequestException, SQLException {
        return crudHandler.batchDelete(modelName, deleteIds);
    }

    /**
     * 撤销删除
     *
     * @param modelName a {@link java.lang.String} object.
     * @param id        a {@link java.lang.Integer} object.
     * @return a {@link cool.scx.vo.Json} object.
     * @throws cool.scx.exception.HttpRequestException if any.
     * @throws java.sql.SQLException                   SQLException
     */
    @ScxMapping(value = ":modelName/revoke-delete/:id", method = Method.GET)
    public Json revokeDelete(String modelName, Integer id) throws HttpRequestException, SQLException {
        return crudHandler.revokeDelete(modelName, id);
    }

    /**
     * 获取自动完成字段
     *
     * @param modelName a {@link java.lang.String} object.
     * @param fieldName a {@link java.lang.String} object.
     * @return a {@link cool.scx.vo.Json} object.
     * @throws cool.scx.exception.HttpRequestException if any.
     * @throws java.sql.SQLException                   SQLException
     */
    @ScxMapping(value = ":modelName/get-auto-complete/:fieldName", method = Method.POST)
    public Json getAutoComplete(String modelName, String fieldName) throws HttpRequestException, SQLException {
        return crudHandler.getAutoComplete(modelName, fieldName);
    }

    /**
     * 校验唯一性
     *
     * @param modelName a {@link java.lang.String} object.
     * @param params    a {@link java.util.Map} object.
     * @return a {@link cool.scx.vo.Json} object.
     * @throws cool.scx.exception.HttpRequestException if any.
     * @throws java.sql.SQLException                   SQLException
     */
    @ScxMapping(value = ":modelName/check-unique", method = Method.POST)
    public Json checkUnique(String modelName, Map<String, Object> params) throws HttpRequestException, SQLException {
        return crudHandler.checkUnique(modelName, params);
    }

}
