package cool.scx.ext.crud;

import cool.scx.Scx;
import cool.scx.ScxContext;
import cool.scx.ScxEasyConfig;
import cool.scx.annotation.FromBody;
import cool.scx.annotation.NoColumn;
import cool.scx.annotation.ScxModel;
import cool.scx.annotation.ScxService;
import cool.scx.base.BaseModel;
import cool.scx.base.BaseService;
import cool.scx.bo.Query;
import cool.scx.enumeration.WhereType;
import cool.scx.exception.BadRequestException;
import cool.scx.exception.CustomHttpRequestException;
import cool.scx.exception.HttpRequestException;
import cool.scx.util.Ansi;
import cool.scx.util.ObjectUtils;
import cool.scx.vo.Json;

import java.sql.SQLException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;

@ScxService
public class CRUDHandlerImpl implements CRUDHandler {

    /**
     * scx bean 名称 和 class 对应映射
     */
    private static final Map<String, Class<?>> SCX_BEAN_CLASS_NAME_MAPPING;

    static {
        SCX_BEAN_CLASS_NAME_MAPPING = new HashMap<>();
        for (var allModule : Scx.getScxModules()) {
            for (var c : allModule.classList()) {
                if (c.isAnnotationPresent(ScxService.class) || c.isAnnotationPresent(ScxModel.class)) {
                    var className = c.getSimpleName().toLowerCase();
                    var aClass = SCX_BEAN_CLASS_NAME_MAPPING.get(className);
                    if (aClass == null) {
                        SCX_BEAN_CLASS_NAME_MAPPING.put(c.getSimpleName().toLowerCase(), c);
                    } else {
                        SCX_BEAN_CLASS_NAME_MAPPING.put(c.getName(), c);
                        Ansi.out().brightRed("检测到重复名称的 class ").brightYellow("[" + aClass.getName() + "] ").blue("[" + c.getName() + "]").brightRed(" 可能会导致根据名称调用时意义不明确 !!! 建议修改 !!!").println();
                    }
                }
            }
        }
    }

    /**
     * 获取 service
     * todo 错误信息待处理
     *
     * @param modelName model 名称
     * @param <T>       model 类型
     * @return service
     * @throws HttpRequestException service 未找到
     */
    @SuppressWarnings("unchecked")
    private static <T extends BaseModel> BaseService<T> getBaseService(String modelName) throws HttpRequestException {
        try {
            var o = ScxContext.getBean(getClassByName(modelName.toLowerCase() + "service"));
            return (BaseService<T>) o;
        } catch (Exception e) {
            throw new CustomHttpRequestException(ctx -> Json.fail("unknown-crud-service").put("service-name", modelName.toLowerCase()).sendToClient(ctx));
        }
    }

    private static BaseModel getBaseModel(Map<String, Object> entityMap, String modelName) throws HttpRequestException {
        try {
            return (BaseModel) ObjectUtils.convertValue(entityMap, getClassByName(modelName));
        } catch (Exception e) {
            //这里一般就是 参数转换错误
            throw new BadRequestException(e);
        }
    }

    /**
     * <p>getClassByName.</p>
     *
     * @param str a {@link java.lang.String} object.
     * @return a {@link java.lang.Class} object.
     */
    public static Class<?> getClassByName(String str) {
        return SCX_BEAN_CLASS_NAME_MAPPING.get(str.toLowerCase());
    }

    /**
     * 获取 query
     *
     * @param limit         l
     * @param page          p
     * @param orderByColumn or
     * @param sortType      so
     * @param whereBodyList wh
     * @return q
     */
    private static Query getQuery(Class<?> modelClass, Integer limit, Integer page, String orderByColumn, String sortType, List<CrudWhereBody> whereBodyList) throws CustomHttpRequestException {
        var query = new Query();
        if (limit != null && limit >= 0) {
            if (page != null && page >= 0) {
                query.setPagination(page, limit);
            } else {
                query.setPagination(limit);
            }
        }
        if (orderByColumn != null && sortType != null) {
            query.addOrderBy(orderByColumn, sortType);
        }
        if (whereBodyList != null) {
            for (var crudWhereBody : whereBodyList) {
                if (crudWhereBody.fieldName != null && crudWhereBody.whereType != null) {
                    //校验 fieldName 是否正确
                    checkFieldName(modelClass, crudWhereBody.fieldName);
                    //检查 whereType 是否正确
                    var whereType = checkWhereType(crudWhereBody.fieldName, crudWhereBody.whereType);
                    //检查参数数量是否正确
                    checkWhereBodyParametersSize(crudWhereBody.fieldName, whereType, crudWhereBody.value1, crudWhereBody.value2);
                    if (whereType.paramSize() == 0) {
                        query.addWhere(crudWhereBody.fieldName, whereType);
                    } else if (whereType.paramSize() == 1) {
                        query.addWhere(crudWhereBody.fieldName, whereType, crudWhereBody.value1);
                    } else if (whereType.paramSize() == 2) {
                        query.addWhere(crudWhereBody.fieldName, whereType, crudWhereBody.value1, crudWhereBody.value2);
                    }
                }
            }
        }
        return query;
    }

    private static void checkFieldName(Class<?> modelClass, String fieldName) throws CustomHttpRequestException {
        try {
            var field = modelClass.getField(fieldName);
            if (field.isAnnotationPresent(NoColumn.class)) {
                throw new CustomHttpRequestException(ctx -> Json.fail("unknown-field-name").put("field-name", fieldName).sendToClient(ctx));
            }
        } catch (Exception e) {
            throw new CustomHttpRequestException(ctx -> Json.fail("unknown-field-name").put("field-name", fieldName).sendToClient(ctx));
        }
    }

    private static WhereType checkWhereType(String fieldName, String strWhereType) throws CustomHttpRequestException {
        try {
            return WhereType.valueOf(strWhereType.toUpperCase());
        } catch (Exception ignored) {
            throw new CustomHttpRequestException(ctx -> Json.fail("unknown-where-type").put("field-name", fieldName).put("where-type", strWhereType).sendToClient(ctx));
        }
    }

    private static void checkWhereBodyParametersSize(String fieldName, WhereType whereType, Object value1, Object value2) throws CustomHttpRequestException {
        AtomicInteger paramSize = new AtomicInteger();
        if (value1 != null) {
            paramSize.set(paramSize.get() + 1);
        }
        if (value2 != null) {
            paramSize.set(paramSize.get() + 1);
        }

        if (whereType.paramSize() != paramSize.get()) {
            throw new CustomHttpRequestException(ctx -> Json.fail("where-body-parameters-size-error")
                    .put("field-name", fieldName)
                    .put("where-type", whereType)
                    .put("need-parameters-size", whereType.paramSize())
                    .put("got-parameters-size", paramSize.get())
                    .sendToClient(ctx));
        }
    }

    /**
     * 列表查询
     *
     * @param modelName     a {@link java.lang.String} object.
     * @param limit         a {@link java.lang.Integer} object.
     * @param page          a {@link java.lang.Integer} object.
     * @param orderByColumn a {@link java.lang.String} object.
     * @param sortType      a {@link java.lang.String} object.
     * @param whereBodyList a {@link java.util.Map} object.
     * @return a {@link cool.scx.vo.Json} object.
     * @throws cool.scx.exception.HttpRequestException if any.
     * @throws SQLException                            if any.
     */
    @Override
    public Json list(String modelName, Integer limit, Integer page, String orderByColumn, String sortType, List<CrudWhereBody> whereBodyList) throws HttpRequestException, SQLException {
        var baseService = getBaseService(modelName);
        var queryParam = getQuery(getClassByName(modelName), limit, page, orderByColumn, sortType, whereBodyList);
        var list = baseService.list(queryParam);
        var count = baseService.count(queryParam);
        return Json.ok().put("items", list).put("total", count);
    }

    /**
     * 获取详细信息
     *
     * @param modelName a {@link java.lang.String} object.
     * @param id        a {@link java.lang.Long} object.
     * @return a {@link cool.scx.vo.Json} object.
     * @throws cool.scx.exception.HttpRequestException if any.
     * @throws SQLException                            if any.
     */
    @Override
    public Json info(String modelName, Long id) throws HttpRequestException, SQLException {
        var baseService = getBaseService(modelName);
        var info = baseService.get(id);
        return Json.ok().put("info", info);
    }

    /**
     * 保存
     *
     * @param modelName a {@link java.lang.String} object.
     * @param entityMap a {@link java.util.Map} object.
     * @return a {@link cool.scx.vo.Json} object.
     * @throws cool.scx.exception.HttpRequestException if any.
     * @throws SQLException                            if any.
     */
    @Override
    public Json save(String modelName, Map<String, Object> entityMap) throws HttpRequestException, SQLException {
        if (entityMap == null) {
            throw new CustomHttpRequestException(ctx -> Json.fail("empty-param").sendToClient(ctx));
        }
        var baseService = getBaseService(modelName);
        var realObject = getBaseModel(entityMap, modelName);
        return Json.ok().put("item", baseService.save(realObject));
    }

    /**
     * 更新
     *
     * @param modelName a {@link java.lang.String} object.
     * @param entityMap a {@link java.util.Map} object.
     * @return a {@link cool.scx.vo.Json} object.
     * @throws HttpRequestException if any.
     * @throws SQLException         if any.
     */
    @Override
    public Json update(String modelName, Map<String, Object> entityMap) throws HttpRequestException, SQLException {
        if (entityMap == null) {
            throw new CustomHttpRequestException(ctx -> Json.fail("empty-param").sendToClient(ctx));
        }
        var baseService = getBaseService(modelName);
        var realObject = getBaseModel(entityMap, modelName);
        return Json.ok().put("item", baseService.update(realObject));
    }

    /**
     * 删除
     *
     * @param modelName a
     * @param id        a
     * @return j
     * @throws HttpRequestException if any.
     * @throws SQLException         if any.
     */
    @Override
    public Json delete(String modelName, Integer id) throws HttpRequestException, SQLException {
        var baseService = getBaseService(modelName);
        var deleteByIds = baseService.delete(id);
        return Json.ok().put("delete-result", deleteByIds == 1);
    }

    /**
     * 批量删除
     *
     * @param modelName a {@link java.lang.String} object.
     * @param deleteIds a {@link java.util.Map} object.
     * @return a {@link cool.scx.vo.Json} object.
     * @throws cool.scx.exception.HttpRequestException if any.
     * @throws java.sql.SQLException                   SQLException
     */
    @Override
    public Json batchDelete(String modelName, @FromBody("deleteIds") long[] deleteIds) throws HttpRequestException, SQLException {
        var baseService = getBaseService(modelName);
        var deletedCount = baseService.delete(deleteIds);
        return Json.ok().put("deletedCount", deletedCount);
    }

    /**
     * 撤销删除
     *
     * @param modelName a {@link java.lang.String} object.
     * @param id        a {@link java.lang.Integer} object.
     * @return a {@link cool.scx.vo.Json} object.
     * @throws cool.scx.exception.HttpRequestException if any.
     * @throws java.sql.SQLException                   SQLException
     */
    @Override
    public Json revokeDelete(String modelName, Integer id) throws HttpRequestException, SQLException {
        if (!ScxEasyConfig.tombstone()) {
            return Json.fail("not-used-tombstone");
        } else {
            var baseService = getBaseService(modelName);
            var revokeDeleteCount = baseService.revokeDelete(Long.valueOf(id));
            return revokeDeleteCount == 1 ? Json.ok() : Json.fail();
        }
    }

    /**
     * 获取自动完成字段
     *
     * @param modelName a {@link java.lang.String} object.
     * @param fieldName a {@link java.lang.String} object.
     * @return a {@link cool.scx.vo.Json} object.
     * @throws cool.scx.exception.HttpRequestException if any.
     * @throws java.sql.SQLException                   SQLException
     */
    @Override
    public Json getAutoComplete(String modelName, String fieldName) throws HttpRequestException, SQLException {
        var baseService = getBaseService(modelName);
        var fieldList = baseService.getFieldList(fieldName);
        return Json.ok().put("fields", fieldList);
    }

    /**
     * 校验唯一性
     *
     * @param modelName a {@link java.lang.String} object.
     * @param params    a {@link java.util.Map} object.
     * @return a {@link cool.scx.vo.Json} object.
     * @throws cool.scx.exception.HttpRequestException if any.
     * @throws java.sql.SQLException                   SQLException
     */
    @Override
    public Json checkUnique(String modelName, Map<String, Object> params) throws HttpRequestException, SQLException {
        var baseService = getBaseService(modelName);
        var model = getBaseModel(params, modelName);
        var modelID = model.id;
        model.id = null;
        var queryParam = new Query().addWhereByObject(model);
        if (modelID != null) {
            queryParam.addWhere("id", WhereType.NOT_EQUAL, modelID);
        }
        var b = baseService.count(queryParam) == 0;
        return Json.ok().put("isUnique", b);
    }

}
