package cool.scx.ext.fixtable;

import cool.scx.Scx;
import cool.scx.ScxDao;
import cool.scx.ScxEasyConfig;
import cool.scx.annotation.Column;
import cool.scx.annotation.ScxModel;
import cool.scx.gui.SQLGUIHandler;
import cool.scx.sql.SQLHelper;
import cool.scx.sql.SQLRunner;
import cool.scx.util.Ansi;
import cool.scx.util.CaseUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * <p>FixTableUtils class.</p>
 *
 * @author scx567888
 * @version 1.3.0
 */
public class FixTableUtils {

    private static final Logger logger = LoggerFactory.getLogger(FixTableUtils.class);

    /**
     * 数据库实例
     */
    private static final String databaseName = ScxEasyConfig.dataSourceDatabase();

    /**
     * <p>fixTable.</p>
     */
    public static void fixTableByScxModel() {
        //如果无法链接数据库 就跳过修复表
        if (!ScxDao.checkDataSource()) {
            return;
        }
        logger.debug("检查数据表结构中...");
        //已经显示过修复表的 gui 这里使用 flag 只显示一次
        boolean alreadyShowConfirmFixTable = false;
        //修复成功的表
        var fixSuccess = 0;
        //修复失败的表
        var fixFail = 0;
        //不需要修复的表
        var noNeedToFix = 0;
        for (var scxModule : Scx.getScxModules()) {
            for (var v : scxModule.classList()) {
                //只对 ScxModel 注解标识的了类进行数据表修复
                if (v.isAnnotationPresent(ScxModel.class) && !v.isInterface()) {
                    //判断是否需要修复
                    if (needFixTable(v)) {
                        //如果已经显示过gui选择界面了就不再显示
                        if (!alreadyShowConfirmFixTable) {
                            //获取用户数据 true 为修复 false 为不修复
                            var cancelFix = !SQLGUIHandler.confirmFixTable();
                            //如果取消修复 直接跳出这个方法
                            if (cancelFix) {
                                logger.warn("已取消修复表...");
                                return;
                            }
                            //设置 flag
                            alreadyShowConfirmFixTable = true;
                        }
                        //获取修复表的结果
                        var r = fixTable(v);
                        if (r == FixTableResult.FIX_SUCCESS) {
                            fixSuccess = fixSuccess + 1;
                        } else if (r == FixTableResult.FIX_FAIL) {
                            fixFail = fixFail + 1;
                        } else if (r == FixTableResult.NO_NEED_TO_FIX) {
                            noNeedToFix = noNeedToFix + 1;
                        }
                    }
                }
            }
        }

        if (fixSuccess != 0) {
            logger.debug("修复成功 {} 张表...", fixSuccess);
        }
        if (fixFail != 0) {
            logger.warn("修复失败 {} 张表...", fixFail);
        }
        if (fixSuccess + fixFail == 0) {
            logger.debug("没有表需要修复...");
        }

    }


    /**
     * 根据类修复表
     *
     * @param clazz a {@link java.lang.Class} object.
     * @return 是否修复
     */
    public static FixTableResult fixTable(Class<?> clazz) {
        var table = SQLHelper.getTableInfo(clazz);
        try (var connection = SQLRunner.getConnection()) {
            //获取当前连接对象的 MetaData
            var dbMetaData = connection.getMetaData();
            //根据表名称获取表
            var nowTable = dbMetaData.getTables(databaseName, databaseName, table.tableName, new String[]{"TABLE"});
            //获取到表
            if (nowTable.next()) {
                var nowColumns = dbMetaData.getColumns(databaseName, databaseName, nowTable.getString("TABLE_NAME"), null);
                var stringArrayList = new ArrayList<>();
                while (nowColumns.next()) {
                    stringArrayList.add(nowColumns.getString("COLUMN_NAME"));
                }
                var nonExistentFields = Stream.of(table.allFields).filter(field ->
                        !stringArrayList.contains(CaseUtils.toSnake(field.getName()))
                ).collect(Collectors.toList());

                if (nonExistentFields.size() != 0) {
                    var columns = nonExistentFields.stream().map(field -> CaseUtils.toSnake(field.getName())).collect(Collectors.joining(" , ", " [ ", " ] "));
                    Ansi.out().brightBlue("未找到表 " + table.tableName + " 中的 " + columns + " 字段 --> 正在自动建立 !!!").println();
                    var addSql = nonExistentFields.stream().map(field -> " ADD " + getSQLColumn(field)).collect(Collectors.joining(",", "", ""));
                    var alertSql = "ALTER TABLE `" + table.tableName + "` " + addSql;
                    var otherSQLByField = getOtherSQL(nonExistentFields.toArray(Field[]::new));
                    if (otherSQLByField.size() > 0) {
                        alertSql += otherSQLByField.stream().map(str -> " ADD " + str).collect(Collectors.joining(",", ",", ";"));
                    } else {
                        alertSql += ";";
                    }
                    SQLRunner.execute(alertSql, null);
                    return FixTableResult.FIX_SUCCESS;
                } else {
                    return FixTableResult.NO_NEED_TO_FIX;
                }
            } else {
                logger.warn("未找到表 {} --> 正在自动建立 !!!", table.tableName);
                var createTableSql = "CREATE TABLE `" + table.tableName + "` ( " + Stream.of(table.allFields).map(field -> getSQLColumn(field) + ",").collect(Collectors.joining("", "", "")) + String.join(",", getOtherSQL(table.allFields)) + ") ;";
                SQLRunner.execute(createTableSql, null);
                return FixTableResult.FIX_SUCCESS;
            }
        } catch (Exception e) {
            e.printStackTrace();
            return FixTableResult.FIX_FAIL;
        }
    }

    /**
     * 检查是否需要修复表
     *
     * @param clazz a {@link java.lang.Class} object.
     * @return 是否修复
     */
    public static boolean needFixTable(Class<?> clazz) {
        var table = SQLHelper.getTableInfo(clazz);
        try (var connection = SQLRunner.getConnection()) {
            //获取当前连接对象的 MetaData
            var dbMetaData = connection.getMetaData();
            //根据表名称获取表
            var nowTable = dbMetaData.getTables(databaseName, databaseName, table.tableName, new String[]{"TABLE"});
            //获取到表
            if (nowTable.next()) {
                var nowColumns = dbMetaData.getColumns(databaseName, databaseName, nowTable.getString("TABLE_NAME"), null);
                var stringArrayList = new ArrayList<>();
                while (nowColumns.next()) {
                    stringArrayList.add(nowColumns.getString("COLUMN_NAME"));
                }
                var nonExistentFields = Stream.of(table.allFields).filter(field -> !stringArrayList.contains(CaseUtils.toSnake(field.getName()))).collect(Collectors.toList());

                return nonExistentFields.size() != 0;
            } else {
                return true;
            }
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        }
    }


    /**
     * 根据 field 构建 特殊的 SQLColumn
     * 如 是否为唯一键 是否添加索引 是否为主键等
     *
     * @param allFields allFields
     * @return 生成的语句片段
     */
    private static List<String> getOtherSQL(Field... allFields) {
        var list = new ArrayList<String>();
        for (Field field : allFields) {
            var column = field.getAnnotation(Column.class);
            if (column != null) {
                var columnName = CaseUtils.toSnake(field.getName());
                if (column.primaryKey()) {
                    list.add("PRIMARY KEY (`" + columnName + "`)");
                }
                if (column.unique()) {
                    list.add("UNIQUE KEY `unique_" + columnName + "`(`" + columnName + "`)");
                }
                if (column.needIndex()) {
                    list.add("KEY `index_" + columnName + "`(`" + columnName + "`)");
                }
            }
        }
        return list;
    }

    /**
     * 根据 field 构建 基本的 SQLColumn
     *
     * @param field field
     * @return 生成的语句片段
     */
    private static String getSQLColumn(Field field) {
        var columnName = "`" + CaseUtils.toSnake(field.getName()) + "` ";
        var type = "";
        var notNull = "";
        var autoIncrement = "";
        var defaultValue = "";
        var onUpdate = "";
        var fieldColumn = field.getAnnotation(Column.class);
        if (fieldColumn != null) {
            type = "".equals(fieldColumn.type()) ? SQLHelper.getMySQLTypeCreateName(field.getType()) : fieldColumn.type();
            notNull = fieldColumn.notNull() ? " NOT NULL" : " NULL";
            if (fieldColumn.autoIncrement()) {
                autoIncrement = " AUTO_INCREMENT ";
            }
            if (fieldColumn.primaryKey()) {
                notNull = " NOT NULL ";
            }
            if (!"".equals(fieldColumn.defaultValue())) {
                defaultValue = " DEFAULT " + fieldColumn.defaultValue();
            }
            if (!"".equals(fieldColumn.onUpdateValue())) {
                onUpdate += " ON UPDATE " + fieldColumn.defaultValue();
            }
        } else {
            type = SQLHelper.getMySQLTypeCreateName(field.getType());
            notNull = " NULL ";
        }
        return columnName + type + notNull + autoIncrement + defaultValue + onUpdate;
    }

}
