package cool.scx.ext.crud;

import cool.scx.Scx;
import cool.scx.ScxContext;
import cool.scx.annotation.NoColumn;
import cool.scx.annotation.ScxModel;
import cool.scx.annotation.ScxService;
import cool.scx.base.BaseModel;
import cool.scx.base.BaseService;
import cool.scx.bo.Query;
import cool.scx.enumeration.WhereType;
import cool.scx.exception.BadRequestException;
import cool.scx.exception.CustomHttpRequestException;
import cool.scx.exception.HttpRequestException;
import cool.scx.util.Ansi;
import cool.scx.util.ObjectUtils;
import cool.scx.vo.Json;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;

public final class CRUDHelper {

    /**
     * scx bean 名称 和 class 对应映射
     */
    private static final Map<String, Class<?>> SCX_BEAN_CLASS_NAME_MAPPING;

    static {
        SCX_BEAN_CLASS_NAME_MAPPING = new HashMap<>();
        for (var allModule : Scx.getScxModules()) {
            for (var c : allModule.classList()) {
                if (c.isAnnotationPresent(ScxService.class) || c.isAnnotationPresent(ScxModel.class)) {
                    var className = c.getSimpleName().toLowerCase();
                    var aClass = SCX_BEAN_CLASS_NAME_MAPPING.get(className);
                    if (aClass == null) {
                        SCX_BEAN_CLASS_NAME_MAPPING.put(c.getSimpleName().toLowerCase(), c);
                    } else {
                        SCX_BEAN_CLASS_NAME_MAPPING.put(c.getName(), c);
                        Ansi.out().brightRed("检测到重复名称的 class ").brightYellow("[" + aClass.getName() + "] ").blue("[" + c.getName() + "]").brightRed(" 可能会导致根据名称调用时意义不明确 !!! 建议修改 !!!").println();
                    }
                }
            }
        }
    }


    /**
     * 获取 service
     * todo 错误信息待处理
     *
     * @param modelName model 名称
     * @param <T>       model 类型
     * @return service
     * @throws HttpRequestException service 未找到
     */
    @SuppressWarnings("unchecked")
    public static <T extends BaseModel> BaseService<T> getBaseService(String modelName) throws HttpRequestException {
        try {
            var o = ScxContext.getBean(getClassByName(modelName.toLowerCase() + "service"));
            return (BaseService<T>) o;
        } catch (Exception e) {
            throw new CustomHttpRequestException(ctx -> Json.fail("unknown-crud-service").put("service-name", modelName.toLowerCase()).sendToClient(ctx));
        }
    }

    public static BaseModel getBaseModel(Map<String, Object> entityMap, String modelName) throws HttpRequestException {
        try {
            return (BaseModel) ObjectUtils.convertValue(entityMap, getClassByName(modelName));
        } catch (Exception e) {
            //这里一般就是 参数转换错误
            throw new BadRequestException(e);
        }
    }

    /**
     * <p>getClassByName.</p>
     *
     * @param str a {@link java.lang.String} object.
     * @return a {@link java.lang.Class} object.
     */
    public static Class<?> getClassByName(String str) {
        return SCX_BEAN_CLASS_NAME_MAPPING.get(str.toLowerCase());
    }

    /**
     * 获取 query
     *
     * @param limit         l
     * @param page          p
     * @param orderByColumn or
     * @param sortType      so
     * @param whereBodyList wh
     * @return q
     */
    public static Query getQuery(Class<?> modelClass, Integer limit, Integer page, String orderByColumn, String sortType, List<CrudWhereBody> whereBodyList) throws CustomHttpRequestException {
        var query = new Query();
        if (limit != null && limit >= 0) {
            if (page != null && page >= 0) {
                query.setPagination(page, limit);
            } else {
                query.setPagination(limit);
            }
        }
        if (orderByColumn != null && sortType != null) {
            query.addOrderBy(orderByColumn, sortType);
        }
        if (whereBodyList != null) {
            for (var crudWhereBody : whereBodyList) {
                if (crudWhereBody.fieldName != null && crudWhereBody.whereType != null) {
                    //校验 fieldName 是否正确
                    checkFieldName(modelClass, crudWhereBody.fieldName);
                    //检查 whereType 是否正确
                    var whereType = checkWhereType(crudWhereBody.fieldName, crudWhereBody.whereType);
                    //检查参数数量是否正确
                    checkWhereBodyParametersSize(crudWhereBody.fieldName, whereType, crudWhereBody.value1, crudWhereBody.value2);
                    if (whereType.paramSize() == 0) {
                        query.addWhere(crudWhereBody.fieldName, whereType);
                    } else if (whereType.paramSize() == 1) {
                        query.addWhere(crudWhereBody.fieldName, whereType, crudWhereBody.value1);
                    } else if (whereType.paramSize() == 2) {
                        query.addWhere(crudWhereBody.fieldName, whereType, crudWhereBody.value1, crudWhereBody.value2);
                    }
                }
            }
        }
        return query;
    }

    public static void checkFieldName(Class<?> modelClass, String fieldName) throws CustomHttpRequestException {
        try {
            var field = modelClass.getField(fieldName);
            if (field.isAnnotationPresent(NoColumn.class)) {
                throw new CustomHttpRequestException(ctx -> Json.fail("unknown-field-name").put("field-name", fieldName).sendToClient(ctx));
            }
        } catch (Exception e) {
            throw new CustomHttpRequestException(ctx -> Json.fail("unknown-field-name").put("field-name", fieldName).sendToClient(ctx));
        }
    }

    public static WhereType checkWhereType(String fieldName, String strWhereType) throws CustomHttpRequestException {
        try {
            return WhereType.valueOf(strWhereType.toUpperCase());
        } catch (Exception ignored) {
            throw new CustomHttpRequestException(ctx -> Json.fail("unknown-where-type").put("field-name", fieldName).put("where-type", strWhereType).sendToClient(ctx));
        }
    }

    public static void checkWhereBodyParametersSize(String fieldName, WhereType whereType, Object value1, Object value2) throws CustomHttpRequestException {
        AtomicInteger paramSize = new AtomicInteger();
        if (value1 != null) {
            paramSize.set(paramSize.get() + 1);
        }
        if (value2 != null) {
            paramSize.set(paramSize.get() + 1);
        }

        if (whereType.paramSize() != paramSize.get()) {
            throw new CustomHttpRequestException(ctx -> Json.fail("where-body-parameters-size-error")
                    .put("field-name", fieldName)
                    .put("where-type", whereType)
                    .put("need-parameters-size", whereType.paramSize())
                    .put("got-parameters-size", paramSize.get())
                    .sendToClient(ctx));
        }
    }

}
